%% Script for plotting
%{
Add the folder containing this script file to MATLAB path, and then paste the file path to the
variable 'BasePath'. All required third party functions should also be in this folder.
Place the source data files for this figure into the same folder as this script.
Tested with MATLAB R2015a. Requires the Statistics and Machine Learning Toolbox (and the code in 
this folder).
%}

%% Initial Plot/Figure Settings

set(groot,'DefaultFigureColor',[1 1 1]);
set(groot,'DefaultFigureWindowStyle','docked');
set(groot,'DefaultAxesFontSize',24);
set(groot,'DefaultTextFontSize',24);
set(groot,'DefaultAxesColor','none');
set(groot,'DefaultAxesXColor',[0,0,0]);
set(groot,'DefaultAxesYColor',[0,0,0]);
set(groot,'DefaultAxesZColor',[0,0,0]);
set(groot,'DefaultAxesLineWidth',2);
set(groot,'DefaultAxesTickLength',0.015+[0,0]);
set(groot,'DefaultAxesTickDir','out');
set(groot,'DefaultAxesTickDirMode','manual');
set(groot,'DefaultLineLineWidth',1);

%% Load Input Data

BasePath = '';
if isempty(BasePath)
    BasePath = fileparts(mfilename('fullpath'));
    if isempty(BasePath)
        BasePath = pwd;
    end
end

load(fullfile(BasePath,'IlluminData.mat'),'IlluminData');
load(fullfile(BasePath,'ControlData.mat'),'ControlData');
load(fullfile(BasePath,'RegularData.mat'),'RegularData');

%% Fig. 5ab: 3D trajectory plots

FigureHandle = figure('NumberTitle','off','Name','Fig. 5ab: Illuminated (L) & Control (R) Families');
AxisHandle = axes('parent',FigureHandle,'NextPlot','add');

ColorOrder = [
    9,106,200
    30,165,138
    198,168,92
    ]./255;

subplot(1,2,1,AxisHandle);
AxisHandle = [AxisHandle; copyobj(AxisHandle,FigureHandle)];
subplot(1,2,2,AxisHandle(2));

colormap(AxisHandle(1),ColorOrder([3,1,2],:));
colormap(AxisHandle(2),ColorOrder([2,1,3],:));

for n1 = 1:numel(IlluminData)
    for n2 = 1:numel(IlluminData(n1).TrajectoryData)
        Coord = IlluminData(n1).TrajectoryData{n2};
        surface('XData',Coord(:,[1,1]),'YData',Coord(:,[2,2]),'ZData',Coord(:,[3,3])./3600,'CData',n1+zeros(size(Coord,1),2),...
            'EdgeColor','interp','FaceColor','none','LineWidth',2,'Parent',AxisHandle(1))
    end
end
arrayfun(@(d) cellfun(@(x) ...
    plot3(x(1,1),x(1,2),x(1,3)./3600,'k*',...
    'MarkerSize',20,'LineWidth',2.1,'Parent',AxisHandle(1)),...
    d.TrajectoryData(1)), IlluminData);


for n1 = 1:numel(ControlData)-1
    for n2 = 1:numel(ControlData(n1).TrajectoryData)
        Coord = ControlData(n1).TrajectoryData{n2};
        surface('XData',Coord(:,[1,1]),'YData',Coord(:,[2,2]),'ZData',Coord(:,[3,3])./3600,'CData',n1+zeros(size(Coord,1),2),...
            'EdgeColor','interp','FaceColor','none','LineWidth',2,'Parent',AxisHandle(2))
    end
end
arrayfun(@(d) cellfun(@(x) ...
    plot3(x(1,1),x(1,2),x(1,3)./3600,'k*',...
    'MarkerSize',20,'LineWidth',2.1,'Parent',AxisHandle(2)),...
    d.TrajectoryData(1)), ControlData(1:3));

set(AxisHandle,'CLim',[1,max(numel(IlluminData),numel(ControlData(1:3)))+1]);

hlink = linkprop(AxisHandle,{'XLim','YLim','ZLim','ZDir','DataAspectRatio','DataAspectRatioMode','LineWidth','CameraPosition','CameraTarget','CameraUpVector','CameraViewAngle',...
    'XLimMode','YLimMode','ZLimMode','XColor','YColor','ZColor','Color','CameraPositionMode','CameraTargetMode','CameraUpVectorMode','CameraViewAngleMode'});

set(FigureHandle,'UserData',struct('LinkProp',hlink));

set(AxisHandle,'XColor','k','YColor','k','ZColor','k','Color','none','LineWidth',2);

set(AxisHandle,'XLim',[min(min(cell2mat(get(findobj(AxisHandle,'type','surface'),'XData')))),max(max(cell2mat(get(findobj(AxisHandle,'type','surface'),'XData'))))]);
set(AxisHandle,'YLim',[min(min(cell2mat(get(findobj(AxisHandle,'type','surface'),'YData')))),max(max(cell2mat(get(findobj(AxisHandle,'type','surface'),'YData'))))]);
set(AxisHandle,'ZLim',[min(min(cell2mat(get(findobj(AxisHandle,'type','surface'),'ZData')))),max(max(cell2mat(get(findobj(AxisHandle,'type','surface'),'ZData'))))]);
set(AxisHandle,'ZDir','reverse');
set(AxisHandle,'DataAspectRatio',[1,1,diff(zlim(AxisHandle(1)))/diff(xlim(AxisHandle(1)))]);
set(AxisHandle,'Box','on');

view(AxisHandle(1),185,18);

arrayfun(@(h)xlabel(h,'X (\mum)'),AxisHandle);
arrayfun(@(h)ylabel(h,'Y (\mum)'),AxisHandle);
arrayfun(@(h)zlabel(h,'Time (h)'),AxisHandle);

FigureHandle = [FigureHandle; copyobj(FigureHandle,0)];

hlink = linkprop(findobj(FigureHandle(2),'type','axes'),{'XLim','YLim','ZLim','ZDir','DataAspectRatio','DataAspectRatioMode','LineWidth','CameraPosition','CameraTarget',...
    'CameraUpVector','CameraViewAngle','XLimMode','YLimMode','ZLimMode','XColor','YColor','ZColor','Color','CameraPositionMode','CameraTargetMode','CameraUpVectorMode','CameraViewAngleMode'});
set(FigureHandle(2),'UserData',struct('LinkProp',hlink));
arrayfun(@(h)view(h,180,90),findobj(FigureHandle(2),'type','axes'));

%% Fig. 5cd: 2D family tree plots

FigureHandle = figure('NumberTitle','off','Name','Fig. 5cd: 2D family tree plots');

h = [
    feval(@(n) plotGenealogyTree_BranchList(IlluminData(n).BranchList,'tUnitConversion',1/3600,'DisplayName',sprintf('Illuminated Family %d',n)), 3)
    feval(@(n) plotGenealogyTree_BranchList(ControlData(n).BranchList,'tUnitConversion',1/3600,'DisplayName',sprintf('Control Family %d',n)), 2)
    ];

ColorOrder = [
    30,165,138
    9,106,200
    ]./255;

arrayfun(@(n)set([h(n).VertLineHandle.Node{:}],'Color',ColorOrder(n,:)),1:numel(h));
arrayfun(@(n)set([h(n).HorzLineHandle.Node{:}],'Color','k'),1:numel(h));
arrayfun(@(n)set([h(n).DetachPointsHandle.Node{:}],'MarkerFaceColor',ColorOrder(n,:)),1:numel(h));

AxisHandle = arrayfun(@(x)copyobj(x.PlotHandle,FigureHandle),h,'un',0);
AxisHandle = arrayfun(@(n)subplot(1,numel(AxisHandle),n,AxisHandle{n}),1:numel(AxisHandle),'un',0);
AxisHandle = cat(1,AxisHandle{:});
arrayfun(@(x)close(x.FigureHandle),h);

set(AxisHandle,'PlotBoxAspectRatio',[1,1,1],'TickDir','out','YGrid','off');
arrayfun(@(x)ylabel(x,'Time (h)'),AxisHandle);
set(AxisHandle,'FontSize',30);

%% Fig. 5 - S3: I_cdGMP vs TreeAsym (all)

FigureHandle = figure('NumberTitle','off','Name','Fig. S12: I_cdGMP vs TreeAsym (all)');
AxisHandle = axes('parent',FigureHandle);
set(AxisHandle,'NextPlot','add','Color','none','XColor','k','YColor','k','LineWidth',2,'TickLength',0.025+[0,0],'TickDir','out');

Data = [[RegularData.TreeAsym];[RegularData.I_cdGMP]]';

plot(Data(:,1),Data(:,2),'parent',AxisHandle,'LineWidth',2,'MarkerSize',15,'LineStyle','none','Marker','o','MarkerEdgeColor','k','MarkerFaceColor',0.75+[0,0,0]);

[~,D,X,Y] = kde2d(Data,2^8,[0,0],[1,1]);
dX = X(1,2)-X(1,1);
dY = Y(2,1)-Y(1,1);

[DS,DSidx] = sort(D(:),'descend');
[~,Didx] = sort(DSidx,'ascend');
CumDS = cumsum(DS.*dX.*dY);
CumDS = CumDS ./ max(CumDS);
CumD = reshape(CumDS(Didx),size(D));

[~,ContourHandle] = contour(AxisHandle,X,Y,CumD,'Fill','off','LineStyle','-','LineWidth',2.1,'Color',0.5+[0,0,0],'parent',AxisHandle);
ContourHandle = [ContourHandle; copyobj(ContourHandle,AxisHandle)];
uistack(ContourHandle,'bottom');
set(ContourHandle(1),'LevelList',0.95,'LineStyle','-');
set(ContourHandle(2),'LevelList',0.5,'LineStyle','--');

[rho,pval] = corr(Data(:,1),Data(:,2),'type','spearman');

text(0.95,0.95,{sprintf('\\rho = %0.3g',rho),sprintf('p-val = %0.2g',pval),''},'units','normalized','parent',AxisHandle,...
    'HorizontalAlignment','right','VerticalAlignment','top','clipping','off','Rotation',0);

set(AxisHandle,'XLim',[0,1],'XTick',0:.25:1);
set(AxisHandle,'YLim',[0,1],'YTick',0:.25:1);
set(AxisHandle,'PlotBoxAspectRatio',[1,1,1]);

xlabel(AxisHandle,'Tree Asymmetry \lambda');
ylabel(AxisHandle,'I_{c-di-GMP}');

Idx = [25,12,21];
for n1 = 1:numel(Idx)
    text(Data(Idx(n1),1)+diff(get(AxisHandle,'XLim')).*0.015,Data(Idx(n1),2)+diff(get(AxisHandle,'YLim')).*0.015,sprintf('%g',n1),'units','data','parent',AxisHandle,...
        'HorizontalAlignment','left','VerticalAlignment','bottom','clipping','off','Rotation',0);
end


plot([IlluminData.TreeAsym],[IlluminData.I_cdGMP],'parent',AxisHandle,'LineWidth',2,'MarkerSize',15,'LineStyle','none','Marker','^','MarkerEdgeColor','k','MarkerFaceColor',[217,83,25]./255);
plot([ControlData.TreeAsym],[ControlData.I_cdGMP],'parent',AxisHandle,'LineWidth',2,'MarkerSize',20,'LineStyle','none','Marker','s','MarkerEdgeColor','k','MarkerFaceColor',[77,190,238]./255);




