##======================================================================================================
##
## Project: Million Death Study - Analysis of Child Diarrhea & Pneumonia, 2005-2013
## Author(s): Danny Farrar 
## 
## MDS Pneumonia-Diarrhea Seasonality Models and Figures
## Source Data for eLife submission
## Filename: 'eLife_seasonality_sourcedata.R'
##
## Note: Input data for seasonality models contains monthly counts of pneumonia/diarrhea
##       mortality by total and subset
##
## Table of Contents:
## Line 51: Model 1 - Pneumonia 1m-14yr
## Line 140: Model 2 - Diarrhea & Typhoid 1m-14yr
## Line 491: Model 3 - Pneumonia adult seasonality
## Line 721: Model 4 - Pneumonia child age subsets
## Line 893: Model 5 - Diarrhea child age subsets
## Line 1183: Model 6 - Pneumonia climate region
## Line 1871: Model 7 - Diarrhea climate region
##
## Manuscript Figures:
## Figure 3 - Line 414
## Figure 3, figure supplement 1 (Panel A) - Line 99
## Figure 3, figure supplement 1 (Panel B) - Line 474
## Figure 3, figure supplement 2 - Line 660
## Figure 4 - Line 1064
## Figure 4, figure supplement 1 (Panel A) - Line 774
## Figure 4, figure supplement 1 (Panel B) - Line 947
## Figure 6 - Line 1701
## Figure 6, figure supplement 1 (Panel A) - Line 1239
## Figure 6, figure supplement 1 (Panel B) - Line 1926
## Figure 8 - Line 2050
##
##======================================================================================================

rm(list=ls())

setwd("D:/Pneumonia&Diarrhoea/R code")

library(rms)
library(ggplot2)
library(mgcv)
library(tsModel)
library(splines)
library(foreign)
library(MASS)
library(gridExtra)

##======================================================================================================
## Model 1: Pneumonia mortality; 1m-14yr (excluding cases with reported measles)
##======================================================================================================

#### Prep pneumonia data

  pneu <- read.csv("ts_pneu1mto14yr.csv", header=TRUE)
  
  colnames(pneu) <- c("Year","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")
  rownames(pneu) <- pneu[,1]
  pneu$Year <- NULL
  pneu <- pneu[-1,] ## Exclude 2004 data due to cancelled records
  rn <- as.numeric(rownames(pneu))
  Years <- rn[1]:rn[length(rn)]
  
  pneu <- stack(pneu)[,2:1]
  names(pneu) <- c("Month", "pneu_deaths")
  pneu <- transform(pneu, Year = (Year <- rep(Years, times = 12)),
                    moy = rep(1:12, each = length(Years)),
                    Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                   format = "%Y-%b-%d"))
  pneu <- pneu[with(pneu, order(Date)), ]
  pneu <- transform(pneu, mo_index = 1:108)

#### Plotting crude, unmodelled monthly pneumonia mortality
  
  par(mfrow = c(1,1))
  
  plot(pneu_deaths ~ mo_index, data = pneu, type = "l", axes = FALSE,  ylim = c(0, 140), xlab = "Time", 
       ylab = "Count of deaths", main = "Crude pneumonia mortality by month")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100, 120, 140))
  abline(v=pneu$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120)],col=grey(0.6),lty=2)

#### Generate time value and harmonics (two sine/cosine pairs) for model
  
  pneu$yearC = as.numeric(difftime(pneu$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), units='days'))/365.25
  pneu$cos12 = cos(2*pi*pneu$yearC)
  pneu$sin12 = sin(2*pi*pneu$yearC)
  pneu$cos6 = cos(2*2*pi*pneu$yearC)
  pneu$sin6 = sin(2*2*pi*pneu$yearC)

#### Poisson Model
  
  pneu$logDaysInMonth = log(Hmisc::monthDays(pneu$Date))
  pneu_fit = gam(pneu_deaths ~ sin12+cos12+sin6+cos6 + s(yearC) + offset(logDaysInMonth), family='poisson', data=pneu)
  summary(pneu_fit)

#### Plotting modelled series --> FIGURE 3, FIGURE SUPPLEMENT 1 (PANEL A: PNEUMONIA)
  
  plot(pneu_fit$fitted.values, data = pneu, type='l', lwd=2, xlim = c(), ylim=c(0,125), 
       axes = FALSE, xaxs = "i", yaxs = "i", ylab="Death counts", xlab="Year", cex.lab=1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE, cex.axis=1.25)
  axis(1, at = c(1, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100, 120), las=1, cex.axis=1.25)
  abline(v=pneu$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
  mtext(paste0("A: Pneumonia (1M-14Y)"), side = 3, adj = 0.015, cex=1.5)

#### Calculate average seasonal trend between 2005-2013

  pneu$logDaysInMonth = log(Hmisc::monthDays(pneu$Date))
  
  pneu_seasonal = glm(pneu_deaths ~ sin12+cos12+sin6+cos6 + offset(logDaysInMonth), family='poisson', data=pneu)
  pneu_2009 = pneu[52:64,]
  pneu_pred = predict(pneu_seasonal, pneu_2009, type='link', se.fit=TRUE)
  pneu_pred2 = pneu_pred$fit

#### Rate ratio calculation
  
  avg_pneu = pneu_2009
  terms=c('sin12','cos12','sin6','cos6')
  pred_avg_pneu = avg_pneu[,terms] - avg_pneu[rep(which(avg_pneu$Month=='Apr')[1], nrow(avg_pneu)), terms]
  avg_pneu$seasonal1 = as.matrix(pred_avg_pneu[,terms]) %*% pneu_seasonal$coef[terms]
  dimnames(pneu_fit$Vp) = list(names(pneu_fit$coef), names(pneu_fit$coef))
  avg_pneu$seasonal1.se = sqrt(diag(
    as.matrix(pred_avg_pneu[,terms]) %*% 
      pneu_fit$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneu[,terms]))))
  seMat = cbind(
    est = c(1,0),
    lower = c(1, qnorm(0.01)),
    upper = c(1, qnorm(0.99))
  )
  
  ## Save RR file
  avg_pneu <- exp(as.matrix(avg_pneu[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
  write.csv(avg_pneu, file = "avg_pneu1mto14yr.csv")


##======================================================================================================
## Model 2: Diarrhea & Typhoid mortality; 1m-14yr by symptom profile (excluding cases with reported measles)
##======================================================================================================
  
#### Diarrhea WITHOUT fever & bloody stool
  
  #### Prep data
  
    diar_noFBS <- read.csv("ts_diar1mto14yr_nofeverblood.csv", header=TRUE)
    
    colnames(diar_noFBS) <- c("Year","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")
    rownames(diar_noFBS) <- diar_noFBS[,1]
    diar_noFBS$Year <- NULL
    diar_noFBS <- diar_noFBS[-1,] ## Exclude 2004 data due to cancelled records
    rn <- as.numeric(rownames(diar_noFBS))
    Years <- rn[1]:rn[length(rn)]
    
    diar_noFBS <- stack(diar_noFBS)[,2:1]
    names(diar_noFBS) <- c("Month", "diar_noFBS_deaths")
    diar_noFBS <- transform(diar_noFBS, Year = (Year <- rep(Years, times = 12)),
                             moy = rep(1:12, each = length(Years)),
                             Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                            format = "%Y-%b-%d"))
    diar_noFBS <- diar_noFBS[with(diar_noFBS, order(Date)), ]
    diar_noFBS <- transform(diar_noFBS, mo_index = 1:108)
  
  #### Plotting crude, unmodelled monthly diarrhea mortality
  
    par(mfrow = c(1,1))
    
    plot(diar_noFBS_deaths ~ mo_index, data = diar_noFBS, type = "l", axes = FALSE,  ylim = c(0, 140), xlab = "Time", 
         ylab = "Count of deaths", main = "Crude diarrhea (no FBS) mortality by month")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120, 140))
    abline(v=diar_noFBS$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120)],col=grey(0.6),lty=2)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
    
    diar_noFBS$yearC = as.numeric(difftime(diar_noFBS$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), units='days'))/365.25
    diar_noFBS$cos12 = cos(2*pi*diar_noFBS$yearC)
    diar_noFBS$sin12 = sin(2*pi*diar_noFBS$yearC)
    diar_noFBS$cos6 = cos(2*2*pi*diar_noFBS$yearC)
    diar_noFBS$sin6 = sin(2*2*pi*diar_noFBS$yearC)
    
  #### Poisson Models
  
    diar_noFBS$logDaysInMonth = log(Hmisc::monthDays(diar_noFBS$Date))
    diar_noFBS_fit = gam(diar_noFBS_deaths ~ sin12+cos12+sin6+cos6 + s(yearC) + offset(logDaysInMonth), family='poisson', data=diar_noFBS)
    summary(diar_noFBS_fit)
    
  #### Plotting modelled series
    
    plot(diar_noFBS_fit)
    plot(diar_noFBS$Date, diar_noFBS_fit$fitted.values, type='l')
    
    plot(diar_noFBS_fit$fitted.values, data = diar_noFBS, type='l', lwd=2, xlim = c(), ylim=c(0,125), 
         axes = FALSE, xaxs = "i", yaxs = "i", ylab="Death counts", xlab="Year", cex.lab=1.25)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE, cex.axis=1.25)
    axis(1, at = c(1, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120), las=1, cex.axis=1.25)
    abline(v=diar_noFBS$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    mtext(paste0("B: Diarrhea (no F&BS; 1M-14Y)"), side = 3, adj = 0.015, cex=1.5)
    
  #### Calculate average seasonal trend between 2005-2013
    
    diar_noFBS$logDaysInMonth = log(Hmisc::monthDays(diar_noFBS$Date))
    
    diar_noFBS_seasonal = glm(diar_noFBS_deaths ~ sin12+cos12+sin6+cos6 + offset(logDaysInMonth), family='poisson', data=diar_noFBS)
    diar_noFBS_2009 = diar_noFBS[52:64,]
    diar_noFBS_pred = predict(diar_noFBS_seasonal, diar_noFBS_2009, type='link', se.fit=TRUE)
    diar_noFBS_pred2 = diar_noFBS_pred$fit
    
  #### Rate ratio calculation
    
    avg_diar_noFBS = diar_noFBS_2009
    terms=c('sin12','cos12','sin6','cos6')
    pred_avg_diar_noFBS = avg_diar_noFBS[,terms] - avg_diar_noFBS[rep(which(avg_diar_noFBS$Month=='Apr')[1], nrow(avg_diar_noFBS)), terms]
    avg_diar_noFBS$seasonal1 = as.matrix(pred_avg_diar_noFBS[,terms]) %*% diar_noFBS_seasonal$coef[terms]
    dimnames(diar_noFBS_fit$Vp) = list(names(diar_noFBS_fit$coef), names(diar_noFBS_fit$coef))
    avg_diar_noFBS$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_diar_noFBS[,terms]) %*% 
        diar_noFBS_fit$Vp[terms,terms] %*% t(as.matrix(pred_avg_diar_noFBS[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar_noFBS <- exp(as.matrix(avg_diar_noFBS[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_diar_noFBS, file = "avg_diar1mto14yr_nofeverblood.csv")
    
    
#### Diarrhea WITH fever & bloody stool
    
  #### Prep data
    
    diar_FBS <- read.csv("ts_diar1mto14yr_feverblood.csv", header=TRUE)
    
    colnames(diar_FBS) <- c("Year","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")
    rownames(diar_FBS) <- diar_FBS[,1]
    diar_FBS$Year <- NULL
    diar_FBS <- diar_FBS[-1,] ## Exclude 2004 data due to cancelled records
    rn <- as.numeric(rownames(diar_FBS))
    Years <- rn[1]:rn[length(rn)]
    
    diar_FBS <- stack(diar_FBS)[,2:1]
    names(diar_FBS) <- c("Month", "diar_FBS_deaths")
    diar_FBS <- transform(diar_FBS, Year = (Year <- rep(Years, times = 12)),
                           moy = rep(1:12, each = length(Years)),
                           Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                          format = "%Y-%b-%d"))
    diar_FBS <- diar_FBS[with(diar_FBS, order(Date)), ]
    diar_FBS <- transform(diar_FBS, mo_index = 1:108)
    
  #### Plotting crude, unmodelled monthly diarrhea mortality
    
    par(mfrow = c(1,1))
    
    plot(diar_FBS_deaths ~ mo_index, data = diar_FBS, type = "l", axes = FALSE,  ylim = c(0, 140), xlab = "Time", 
         ylab = "Count of deaths", main = "Crude diarrhea (with FBS) mortality by month")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120, 140))
    abline(v=diar_FBS$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120)],col=grey(0.6),lty=2)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
    
    diar_FBS$yearC = as.numeric(difftime(diar_FBS$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), units='days'))/365.25
    diar_FBS$cos12 = cos(2*pi*diar_FBS$yearC)
    diar_FBS$sin12 = sin(2*pi*diar_FBS$yearC)
    diar_FBS$cos6 = cos(2*2*pi*diar_FBS$yearC)
    diar_FBS$sin6 = sin(2*2*pi*diar_FBS$yearC)
    
  #### Poisson Model
    
    diar_FBS$logDaysInMonth = log(Hmisc::monthDays(diar_FBS$Date))
    diar_FBS_fit = gam(diar_FBS_deaths ~ sin12+cos12+sin6+cos6 + s(yearC) + offset(logDaysInMonth), family='poisson', data=diar_FBS)
    summary(diar_FBS_fit)
    
  #### Plotting modelled series
    
    plot(diar_FBS_fit)
    plot(diar_FBS$Date, diar_FBS_fit$fitted.values, type='l')
    
    plot(diar_FBS_fit$fitted.values, data = diar_FBS, type='l', lwd=2, xlim = c(), ylim=c(0,125), 
         axes = FALSE, xaxs = "i", yaxs = "i", ylab="Death counts", xlab="Year", cex.lab=1.25)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE, cex.axis=1.25)
    axis(1, at = c(1, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120), las=1, cex.axis=1.25)
    abline(v=diar_FBS$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    mtext(paste0("B: Diarrhea (with F&BS; 1M-14Y)"), side = 3, adj = 0.015, cex=1.5)
    
  #### Calculate average seasonal trend between 2005-2013
    
    diar_FBS$logDaysInMonth = log(Hmisc::monthDays(diar_FBS$Date))
    
    diar_FBS_seasonal = glm(diar_FBS_deaths ~ sin12+cos12+sin6+cos6 + offset(logDaysInMonth), family='poisson', data=diar_FBS)
    diar_FBS_2009 = diar_FBS[52:64,]
    diar_FBS_pred = predict(diar_FBS_seasonal, diar_FBS_2009, type='link', se.fit=TRUE)
    diar_FBS_pred2 = diar_FBS_pred$fit
    
  #### Rate ratio calculation

    avg_diar_FBS = diar_FBS_2009
    terms=c('sin12','cos12','sin6','cos6')
    pred_avg_diar_FBS = avg_diar_FBS[,terms] - avg_diar_FBS[rep(which(avg_diar_FBS$Month=='Apr')[1], nrow(avg_diar_FBS)), terms]
    avg_diar_FBS$seasonal1 = as.matrix(pred_avg_diar_FBS[,terms]) %*% diar_FBS_seasonal$coef[terms]
    dimnames(diar_FBS_fit$Vp) = list(names(diar_FBS_fit$coef), names(diar_FBS_fit$coef))
    avg_diar_FBS$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_diar_FBS[,terms]) %*% 
        diar_FBS_fit$Vp[terms,terms] %*% t(as.matrix(pred_avg_diar_FBS[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar_FBS <- exp(as.matrix(avg_diar_FBS[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_diar_FBS, file = "avg_diar1mto14yr_feverblood.csv")
    
    
#### Typhoid and paratyphoid fever
    
  #### Prepare data
    
    typhoid <- read.csv("ts_diar1mto14yr_typhoid.csv", header=TRUE)
    
    colnames(typhoid) <- c("Year","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")
    rownames(typhoid) <- typhoid[,1]
    typhoid$Year <- NULL
    typhoid <- typhoid[-1,] ## Exclude 2004 data due to cancelled records
    rn <- as.numeric(rownames(typhoid))
    Years <- rn[1]:rn[length(rn)]
    
    typhoid <- stack(typhoid)[,2:1]
    names(typhoid) <- c("Month", "typhoid_deaths")
    typhoid <- transform(typhoid, Year = (Year <- rep(Years, times = 12)),
                         moy = rep(1:12, each = length(Years)),
                         Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                        format = "%Y-%b-%d"))
    typhoid <- typhoid[with(typhoid, order(Date)), ]
    typhoid <- transform(typhoid, mo_index = 1:108)
    
  #### Plotting crude, unmodelled monthly typhoid mortality
    
    par(mfrow = c(1,1))
    
    plot(typhoid_deaths ~ mo_index, data = typhoid, type = "l", axes = FALSE,  ylim = c(0, 140), xlab = "Time", 
         ylab = "Count of deaths", main = "Crude typhoid mortality by month")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120, 140))
    abline(v=typhoid$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120)],col=grey(0.6),lty=2)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
    
    typhoid$yearC = as.numeric(difftime(typhoid$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), units='days'))/365.25
    typhoid$cos12 = cos(2*pi*typhoid$yearC)
    typhoid$sin12 = sin(2*pi*typhoid$yearC)
    typhoid$cos6 = cos(2*2*pi*typhoid$yearC)
    typhoid$sin6 = sin(2*2*pi*typhoid$yearC)
    
  #### Poisson Model
    
    typhoid$logDaysInMonth = log(Hmisc::monthDays(typhoid$Date))
    typhoid_fit = gam(typhoid_deaths ~ sin12+cos12+sin6+cos6 + s(yearC) + offset(logDaysInMonth), family='poisson', data=typhoid)
    summary(typhoid_fit)
    
  #### Plotting modelled series
    
    plot(typhoid_fit)
    plot(typhoid$Date, typhoid_fit$fitted.values, type='l')
    
    plot(typhoid_fit$fitted.values, data = typhoid, type='l', lwd=2, xlim = c(), ylim=c(0,125), 
         axes = FALSE, xaxs = "i", yaxs = "i", ylab="Death counts", xlab="Year", cex.lab=1.25)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE, cex.axis=1.25)
    axis(1, at = c(1, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 20, 40, 60, 80, 100, 120), las=1, cex.axis=1.25)
    abline(v=typhoid$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    mtext(paste0("B: Typhoid (1M-14Y)"), side = 3, adj = 0.015, cex=1.5)
    
  #### Calculate average seasonal trend between 2005-2013
    
    typhoid$logDaysInMonth = log(Hmisc::monthDays(typhoid$Date))
    
    typhoid_seasonal = glm(typhoid_deaths ~ sin12+cos12+sin6+cos6 + offset(logDaysInMonth), family='poisson', data=typhoid)
    typhoid_2009 = diar_FBS[52:64,]
    typhoid_pred = predict(typhoid_seasonal, diar_FBS_2009, type='link', se.fit=TRUE)
    typhoid_pred2 = typhoid_pred$fit
    
  #### Rate ratio calculation

    avg_typhoid = typhoid_2009
    terms=c('sin12','cos12','sin6','cos6')
    pred_avg_typhoid = avg_typhoid[,terms] - avg_typhoid[rep(which(avg_typhoid$Month=='Apr')[1], nrow(avg_typhoid)), terms]
    avg_typhoid$seasonal1 = as.matrix(pred_avg_typhoid[,terms]) %*% typhoid_seasonal$coef[terms]
    dimnames(typhoid_fit$Vp) = list(names(typhoid_fit$coef), names(typhoid_fit$coef))
    avg_typhoid$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_typhoid[,terms]) %*% 
        typhoid_fit$Vp[terms,terms] %*% t(as.matrix(pred_avg_typhoid[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_typhoid <- exp(as.matrix(avg_typhoid[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_typhoid, file = "avg_typhoid1mto14yr.csv")
    

##======================================================================================================
## FIGURE 3: Pneumonia&Diarrhea 1m-14yr average annual seasonal pattern
##======================================================================================================
    
  months <- c(month.abb[4], month.abb[7], month.abb[1], month.abb[4])
  
  figure3a_data <- data.frame(x = 1:nrow(pneu_2009), y = exp(pneu_pred2))
  figure3a <- qplot() + geom_line(data = figure3a_data, aes(x=x, y=y), size = 1) +
    ggtitle("A: Pneumonia (1M-14Y)") +
    expand_limits(x = c(1,13), y = c(0,94)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.01,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 20, 40, 60, 80), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=65.40844), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=88.94629), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=65.40844), size=2) + 
    geom_point(aes(x=10, y=88.94629), size=2) + 
    geom_segment(aes(x = 4, y = 70.40844, xend = 4, yend = 66.40844), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=72.40844, label= "RR = 1.22", size = 5) + 
    geom_segment(aes(x = 10, y = 91.94629, xend = 10, yend = 89.75), arrow = arrow(length = unit(0.0125, "npc"), type = "closed")) + 
    annotate("text", x=10, y=93, label= "RR = 1.66", size = 5)
  
  diar1 <- data.frame(x = 1:nrow(diar_FBS_2009), y = exp(diar_FBS_pred2))
  diar1$group <- "FandBS"
  diar2 <- data.frame(x = 1:nrow(diar_noFBS_2009), y = exp(diar_noFBS_pred2))
  diar2$group <- "NoFandBS"
  diar3 <- data.frame(x = 1:nrow(typhoid_2009), y = exp(typhoid_pred2))
  diar3$group <- "Typhoid"
  alldiar <- rbind(diar2, diar1, diar3)
  alldiar$group <- as.character(alldiar$group)
  alldiar$group <- factor(alldiar$group, levels=unique(alldiar$group))
  
  figure3b_data <- alldiar
  figure3b <- qplot() + geom_line(data = figure3b_data, aes(x=x, y=y, group=group, color=group, linetype=group), size = 1) + 
    scale_color_manual(labels = c("No fever & bloody stool", "With fever & bloody stool", "Typhoid"), 
                       values=c("blue", "red", "purple")) + 
    scale_linetype_manual(labels = c("No fever & bloody stool", "With fever & bloody stool", "Typhoid"), 
                          values=c("solid", "solid", "twodash")) +
    ggtitle("B: Diarrhea (1M-14Y)") + 
    expand_limits(x = c(1,13), y = c(0,80)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, 
                       expand=c(0.01,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 20, 40, 60, 80), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20), 
          legend.position = c(0.8, 0.925), legend.title = element_blank(), legend.key=element_blank(),
          legend.text=element_text(size=14)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=64.442212), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=53.429006), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=64.442212), size=2) + 
    geom_point(aes(x=10, y=53.429006), size=2) + 
    geom_segment(aes(x = 4, y = 69.442212, xend = 4, yend = 65.442212), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=71.442212, label= "RR = 1.66", size = 5) + 
    geom_segment(aes(x = 10, y = 58.429006, xend = 10, yend = 54.429006), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=60.429006, label= "RR = 1.37", size = 5)
  
  grid.arrange(figure3a, figure3b)

  
##======================================================================================================
## FIGURE 3, figure supplement 1 (panel B - DIARRHEA): Pneumonia&Diarrhea modelled time series
##======================================================================================================

  plot(diar_FBS_fit$fitted.values, type='l', lwd=2, col="red", 
       xlim = c(), ylim=c(0,125), axes = FALSE, xaxs = "i", yaxs = "i", 
       ylab="Death counts", xlab="Year", cex.lab=1.25)
  lines(diar_noFBS_fit$fitted.values, lwd=2, col="blue")
  lines(typhoid_fit$fitted.values, lwd=2, lty = 2, col="purple")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis=1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100, 120), las=1, cex.axis=1.25)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  legend("topright", inset = c(0.005, 0.025), legend=c("No fever & bloody stool", "With fever & bloody stool", "Typhoid"), 
         col=c("blue", "red", "purple"), lty=1, lwd=2, box.col="white", cex=1.25)
  mtext(paste0("B: Diarrhea (1M-14Y)"), side = 3, adj = 0.015, cex=1.5)

##======================================================================================================
## Model 3: Pneumonia adult mortality; 15-69 Years
##======================================================================================================
  
#### Prep data
  
  pneu_adult_age <- read.csv("ts_pneu15to69yr.csv", header=TRUE)
  
  pneu_adult_age <- pneu_adult_age[!pneu_adult_age$Year == 2004,] ## Exclude 2004 data due to cancelled records
  pneu_adult_age$mo_index <- pneu_adult_age$mo_index - 12
  
#### Crude plot of pneumonia deaths
  
  age <- c(1,2,3)
  r <- as.list(rep("", 3)) 
  for(i in age) { 
    r[[i]] <- subset(pneu_adult_age, age_subset == i)
    r[[i]]$age_subset <- NULL
  }
  
  par(mfrow = c(1,1))
  plot(pneu_deaths ~ mo_index, data = r[[1]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 100), 
       xlab = "Time", ylab = "Count of deaths", main = "Crude pneumonia mortality by month", 
       col="gold")
  lines(pneu_deaths ~ mo_index, data = r[[2]], lwd=2, col="darkorchid3")
  lines(pneu_deaths ~ mo_index, data = r[[3]], lwd=2, col="palevioletred1")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100))
  abline(v=r[[1]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
  legend("topright", inset = 0.025, legend=c("15 - 29 Years", "30 - 59 Years", "60 - 69 Years"), 
         col=c("gold", "darkorchid3", "palevioletred1"), lty=1, lwd=2, cex=0.8)
  
#### Generate time value and harmonics (two sine/cosine pairs) for model
  
  pneu_adult_age <- transform(pneu_adult_age, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                             format = "%Y-%b-%d"))
  
  pneu_adult_age$yearC = as.numeric(difftime(pneu_adult_age$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                             units='days'))/365.25
  
  pneu_adult_age$cos12 = cos(2*pi*pneu_adult_age$yearC)
  pneu_adult_age$sin12 = sin(2*pi*pneu_adult_age$yearC)
  pneu_adult_age$cos6 = cos(2*2*pi*pneu_adult_age$yearC)
  pneu_adult_age$sin6 = sin(2*2*pi*pneu_adult_age$yearC)
  
#### Poisson Model
  
  pneu_adult_age$age_subset <- factor(pneu_adult_age$age_subset)
  
  pneu_adult_age$logDaysInMonth = log(Hmisc::monthDays(pneu_adult_age$Date))
  pneu_adult_agebase = gam(pneu_deaths ~ age_subset*(sin12+cos12+sin6+cos6) + s(yearC, by=age_subset, id=1)
                           + offset(logDaysInMonth), family='poisson', data=pneu_adult_age)
  
#### Plotting modelled series
  
  plot(pneu_adult_agebase$fitted.values[which(pneu_adult_age$age_subset==1)], type='l', lwd=2, col="gold", 
       xlim = c(), ylim=c(0,80), axes = FALSE, xaxs = "i", yaxs = "i", 
       ylab="Death counts", xlab="Year", cex.lab=1.25)
  lines(pneu_adult_agebase$fitted.values[which(pneu_adult_age$age_subset==2)], lwd=2, col="darkorchid3")
  lines(pneu_adult_agebase$fitted.values[which(pneu_adult_age$age_subset==3)], type='l', lwd=2, col="palevioletred1")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis=1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80), las=1, cex.axis=1.25)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  legend("topright", inset = c(0.005, 0.025), legend=c("15 - 29 Years", "30 - 59 Years", "60 - 69 Years"), 
         col=c("dodgerblue2", "firebrick2", "lawngreen"), lty=1, lwd=2, box.col="white", cex=1.25)
  mtext(paste0("A: Pneumonia"), side = 3, adj = 0.015, cex=1.5)
  
#### Calculate average seasonal trend between 2005-2013
  
  pneu_adult_age$logDaysInMonth = log(Hmisc::monthDays(pneu_adult_age$Date))
  
  pneuadultage_seasonal = glm(pneu_deaths ~ age_subset*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=pneu_adult_age)
  
  pneuadultage_2009_1 = pneu_adult_age[pneu_adult_age$age_subset==1,][52:64,]
  pneuadultage_pred_1 = predict(pneuadultage_seasonal, pneuadultage_2009_1, type='link', se.fit=TRUE)
  pneuadultage_pred2_1 = pneuadultage_pred_1$fit
  
  pneuadultage_2009_2 = pneu_adult_age[pneu_adult_age$age_subset==2,][52:64,]
  pneuadultage_pred_2 = predict(pneuadultage_seasonal, pneuadultage_2009_2, type='link', se.fit=TRUE)
  pneuadultage_pred2_2 = pneuadultage_pred_2$fit
  
  pneuadultage_2009_3 = pneu_adult_age[pneu_adult_age$age_subset==3,][52:64,]
  pneuadultage_pred_3 = predict(pneuadultage_seasonal, pneuadultage_2009_3, type='link', se.fit=TRUE)
  pneuadultage_pred2_3 = pneuadultage_pred_3$fit

#### Rate ratio calculations 
  
  ## Pneumonia; 15-29 Years

    avg_pneuadultage1 = pneuadultage_2009_1
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("age_subset2:",terms,sep=''))
    terms3 = c(terms, paste("age_subset3:",terms,sep=''))
    pred_avg_pneuadultage1 = avg_pneuadultage1[,terms] - avg_pneuadultage1[rep(which(avg_pneuadultage1$Month=='Apr')[1], 
                                                                                  nrow(avg_pneuadultage1)), terms]
    avg_pneuadultage1$seasonal1 = as.matrix(pred_avg_pneuadultage1[,terms]) %*% pneuadultage_seasonal$coef[terms]
    avg_pneuadultage1$seasonal2 = as.matrix(pred_avg_pneuadultage1[,c(terms, terms)]) %*% 
      pneuadultage_seasonal$coef[terms2]
    avg_pneuadultage1$seasonal3 = as.matrix(pred_avg_pneuadultage1[,c(terms,terms)]) %*% 
      pneuadultage_seasonal$coef[terms3]
    dimnames(pneu_adult_agebase$Vp) = list(names(pneu_adult_agebase$coef), names(pneu_adult_agebase$coef))
    avg_pneuadultage1$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_pneuadultage1[,terms]) %*% 
        pneu_adult_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuadultage1[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_pneu15to29yr <- exp(as.matrix(avg_pneuadultage1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_pneu15to29yr, file = "avg_pneu15to29yr.csv")
  
  ## Pneumonia; 30-59 Years

    avg_pneuadultage2 = pneuadultage_2009_2
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("age_subset2:",terms,sep=''))
    terms3 = c(terms, paste("age_subset3:",terms,sep=''))
    pred_avg_pneuadultage2 = avg_pneuadultage2[,terms] - avg_pneuadultage2[rep(which(avg_pneuadultage2$Month=='Apr')[1], 
                                                                                  nrow(avg_pneuadultage2)), terms]
    avg_pneuadultage2$seasonal1 = as.matrix(pred_avg_pneuadultage2[,terms]) %*% pneuadultage_seasonal$coef[terms]
    avg_pneuadultage2$seasonal2 = as.matrix(pred_avg_pneuadultage2[,c(terms, terms)]) %*% 
      pneuadultage_seasonal$coef[terms2]
    avg_pneuadultage2$seasonal3 = as.matrix(pred_avg_pneuadultage2[,c(terms,terms)]) %*% 
      pneuadultage_seasonal$coef[terms3]
    dimnames(pneu_adult_agebase$Vp) = list(names(pneu_adult_agebase$coef), names(pneu_adult_agebase$coef))
    avg_pneuadultage2$seasonal2.se = sqrt(diag(
      as.matrix(pred_avg_pneuadultage2[,terms]) %*% 
        pneu_adult_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuadultage2[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR files
    avg_pneu30to59yr <- exp(as.matrix(avg_pneuadultage2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
    write.csv(avg_pneu30to59yr, file = "avg_pneu30to59yr.csv")
  
  ## Pneumonia; 60-69 Years

    avg_pneuadultage3 = pneuadultage_2009_3
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("age_subset2:",terms,sep=''))
    terms3 = c(terms, paste("age_subset3:",terms,sep=''))
    pred_avg_pneuadultage3 = avg_pneuadultage3[,terms] - avg_pneuadultage3[rep(which(avg_pneuadultage3$Month=='Apr')[1], 
                                                                                  nrow(avg_pneuadultage3)), terms]
    avg_pneuadultage3$seasonal1 = as.matrix(pred_avg_pneuadultage3[,terms]) %*% pneuadultage_seasonal$coef[terms]
    avg_pneuadultage3$seasonal2 = as.matrix(pred_avg_pneuadultage3[,c(terms, terms)]) %*% 
      pneuadultage_seasonal$coef[terms2]
    avg_pneuadultage3$seasonal3 = as.matrix(pred_avg_pneuadultage3[,c(terms,terms)]) %*% 
      pneuadultage_seasonal$coef[terms3]
    dimnames(pneu_adult_agebase$Vp) = list(names(pneu_adult_agebase$coef), names(pneu_adult_agebase$coef))
    avg_pneuadultage3$seasonal3.se = sqrt(diag(
      as.matrix(pred_avg_pneuadultage3[,terms]) %*% 
        pneu_adult_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuadultage3[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_pneu60to69yr <- exp(as.matrix(avg_pneuadultage3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
    write.csv(avg_pneu60to69yr, file = "avg_pneu60to69yr.csv")

#### Plot average annual seasonal pattern - Figure 3, figure supplement 2
    
  fig3sup2_pneu15to29y_data <- data.frame(x = 1:nrow(pneuadultage_2009_1), y = exp(pneuadultage_pred2_1))
  fig3sup2_pneu15to29y <- qplot(x, y, data = fig3sup2_pneu15to29y_data, geom = "line") +
    ggtitle("A: 15-29 years") + 
    geom_line(size = 1, colour = "gold") + 
    expand_limits(x = c(1,13), y = c(0,55)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0,10,20,30,40,50), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=14.75095), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=20.25514), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=14.75095), size=2) + 
    geom_point(aes(x=9, y=21.15232), size=2) + 
    geom_segment(aes(x = 4, y = 19.75095, xend = 4, yend = 15.75095), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=21.75095, label= "RR = 1.22", size = 5) + 
    geom_segment(aes(x = 9, y = 26.15232, xend = 9, yend = 22.15232), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=9, y=28.15232, label= "RR = 1.76", size = 5)
  
  fig3sup2_pneu30to59y_data <- data.frame(x = 1:nrow(pneuadultage_2009_2), y = exp(pneuadultage_pred2_2))
  fig3sup2_pneu30to59y <- qplot(x, y, data = fig3sup2_pneu30to59y_data, geom = "line") +
    ggtitle("B: 30-59 years") + 
    geom_line(size = 1, colour = "darkorchid3") + 
    expand_limits(x = c(1,13), y = c(0,55)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "", breaks = c(0,10,20,30,40,50), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=39.87815), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=49.90721), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=39.87815), size=2) + 
    geom_point(aes(x=10, y=49.90721), size=2) + 
    geom_segment(aes(x = 4, y = 44.87815, xend = 4, yend = 40.87815), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=46.87815, label= "RR = 1.28", size = 5) + 
    geom_segment(aes(x = 10, y = 52.90721, xend = 10, yend = 50.70721), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=53.90721, label= "RR = 1.61", size = 5)
  
  fig3sup2_pneu60to69_data <- data.frame(x = 1:nrow(pneuadultage_2009_3), y = exp(pneuadultage_pred2_3))
  fig3sup2_pneu60to69 <- qplot(x, y, data = fig3sup2_pneu60to69_data, geom = "line") +
    ggtitle("C: 60-69 years") + 
    geom_line(size = 1, colour = "palevioletred1") + 
    expand_limits(x = c(1,13), y = c(0,55)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "", breaks = c(0,10,20,30,40,50), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=6.259349), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=8.674082), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=10, y=8.674082), size=2) + 
    geom_segment(aes(x = 10, y = 13.674082, xend = 10, yend = 9.674082), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=15.674082, label= "RR = 1.65", size = 5)
  
  grid.arrange(fig3sup2_pneu15to29y, fig3sup2_pneu30to59y, fig3sup2_pneu60to69, ncol=3)


##======================================================================================================
## Model 4: Pneumonia mortality by child age subset (excluding cases with reported measles)
##======================================================================================================

#### Prep data
  
  pneu_age <- read.csv("ts_pneu1mto14yr_agegroups.csv", header=TRUE)
  
  pneu_age <- pneu_age[!pneu_age$Year == 2004,] ## Exclude 2004 data due to cancelled records
  pneu_age$mo_index <- pneu_age$mo_index - 12
  
#### Plotting crude, unmodelled monthly pneumonia mortality
  
  age <- c(1,2,3)
  r <- as.list(rep("", 3)) 
  for(i in age) { 
    r[[i]] <- subset(pneu_age, flag1to12m == i)
    r[[i]]$flag1to12m <- NULL
  }
  
  par(mfrow = c(1,1))
  plot(pneu_deaths ~ mo_index, data = r[[1]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 100), 
       xlab = "Time", ylab = "Count of deaths", main = "Crude pneumonia mortality by month", 
       col="blue")
  lines(pneu_deaths ~ mo_index, data = r[[2]], lwd=2, col="red")
  lines(pneu_deaths ~ mo_index, data = r[[3]], lwd=2, col="green")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100))
  abline(v=r[[1]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
  legend("topright", inset = 0.025, legend=c("1 - 12 Months", "12 - 59 Months", "5 - 14 Years"), 
         col=c("blue", "red", "green"), lty=1, lwd=2, cex=0.8)

#### Generate time value and harmonics (two sine/cosine pairs) for model
  
  pneu_age <- transform(pneu_age, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                 format = "%Y-%b-%d"))
  
  pneu_age$yearC = as.numeric(difftime(pneu_age$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                       units='days'))/365.25
  
  pneu_age$cos12 = cos(2*pi*pneu_age$yearC)
  pneu_age$sin12 = sin(2*pi*pneu_age$yearC)
  pneu_age$cos6 = cos(2*2*pi*pneu_age$yearC)
  pneu_age$sin6 = sin(2*2*pi*pneu_age$yearC)
  
#### Poisson Model
  
  pneu_age$flag1to12m <- factor(pneu_age$flag1to12m)
  
  pneu_age$logDaysInMonth = log(Hmisc::monthDays(pneu_age$Date))
  pneu_agebase = gam(pneu_deaths ~ flag1to12m*(sin12+cos12+sin6+cos6) + s(yearC, by=flag1to12m, id=1)
                     + offset(logDaysInMonth), family='poisson', data=pneu_age)

#### Plotting modelled series --> FIGURE 4, FIGURE SUPPLEMENT 1 (PANEL A: PNEUMONIA)
  
  plot(pneu_agebase$fitted.values[which(pneu_age$flag1to12m==2)], type='l', lwd=2, col="firebrick2", 
       xlim = c(), ylim=c(0,75), axes = FALSE, xaxs = "i", yaxs = "i", 
       ylab="Death counts", xlab="Year", cex.lab=1.25)
  lines(pneu_agebase$fitted.values[which(pneu_age$flag1to12m==3)], type='l', lwd=2, col="lawngreen")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis=1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60), las=1, cex.axis=1.25)
  axis(2, at = c(0, 20, 40, 60, 75), labels = FALSE, lwd.tick=0)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  legend("topright", inset = c(-0.15, -0.01), legend=c("1 - 11 Months", "1 - 4 Years", "5 - 14 Years"), 
         col=c("dodgerblue2", "firebrick2", "lawngreen"), lty=1, lwd=2, box.col="white", cex=0.8)
  lines(pneu_agebase$fitted.values[which(pneu_age$flag1to12m==1)], lwd=2, col="dodgerblue2")
  mtext(paste0("A: Pneumonia"), side = 3, adj = 0.015, cex=1.5)
  
#### Calculate average seasonal trend between 2005-2013
  
  pneu_age$logDaysInMonth = log(Hmisc::monthDays(pneu_age$Date))
  
  pneuage_seasonal = glm(pneu_deaths ~ flag1to12m*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=pneu_age)
  
  pneuage_2009_1 = pneu_age[pneu_age$flag1to12m==1,][52:64,]
  pneuage_pred_1 = predict(pneuage_seasonal, pneuage_2009_1, type='link', se.fit=TRUE)
  pneuage_pred2_1 = pneuage_pred_1$fit
  
  pneuage_2009_2 = pneu_age[pneu_age$flag1to12m==2,][52:64,]
  pneuage_pred_2 = predict(pneuage_seasonal, pneuage_2009_2, type='link', se.fit=TRUE)
  pneuage_pred2_2 = pneuage_pred_2$fit
  
  pneuage_2009_3 = pneu_age[pneu_age$flag1to12m==3,][52:64,]
  pneuage_pred_3 = predict(pneuage_seasonal, pneuage_2009_3, type='link', se.fit=TRUE)
  pneuage_pred2_3 = pneuage_pred_3$fit

#### Rate ratio calculations
  
  ## Pneumonia; 1-11 Months

    avg_pneuage1 = pneuage_2009_1
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_pneuage1 = avg_pneuage1[,terms] - avg_pneuage1[rep(which(avg_pneuage1$Month=='Apr')[1], 
                                                                   nrow(avg_pneuage1)), terms]
    avg_pneuage1$seasonal1 = as.matrix(pred_avg_pneuage1[,terms]) %*% pneuage_seasonal$coef[terms]
    avg_pneuage1$seasonal2 = as.matrix(pred_avg_pneuage1[,c(terms, terms)]) %*% 
      pneuage_seasonal$coef[terms2]
    avg_pneuage1$seasonal3 = as.matrix(pred_avg_pneuage1[,c(terms,terms)]) %*% 
      pneuage_seasonal$coef[terms3]
    dimnames(pneu_agebase$Vp) = list(names(pneu_agebase$coef), names(pneu_agebase$coef))
    avg_pneuage1$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_pneuage1[,terms]) %*% 
        pneu_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuage1[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_pneu1to11m <- exp(as.matrix(avg_pneuage1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_pneu1to11m, file = "avg_pneu1to11m.csv")
  
  ## Pneumonia; 1-4 Years

    avg_pneuage2 = pneuage_2009_2
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_pneuage2 = avg_pneuage2[,terms] - avg_pneuage2[rep(which(avg_pneuage2$Month=='Apr')[1], 
                                                                   nrow(avg_pneuage2)), terms]
    avg_pneuage2$seasonal1 = as.matrix(pred_avg_pneuage2[,terms]) %*% pneuage_seasonal$coef[terms]
    avg_pneuage2$seasonal2 = as.matrix(pred_avg_pneuage2[,c(terms, terms)]) %*% 
      pneuage_seasonal$coef[terms2]
    avg_pneuage2$seasonal3 = as.matrix(pred_avg_pneuage2[,c(terms,terms)]) %*% 
      pneuage_seasonal$coef[terms3]
    dimnames(pneu_agebase$Vp) = list(names(pneu_agebase$coef), names(pneu_agebase$coef))
    avg_pneuage2$seasonal2.se = sqrt(diag(
      as.matrix(pred_avg_pneuage2[,terms]) %*% 
        pneu_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuage2[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_pneu1to4y <- exp(as.matrix(avg_pneuage2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
    write.csv(avg_pneu1to4y, file = "avg_pneu1to4y.csv")
  
  ## Pneumonia; 5-14 Years

    avg_pneuage3 = pneuage_2009_3
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_pneuage3 = avg_pneuage3[,terms] - avg_pneuage3[rep(which(avg_pneuage3$Month=='Apr')[1], 
                                                                   nrow(avg_pneuage3)), terms]
    avg_pneuage3$seasonal1 = as.matrix(pred_avg_pneuage3[,terms]) %*% pneuage_seasonal$coef[terms]
    avg_pneuage3$seasonal2 = as.matrix(pred_avg_pneuage3[,c(terms, terms)]) %*% 
      pneuage_seasonal$coef[terms2]
    avg_pneuage3$seasonal3 = as.matrix(pred_avg_pneuage3[,c(terms,terms)]) %*% 
      pneuage_seasonal$coef[terms3]
    dimnames(pneu_agebase$Vp) = list(names(pneu_agebase$coef), names(pneu_agebase$coef))
    avg_pneuage3$seasonal3.se = sqrt(diag(
      as.matrix(pred_avg_pneuage3[,terms]) %*% 
        pneu_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuage3[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_pneu5to14y <- exp(as.matrix(avg_pneuage3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
    write.csv(avg_pneu5to14y, file = "avg_pneu5to14y.csv")

    
##======================================================================================================
## Model 5: Diarrhea mortality by child age subset (excluding typhoid & cases with reported measles)
##======================================================================================================

#### Prep data
    
  diar_age <- read.csv("ts_diar1mto14yr_agegroups.csv", header=TRUE)
  
  diar_age <- diar_age[!diar_age$Year == 2004,] ## Exclude 2004 data due to cancelled records
  diar_age$mo_index <- diar_age$mo_index - 12
  
#### Plotting crude, unmodelled monthly diarrhea mortality
  
  age <- c(1,2,3)
  r <- as.list(rep("", 3)) 
  for(i in age) { 
    r[[i]] <- subset(diar_age, flag1to12m == i)
    r[[i]]$flag1to12m <- NULL
  }
  
  par(mfrow = c(1,1))
  
  plot(diar_deaths ~ mo_index, data = r[[1]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 100), 
       xlab = "Time", ylab = "Count of deaths", main = "Crude diarrhea mortality by month", 
       col="blue")
  lines(diar_deaths ~ mo_index, data = r[[2]], lwd=2, col="red")
  lines(diar_deaths ~ mo_index, data = r[[3]], lwd=2, col="green")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA,2005:2013), tick=FALSE)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60, 80, 100))
  abline(v=r[[1]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
  legend("topright", inset = 0.025, legend=c("1 - 12 Months", "12 - 59 Months", "5 - 14 Years"), 
         col=c("blue", "red", "green"), lty=1, lwd=2, cex=0.8)

#### Generate time value and harmonics (two sine/cosine pairs) for model
  
  diar_age <- transform(diar_age, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                 format = "%Y-%b-%d"))
  
  diar_age$yearC = as.numeric(difftime(diar_age$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                       units='days'))/365.25
  
  diar_age$cos12 = cos(2*pi*diar_age$yearC)
  diar_age$sin12 = sin(2*pi*diar_age$yearC)
  diar_age$cos6 = cos(2*2*pi*diar_age$yearC)
  diar_age$sin6 = sin(2*2*pi*diar_age$yearC)
  
#### Poisson Model
  
  diar_age$flag1to12m <- factor(diar_age$flag1to12m)
  
  diar_age$logDaysInMonth = log(Hmisc::monthDays(diar_age$Date))
  diar_agebase = gam(diar_deaths ~ flag1to12m*(sin12+cos12+sin6+cos6) + s(yearC, by=flag1to12m, id=1)
                     + offset(logDaysInMonth), family='poisson', data=diar_age)
  
#### Plotting modelled series --> FIGURE 4, FIGURE SUPPLEMENT 1 (PANEL A: DIARRHEA)

  plot(diar_agebase$fitted.values[which(diar_age$flag1to12m==1)], type='l', lwd=2, col="dodgerblue2", 
       xlim = c(), ylim=c(0,60), axes = FALSE, xaxs = "i", yaxs = "i", 
       ylab="Death counts", xlab="Year", cex.lab=1.25)
  lines(diar_agebase$fitted.values[which(diar_age$flag1to12m==2)], lwd=2, col="firebrick2")
  lines(diar_agebase$fitted.values[which(diar_age$flag1to12m==3)], type='l', lwd=2, col="lawngreen")
  lines(diar$mo_index, diar_fit$fitted.values, type='l', lwd=2, col="black")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis=1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60), las=1, cex.axis=1.25)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  mtext(paste0("B: Diarrhoea"), side = 3, adj = 0.015, cex=1.5)
  
#### Calculate average seasonal trend between 2005-2013
  
  diar_age$logDaysInMonth = log(Hmisc::monthDays(diar_age$Date))
  
  diarage_seasonal = glm(diar_deaths ~ flag1to12m*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), 
                         family='poisson', data=diar_age)
  
  diarage_2009_1 = diar_age[diar_age$flag1to12m==1,][52:64,]
  diarage_pred_1 = predict(diarage_seasonal,diarage_2009_1, type='link', se.fit=TRUE)
  diarage_pred2_1 = diarage_pred_1$fit
  
  diarage_2009_2 = diar_age[diar_age$flag1to12m==2,][52:64,]
  diarage_pred_2 = predict(diarage_seasonal, diarage_2009_2, type='link', se.fit=TRUE)
  diarage_pred2_2 = diarage_pred_2$fit
  
  diarage_2009_3 = diar_age[diar_age$flag1to12m==3,][52:64,]
  diarage_pred_3 = predict(diarage_seasonal, diarage_2009_3, type='link', se.fit=TRUE)
  diarage_pred2_3 = diarage_pred_3$fit
  
#### Rate ratio calculations
  
  ## Diarrhea; 1-11 Months

    avg_diarage1 = diarage_2009_1
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_diarage1 = avg_diarage1[,terms] - avg_diarage1[rep(which(avg_diarage1$Month=='Apr')[1], 
                                                                   nrow(avg_diarage1)), terms]
    avg_diarage1$seasonal1 = as.matrix(pred_avg_diarage1[,terms]) %*% diarage_seasonal$coef[terms]
    avg_diarage1$seasonal2 = as.matrix(pred_avg_diarage1[,c(terms, terms)]) %*% 
      diarage_seasonal$coef[terms2]
    avg_diarage1$seasonal3 = as.matrix(pred_avg_diarage1[,c(terms,terms)]) %*% 
      diarage_seasonal$coef[terms3]
    dimnames(diar_agebase$Vp) = list(names(diar_agebase$coef), names(diar_agebase$coef))
    avg_diarage1$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_diarage1[,terms]) %*% 
        diar_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarage1[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar1to11m <- exp(as.matrix(avg_diarage1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_diar1to11m, file = "avg_diar1to11m.csv")
  
  ## Diarrhea; 1-4 Years

    avg_diarage2 = diarage_2009_2
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_diarage2 = avg_diarage2[,terms] - avg_diarage2[rep(which(avg_diarage2$Month=='Apr')[1], 
                                                                   nrow(avg_diarage2)), terms]
    avg_diarage2$seasonal1 = as.matrix(pred_avg_diarage2[,terms]) %*% diarage_seasonal$coef[terms]
    avg_diarage2$seasonal2 = as.matrix(pred_avg_diarage2[,c(terms, terms)]) %*% 
      diarage_seasonal$coef[terms2]
    avg_diarage2$seasonal3 = as.matrix(pred_avg_diarage2[,c(terms,terms)]) %*% 
      diarage_seasonal$coef[terms3]
    dimnames(diar_agebase$Vp) = list(names(diar_agebase$coef), names(diar_agebase$coef))
    avg_diarage2$seasonal2.se = sqrt(diag(
      as.matrix(pred_avg_diarage2[,terms]) %*% 
        diar_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarage2[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar1to4y <- exp(as.matrix(avg_diarage2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
    write.csv(avg_diar1to4y, file = "avg_diar1to4y.csv")
  
  ## Diarrhea; 5-14 Years

    avg_diarage3 = diarage_2009_3
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("flag1to12m2:",terms,sep=''))
    terms3 = c(terms, paste("flag1to12m3:",terms,sep=''))
    pred_avg_diarage3 = avg_diarage3[,terms] - avg_diarage3[rep(which(avg_diarage3$Month=='Apr')[1], 
                                                                   nrow(avg_diarage3)), terms]
    avg_diarage3$seasonal1 = as.matrix(pred_avg_diarage3[,terms]) %*% diarage_seasonal$coef[terms]
    avg_diarage3$seasonal2 = as.matrix(pred_avg_diarage3[,c(terms, terms)]) %*% 
      diarage_seasonal$coef[terms2]
    avg_diarage3$seasonal3 = as.matrix(pred_avg_diarage3[,c(terms,terms)]) %*% 
      diarage_seasonal$coef[terms3]
    dimnames(diar_agebase$Vp) = list(names(diar_agebase$coef), names(diar_agebase$coef))
    avg_diarage3$seasonal3.se = sqrt(diag(
      as.matrix(pred_avg_diarage3[,terms]) %*% 
        diar_agebase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarage3[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar5to14y <- exp(as.matrix(avg_diarage3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
    write.csv(avg_diar5to14y, file = "avg_diar5to14y.csv")
  
##======================================================================================================
## FIGURE 4: Pneumonia&Diarrhea Child Age Subsets - average annual seasonal pattern
##======================================================================================================
    
  fig4sup1_pneu1to11m_data <- data.frame(x = 1:nrow(pneuage_2009_1), y = exp(pneuage_pred2_1))
  fig4sup1_pneu1to11m <- qplot(x, y, data = fig4sup1_pneu1to11m_data, geom = "line") +
    ggtitle("A: 1-11 month") + 
    geom_line(size = 1, colour = "dodgerblue2") + 
    expand_limits(x = c(1,13), y = c(0,60)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 20, 40, 60), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=35.72166), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=54.50912), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=35.72166), size=2) + 
    geom_point(aes(x=9, y=54.91761), size=2) + 
    geom_segment(aes(x = 4, y = 40.72166, xend = 4, yend = 36.72166), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=42.72166, label= "RR = 1.12", size = 5) + 
    geom_segment(aes(x = 9, y = 57.41761, xend = 9, yend = 55.91761), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=9, y=58.91761, label= "RR = 1.72", size = 5)
  
  fig4sup1_pneu1to4y_data <- data.frame(x = 1:nrow(pneuage_2009_2), y = exp(pneuage_pred2_2))
  fig4sup1_pneu1to4y <- qplot(x, y, data = fig4sup1_pneu1to4y_data, geom = "line") +
    ggtitle("C: 1-4 years") + 
    geom_line(size = 1, colour = "firebrick2") + 
    expand_limits(x = c(1,13), y = c(0,40)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=23.90295), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=26.81451), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=5, y=24.06063), size=2) + 
    geom_point(aes(x=10, y=26.81451), size=2) + 
    geom_segment(aes(x = 5, y = 29.06063, xend = 5, yend = 25.06063), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=5, y=31.06063, label= "RR = 1.42", size = 5) + 
    geom_segment(aes(x = 10, y = 31.81451, xend = 10, yend = 27.81451), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=33.81451, label= "RR = 1.59", size = 5)
  
  fig4sup1_pneu5to14y_data <- data.frame(x = 1:nrow(pneuage_2009_3), y = exp(pneuage_pred2_3))
  fig4sup1_pneu5to14y <- qplot(x, y, data = fig4sup1_pneu5to14y_data, geom = "line") +
    ggtitle("E: 5-14 years") + 
    geom_line(size = 1, colour = "forestgreen") + 
    expand_limits(x = c(1,13), y = c(0,20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=5.773266), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=7.501483), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=5, y=6.091081), size=2) + 
    geom_segment(aes(x = 5, y = 9.091081, xend = 5, yend = 6.591081), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=5, y=10.091081, label= "RR = 1.24", size = 5) + 
    geom_point(aes(x=10, y=7.501483), size=2) + 
    geom_segment(aes(x = 10, y = 10.501483, xend = 10, yend = 8.001483), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=11.501483, label= "RR = 1.52", size = 5)

  fig4sup1_diar1to11m_data <- data.frame(x = 1:nrow(diarage_2009_1), y = exp(diarage_pred2_1))
  fig4sup1_diar1to11m <- qplot(x, y, data = fig4sup1_diar1to11m_data, geom = "line") + 
    ggtitle("B: 1-11 month") + 
    geom_line(size = 1, colour = "dodgerblue2") + 
    expand_limits(x = c(1,13), y = c(0,60)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "", breaks = c(0, 20, 40, 60), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +  
    geom_segment(aes(x=4, y=0, xend=4, yend=23.23769), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=24.37820), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=23.23769), size=2) + 
    geom_point(aes(x=10, y=24.37820), size=2) + 
    geom_segment(aes(x = 4, y = 28.23769, xend = 4, yend = 24.23769), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=30.23769, label= "RR = 1.37", size = 5) + 
    geom_segment(aes(x = 10, y = 29.37820, xend = 10, yend = 25.37820), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=31.37820, label= "RR = 1.43", size = 5)
  
  fig4sup1_diar1to4y_data <- data.frame(x = 1:nrow(diarage_2009_2), y = exp(diarage_pred2_2))
  fig4sup1_diar1to4y <- qplot(x, y, data = fig4sup1_diar1to4y_data, geom = "line") + 
    ggtitle("D: 1-4 years") + 
    geom_line(size = 1, colour = "firebrick2") + 
    expand_limits(x = c(1,13), y = c(0,40)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=27.71726), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=21.76463), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=27.71726), size=2) + 
    geom_point(aes(x=10, y=21.76463), size=2) + 
    geom_segment(aes(x = 4, y = 32.71726, xend = 4, yend = 28.71726), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=34.71726, label= "RR = 1.88", size = 5) + 
    geom_segment(aes(x = 10, y = 26.76463, xend = 10, yend = 22.76463), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=28.76463, label= "RR = 1.48", size = 5)
  
  fig4sup1_diar5to14y_data <- data.frame(x = 1:nrow(diarage_2009_3), y = exp(diarage_pred2_3))
  fig4sup1_diar5to14y <- qplot(x, y, data = fig4sup1_diar5to14y_data, geom = "line") + 
    ggtitle("F: 5-14 years") + 
    geom_line(size = 1, colour = "forestgreen") + 
    expand_limits(x = c(1,13), y = c(0,20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.04,0)) + 
    scale_y_continuous(name = "", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=13.353789), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=7.305075), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=13.353789), size=2) + 
    geom_segment(aes(x = 4, y = 15.353789, xend = 4, yend = 13.853789), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=16.353789, label= "RR = 1.84", size = 5)

  grid.arrange(fig4sup1_pneu1to11m, fig4sup1_diar1to11m, fig4sup1_pneu1to4y, fig4sup1_diar1to4y, fig4sup1_pneu5to14y, fig4sup1_diar5to14y, ncol = 2)


  
##======================================================================================================
## Model 6: Pneumonia mortality by climate region; child & adult (excluding cases with reported measles)
##======================================================================================================
  
#### Pneumonia; 1M-14Y

  #### Prepare data
  
    pneu_climate <- read.csv("ts_pneu1mto14yr_climate.csv", header=TRUE)
    
    pneu_climate <- pneu_climate[!pneu_climate$Year == 2004,] ## Exclude 2004 data due to cancelled records
    pneu_climate$mo_index <- pneu_climate$mo_index - 12
    
  #### Plotting crude, unmodelled monthly pneumonia mortality
    
    region <- c(3,6,11)
    r <- as.list(rep("", 3)) 
    for(i in region) { 
      r[[i]] <- subset(pneu_climate, kg_val == i)
      r[[i]]$kg_val <- NULL
    }
    
    par(mfrow = c(1,1))
    
    plot(pneu_deaths ~ mo_index, data = r[[3]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 80), 
         xlab = "Time", ylab = "Count of deaths", main = "Crude pneumonia mortality by month", 
         col="steelblue")
    lines(pneu_deaths ~ mo_index, data = r[[6]], lwd=2, col="orange")
    lines(pneu_deaths ~ mo_index, data = r[[11]], lwd=2, col="springgreen")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = c(NA,2005:2013))
    axis(2, at = c(0, 20, 40, 60, 80))
    abline(v=r[[3]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    legend("topright", inset = 0.025, legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
           col=c("steelblue", "orange", "springgreen"), lty=c(1,1,1), lwd=2, cex=0.8)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
  
  pneu_climate <- transform(pneu_climate, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                         format = "%Y-%b-%d"))
  
  pneu_climate$yearC = as.numeric(difftime(pneu_climate$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                           units='days'))/365.25
  
  pneu_climate$cos12 = cos(2*pi*pneu_climate$yearC)
  pneu_climate$sin12 = sin(2*pi*pneu_climate$yearC)
  pneu_climate$cos6 = cos(2*2*pi*pneu_climate$yearC)
  pneu_climate$sin6 = sin(2*2*pi*pneu_climate$yearC)
  
  ## Poisson Model
  
  pneu_climate$kg_val <- ifelse(pneu_climate$kg_val == 3, 1, ifelse(pneu_climate$kg_val == 6, 2, 3))
  pneu_climate$kg_val <- factor(pneu_climate$kg_val)
  
  pneu_climate$logDaysInMonth = log(Hmisc::monthDays(pneu_climate$Date))
  pneu_climbase = gam(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + s(yearC, by=kg_val, id=1)
                      + offset(logDaysInMonth), family='poisson', data=pneu_climate)
  
  #### Plotting modelled series --> FIGURE 6, FIGURE SUPPLEMENT 1 (PANEL A: PNEUMONIA)
  
  plot(pneu_climbase$fitted.values[which(pneu_climate$kg_val==1)], type='l', lwd=2, col="steelblue", 
       xlim = c(), ylim=c(0,63.5), axes = FALSE, xaxs = "i", yaxs = "i", 
       ylab="Death counts", xlab="Year", cex.lab=1.25)
  lines(pneu_climbase$fitted.values[which(pneu_climate$kg_val==2)], type='l', lwd=2, col="orange")
  lines(pneu_climbase$fitted.values[which(pneu_climate$kg_val==3)], type='l', lwd=2, col="springgreen")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis = 1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60), las=1, cex.axis = 1.25)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  legend("topright", inset = c(-0.25, 0.025), legend=c("Humid subtropical", "Tropical savannah", "Hot semi-arid"), 
         col=c("springgreen", "steelblue", "orange"), lty=1, lwd=2, cex=1.05, box.col="white")
  mtext(paste0("A: Pneumonia (1M-14Y)"), side = 3, adj = 0.015, cex=1.5)
  
  #### Calculate average seasonal trend between 2005-2013
  
  pneu_climate$logDaysInMonth = log(Hmisc::monthDays(pneu_climate$Date))
  
  pneuclim_seasonal = glm(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=pneu_climate)
  
  pneuclim_2009_1 = pneu_climate[pneu_climate$kg_val==1,][52:64,]
  pneuclim_pred_1 = predict(pneuclim_seasonal, pneuclim_2009_1, type='link', se.fit=TRUE)
  pneuclim_pred2_1 = pneuclim_pred_1$fit
  
  pneuclim_2009_2 = pneu_climate[pneu_climate$kg_val==2,][52:64,]
  pneuclim_pred_2 = predict(pneuclim_seasonal, pneuclim_2009_2, type='link', se.fit=TRUE)
  pneuclim_pred2_2 = pneuclim_pred_2$fit
  
  pneuclim_2009_3 = pneu_climate[pneu_climate$kg_val==3,][52:64,]
  pneuclim_pred_3 = predict(pneuclim_seasonal, pneuclim_2009_3, type='link', se.fit=TRUE)
  pneuclim_pred2_3 = pneuclim_pred_3$fit
  
  #### Rate ratio calculations
  
    ## Pneumonia 1M-14Y; Tropical savannah

      avg_pneuclim1 = pneuclim_2009_1
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      terms4 = c(terms, paste("kg_val4:",terms,sep=''))
      pred_avg_pneuclim1 = avg_pneuclim1[,terms] - avg_pneuclim1[rep(which(avg_pneuclim1$Month=='Apr')[1], 
                                                                    nrow(avg_pneuclim1)), terms]
      avg_pneuclim1$seasonal1 = as.matrix(pred_avg_pneuclim1[,terms]) %*% pneuclim_seasonal$coef[terms]
      avg_pneuclim1$seasonal2 = as.matrix(pred_avg_pneuclim1[,c(terms, terms)]) %*% pneuclim_seasonal$coef[terms2]
      avg_pneuclim1$seasonal3 = as.matrix(pred_avg_pneuclim1[,c(terms,terms)]) %*% pneuclim_seasonal$coef[terms3]
      avg_pneuclim1$seasonal4 = as.matrix(pred_avg_pneuclim1[,c(terms,terms)]) %*% pneuclim_seasonal$coef[terms4]
      dimnames(pneu_climbase$Vp) = list(names(pneu_climbase$coef), names(pneu_climbase$coef))
      avg_pneuclim1$seasonal1.se = sqrt(diag(
                                    as.matrix(pred_avg_pneuclim1[,terms]) %*% 
                                    pneu_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim1[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
    
      ## Save RR file
      avg_pneu_TSregion <- exp(as.matrix(avg_pneuclim1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
      write.csv(avg_pneu_TSregion, file = "avg_pneu1mto14yr_AmAw.csv")
    
    ## Pneumonia 1M-14Y; Hot Semi-Arid

      avg_pneuclim2 = pneuclim_2009_2
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      terms4 = c(terms, paste("kg_val4:",terms,sep=''))
      pred_avg_pneuclim2 = avg_pneuclim2[,terms] - avg_pneuclim2[rep(which(avg_pneuclim2$Month=='Apr')[1], 
                                                                        nrow(avg_pneuclim2)), terms]
      avg_pneuclim2$seasonal1 = as.matrix(pred_avg_pneuclim2[,terms]) %*% pneuclim_seasonal$coef[terms]
      avg_pneuclim2$seasonal2 = as.matrix(pred_avg_pneuclim2[,c(terms, terms)]) %*% 
        pneuclim_seasonal$coef[terms2]
      avg_pneuclim2$seasonal3 = as.matrix(pred_avg_pneuclim2[,c(terms,terms)]) %*% 
        pneuclim_seasonal$coef[terms3]
      avg_pneuclim2$seasonal4 = as.matrix(pred_avg_pneuclim2[,c(terms,terms)]) %*%
        pneuclim_seasonal$coef[terms4]
      dimnames(pneu_climbase$Vp) = list(names(pneu_climbase$coef), names(pneu_climbase$coef))
      avg_pneuclim2$seasonal2.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim2[,terms]) %*% 
          pneu_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim2[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu_HSAregion <- exp(as.matrix(avg_pneuclim2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
      write.csv(avg_pneu_HSAregion, file = "avg_pneu1mto14yr_BshBwh.csv")
    
    ## Pneumonia 1M-14Y; Humid Subtropical

      avg_pneuclim3= pneuclim_2009_3
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      terms4 = c(terms, paste("kg_val4:",terms,sep=''))
      pred_avg_pneuclim3 = avg_pneuclim3[,terms] - avg_pneuclim3[rep(which(avg_pneuclim3$Month=='Apr')[1], 
                                                                        nrow(avg_pneuclim3)), terms]
      avg_pneuclim3$seasonal1 = as.matrix(pred_avg_pneuclim3[,terms]) %*% pneuclim_seasonal$coef[terms]
      avg_pneuclim3$seasonal2 = as.matrix(pred_avg_pneuclim3[,c(terms, terms)]) %*% 
        pneuclim_seasonal$coef[terms2]
      avg_pneuclim3$seasonal3 = as.matrix(pred_avg_pneuclim3[,c(terms,terms)]) %*% 
        pneuclim_seasonal$coef[terms3]
      avg_pneuclim3$seasonal4 = as.matrix(pred_avg_pneuclim3[,c(terms,terms)]) %*%
        pneuclim_seasonal$coef[terms4]
      dimnames(pneu_climbase$Vp) = list(names(pneu_climbase$coef), names(pneu_climbase$coef))
      avg_pneuclim3$seasonal3.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim3[,terms]) %*% 
          pneu_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim3[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu_HSTregion <- exp(as.matrix(avg_pneuclim3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
      write.csv(avg_pneu_HSTregion, file = "avg_pneu1mto14yr_CwaCwb.csv")

#### Pneumonia; 15-29 Years

  #### Prep data
  
    pneuclim_15to29 <- read.csv("ts_pneu15to29yr_climate.csv", header=TRUE)
    
    pneuclim_15to29 <- pneuclim_15to29[!pneuclim_15to29$Year == 2004,] ## Exclude 2004 data due to cancelled records
    pneuclim_15to29$mo_index <- pneuclim_15to29$mo_index - 12
    
  #### Plotting crude, unmodelled monthly pneumonia mortality
    
    region <- c(3,6,11)
    r <- as.list(rep("", 3)) 
    for(i in region) { 
      r[[i]] <- subset(pneuclim_15to29, kg_val == i)
      r[[i]]$kg_val <- NULL
    }
    
    par(mfrow = c(1,1))
    
    plot(pneu_deaths ~ mo_index, data = r[[3]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 20), 
         xlab = "Time", ylab = "Count of deaths", main = "Crude pneumonia mortality by month", 
         col="steelblue")
    lines(pneu_deaths ~ mo_index, data = r[[6]], lwd=2, col="orange")
    lines(pneu_deaths ~ mo_index, data = r[[11]], lwd=2, col="springgreen")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = c(NA,2005:2013))
    axis(2, at = c(0, 5, 10, 15, 20))
    abline(v=r[[3]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    legend("topright", inset = 0.025, legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
           col=c("steelblue", "orange", "springgreen"), lty=c(1,1,1), lwd=2, cex=0.8)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
  
    pneuclim_15to29 <- transform(pneuclim_15to29, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                                 format = "%Y-%b-%d"))
    
    pneuclim_15to29$yearC = as.numeric(difftime(pneuclim_15to29$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                                units='days'))/365.25
    
    pneuclim_15to29$cos12 = cos(2*pi*pneuclim_15to29$yearC)
    pneuclim_15to29$sin12 = sin(2*pi*pneuclim_15to29$yearC)
    pneuclim_15to29$cos6 = cos(2*2*pi*pneuclim_15to29$yearC)
    pneuclim_15to29$sin6 = sin(2*2*pi*pneuclim_15to29$yearC)
    
  #### Poisson Model
      
    pneuclim_15to29$kg_val <- ifelse(pneuclim_15to29$kg_val == 3, 1, ifelse(pneuclim_15to29$kg_val == 6, 2, 3))
    pneuclim_15to29$kg_val <- factor(pneuclim_15to29$kg_val)
    
    pneuclim_15to29$logDaysInMonth = log(Hmisc::monthDays(pneuclim_15to29$Date))
    pneu_clim15to29base = gam(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + s(yearC, by=kg_val, id=1)
                              + offset(logDaysInMonth), family='poisson', data=pneuclim_15to29)
  
  #### Plotting modelled series
    
    plot(pneu_clim15to29base$fitted.values[which(pneuclim_15to29$kg_val==1)], type='l', lwd=2, col="steelblue", 
         xlim = c(), ylim=c(0,20), axes = FALSE, xaxs = "i", yaxs = "i", 
         ylab="Death counts", xlab="Year", cex.lab=1.25)
    lines(pneu_clim15to29base$fitted.values[which(pneuclim_15to29$kg_val==2)], type='l', lwd=2, col="orange")
    lines(pneu_clim15to29base$fitted.values[which(pneuclim_15to29$kg_val==3)], type='l', lwd=2, col="springgreen")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis = 1.25)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 5, 10, 15, 20), las=1, cex.axis = 1.25)
    abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
    legend("topright", inset = c(0.005, 0.025), legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
           col=c("steelblue", "orange", "springgreen"), lty=1, lwd=2, cex=1.25, box.col="white")
    mtext(paste0("A: Pneumonia"), side = 3, adj = 0.015, cex=1.5)
      
  #### Calculate average seasonal trend between 2005-2013
  
    pneuclim_15to29$logDaysInMonth = log(Hmisc::monthDays(pneuclim_15to29$Date))
    
    pneuclim15to29_seasonal = glm(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=pneuclim_15to29)
    
    pneuclim15to29_2009_1 = pneuclim_15to29[pneuclim_15to29$kg_val==1,][52:64,]
    pneuclim15to29_pred_1 = predict(pneuclim15to29_seasonal, pneuclim15to29_2009_1, type='link', se.fit=TRUE)
    pneuclim15to29_pred2_1 = pneuclim15to29_pred_1$fit
    
    pneuclim15to29_2009_2 = pneuclim_15to29[pneuclim_15to29$kg_val==2,][52:64,]
    pneuclim15to29_pred_2 = predict(pneuclim15to29_seasonal, pneuclim15to29_2009_2, type='link', se.fit=TRUE)
    pneuclim15to29_pred2_2 = pneuclim15to29_pred_2$fit
    
    pneuclim15to29_2009_3 = pneuclim_15to29[pneuclim_15to29$kg_val==3,][52:64,]
    pneuclim15to29_pred_3 = predict(pneuclim15to29_seasonal, pneuclim15to29_2009_3, type='link', se.fit=TRUE)
    pneuclim15to29_pred2_3 = pneuclim15to29_pred_3$fit

  #### Rate ratio calculations
    
    ## Pneumonia 15-29Y; Tropical savannah

      avg_pneuclim15to29_1 = pneuclim15to29_2009_1
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim15to29_1 = avg_pneuclim15to29_1[,terms] - avg_pneuclim15to29_1[rep(which(avg_pneuclim15to29_1$Month=='Apr')[1], 
                                                                                         nrow(avg_pneuclim15to29_1)), terms]
      avg_pneuclim15to29_1$seasonal1 = as.matrix(pred_avg_pneuclim15to29_1[,terms]) %*% pneuclim15to29_seasonal$coef[terms]
      avg_pneuclim15to29_1$seasonal2 = as.matrix(pred_avg_pneuclim15to29_1[,c(terms, terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms2]
      avg_pneuclim15to29_1$seasonal3 = as.matrix(pred_avg_pneuclim15to29_1[,c(terms,terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms3]
      dimnames(pneu_clim15to29base$Vp) = list(names(pneu_clim15to29base$coef), names(pneu_clim15to29base$coef))
      avg_pneuclim15to29_1$seasonal1.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim15to29_1[,terms]) %*% 
          pneu_clim15to29base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim15to29_1[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu15to29_TSregion <- exp(as.matrix(avg_pneuclim15to29_1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
      write.csv(avg_pneu15to29_TSregion, file = "avg_pneu15to29yr_AmAw.csv")
    
    ## Pneumonia 15-29Y; Hot Semi-Arid

      avg_pneuclim15to29_2 = pneuclim15to29_2009_2
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim15to29_2 = avg_pneuclim15to29_2[,terms] - avg_pneuclim15to29_2[rep(which(avg_pneuclim15to29_2$Month=='Apr')[1], 
                                                                                         nrow(avg_pneuclim15to29_2)), terms]
      avg_pneuclim15to29_2$seasonal1 = as.matrix(pred_avg_pneuclim15to29_2[,terms]) %*% pneuclim15to29_seasonal$coef[terms]
      avg_pneuclim15to29_2$seasonal2 = as.matrix(pred_avg_pneuclim15to29_2[,c(terms, terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms2]
      avg_pneuclim15to29_2$seasonal3 = as.matrix(pred_avg_pneuclim15to29_2[,c(terms,terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms3]
      dimnames(pneu_clim15to29base$Vp) = list(names(pneu_clim15to29base$coef), names(pneu_clim15to29base$coef))
      avg_pneuclim15to29_2$seasonal2.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim15to29_2[,terms]) %*% 
          pneu_clim15to29base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim15to29_2[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu15to29_HSAregion <- exp(as.matrix(avg_pneuclim15to29_2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
      write.csv(avg_pneu15to29_HSAregion, file = "avg_pneu15to29yr_BshBwh.csv")
    
    ## Pneumonia 15-29Y; Humid Subtropical
      
      avg_pneuclim15to29_3= pneuclim15to29_2009_3
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim15to29_3 = avg_pneuclim15to29_3[,terms] - avg_pneuclim15to29_3[rep(which(avg_pneuclim15to29_3$Month=='Apr')[1], 
                                                                                         nrow(avg_pneuclim15to29_3)), terms]
      avg_pneuclim15to29_3$seasonal1 = as.matrix(pred_avg_pneuclim15to29_3[,terms]) %*% pneuclim15to29_seasonal$coef[terms]
      avg_pneuclim15to29_3$seasonal2 = as.matrix(pred_avg_pneuclim15to29_3[,c(terms, terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms2]
      avg_pneuclim15to29_3$seasonal3 = as.matrix(pred_avg_pneuclim15to29_3[,c(terms,terms)]) %*% 
        pneuclim15to29_seasonal$coef[terms3]
      dimnames(pneu_clim15to29base$Vp) = list(names(pneu_clim15to29base$coef), names(pneu_clim15to29base$coef))
      avg_pneuclim15to29_3$seasonal3.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim15to29_3[,terms]) %*% 
          pneu_clim15to29base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim15to29_3[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu15to29_HSTregion <- exp(as.matrix(avg_pneuclim15to29_3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
      write.csv(avg_pneu15to29_HSTregion, file = "avg_pneu15to29yr_CwaCwb.csv")
    
#### Pneumonia; 30-59 Years
      
  #### Prep data
    
    pneuclim_30to59 <- read.csv("ts_pneu30to59yr_climate.csv", header=TRUE)
    
    pneuclim_30to59 <- pneuclim_30to59[!pneuclim_30to59$Year == 2004,] ## Exclude 2004 data due to cancelled records
    pneuclim_30to59$mo_index <- pneuclim_30to59$mo_index - 12
    
  #### Plotting crude, unmodelled monthly pneumonia mortality
    
    region <- c(3,6,11)
    r <- as.list(rep("", 3)) 
    for(i in region) { 
      r[[i]] <- subset(pneuclim_30to59, kg_val == i)
      r[[i]]$kg_val <- NULL
    }
    
    par(mfrow = c(1,1))
    
    plot(pneu_deaths ~ mo_index, data = r[[3]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 20), 
         xlab = "Time", ylab = "Count of deaths", main = "Crude pneumonia mortality by month", 
         col="steelblue")
    lines(pneu_deaths ~ mo_index, data = r[[6]], lwd=2, col="orange")
    lines(pneu_deaths ~ mo_index, data = r[[11]], lwd=2, col="springgreen")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = c(NA,2005:2013))
    axis(2, at = c(0, 5, 10, 15, 20))
    abline(v=r[[3]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
    legend("topright", inset = 0.025, legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
           col=c("steelblue", "orange", "springgreen"), lty=c(1,1,1), lwd=2, cex=0.8)
    
  #### Generate time value and harmonics (two sine/cosine pairs) for model
    
    pneuclim_30to59 <- transform(pneuclim_30to59, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                                 format = "%Y-%b-%d"))
    
    pneuclim_30to59$yearC = as.numeric(difftime(pneuclim_30to59$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                                units='days'))/365.25
    
    pneuclim_30to59$cos12 = cos(2*pi*pneuclim_30to59$yearC)
    pneuclim_30to59$sin12 = sin(2*pi*pneuclim_30to59$yearC)
    pneuclim_30to59$cos6 = cos(2*2*pi*pneuclim_30to59$yearC)
    pneuclim_30to59$sin6 = sin(2*2*pi*pneuclim_30to59$yearC)
    
  #### Poisson Model
    
    pneuclim_30to59$kg_val <- ifelse(pneuclim_30to59$kg_val == 3, 1, ifelse(pneuclim_30to59$kg_val == 6, 2, 3))
    pneuclim_30to59$kg_val <- factor(pneuclim_30to59$kg_val)
    
    pneuclim_30to59$logDaysInMonth = log(Hmisc::monthDays(pneuclim_30to59$Date))
    pneu_clim30to59base = gam(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + s(yearC, by=kg_val, id=1)
                              + offset(logDaysInMonth), family='poisson', data=pneuclim_30to59)
    
  #### Plotting modelled series
    
    plot(pneu_clim30to59base$fitted.values[which(pneuclim_30to59$kg_val==1)], type='l', lwd=2, col="steelblue", 
         xlim = c(), ylim=c(0,30), axes = FALSE, xaxs = "i", yaxs = "i", 
         ylab="Death counts", xlab="Year", cex.lab=1.25)
    lines(pneu_clim30to59base$fitted.values[which(pneuclim_30to59$kg_val==2)], type='l', lwd=2, col="orange")
    lines(pneu_clim30to59base$fitted.values[which(pneuclim_30to59$kg_val==3)], type='l', lwd=2, col="springgreen")
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis = 1.25)
    axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
    axis(2, at = c(0, 5, 10, 15, 20, 25, 30), las=1, cex.axis = 1.25)
    abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
    legend("topright", inset = c(0.005, 0.025), legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
           col=c("steelblue", "orange", "springgreen"), lty=1, lwd=2, cex=1.25, box.col="white")
    mtext(paste0("A: Pneumonia"), side = 3, adj = 0.015, cex=1.5)
    
  #### Calculate average seasonal trend between 2005-2013
    
    pneuclim_30to59$logDaysInMonth = log(Hmisc::monthDays(pneuclim_30to59$Date))
    
    pneuclim30to59_seasonal = glm(pneu_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=pneuclim_30to59)
    
    pneuclim30to59_2009_1 = pneuclim_30to59[pneuclim_30to59$kg_val==1,][52:64,]
    pneuclim30to59_pred_1 = predict(pneuclim30to59_seasonal, pneuclim30to59_2009_1, type='link', se.fit=TRUE)
    pneuclim30to59_pred2_1 = pneuclim30to59_pred_1$fit
    
    pneuclim30to59_2009_2 = pneuclim_30to59[pneuclim_30to59$kg_val==2,][52:64,]
    pneuclim30to59_pred_2 = predict(pneuclim30to59_seasonal, pneuclim30to59_2009_2, type='link', se.fit=TRUE)
    pneuclim30to59_pred2_2 = pneuclim30to59_pred_2$fit
    
    pneuclim30to59_2009_3 = pneuclim_30to59[pneuclim_30to59$kg_val==3,][52:64,]
    pneuclim30to59_pred_3 = predict(pneuclim30to59_seasonal, pneuclim30to59_2009_3, type='link', se.fit=TRUE)
    pneuclim30to59_pred2_3 = pneuclim30to59_pred_3$fit
  
  #### Rate ratio calculations
    
    ## Pneumonia 30-59Y; Tropical savannah
      
      avg_pneuclim30to59_1 = pneuclim30to59_2009_1
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim30to59_1 = avg_pneuclim30to59_1[,terms] - avg_pneuclim30to59_1[rep(which(avg_pneuclim30to59_1$Month=='Apr')[1], 
                                                                                             nrow(avg_pneuclim30to59_1)), terms]
      avg_pneuclim30to59_1$seasonal1 = as.matrix(pred_avg_pneuclim30to59_1[,terms]) %*% pneuclim30to59_seasonal$coef[terms]
      avg_pneuclim30to59_1$seasonal2 = as.matrix(pred_avg_pneuclim30to59_1[,c(terms, terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms2]
      avg_pneuclim30to59_1$seasonal3 = as.matrix(pred_avg_pneuclim30to59_1[,c(terms,terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms3]
      dimnames(pneu_clim30to59base$Vp) = list(names(pneu_clim30to59base$coef), names(pneu_clim30to59base$coef))
      avg_pneuclim30to59_1$seasonal1.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim30to59_1[,terms]) %*% 
          pneu_clim30to59base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim30to59_1[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu30to59_TSregion <- exp(as.matrix(avg_pneuclim30to59_1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
      write.csv(avg_pneu30to59_TSregion, file = "avg_pneu30to59yr_AmAw.csv")
    
    ## Pneumonia 30-59Y; Hot Semi-Arid
      
      avg_pneuclim30to59_2 = pneuclim30to59_2009_2
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim30to59_2 = avg_pneuclim30to59_2[,terms] - avg_pneuclim30to59_2[rep(which(avg_pneuclim30to59_2$Month=='Apr')[1], 
                                                                                             nrow(avg_pneuclim30to59_2)), terms]
      avg_pneuclim30to59_2$seasonal1 = as.matrix(pred_avg_pneuclim30to59_2[,terms]) %*% pneuclim30to59_seasonal$coef[terms]
      avg_pneuclim30to59_2$seasonal2 = as.matrix(pred_avg_pneuclim30to59_2[,c(terms, terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms2]
      avg_pneuclim30to59_2$seasonal3 = as.matrix(pred_avg_pneuclim30to59_2[,c(terms,terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms3]
      dimnames(pneu_clim30to59base$Vp) = list(names(pneu_clim30to59base$coef), names(pneu_clim30to59base$coef))
      avg_pneuclim30to59_2$seasonal2.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim30to59_2[,terms]) %*% 
          pneu_clim30to59base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim30to59_2[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu30to59_HSAregion <- exp(as.matrix(avg_pneuclim30to59_2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
      write.csv(avg_pneu30to59_HSAregion, file = "avg_pneu30to59yr_BshBwh.csv")
    
    ## Pneumonia 30-59Y; Humid Subtropical
      
      avg_pneuclim30to59_3= pneuclim30to59_2009_3
      terms=c('sin12','cos12','sin6','cos6')
      terms2 = c(terms, paste("kg_val2:",terms,sep=''))
      terms3 = c(terms, paste("kg_val3:",terms,sep=''))
      pred_avg_pneuclim30to59_3 = avg_pneuclim30to59_3[,terms] - avg_pneuclim30to59_3[rep(which(avg_pneuclim30to59_3$Month=='Apr')[1], 
                                                                                             nrow(avg_pneuclim30to59_3)), terms]
      avg_pneuclim30to59_3$seasonal1 = as.matrix(pred_avg_pneuclim30to59_3[,terms]) %*% pneuclim30to59_seasonal$coef[terms]
      avg_pneuclim30to59_3$seasonal2 = as.matrix(pred_avg_pneuclim30to59_3[,c(terms, terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms2]
      avg_pneuclim30to59_3$seasonal3 = as.matrix(pred_avg_pneuclim30to59_3[,c(terms,terms)]) %*% 
        pneuclim30to59_seasonal$coef[terms3]
      dimnames(pneu_clim30to59base$Vp) = list(names(pneu_clim30to59base$coef), names(pneu_clim30to59base$coef))
      avg_pneuclim30to59_3$seasonal3.se = sqrt(diag(
        as.matrix(pred_avg_pneuclim30to59_3[,terms]) %*% 
          pneu_clim30to59base$Vp[terms,terms] %*% t(as.matrix(pred_avg_pneuclim30to59_3[,terms]))))
      seMat = cbind(
        est = c(1,0),
        lower = c(1, qnorm(0.01)),
        upper = c(1, qnorm(0.99))
      )
      
      ## Save RR file
      avg_pneu30to59_HSTregion <- exp(as.matrix(avg_pneuclim30to59_3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
      write.csv(avg_pneu30to59_HSTregion, file = "avg_pneu30to59yr_CwaCwb.csv")
    
      
##======================================================================================================
## FIGURE 6: Pneumonia Climate Regions - average annual seasonal pattern
##======================================================================================================

  figure6_pneu1mto14y_TS_data <- data.frame(x = 1:nrow(pneuclim_2009_1), y = exp(pneuclim_pred2_1))
  figure6_pneu1mto14y_TS <- qplot(x, y, data = figure6_pneu1mto14y_TS_data, geom = "line") +
    ggtitle("B: Tropical Savannah") + 
    geom_line(size = 1, colour = "steelblue") + 
    expand_limits(x = c(1,13), y = c(0,40)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=16.66735), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=18.43073), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=8, y=20.01340), size=2) + 
    geom_segment(aes(x = 8, y = 23.51340, xend = 8, yend = 21.01340), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=8, y=24.51340, label= "RR = 1.39", size = 5) + 
    geom_label(x=7, y=3, label="Study deaths = 1858", label.size = NA, size=5)
  
  figure6_pneu1mto14y_HSA_data <- data.frame(x = 1:nrow(pneuclim_2009_2), y = exp(pneuclim_pred2_2))
  figure6_pneu1mto14y_HSA <- qplot(x, y, data = figure6_pneu1mto14y_HSA_data, geom = "line") +
    ggtitle("C: Hot Semi-Arid") + 
    geom_line(size = 1, colour = "orange") + 
    expand_limits(x = c(1,13), y = c(0,40)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=12.51340), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=23.19776), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=10, y=23.19776), size=2) + 
    geom_segment(aes(x = 10, y = 26.69776, xend = 10, yend = 24.19776), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=27.69776, label= "RR = 1.75", size = 5) + 
    geom_label(x=7, y=3, label="Study deaths = 1676", label.size = NA, size=5)
  
  figure6_pneu1mto14y_HST_data <- data.frame(x = 1:nrow(pneuclim_2009_3), y = exp(pneuclim_pred2_3))
  figure6_pneu1mto14y_HST <- qplot(x, y, data = figure6_pneu1mto14y_HST_data, geom = "line") +
    ggtitle("A: Humid Subtropical") + 
    geom_line(size = 1, colour = "springgreen") + 
    expand_limits(x = c(1,13), y = c(0,46)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=32.72191), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=44.82749), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=32.72191), size=2) + 
    geom_point(aes(x=10, y=44.82749), size=2) + 
    geom_segment(aes(x = 4, y = 37.72191, xend = 4, yend = 33.72191), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=39.72191, label= "RR = 1.39", size = 5) + 
    geom_segment(aes(x = 9.2, y = 44.82749, xend = 9.7, yend = 44.82749), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=7.65, y=44.82749, label= "RR = 1.91", size = 5) + 
    geom_label(x=7, y=3, label="Study deaths = 3313", label.size = NA, size=5)
  
  figure6_pneu15to29y_TS_data <- data.frame(x = 1:nrow(pneuclim15to29_2009_1), y = exp(pneuclim15to29_pred2_1))
  figure6_pneu15to29y_TS <- qplot(x, y, data = figure6_pneu15to29y_TS_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "steelblue") + 
    expand_limits(x = c(1,13), y = c(0,22.5)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=5.842540), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=6.560328), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=9, y=6.938152), size=2) + 
    geom_segment(aes(x = 9, y = 9.438152, xend = 9, yend = 7.438152), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=9, y=9.938152, label= "RR = 1.46", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 609", label.size = NA, size=5)
  
  figure6_pneu15to29y_HSA_data <- data.frame(x = 1:nrow(pneuclim15to29_2009_2), y = exp(pneuclim15to29_pred2_2))
  figure6_pneu15to29y_HSA <- qplot(x, y, data = figure6_pneu15to29y_HSA_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "orange") + 
    expand_limits(x = c(1,13), y = c(0,20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=2.691562), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=4.010350), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=9, y=4.436104), size=2) + 
    geom_segment(aes(x = 9, y = 6.936104, xend = 9, yend = 4.936104), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=9, y=7.436104, label= "RR = 2.25", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 308", label.size = NA, size=5)
  
  figure6_pneu15to29y_HST_data <- data.frame(x = 1:nrow(pneuclim15to29_2009_3), y = exp(pneuclim15to29_pred2_3))
  figure6_pneu15to29y_HST <- qplot(x, y, data = figure6_pneu15to29y_HST_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "springgreen") + 
    expand_limits(x = c(1,13), y = c(0,20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "Death counts", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.text=element_text(size=16), axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
    geom_segment(aes(x=4, y=0, xend=4, yend=5.739503), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=8.766249), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=10, y=8.766249), size=2) + 
    geom_segment(aes(x = 10, y = 11.266249, xend = 10, yend = 9.266249), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=11.766249, label= "RR = 1.94", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 628", label.size = NA, size=5)
  
  figure6_pneu30to59y_TS_data <- data.frame(x = 1:nrow(pneuclim30to59_2009_1), y = exp(pneuclim30to59_pred2_1))
  figure6_pneu30to59y_TS <- qplot(x, y, data = figure6_pneu30to59y_TS_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "steelblue") + 
    expand_limits(x = c(1,13), y = c(0,22.5)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "", breaks = c(0,5,10,15,20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=16.91162), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=21.18113), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=16.91162), size=2) +  
    geom_point(aes(x=9, y=21.83366), size=2) + 
    geom_segment(aes(x = 4, y = 19.41162, xend = 4, yend = 17.41162), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=19.91162, label= "RR = 1.30", size = 5) + 
    geom_segment(aes(x = 8.2, y = 21.83366, xend = 8.8, yend = 21.83366), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=7.2, y=21.83366, label= "RR = 1.67", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 1785", label.size = NA, size=5)
  
  figure6_pneu30to59y_HSA_data <- data.frame(x = 1:nrow(pneuclim30to59_2009_2), y = exp(pneuclim30to59_pred2_2))
  figure6_pneu30to59y_HSA <- qplot(x, y, data = figure6_pneu30to59y_HSA_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "orange") + 
    expand_limits(x = c(1,13), y = c(0,20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "", breaks = c(0,5,10,15,20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=6.975257), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=9.292342), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=5, y=7.498217), size=2) + 
    geom_point(aes(x=10, y=9.292342), size=2) + 
    geom_segment(aes(x = 5, y = 9.998217, xend = 5, yend = 7.998217), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=5, y=10.498217, label= "RR = 1.45", size = 5) + 
    geom_segment(aes(x = 10, y = 11.792342, xend = 10, yend = 9.792342), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=12.292342, label= "RR = 1.79", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 748", label.size = NA, size=5)
  
  figure6_pneu30to59y_HST_data <- data.frame(x = 1:nrow(pneuclim30to59_2009_3), y = exp(pneuclim30to59_pred2_3))
  figure6_pneu30to59y_HST <- qplot(x, y, data = figure6_pneu30to59y_HST_data, geom = "line") +
    ggtitle("") + 
    geom_line(size = 1, colour = "springgreen") + 
    expand_limits(x = c(1,13), y = c(0, 20)) + 
    scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
    scale_y_continuous(name = "", breaks = c(0,5,10,15,20), minor_breaks = NULL, expand=c(0,0)) + 
    theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
          axis.line.y = element_line(colour = "transparent"), axis.text.y=element_text(colour="transparent"),
          axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
          axis.title=element_text(size=16), plot.title = element_text(size = 20)) +
    geom_segment(aes(x=4, y=0, xend=4, yend=13.15881), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_segment(aes(x=10, y=0, xend=10, yend=16.56578), color=grey(0.7), lwd=1.05, lty="dashed") +
    geom_point(aes(x=4, y=13.15881), size=2) + 
    geom_point(aes(x=10, y=16.56578), size=2) + 
    geom_segment(aes(x = 4, y = 15.65881, xend = 4, yend = 13.65881), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=4, y=16.15881, label= "RR = 1.22", size = 5) + 
    geom_segment(aes(x = 10, y = 19.06578, xend = 10, yend = 17.06578), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
    annotate("text", x=10, y=19.56578, label= "RR = 1.54", size = 5) + 
    geom_label(x=7, y=1, label="Study deaths = 1362", label.size = NA, size=5)
  
  grid.arrange(figure6_pneu1mto14y_HST, figure6_pneu15to29y_HST, figure6_pneu30to59y_HST, 
               figure6_pneu1mto14y_TS, figure6_pneu15to29y_TS, figure6_pneu30to59y_TS, 
               figure6_pneu1mto14y_HSA, figure6_pneu15to29y_HSA, figure6_pneu30to59y_HSA, ncol=3)

  
##======================================================================================================
## Model 7: Diarrhea mortality by climate region; child only (excluding typhoid and cases with reported measles)
##======================================================================================================

#### Prep data
  
  diar_climate <- read.csv("ts_diar1mto14yr_climate.csv", header=TRUE)
  
  diar_climate <- diar_climate[!diar_climate$Year == 2004,] ## Exclude 2004 data due to cancelled records
  diar_climate$mo_index <- diar_climate$mo_index - 12
  
#### Plotting crude, unmodelled monthly pneumonia mortality
  
  region <- c(3,6,11)
  r <- as.list(rep("", 3)) 
  for(i in region) { 
    r[[i]] <- subset(diar_climate, kg_val == i)
    r[[i]]$kg_val <- NULL
  }
  
  par(mfrow = c(1,1))
  
  plot(diar_deaths ~ mo_index, data = r[[3]], type = "l", lwd = 2, axes = FALSE, ylim = c(0, 80), 
       xlab = "Time", ylab = "Count of deaths", main = "Crude diarrhea mortality by month", 
       col="steelblue")
  lines(diar_deaths ~ mo_index, data = r[[6]], lwd=2, col="orange")
  lines(diar_deaths ~ mo_index, data = r[[11]], lwd=2, col="springgreen")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = c(NA,2005:2013))
  axis(2, at = c(0, 20, 40, 60, 80))
  abline(v=r[[3]]$mo_index[c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)],col=grey(0.6),lty=2)
  legend("topright", inset = 0.025, legend=c("Tropical savannah", "Hot semi-arid", "Humid subtropical"), 
         col=c("steelblue", "orange", "springgreen"), lty=c(1,1,1), lwd=2, cex=0.8)
  
#### Generate time value and harmonics (two sine/cosine pairs) for model
  
  diar_climate <- transform(diar_climate, Date = as.Date(paste(Year, Month, "15", sep = "-"),
                                                         format = "%Y-%b-%d"))
  
  diar_climate$yearC = as.numeric(difftime(diar_climate$Date, as.Date(ISOdate(2010,1,15,0,0,0, tz='UTC')), 
                                           units='days'))/365.25
  
  diar_climate$cos12 = cos(2*pi*diar_climate$yearC)
  diar_climate$sin12 = sin(2*pi*diar_climate$yearC)
  diar_climate$cos6 = cos(2*2*pi*diar_climate$yearC)
  diar_climate$sin6 = sin(2*2*pi*diar_climate$yearC)
  
#### Poisson Model
  
  diar_climate$kg_val <- ifelse(diar_climate$kg_val == 3, 1, ifelse(diar_climate$kg_val == 6, 2, 3))
  
  diar_climate$logDaysInMonth = log(Hmisc::monthDays(diar_climate$Date))
  diar_climate$kg_val <- factor(diar_climate$kg_val)
  
  diar_climbase = gam(diar_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + s(yearC, by=kg_val, id=1)
                      + offset(logDaysInMonth), family='poisson', data=diar_climate)
  
#### Plotting modelled series --> FIGURE 6, FIGURE SUPPLEMENT 1 (PANEL B: DIARRHEA)
  
  plot(diar_climbase$fitted.values[which(diar_climate$kg_val==1)], type='l', lwd=2, col="steelblue", 
       xlim = c(), ylim=c(0,62), axes = FALSE, xaxs = "i", yaxs = "i", ylab = "Death counts" , xlab="Year", cex.lab=1.25)
  lines(diar_climbase$fitted.values[which(diar_climate$kg_val==2)], type='l', lwd=2, col="orange")
  lines(diar_climbase$fitted.values[which(diar_climate$kg_val==3)], type='l', lwd=2, col="springgreen")
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108)-5.5, labels = c(NA, 2005:2013), tick=FALSE, cex.axis = 1.25)
  axis(1, at = c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108), labels = FALSE, tick=TRUE)
  axis(2, at = c(0, 20, 40, 60), las=1, cex.axis = 1.25)
  abline(v=c(0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120),col=grey(0.6),lty=2)
  mtext(paste0("B: Diarrhoea (1M-14Y)"), side = 3, adj = 0.015, cex=1.5)
  
#### Calculate average seasonal trend between 2005-2013
  
  diar_climate$logDaysInMonth = log(Hmisc::monthDays(diar_climate$Date))
  
  diarclim_seasonal = glm(diar_deaths ~ kg_val*(sin12+cos12+sin6+cos6) + offset(logDaysInMonth), family='poisson', data=diar_climate)
  
  diarclim_2009_1 = diar_climate[diar_climate$kg_val==1,][52:64,]
  diarclim_pred_1 = predict(diarclim_seasonal, diarclim_2009_1, type='link', se.fit=TRUE)
  diarclim_pred2_1 = diarclim_pred_1$fit
  
  diarclim_2009_2 = diar_climate[diar_climate$kg_val==2,][52:64,]
  diarclim_pred_2 = predict(diarclim_seasonal, diarclim_2009_2, type='link', se.fit=TRUE)
  diarclim_pred2_2 = diarclim_pred_2$fit
  
  diarclim_2009_3 = diar_climate[diar_climate$kg_val==3,][52:64,]
  diarclim_pred_3 = predict(diarclim_seasonal, diarclim_2009_3, type='link', se.fit=TRUE)
  diarclim_pred2_3 = diarclim_pred_3$fit
  
#### Rate ratio calculations
  
  ## Diarrhea 1M-14Y; Tropical savannah
    
    avg_diarclim1 = diarclim_2009_1
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("kg_val2:",terms,sep=''))
    terms3 = c(terms, paste("kg_val3:",terms,sep=''))
    terms4 = c(terms, paste("kg_val4:",terms,sep=''))
    pred_avg_diarclim1 = avg_diarclim1[,terms] - avg_diarclim1[rep(which(avg_diarclim1$Month=='Apr')[1], 
                                                                      nrow(avg_diarclim1)), terms]
    avg_diarclim1$seasonal1 = as.matrix(pred_avg_diarclim1[,terms]) %*% diarclim_seasonal$coef[terms]
    avg_diarclim1$seasonal2 = as.matrix(pred_avg_diarclim1[,c(terms, terms)]) %*% 
      diarclim_seasonal$coef[terms2]
    avg_diarclim1$seasonal3 = as.matrix(pred_avg_diarclim1[,c(terms,terms)]) %*% 
      diarclim_seasonal$coef[terms3]
    avg_diarclim1$seasonal4 = as.matrix(pred_avg_diarclim1[,c(terms,terms)]) %*%
      diarclim_seasonal$coef[terms4]
    dimnames(diar_climbase$Vp) = list(names(diar_climbase$coef), names(diar_climbase$coef))
    avg_diarclim1$seasonal1.se = sqrt(diag(
      as.matrix(pred_avg_diarclim1[,terms]) %*% 
        diar_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarclim1[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR files
    avg_diar_TSregion <- exp(as.matrix(avg_diarclim1[,c('seasonal1', 'seasonal1.se')]) %*% seMat)
    write.csv(avg_diar_TSregion, file = "avg_diar1mto14yr_AmAw.csv")
  
  ## Diarrhea 1M-14Y; Hot Semi-Arid
    
    avg_diarclim2 = diarclim_2009_2
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("kg_val2:",terms,sep=''))
    terms3 = c(terms, paste("kg_val3:",terms,sep=''))
    terms4 = c(terms, paste("kg_val4:",terms,sep=''))
    pred_avg_diarclim2 = avg_diarclim2[,terms] - avg_diarclim2[rep(which(avg_diarclim2$Month=='Apr')[1], 
                                                                      nrow(avg_diarclim2)), terms]
    avg_diarclim2$seasonal1 = as.matrix(pred_avg_diarclim2[,terms]) %*% diarclim_seasonal$coef[terms]
    avg_diarclim2$seasonal2 = as.matrix(pred_avg_diarclim2[,c(terms, terms)]) %*% 
      diarclim_seasonal$coef[terms2]
    avg_diarclim2$seasonal3 = as.matrix(pred_avg_diarclim2[,c(terms,terms)]) %*% 
      diarclim_seasonal$coef[terms3]
    avg_diarclim2$seasonal4 = as.matrix(pred_avg_diarclim2[,c(terms,terms)]) %*%
      diarclim_seasonal$coef[terms4]
    dimnames(diar_climbase$Vp) = list(names(diar_climbase$coef), names(diar_climbase$coef))
    avg_diarclim2$seasonal2.se = sqrt(diag(
      as.matrix(pred_avg_diarclim2[,terms]) %*% 
        diar_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarclim2[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR files
    avg_diar_HSAregion <- exp(as.matrix(avg_diarclim2[,c('seasonal2', 'seasonal2.se')]) %*% seMat)
    write.csv(avg_diar_HSAregion, file = "avg_diar1mto14yr_BshBwh.csv")
  
  ## Diarrhea 1M-14Y; Humid subtropical
    
    avg_diarclim3= diarclim_2009_3
    terms=c('sin12','cos12','sin6','cos6')
    terms2 = c(terms, paste("kg_val2:",terms,sep=''))
    terms3 = c(terms, paste("kg_val3:",terms,sep=''))
    terms4 = c(terms, paste("kg_val4:",terms,sep=''))
    pred_avg_diarclim3 = avg_diarclim3[,terms] - avg_diarclim3[rep(which(avg_diarclim3$Month=='Apr')[1], 
                                                                      nrow(avg_diarclim3)), terms]
    avg_diarclim3$seasonal1 = as.matrix(pred_avg_diarclim3[,terms]) %*% diarclim_seasonal$coef[terms]
    avg_diarclim3$seasonal2 = as.matrix(pred_avg_diarclim3[,c(terms, terms)]) %*% 
      diarclim_seasonal$coef[terms2]
    avg_diarclim3$seasonal3 = as.matrix(pred_avg_diarclim3[,c(terms,terms)]) %*% 
      diarclim_seasonal$coef[terms3]
    avg_diarclim3$seasonal4 = as.matrix(pred_avg_diarclim3[,c(terms,terms)]) %*%
      diarclim_seasonal$coef[terms4]
    dimnames(diar_climbase$Vp) = list(names(diar_climbase$coef), names(diar_climbase$coef))
    avg_diarclim3$seasonal3.se = sqrt(diag(
      as.matrix(pred_avg_diarclim3[,terms]) %*% 
        diar_climbase$Vp[terms,terms] %*% t(as.matrix(pred_avg_diarclim3[,terms]))))
    seMat = cbind(
      est = c(1,0),
      lower = c(1, qnorm(0.01)),
      upper = c(1, qnorm(0.99))
    )
    
    ## Save RR file
    avg_diar_HSAregion <- exp(as.matrix(avg_diarclim3[,c('seasonal3', 'seasonal3.se')]) %*% seMat)
    write.csv(avg_diar_HSAregion, file = "avg_diar1mto14yr_CwaCwb.csv")

    
##======================================================================================================
## FIGURE 8: Diarrhea Climate Regions - average annual seasonal pattern
##======================================================================================================
    
  figure8_diar1mto14yr_TS_data <- data.frame(x = 1:nrow(diarclim_2009_1), y = exp(diarclim_pred2_1))
  figure8_diar1mto14yr_TS <- qplot(x, y, data = figure8_diar1mto14yr_TS_data, geom = "line") +
  ggtitle("B: Tropical Savannah") + 
  geom_line(size = 1, colour = "steelblue") + 
  expand_limits(x = c(1,13), y = c(0,20)) + 
  scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
  scale_y_continuous(name = "", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
  theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
        axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
        axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
  geom_segment(aes(x=4, y=0, xend=4, yend=12.901367), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_segment(aes(x=10, y=0, xend=10, yend=9.847006), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_point(aes(x=4, y=12.901367), size=2) + 
  geom_segment(aes(x = 4, y = 16.401367, xend = 4, yend = 13.901367), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
  annotate("text", x=4, y=17.401367, label= "RR = 1.51", size = 5) + 
  geom_label(x=7, y=1.5, label="Study deaths = 1068", label.size = NA, size=5)
  
  figure8_diar1mto14yr_HSA_data <- data.frame(x = 1:nrow(diarclim_2009_2), y = exp(diarclim_pred2_2))
  figure8_diar1mto14yr_HSA <- qplot(x, y, data = figure8_diar1mto14yr_HSA_data, geom = "line") +
  ggtitle("C: Hot Semi-Arid") +
  geom_line(size = 1, colour = "orange") + 
  expand_limits(x = c(1,13), y = c(0,20)) + 
  scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
  scale_y_continuous(name = "", breaks = c(0, 5, 10, 15, 20), minor_breaks = NULL, expand=c(0,0)) + 
  theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
        axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
        axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
  geom_segment(aes(x=4, y=0, xend=4, yend=11.847856), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_segment(aes(x=10, y=0, xend=10, yend=9.788298), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_point(aes(x=4, y=11.847856), size=2) + 
  geom_point(aes(x=10, y=9.788298), size=2) + 
  geom_segment(aes(x = 4, y = 15.347856, xend = 4, yend = 12.847856), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
  annotate("text", x=4, y=16.347856, label= "RR = 1.61", size = 5) + 
  geom_segment(aes(x = 10, y = 13.288298, xend = 10, yend = 10.788298), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
  annotate("text", x=10, y=14.288298, label= "RR = 1.32", size = 5) + 
  geom_label(x=7, y=1.5, label="Study deaths = 982", label.size = NA, size=5)
  
  figure8_diar1mto14yr_HST_data <- data.frame(x = 1:nrow(diarclim_2009_3), y = exp(diarclim_pred2_3))
  figure8_diar1mto14yr_HST <- qplot(x, y, data = figure8_diar1mto14yr_HST_data, geom = "line") +
  ggtitle("A: Humid Subtropical") + 
  geom_line(size = 1, colour = "springgreen") + 
  expand_limits(x = c(1,13), y = c(0,46)) + 
  scale_x_continuous(name = "Month", breaks = c(1,4,10,13), minor_breaks = NULL, labels = months, expand=c(0.02,0)) + 
  scale_y_continuous(name = "", breaks = c(0, 10, 20, 30, 40), minor_breaks = NULL, expand=c(0,0)) + 
  theme(panel.background = element_blank(), panel.grid.major = element_blank(), axis.line = element_line(size = 0), 
        axis.ticks.y = element_blank(), axis.text=element_text(size=16), 
        axis.title=element_text(size=16), plot.title = element_text(size = 20)) + 
  geom_segment(aes(x=4, y=0, xend=4, yend=37.03179), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_segment(aes(x=10, y=0, xend=10, yend=31.11784), color=grey(0.7), lwd=1.05, lty="dashed") +
  geom_point(aes(x=4, y=37.03179), size=2) + 
  geom_point(aes(x=10, y=31.11784), size=2) + 
  geom_segment(aes(x = 4, y = 42.03179, xend = 4, yend = 38.03179), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
  annotate("text", x=4, y=44.03179, label= "RR = 1.77", size = 5) + 
  geom_segment(aes(x = 10, y = 36.11784, xend = 10, yend = 32.11784), arrow = arrow(length = unit(0.02, "npc"), type = "closed")) + 
  annotate("text", x=10, y=38.11784, label= "RR = 1.49", size = 5) + 
  geom_label(x=7, y=3, label="Study deaths = 2893", label.size = NA, size=5)
  
  grid.arrange(figure8_diar1mto14yr_HST, figure8_diar1mto14yr_TS, figure8_diar1mto14yr_HSA, ncol=3)



  