%race_model_demo Race-to-threshold model for compelled-anti-saccade task
%
% In the CAS task model, the activities of two neuronal populations
% begin building up at random rates (rL and rR, below). When the cue
% is detected, both motor plans are briefly interrupted. The plan
% toward the cue is interrupted very briefly, and is then accelerated
% by the cue (the neural correlate of attentional capture). The plan
% toward the target is interrupted for a somewhat longer period. At
% the end of the interruption/cue-capture interval, the plan toward
% the cue (incorrect location) is decelerated and that toward the
% (correct) anti location is accelerated; this corresponds to the
% location of the cue being identified and interpreted according to
% the rules of the antisaccade task.
%
% This script can run in two modes:
%
% -- Normally, if the variable do_batch does not exist (or is equal to
%    zero), individual trials are run one by one. Plots are shown on
%    two windows while the script is running. Figure 1 shows the
%    simulated neural responses, xR and xL, as functions of time in
%    each single trial, whereas Figure 2 shows the updated
%    psychometric data (psychometric, chronometric, and tachometric
%    curves) for all trials accumulated so far. In this mode, hitting
%    enter runs one trial, whereas entering an integer n runs that
%    number of trials and updates Figure 2. Enter 0 trials to exit the
%    script.
%
% -- Alternatively, when the variable do_batch exists and is set to an
%    integer n, n trials are run without any graphical output. This is
%    considerably faster.
%
% Regardless of the value of do_batch, once the script finishes
% running, all the "psychophysical" results are saved in an array
% called "data", which contains one trial per row, where each row
% includes four columns
%
%   | gap | RT | hit {1,0} | target location {-1,1}
%
% where RT is the reaction time, the variable hit is 0 for incorrect
% and 1 for correct choices, and target location can be -1 (for target
% left/cue right) or +1 (for target right/cue left). The target
% location is determined by the parameter tloc below. Recall that the
% raw processing time, or cue viewing time is 
%
%   rPT = RT - gap
%
% Note that, here, the afferent delays for each trial are precisely
% known, so we know what the effective processing time (ePT) is; this
% is the actual cue viewing time discounting the transmission delays,
% so
%
%   ePT = RT - gap - T_afferent_cue - T_efferent 
%
%       = rPT - T_afferent_delay - T_efferent
%
% The ePT is marked and easy to see in the single-trial plots of
% Figure 1 --- but all analyses on the experimental (and simulated)
% data are based on rPT, which is the measurable quantity.
%
% In addition to the three plots generated by this script while it is
% running, the data can be visualized in more detail by calling
%
% >> psych_analysis_demo(data)
%
% at the Matlab prompt, once this script is finished.
%
% The plots look better with a black background, which can be set by
% calling
%
% >> close all
% >> colordef none
%
% before running this script.

% Copyright (c) 2018 Emilio Salinas

 %====================
 % set race parameters
 %====================

 %
 % primary parameters; these vary by participant and luminance
 % condition (high, medium, or low); whichever line is uncommented
 % runs the corresponding condition
 %

 %
 % Pooled data 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 race_par = [1.4 14 -0.95 0.17 -0.70 51 36  76  5 24  4 10 0 0.96 0.010]; 
 %race_par = [1.4 14 -0.95 0.17 -0.54 51 36 104 13 24  3 14 0 1.15 0.010]; 
 %race_par = [1.4 14 -0.95 0.14 -0.29 51 36 126 19 24 10 14 0 0.58 0.050]; 

 % hi lum, constrained to zero cue-driven acceleration
 %race_par = [2.1 12.6 -0.92 0.24 -0.46 49 41 79 8 41 11 0 0 0 0.020]; 

 % hi lum, constrained to no interruption of target plan
 %race_par = [1.1 15.3 -0.98 0.18 -0.65 53 43 70 8 29 2 15 1 1.03 0.010]; 

 %
 % Participant 6 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [4.5 5.6 -0.81 0.16 -0.90 44 34  88  7 16  1 6 -0.5 0.60 0.015]; 
 %race_par = [4.5 5.6 -0.81 0.19 -0.80 44 34 113 13 17  2 8 -0.8 0.55 0.025]; 
 %race_par = [4.5 5.6 -0.81 0.09 -0.64 44 34 129 10 11 13 7 -0.4 0.50 0.180]; 

 %
 % Participant 5 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [0.5 3.6 -0.95 0.13 -1.27 46 47  67  8 20 14 13 -0.6 1.50 0.015]; 
 %race_par = [0.5 3.6 -0.95 0.11 -0.65 46 47  91  7 17  9  9 -0.3 1.36 0.020]; 
 %race_par = [0.5 3.6 -0.95 0.13 -0.87 46 47 117 12 24 12  7 -0.1 0.62 0.010]; 

 %
 % Participant 4 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [2.3 3.3 -0.92 0.11 -0.68 54 35  78  6 24 5 12 -0.4 0.81 0.015]; 
 %race_par = [2.3 3.3 -0.92 0.11 -0.22 54 35 105  5 21 4 16 -0.2 1.00     0]; 
 %race_par = [2.3 3.3 -0.92 0.10 -0.23 54 35 129 19 24 7 11 -0.4 0.43     0]; 

 %
 % Participant 3 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [4.1 16.0 -0.92 0.36 -0.60 48 40  76  8 20 2 2 -0.6 0.23 0.000]; 
 %race_par = [4.1 16.0 -0.92 0.39 -0.58 48 40 104 15 24 2 1 -0.5 0.22 0.000]; 
 %race_par = [4.1 16.0 -0.92 0.15 -0.20 48 40 128  4  9 8 3 -0.1 0.42 0.010]; 

 %
 % Participant 2 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [2.6 1.8 -0.97 0.09 -1.05 55 41  75  2 14 8 5 -0.9 1.86 0.000]; 
 %race_par = [2.6 1.8 -0.97 0.08 -0.72 55 41 111  4 17 6 4 -0.1 1.05 0.000]; 
 %race_par = [2.6 1.8 -0.97 0.04 -0.22 55 41 154 22 16 6 9  0.0 0.76 0.075]; 

 %
 % Participant 1 
 %
 % full series: hi, med, and lo luminance with identical motor parameters
 %race_par = [1.7 2.9 -0.91 0.18 -0.59 47 40  73 11 26 6 15 0 0.90 0.005]; 
 %race_par = [1.7 2.9 -0.91 0.16 -0.18 47 40 100 20 22 2 17 0 1.03 0.000]; 
 %race_par = [1.7 2.9 -0.91 0.11 -0.27 47 40 114 29 22 9 10 0 0.44 0.035]; 

 %
 % definitions of main race parameters
 %
 race_par_names = { ...
 'rG'; ...       % mean of stage 1 accumulation rate 
 'sdG'; ...      % variance of stage 1 accumulation rate
 'rhoG'; ...     % correlation tw winner and loser rates
 'AccT'; ...     % acceleration for target (threshold units/ms^2)
 'AccD'; ...     % acceleration for distracter (threshold units/ms^2)
 'Ta1'; ...      % mean afferent delay of GO signal (ms)
 'sdTa1'; ...    % SD of afferent delay of GO signal (ms)
 'Ta2'; ...      % mean afferent delay of CUE detection (ms)
 'sdTa2'; ...    % SD of afferent delay of CUE detection (ms)
 'Ilen'; ...     % interruption duration for target plan (ms)
 'sdIlen'; ...   % SD of interruption duration for target plan (ms)
 'Ilen1'; ...    % interruption duration before cue acceleration (ms)
 'vfacI'; ...    % build-up attenuation factor during interrupt 1
 'AccI'; ...     % acceleration of cue plan during interrupt 2
 'Pconf'; ...    % probability of confusion (lapse rate)
 };

 %
 % secondary parameters; for the data sets above, they are always the
 % same 
 %
 second_par = [600 0 20 -8 20 20 0.333 0 2];

 %
 % definitions of secondary parameters
 %
 second_par_names = {...
 'RTmax'; ...            % post simulation: max acceptable RT (in ms)
 'RTmin'; ...            % post simulation: min acceptable RT (in ms)
 'rT'; ...               % max uninformed build-up rate for target (must be > 0)
 'rD'; ...               % min uninformed build-down rate for distracter (must be < 0)
 'Teff'; ...             % efferent delay
 'Taff_min'; ...         % minimum afferent delay (visual latency)
 'sdAccT_fac'; ...       % defines SD of AccT = AccT*sdAccT_fac
 'xdip'; ...             % minimum allowed activity level
 'lapse_type'; ...       % {1,2} --> wrong (1) vs undefined (2) spatial assignments
 };

 % assign parameter values in struct fields
 race_par_s = cell2struct(num2cell(race_par), race_par_names, 2);
 second_par_s = cell2struct(num2cell(second_par), second_par_names, 2);

 % create variables in workspace with respective names
 unpack_struct(race_par_s)
 unpack_struct(second_par_s)

 %
 % run parameters
 %
 Tslow = 0.0001;       % pause length (for single trials)
 withchirp = 0;        % play sounds on hit/error (for single trials)
 do_maxlen_warn = 0;   % warn that maximum trial length was exceeded

 % normally, the simulation starts with N1 trials that are not shown,
 % and then becomes interactive; however, if the parameter do_batch
 % exists and is larger than zero, then do_batch trials of the model
 % are run somewhat more quickly and without any graphics; to use this
 % option to simulate, say, 10,000 trials, simply type
 % 
 % >> do_batch = 10000;
 % >> race_model_demo;
 % 
 % at the Matlab prompt, and then
 % 
 % >> psych_analysis_demo(data)
 % 
 if ~exist('do_batch', 'var') || (do_batch <= 0)
     N1 = 20;      
     do_batch = 0;
     Ntrial_disp = 1000;
     do_skip_guess = 0;    % show rise to threshold from the start 
 else
     N1 = do_batch;   
     str1 = num2str(N1);
     Ntrial_disp = 10000;
     do_skip_guess = 1;    % do not integrate linear part of rise to threshold
 end

 %
 % other parameters
 %
 x_th = 1000;      % threshold for race processes
 tloc = +1;        % -1 --> target on the left, +1 --> target on the right 
 Tmax = 550;       % single trial max duration (for plotting)
 Tlead = 100;      % lead time from trial start to GO signal
 
 % gap: independent variable, in ms
 gap = [0 75:25:200 250 350];
 Ngap = length(gap);   

 %
 % output arrays
 %
 data = zeros(N1,4);       % saved run results go here
 rBdata = zeros(N1,2);     % saved initial build-up rates

 %====================
 % initialize graphics
 %====================
 
 if do_batch == 0
	 % parameters for racer's plot
	 shad1 = 0.12*[1 1 1];
	 lw1 = 1;
	 lw2 = 3;
	 msz1 = 8;
     clL = [1 0 0];       % color for left-saccade population
     clR = [0.25 0 1];    % color for right-saccade population
	 cl1 = 0.7*[1 1 0];
	 ylo = -0.3*x_th;
	 yhi = 1.6*x_th;
	 ylo1 = 0.996*ylo;
	 y1 = x_th*1.00;      % for Go and Sac triangles
	 y2 = x_th*1.02;      % for Go and Sac labels
	 y3 = 0.8*ylo;        % for Gap label
	 y4 = x_th*1.07;      % for ePT label
	 y5 = x_th*1.04;      % for ePT length bar
	 % parameters for subject's display
	 xd1 = 160;
	 xd2 = 240;
	 yd1 = yhi - 350;
	 yd2 = yhi;
	 xd3 = (xd1 + xd2)/2;
	 yd3 = (yd1 + yd2)/2;
	 xd4 = xd3 - (xd2 - xd1)/2*0.7; 
	 xd5 = xd3 + (xd2 - xd1)/2*0.7; 
	 xd4a = xd3 - (xd2 - xd1)/2*0.58; 
	 xd5a = xd3 + (xd2 - xd1)/2*0.58; 
	 lw3 = 2;
	 szt = 40;
	 clc = [0 1 0];          % cue color
	 shad3 = 0.30*[1 1 1];   % background 'monitor' color
     clct = shad3*0.90;      % color of choice targets
	 % initialize figure windows
	 figure(2)
	 clf
	 figure(1)
	 clf
	 axis([-Tlead Tmax ylo yhi])
	 set(gca, 'tickdir', 'out', 'ytick', [0 x_th])
	 hold on
 end

 %================================
 % run model: loop over gap values
 %================================

 do_show=0; 
 go=N1;
 j1=0; jj=0;
 jdel=[];
 while go > 0
     % trial number
     jj = jj + 1;
     % choose a gap
     j1 = j1 + 1;
     if j1 > Ngap
         j1 = 1;
     end
     Tgap = gap(j1);
     % afferent delay of go signal
     Taff1 = Ta1 + sdTa1*randn;
     if Taff1 < Taff_min
         Taff1 = Taff_min + abs(Taff1 - Taff_min);
     end
     Taff1 = round(Taff1);
     % afferent delay of cue detection
     Taff2 = Ta2 + sdTa2*randn;
     if Taff2 < Taff_min
         Taff2 = Taff_min + abs(Taff2 - Taff_min);
     end
     Taff2 = round(Taff2);
     % cue detection time
     Tdet = Tgap + Taff2;
     % afferent delay of cue interpretation (i.e., amount of time
     % needed to determine spatial location)
     Taff3 = Taff2 + Ilen + sdIlen*randn;
     Taff3 = round(Taff3);
	 % cue interpretation time: point at which final (top-down)
	 % acceleration/deceleration starts
     Tacc = Tgap + Taff3;
	 % avoid inconsistencies between event time points; endogenous
	 % acceleration/deceleration always comes after cue detection
     if Tdet < Taff1
         Tdet = Taff1 + 1;
     end
     if Tacc < Tdet
         Tacc = Tdet + 1;
     end
     % effective gap for the trial
     Tgap1 = Tacc - Taff1;
	 % draw initial build-up rates (w/o sensory evidence) using
	 % anticorrelated Gaussians
	 rr = mvnrnd(rG*[1 1],sdG*[1 rhoG; rhoG 1], 1);
     % uncomment to make the build-up rates constant across all trials
     %rr = [1.5 4.0];
     % apply hard limits on build-up rates
     rr(rr > rT) = rT;
	 rr(rr < rD) = rD;
     % assign L/R build-up rates
	 rR = rr(1);
	 rL = rr(2);
     % set acceleration/deceleration
     if tloc > 0
	     aR = AccT + (AccT*sdAccT_fac)*randn;
         aR(aR < 0) = AccT;
         aL = AccD;
     else
         aR = AccD;
         aL = AccT;
	     aL = AccT + (AccT*sdAccT_fac)*randn;
     end
	 % account for the effect of lapses; was the subject confused in
	 % this trial?
     yes_lapse=0;
     if Pconf > 0
         if lapse_type == 1
		     % type 1 lapse: target/cue locations are reversed
			 if rand < Pconf
				 temp = aR;
				 aR = aL;
				 aL = temp;
                 yes_lapse = 1;
			 end
         elseif lapse_type == 2
		     % type 2 lapse: target location is never defined
			 if rand < 2*Pconf
                 aR = 0;
                 aL = 0;
                 yes_lapse = 1;
             end
         else
             error('Invalid lapse_type value ({1,2})')
         end
     end
     % refresh graph if it's an individual trial
     if do_show == 1
         figure(1), cla
         xlabel('Time (ms)')
         ylabel('\itx_L, x_R')
         plot([-Tlead -Tlead], [x_th*1.002 yhi], 'k-', 'linewidth', 2)
		 hr = rectangle('position', [Taff1 ylo1 (Tacc-Taff1) (x_th-ylo1)]); 
		 set(hr, 'edgecolor', shad1, 'facecolor', shad1)
         plot([-Tlead Tmax], x_th*[1 1], 'w-')
         plot([0 0], [ylo y1], 'm:', 'linewidth', lw1)
         plot(0, y1, 'vm', 'markersize', msz1, 'markerfacecolor', 'm')
         text(0, y2, 'Go', 'VerticalAlignment', 'bottom', ...
              'HorizontalAlignment', 'center')
         mssg = ['Gap = ' num2str(Tgap)];
         text(Taff1+Tgap/2, y3, mssg, 'VerticalAlignment', 'middle', ...
              'HorizontalAlignment', 'center')
         % subject's display
         hr = rectangle('position', [xd1 yd1 (xd2-xd1) (yd2-yd1)]); 
         set(hr, 'edgecolor', 'w', 'facecolor', shad3, 'linewidth', lw3, ...
             'clipping', 'off')
         plot(xd3, yd3, '.', 'color', clc, 'markersize', szt)
         plot([xd4 xd5], yd3, '.', 'color', clct, 'markersize', szt)
     end
     % set default initial conditions
     Tstart = -Tlead;
	 vL=rL; vR=rR;
	 xL=0; xR=0; 
     % when shall integration over time start?
     if do_skip_guess
		 % start integrating at interruption onset, or at threshold
		 % crossing, if the trial is a guess; this makes the program
		 % run quite faster
		 if rL > 0
			 txL = Taff1 + x_th/rL;
		 else
			 txL = Inf;
		 end
		 if rR > 0
			 txR = Taff1 + x_th/rR;
		 else
			 txR = Inf;
		 end
		 Tstart = min([txL txR Tdet]) - 1;
         % revise initial conditions
         if Tstart > Taff1
		     xL = rL*(Tstart - Taff1); 
             xR = rR*(Tstart - Taff1); 
             if xL < xdip
                 xL = xdip;
             end
             if xR < xdip
                 xR = xdip;
             end
         end
     end
     %------------------------------------
     % now advance one time step at a time
     %------------------------------------
     k=0; 
     for t=Tstart:2*Tmax
         % before cue detection
		 if (t >= Taff1) && (t < Tdet)
             xL = xL + rL;
             xR = xR + rR;
         end
         % after detection, during interruption state 1, plans advance
         % at lower rates, if at all
         if vfacI
             if (t >= Tdet) && (t < Tdet + Ilen1)
                 xL = xL + rL*vfacI;
                 xR = xR + rR*vfacI;
             end
         end
         % after detection, during interruption state 2, cue plan
         % accelerates, target plan advances slowly, or not at all
		 if (t >= Tdet + Ilen1) && (t <= Tacc)
			 if tloc > 0
                 vL = vL + AccI;
                 xL = xL + vL + 0.5*AccI;
                 if vfacI
                     xR = xR + rR*vfacI;
                 end
             else
                 vR = vR + AccI;
                 xR = xR + vR + 0.5*AccI;
                 if vfacI
                     xL = xL + rL*vfacI;
                 end
             end
         end
		 % after spatial discrimination, during top-down control
         if t > Tacc
			 if tloc > 0
                 % cue plan decelerates
                 vL = vL + aL;
                 xL = xL + vL + 0.5*aL;
                 % target plan accelerates
                 vR = vR + aR;
                 xR = xR + vR + 0.5*aR;
             else
                 % cue plan decelerates
                 vR = vR + aR;
                 xR = xR + vR + 0.5*aR;
                 % target plan accelerates
                 vL = vL + aL;
                 xL = xL + vL + 0.5*aL;
             end
         end
         % don't allow dips in activity below min value
         if xL < xdip
             xL = xdip;
         end
         if xR < xdip
             xR = xdip;
         end
         % check for threshold crossing
         if any([xL xR] >= x_th)
             % break any ties; very unlikely...
             if xL == xR
                 xL = xL + (-1 + 2*(rand < 0.5));
             end
             if xL > xR
                 tL = t;
                 tR = Inf;
             else
                 tR = t;
                 tL = Inf;
             end
             break
         end
         % update the graph
         if do_show == 1
             % racers
             plot(t, xL, '.', 'color', clL)
             plot(t, xR, '.', 'color', clR)
             if Tslow > 0
                 k = k + 1;
                 if k > 5
				     % using a sizable Tslow might be necessary if the
					 % display is too fast
                     pause(Tslow)
                     k = 0;
                 end
             end
             % subject's monitor display
             if t == 0
                 % erase the the choice targets, if they were shown
                 plot(xd3, yd3, '.', 'color', shad3, 'markersize', szt)
             end
             if t == Tgap
                 % reveal the cue
                 if tloc == -1
                     plot(xd5, yd3, '.', 'color', clc, 'markersize', szt)
                 else
                     plot(xd4, yd3, '.', 'color', clc, 'markersize', szt)
                 end
             end
         end
     end
     % trial timed out
     if t == 2*Tmax
         tL = Inf;
         tR = Inf;
         jdel = [jdel; jj];
     end
     % see if the choice was correct, and compute reaction time
     if tL < tR 
         RT = tL;
         if tloc == -1
             hit = 1;
         else
             hit = 0;
         end
     else
         RT = tR;
         if tloc == 1
             hit = 1;
         else
             hit = 0;
         end
     end
     % add efferent delay to RT
     RT = RT + Teff;
     % store trial results 
     data(jj,:) = [Tgap RT hit tloc];
     rBdata(jj,:) = [rL rR];
     % after threshold crossing, draw saccade point & ePT box
     if do_show == 1
         % draw saccade time point
         plot((t+Teff)*[1 1], [ylo y1], 'm:', 'linewidth', lw1)
         plot(t+Teff, y1, 'vm', 'markersize', msz1, 'markerfacecolor', 'm')
         % draw saccade arrow
         if tL < tR
             plot([xd3 xd4a], [yd3 yd3], 'k-', 'linewidth', lw3)
             plot(xd4a, yd3, 'k<', 'markerfacecolor', 'k', 'markersize', msz1)
         else
             plot([xd3 xd5a], [yd3 yd3], 'k-', 'linewidth', lw3)
             plot(xd5a, yd3, 'k>', 'markerfacecolor', 'k', 'markersize', msz1)
         end
         % draw ePT box for the actual amount of cue viewing time 
         ePT1 = RT - Tacc - Teff;
         if ePT1 > 0
             text(t+Teff, y2, 'Sac', 'VerticalAlignment', 'bottom', ...
                  'HorizontalAlignment', 'center')
             hr = rectangle('position', [Tacc ylo1 ePT1 (x_th-ylo1)]); 
             set(hr, 'edgecolor', 'y', 'facecolor', 'none', 'linewidth', lw1)
             plot([Tacc Tacc+ePT1], [y5 y5], 'y-', 'linewidth', lw2)
         elseif ePT1 < 0
             hr = rectangle('position', [Tacc+ePT1 ylo1 -ePT1 (x_th-ylo1)]); 
             set(hr, 'edgecolor', cl1, 'facecolor', 'none', 'linewidth', lw1)
             plot([Tacc Tacc+ePT1], [y5 y5], '-', 'linewidth', lw2, 'color', cl1)
         end
         mssg = ['ePT = ' num2str(ePT1)];
         text(t-ePT1/2, y4, mssg, 'VerticalAlignment', 'bottom', ...
              'HorizontalAlignment', 'center')
        % draw last points
        plot(t, xL, '.', 'color', clL)
        plot(t, xR, '.', 'color', clR)
     end
     % display individual trial results on command window
     if do_show == 1
         mssg = ['trial ' num2str(jj)];
         disp(mssg)
         if yes_lapse
             mssg = '  a lapse occurred!';
             disp(mssg)
         end
         mssg = ['  gap = ' num2str(Tgap) ' (' num2str(Tgap1) ')'];
         disp(mssg)
         mssg = ['  RT = ' num2str(RT)];
         rPT = RT - Tgap;
         disp(mssg)
         mssg = ['  rPT = ' num2str(rPT)];
         disp(mssg)
         mssg = ['  ePT = ' num2str(ePT1)];
         disp(mssg)
         if hit
             disp('  hit')
             if withchirp 
                 chirp
             end
         else
             disp('  error')
             if withchirp 
                 chirp1
             end
         end
     else
         if rem(go, Ntrial_disp) == 0
             mssg = ['trials left: ' num2str(go) '/' str1];
             disp(mssg)
         end
     end
     % update second figure window with psychometric curves
     if (go == 1) && (do_batch == 0)
         figure(2)
         psych_analysis_demo(data, 1)
     end
     % check user's next instruction, ie., number of trials to do next
     if go == 1
         if do_batch 
             break
         else
			 ui = input('number of trials (return=1): ');
			 if isempty(ui) || ui == 1
				 go = 1;
				 do_show = 1;
			 elseif ui <= 0
				 break
			 else
				 go = ui;
				 str1 = num2str(go);
				 do_show = 0;
			 end
			 data = [data; zeros(go,4)];   
			 rBdata = [rBdata; zeros(go,2)];
         end
     else
         go = go - 1;
     end
 end

 %===========================
 % post-simulation processing
 %===========================

 % eliminate trials that did not reach threshold; i.e., timed out
 Ndel = length(jdel);
 if Ndel
     data(jdel,:) = [];
     rBdata(jdel,:) = [];
     disp([num2str(Ndel) ' trials timed out before reaching threshold'])
 end

 % eliminate trials that exceed a max RT value or do not meet a
 % minimum
 rt = data(:,2);
 jdel1 = find(rt > RTmax);
 jdel2 = find(rt < RTmin);
 Ndel1 = length(jdel1);
 Ndel2 = length(jdel2);
 if Ndel1 || Ndel2
     jdel = union(jdel1, jdel2);
     data(jdel,:) = [];
     rBdata(jdel,:) = [];
     disp([num2str(Ndel2) ' trials did not meet the minimum RT allowed (' ...
           num2str(RTmin) ' ms)'])
     disp([num2str(Ndel1) ' trials exceeded the maximum RT allowed (' ...
           num2str(RTmax) ' ms)'])
 end

 % farewell message
 disp('results stored in matrix "data"')
 disp('use psych_analysis_demo(data) to view them')
 disp([char(10) 'That''s all folks!' char(10)])



