function [] = psych_analysis_demo(data, style)

%psych_analysis_demo Analysis of data from compelled antisaccade task
%
% [] = psych_analysis_demo(data, [style])
%
% This function generates 
%
%    (1) psychometric, chronometric and tachometric curves
%    (2) processing time histograms (correct and error) for all trials
%    (3) reaction time histograms (correct and error) for each gap
%
% The curves are calculated first and then are displayed.
%
% All this is computed from a matrix called "data", which is the
% result of running race_model_demo. 
%
% This function runs in two modes, according to its second, optional
% parameter:
%
%    style = 1 --> only psychometric, chronometric and tachometric
%                  curves are shown, all in one shot
%
%    style = 2 --> the above plus all RT histograms are shown, one by
%                  one (default)
%
% The style = 1 option is used by race_model_demo. To see the full
% analysis, after running race_model_demo, call psych_analysis_demo
% from the Matlab prompt like this:
%
% >> psych_analysis_demo(data)
%
% All time axes are assumed to vary in steps of 1 ms. 

% Copyright (c) 2018 Emilio Salinas

 if nargin < 2
     style = 2;
 end

 if style == 1
     verbose = 0;
 else
     verbose = 1;
 end

 % This is the mean non-decision time; when it is zero, the processing
 % time is the "raw" one, the maximum cue viewing time, which is what
 % is normally used. When Tnd is the average total delay (cue afferent
 % delay, which is ~75 ms for high luminance, plus efferent delay,
 % which is ~20 ms), the processing time axis is shifted to
 % approximate the "effective" processing time, or actual cue viewing
 % time. This can only be an approximation because the afferent delays
 % vary across trials.
 Tnd = 0;

 % get rid of any Inf values
 noInfs = 1;

 if noInfs
     ii = find(isinf(data(:,2)));
     if ~isempty(ii)
         data(ii,:) = [];
     end
 end 

 %=============================
 % compute all the curves first
 %=============================

 %
 % some analysis parameters
 %
 wbin1 = 15;                  % bin width for PT axis (ms) 
 PTrng = [-200 400] - Tnd;    % PT range for histograms
 PTrng1= [-100 300] - Tnd;    % PT range for tachometric curve
 wbin2 = 41;                  % bin width for RT histograms (ms)
 RTrng = [0 600];             % range for RT histograms

 % min number of trials per PT bin
 N1 = size(data,1);
 if N1 > 200
     Nmin = 20;
 elseif N1 > 100
     Nmin = 10;
 else
     Nmin = 5;
 end
 str1 = num2str(N1);
    
 %
 % extract data; data columns are
 %
 %  gap | RT | hit {0,1} | target location {-1,1} 
 % 
 %
 gap1 = data(:,1);
 rt = data(:,2);
 hit = (data(:,3) > 0);
 tloc = data(:,4);

 gap = [unique(gap1)]';
 Ng = length(gap);
 ept = rt - gap1 - Tnd;

 %
 % produce mean data as a function of gap
 %
 if verbose
     disp('computing psychometric and chronometric curves...')
 end
 mRT = zeros(1,Ng);   % mean RT
 sdRT = zeros(1,Ng);  % SD of RT
 pC = zeros(1,Ng);    % probability correct
 Nt = zeros(1,Ng);    % number of trials
 for j=1:Ng
     ii = (gap1 == gap(j));
     mRT(j) = nanmean(rt(ii));
     sdRT(j) = nanstd(rt(ii));
     pC(j) = sum(hit(ii))/sum(ii);
     Nt(j) = sum(ii);
 end
 if verbose
     disp('done')
 end

 %
 % generate PT distributions and tachometric curve
 %
 % Note that here we combine the functions hist and conv to produce
 % running histograms; this is fast.
 %
 if verbose
     disp('computing PT distributions and tachometric curve...')
 end
 % processing times (ms)
 PTbin = PTrng(1):1:PTrng(2); 
 % define square sliding window for averaging
 fwin = ones(wbin1,1);
 % PT distribution for correct trials
 temp = hist(ept(hit), PTbin);
 PTc = conv(temp, fwin, 'same');
 % PT distribution for error trials
 temp = hist(ept(~hit), PTbin);
 PTe = conv(temp, fwin, 'same');
 % tachometric curve
 PTn = PTc + PTe;
 PTpC = NaN*PTc;
 ii = (PTn >= Nmin);
 PTpC(ii) = PTc(ii)./PTn(ii);
 if verbose
     disp('done')
     drawnow
 end
 
 if style > 1
     %
     % generate 2 RT distributions per gap
     %
     if verbose
         disp('computing RT histograms...')
     end
     % RTs (ms)
     RTbin = RTrng(1):1:RTrng(2);  
     Nbins = length(RTbin);
     RTc = zeros(Ng,Nbins);       % for correct trials
     RTe = zeros(Ng,Nbins);       % for error trials
     % define square sliding window for averaging
	 fwin = ones(wbin2,1);
     % loop over gaps
	 for j=1:Ng
		 ii = (gap1 == gap(j));
		 % for correct trials
		 temp = hist(rt(ii & hit), RTbin);
		 RTc(j,:) = conv(temp, fwin, 'same');
		 % for incorrect trials
		 temp = hist(rt(ii & ~hit), RTbin);
		 RTe(j,:) = conv(temp, fwin, 'same');
	 end
     if verbose
         disp('done')
     end
 end

 %=====================
 % now plot the results
 %=====================

 %
 % these are the relevant output (processed) quantities:
 %
 % gap    --> unique gap values
 % pC     --> probability correct as a function of gap
 % mRT    --> mean RT as a function of gap
 % sdRT   --> SD in RT as a function of gap
 % Nt     --> number of trials per gap
 % PTbin  --> possible PT values
 % PTpC   --> probability correct as a function of PT
 % PTc    --> number of correct trials as a function of PT
 % PTe    --> number of error trials as a function of PT
 % RTbin  --> possible RT values
 % RTc    --> matrix with numbers of RTs per bin value for each gap,
 %            for correct trials
 % RTe    --> matrix with numbers of RTs per bin value for each gap,
 %            for error trials

 if style ==  1
     % Psychometric curve: P(correct) vs gap (in ms)
     subplot(1,3,1)
     cla
     hold on
     plot([min(gap)-5 max(gap)+5], [0.5 0.5], 'w:')
     ylim([min(min(pC), 0.4) 1.05])
     xlim([min(gap)-5 max(gap)+5]);
     plot(gap, pC, 'ro-')
     set(gca, 'box', 'on')
     ylabel('P(correct)')
     xlabel('Gap (ms)')
     title(['Psychometric curve after ' str1 ' trials'])

     % chronometric curve: <RT> vs gap (in ms)
     subplot(1,3,2)
     cla
     hold on
     line([gap; gap],[mRT-sdRT; mRT+sdRT],'color', 'b')
     xlim([min(gap)-5 max(gap)+5]);
     plot(gap, mRT, 'co-')
     set(gca, 'box', 'on')
     xlabel('Gap (ms)')
     ylabel('RT (ms)')
     title(['Chronometric curve after ' str1 ' trials'])

     % tachometric curve: P(correct) vs processing time
     subplot(1,3,3)
     cla
     hold on
     xlim([PTrng1]);
     ylim([0 1.05]);
     plot(PTrng1, [0.5 0.5], 'w:')
     plot(PTbin, PTpC, 'mo-')
     set(gca, 'box', 'on')
     if Tnd == 0
         xlabel('Raw processing time (ms)')
     else
         xlabel('Effective processing time (ms)')
     end
     ylabel('P(correct)')
     title(['Tachometric curve after ' str1 ' trials'])
 else
     % Psychometric curve: P(correct) vs gap (in ms)
     clf
     subplot(1,2,1)
     hold on
     plot([min(gap)-5 max(gap)+5], [0.5 0.5], 'w:')
     ylim([min(min(pC),0.4) 1.05])
     xlim([min(gap)-5 max(gap)+5]);
     plot(gap, pC, 'ro-')
     set(gca, 'box', 'on')
     xlabel('Gap (ms)')
     ylabel('P(correct)')
     title(['Psychometric curve after ' str1 ' trials'])

     % chronometric curve: <RT> vs gap (in ms)
     subplot(1,2,2)
     hold on
     line([gap; gap],[mRT-sdRT; mRT+sdRT],'color', 'r')
     xlim([min(gap)-5 max(gap)+5]);
     plot(gap, mRT, 'ro-')
     set(gca, 'box', 'on')
     xlabel('Gap (ms)')
     ylabel('RT (ms)')
     title(['Chronometric curve after ' str1 ' trials'])
     hold off
     disp('press a key...')
     pause

     % tachometric curve: P(correct) vs processing time
     clf
     hold on
     ii = (isnan(PTpC) == 0);
     xlim([PTrng1]);
     plot(PTrng1, [0.5 0.5], 'w:')
     %xlim([min(PTbin(ii))-5 max(PTbin(ii))+5]);
     %plot([min(PTbin(ii))-5 max(PTbin(ii))+5], [0.5 0.5], 'w:')
     plot(PTbin, PTpC, 'ro-')
     set(gca, 'box', 'on')
     ylim([0 1.05]);
     if Tnd == 0
         xlabel('Raw processing time (ms)')
     else
         xlabel('Effective processing time (ms)')
     end
     ylabel('P(correct)')
     title(['Tachometric curve after ' str1 ' trials'])
     hold off
     disp('press a key...')
     pause

     % overall distributions of processing times
     clf
     hold on
     xlim(PTrng)
     plot(PTbin, PTc/max(PTc),'r.-')
     plot(PTbin, PTe/max(PTc),'g.-')
     set(gca, 'box', 'on')
     if Tnd == 0
         xlabel('Raw processing time (ms)')
         title('rPT distributions')
     else
         xlabel('Effective processing time (ms)')
         title('ePT distributions')
     end
     ylabel('Normalized frequency')
     plot(PTrng(1)+[30 80], [0.9 0.9], 'r-', 'linewidth', 2)
     text(PTrng(1)+90, 0.9, 'correct trials', 'verticalalignment', 'middle')
     plot(PTrng(1)+[30 80], [0.8 0.8], 'g-', 'linewidth', 2)
     text(PTrng(1)+90, 0.8, 'error trials', 'verticalalignment', 'middle')
     hold off
     disp('press a key...')
     pause

     % reaction time distributions for hit and error trials
     clf
     for j=1:length(gap)
         nfac = max([RTc(j,:) RTe(j,:)]);
         plot(RTbin, RTc(j,:)/nfac, 'r.-', RTbin, RTe(j,:)/nfac, 'g.-')
         set(gca, 'box', 'on')
         xlabel('Reaction time (ms)')
         ylabel('Normalized frequency')
         mssg = ['RT distributions, gap = ' num2str(gap(j)) ' ms'];
         title(mssg)
         if j < length(gap)
             disp('press a key...')
             pause
         end
     end
     disp(['done' char(10)])
 end



 
