#ifndef CPM_Cell_hpp
#define CPM_Cell_hpp

#include <algorithm>

#include <gsl/gsl_rng.h>

#include "CPM_Tile.hpp"
#include "CPM_Edge.hpp"

#include "IO_Parameters.hpp"

class Cell {

public:

    /*!
     *  Migratory state of the cell.
     */
    enum class MotilityState {Motile, Immotile};

    /*!
     *  Proliferation state of the cell.
     */
    enum class ProliferationState {Presimulation, Quiescence, Growing, Dividing, DivisionSignal};

    /*!
     *  Struct storing the eigenvalues and eigenvectors correspinding to the Principal Component Analysis of the Cell.
     */
    struct PrincipalComponents {
        double_t Eigenvalue1;   ///< eigenvalue (>)
        double_t Eigenvalue2;   ///< eigenvalue (<)
        Vector Eigenvector1;    ///< eigenvector (>)
        Vector Eigenvector2;    ///< eigenvector (<)
    };

    /*!
     *  Default constructor is not allowed.
     */
    Cell() = delete;

    /*!
     *  Copy constructor is not allowed.
     */
    Cell(Cell& base) = delete;

    /*!
     *  Constructor of the Cell.
     *  @param identifier index of the new Cell
     *  @param tile Tile that the cell is seeded on
     *  @param parameters Parameters of the Cell
     *  @param RNG pointer to the random number generator
     */
    Cell(size_t identifier, Tile* tile, Parameters* parameters, gsl_rng* RNG)
    : m_X(tile->m_X)
    , m_MotilityState(MotilityState::Immotile)
    , m_ProliferationState(ProliferationState::Presimulation)
    , m_Index(identifier)
    , m_BulkArea(0.)
    , m_MembraneLength(0.)
    , m_BulkStiffness(parameters->C_BulkStiffness() / parameters->Temperature())
    , m_MembraneStiffness(parameters->C_MembraneStiffness() / parameters->Temperature())
    , m_SubstrateAdhesionMax((parameters->C_SubstrateAdhesion() + 0.5* parameters->C_SubstrateAdhesionPolarizability()) / parameters->Temperature())
    , m_SubstrateAdhesionMin((parameters->C_SubstrateAdhesion() - 0.5* parameters->C_SubstrateAdhesionPolarizability()) / parameters->Temperature())
    , m_UpdateRate(parameters->C_PolarizationUpdateRate())
    , m_SignallingRadius2(parameters->C_SignallingRadius2())
    , m_BulkStiffnessQuiescence(m_BulkStiffness)
    , m_MembraneStiffnessQuiescence(m_MembraneStiffness)
    , m_AreaThresholdGrowth(parameters->C_AreaThresholdGrowth())
    , m_AreaThresholdDivision(parameters->C_AreaThresholdDivision())
    , m_GrowthDuration(parameters->C_GrowthDuration())
    , m_DivisionDuration(parameters->C_DivisionDuration())
    , m_Timer(0)
    , g_RNG(RNG)
    {
        // initialize cell bulk
        tile->m_Cell = this;
        tile->m_IndexCell = m_Bulk.size();
        tile->m_CellSubstrateAdhesion = 0.5 * (m_SubstrateAdhesionMax+m_SubstrateAdhesionMin);
        m_Bulk.push_back(tile);

        // initialize cell membrane
        for(size_t i=0; i<tile->m_Edges.size(); ++i) {
            Edge* edge = &tile->m_Edges[i];
            edge->m_Cell = this;
            edge->m_IndexCell = m_Membrane.size();
            //edge->m_Prev = &tile->m_Edges[i>=1 ? i-1 : i+5];
            //edge->m_Next = &tile->m_Edges[i>=5 ? i-5 : i+1];
            m_Membrane.push_back(edge);
        }

        // initialize cell area and perimeter
        UpdateMorphology();
    };

    /*!
     *  Cell division operator by explicit copy construction. It divides the Cell along its short axis and distributes the substrate tiles left and right of the division axis to the corresponding daughter Cells.
     *  @param identifier index of the new Cell
     *  @param base Cell to be divided
     */
    Cell(size_t identifier, Cell& base);

    /*!
     *  Update morphological information (e.g. cell area and perimeter). Useful if the shape of the tiles has changed.
     */
    void UpdateMorphology() {
        // update area
        m_BulkArea = 0.;
        for (const auto& tile : m_Bulk) {
            m_BulkArea += tile->m_Area;
        }

        // update perimeter
        m_MembraneLength = 0.;
        for (const auto& edge : m_Membrane) {
            m_MembraneLength += edge->m_Length;
        }
    };

    /*!
     *  Randomly choose an edge from the cell membrane. The probability is proportional to the edge length.
     */
    Edge* RandomEdge() {
        // roll random number
        double_t rnd = gsl_rng_uniform(g_RNG) * m_MembraneLength;

        // search for edge corresponding to the rolled random number
        double_t sum = 0.;
        for (const auto& edge : m_Membrane) {
            sum += edge->m_Length;
            if (sum >= rnd) {
                return edge;
            }
        }

        // if no corresponding edge is found, throw!
        throw std::logic_error("RandomEdge algorithm erroneous!");
    };


    /*!
     *  Perform a protrution Event.
     *  @param input Edge along which the cell protrudes
     */
    void Protrude(Edge* input);

    /*!
     *  Perform a retraction Event.
     *  @param input Edge along which the cell retracts
     */
    void Retract(Edge* input);

    /*!
     *  Compute the movement vector of the Cell, then move the Cell.
     */
    const Vector MoveVector() {
        Vector displacement (0.,0.);
        double_t total_weight = 0.;
        for (auto& tile : m_Bulk) {
            Vector vec = (tile->X() - this->X());
            double_t weight = tile->m_Area;
            displacement += weight * vec;
            total_weight += weight;
        }
        displacement /= total_weight;
        m_X += displacement;
        return displacement;
    }

    /*!
     *  Compute polarization vector of the Cell.
     */
    const Vector PolarizationVector() const {
        Vector axis (0.,0.);
        double_t total_weight = 0.;
        for (auto& tile : m_Bulk) {
            Vector vec = (tile->X() - this->X());
            double_t weight = tile->m_CellSubstrateAdhesion;
            axis += weight * vec;
            total_weight += weight;
        }
        return axis / total_weight;
    }

    /*!
     *  Perform Principal Components Analysis of the Cell.
     */
    const PrincipalComponents PrincipalComponentAnalysis() const {

        //compute covariance matrix
        double_t covxx = 0.;
        double_t covxy = 0.;
        double_t covyy = 0.;
        double_t total_weight = 0.;
        for (auto& tile : m_Bulk) {
            Vector pos = (tile->X() - this->X());
            double_t weight = tile->Area();
            covxx += weight * pos[0] * pos[0];
            covxy += weight * pos[0] * pos[1];
            covyy += weight * pos[1] * pos[1];
            total_weight += weight;
        }
        covxx /= total_weight;
        covxy /= total_weight;
        covyy /= total_weight;

        //compute determinant
        double_t D = std::sqrt( (covxx-covyy) * (covxx-covyy) + 4.* covxy * covxy);

        //compute eigenvalues and eigenvectors
        double_t EV1 = 0.5 * (covxx + covyy + D);
        double_t EV2 = 0.5 * (covxx + covyy - D);
        Vector EVEC1 ((covxx - covyy + D) / (2*covxy), 1);
        Vector EVEC2 ((covxx - covyy - D) / (2*covxy), 1);

        const PrincipalComponents PC ({EV1, EV2, EVEC1, EVEC2});
        return PC;

    };

    /*!
     *  Change proliferation state of the cell
     */
    void SetProliferationState(const ProliferationState& state) {
        m_ProliferationState = state;
    }

    /*!
     *  Get proliferation state of the cell
     */
    const Cell::ProliferationState GetProliferationState() const {
        return m_ProliferationState;
    }

    /*!
     *  Change motility state of the cell
     */
    void SetMotilityState(const MotilityState& state) {
        m_MotilityState = state;
    };

    /*!
     *  Return motility state of the cell.
     */
    const MotilityState& CheckMotility() const {
        return m_MotilityState;
    }

    /*!
     *  Return proliferation state of the cell.
     */
    const ProliferationState& CheckProliferation() const {
        return m_ProliferationState;
    }

    /*!
     *  Get current position of the Cell.
     */
    const Point& X() const {
        return m_X;
    }

    /*!
     *  Get index of the Cell
     */
    const size_t& Index() const {
        return m_Index;
    };

    /*!
     *  Get area of the Cell
     */
    const double_t& Area() const {
        return m_BulkArea;
    };

    /*!
     *  Get perimeter of the Cell
     */
    const double_t& Perimeter() const {
        return m_MembraneLength;
    };

    /*!
     *  Get maximal substrate adhesion of the Cell
     */
    const double_t& SubstrateAdhesionMax() const {
        return m_SubstrateAdhesionMax;
    };

    /*!
     *  Get minimal substrate adhesion of the Cell
     */
    const double_t& SubstrateAdhesionMin() const {
        return m_SubstrateAdhesionMin;
    };

    /*!
     *  Get bulk stiffness of the Cell
     */
    const double_t& BulkStiffness() const {
        return m_BulkStiffness;
    };

    /*!
     *  Get membrane stiffness of the Cell
     */
    const double_t& MembraneStiffness() const {
        return m_MembraneStiffness;
    };

    /*!
     *  Get vector of bulk Tiles of the Cell
     */
    const std::vector<Tile*>& Bulk() const {
        return m_Bulk;
    };

    /*!
     *  Get vector of membrane Edges of the Cell
     */
    const std::vector<Edge*>& Membrane() const {
        return m_Membrane;
    };

    //=============================================//
    // Polarization Models                         //
    //=============================================//


    /*!
     *  Update the Cell adhesion field at the end of a Monte-Carlo step.
     */
    void UpdatePolarization() {
        for (auto& tile : m_Bulk) {
            if (m_MotilityState == MotilityState::Immotile){
                tile->m_CellSubstrateAdhesion += m_UpdateRate * (0.5* (m_SubstrateAdhesionMax + m_SubstrateAdhesionMin) - tile->m_CellSubstrateAdhesion);
            }
            else if (tile->m_Signal < -0) {
                tile->m_CellSubstrateAdhesion += m_UpdateRate * (m_SubstrateAdhesionMin - tile->m_CellSubstrateAdhesion);
            }
            else if (tile->m_Signal > +0) {
                tile->m_CellSubstrateAdhesion += m_UpdateRate * (m_SubstrateAdhesionMax - tile->m_CellSubstrateAdhesion);
            }
            else{
                tile->m_CellSubstrateAdhesion += m_UpdateRate * (0.5* (m_SubstrateAdhesionMax + m_SubstrateAdhesionMin) - tile->m_CellSubstrateAdhesion);
            }
            tile->m_Signal = 0;
        }
    }

    //=============================================//
    // Proliferation Models                        //
    //=============================================//

    /*!
     *  Advance in the cell cycle with probabilities derived from average dwelling times in each phase
     */
    void UpdateCellCycle() {

        // Exit function of invalid value of average cell growth time
        if(m_GrowthDuration <= 0.) return;

        // Exit function of invalid value of average cell division time
        if(m_DivisionDuration <= 0.) return;

        // Check if Cell crosses a region where divisions are not allowed
        for (auto& tile : m_Bulk) {
            if(!tile->CheckDivisions()) {
                m_ProliferationState = ProliferationState::Quiescence;
                m_BulkStiffness = m_BulkStiffnessQuiescence;
                m_MembraneStiffness = m_MembraneStiffnessQuiescence;
                return;
            };
        };

        switch (m_ProliferationState) {
            case ProliferationState::Presimulation: {
                // do nothing in the presimulations
                break;
            }

            case ProliferationState::DivisionSignal: {
                // cell has just divided
                // immediately advance cell cycle back to growing state
                // reset cell shape parameters
                m_ProliferationState = ProliferationState::Quiescence;
                m_MotilityState = MotilityState::Motile;
                m_BulkStiffness = m_BulkStiffnessQuiescence;
                m_MembraneStiffness = m_MembraneStiffnessQuiescence;
            }

            case ProliferationState::Quiescence: {
                // cell had perviously the signal to stop growing (e.g. due to crossing areas where growth is prohibited)
                // in the next step, it will attempt to grow unless it again crosses such an area
                if(Area()>m_AreaThresholdGrowth) {
                    m_ProliferationState = ProliferationState::Growing;
                }
                break;
            }

            case ProliferationState::Growing: {
                // grow cell exponentially by halving bulk stiffness over the average growth duration
                // too keep cell motility approximately the same, membrane stiffness has to be reduced by a factor of sqrt(0.5)
                m_BulkStiffness *= std::pow(0.5,1./m_GrowthDuration);
                m_MembraneStiffness *= std::pow(0.5,0.5/m_GrowthDuration);
                //double_t probability = 1. - std::exp(-1./m_GrowthDuration);
                //if(Area()>=m_AreaThresholdDivision) {
                ++m_Timer;
                if(m_Timer>=m_GrowthDuration) {
                    m_ProliferationState = ProliferationState::Dividing;
                    m_MotilityState = MotilityState::Immotile;
                    m_Timer = 0;
                };
                break;
            }
            case ProliferationState::Dividing: {
                // in this state, the cell is immotile
                // determine by the gillespie algorithm when the cell will give the signal to divide
                //double_t probability = 1. - std::exp(-1./m_DivisionDuration);
                ++m_Timer;
                if(m_Timer>=m_DivisionDuration) {
                    m_ProliferationState = ProliferationState::DivisionSignal;
                    m_Timer = 0;
                };
                break;
            }

        }
    }

private:

    /*!
     *  Find the largest continuous blob of connected subtrate tiles and initialize the Cell from it.
     */
    void InitializeFromPool(std::vector<Tile*> pool);

    const size_t m_Index;                               ///< Index of the cell

    MotilityState m_MotilityState;                      ///< Flag indicating whether cell can polarize
    ProliferationState m_ProliferationState;            ///< Flag indicating current state in cell cycle

    Point m_X;                                          ///< Current position of the cell

    double_t m_BulkArea;                                ///< Area of the cell
    double_t m_MembraneLength;                          ///< Perimeter of the cell
    std::vector<Tile*> m_Bulk;                          ///< List of cell bulk elements
    std::vector<Edge*> m_Membrane;                      ///< List of cell membrane elements

    gsl_rng* g_RNG;                                     ///< Pointer to random number generator


    double_t m_BulkStiffness;                           ///< current area stiffness of cell
    const double_t m_BulkStiffnessQuiescence;           ///< current area stiffness of cell
    double_t m_MembraneStiffness;                 ///< Perimeter stiffness of the cell
    const double_t m_MembraneStiffnessQuiescence;           ///< current area stiffness of cell
    const double_t m_AreaThresholdGrowth;                     ///< Area threshold above which cell switches to interphase
    const double_t m_AreaThresholdDivision;                     ///< Area threshold above which cell divides
    const double_t m_GrowthDuration;                    ///< Probability to grow in this timestep
    const double_t m_DivisionDuration;                  ///< Rate of cell growth given that the cell grows
    double_t m_Timer;

    const double_t m_SubstrateAdhesionMax;              ///< Maximal cell polarization
    const double_t m_SubstrateAdhesionMin;              ///< Minimal cell polarization
    const double_t m_UpdateRate;                        ///< Update rate of cell polarization
    const double_t m_SignallingRadius2;                 ///< Squared signalling radius of the internal cell dynamics

};

#endif /* CPM_Cell_hpp */
