#include <hdf5.h>
#include <H5Cpp.h>

#include "CPM_Grid.hpp"

/*!
 *  Construct Grid from Parameters and the Random Number Generator
 *  @param  parameters   Parameters indicating substrate properties and size
 *  @param  RNG          Pointer to the Random Number Generator
 */
Grid::Grid(Parameters* parameters, gsl_rng* RNG)
: m_NCols(parameters->N_Columns())
, m_NRows(parameters->N_Rows())
, g_RNG(RNG)
{

    m_Tiles.reserve(m_NCols*m_NRows);
    // Build Tiles
    for (size_t i = 0; i<m_NRows; ++i) {
        for (size_t j = 0; j<m_NCols; ++j) {
            m_Tiles.emplace_back(j,i,m_NCols,m_NRows);
        }
    }

    // Initialize Neighborhood
    for (size_t i = 0; i<m_NRows; ++i) {
        for (size_t j = 0; j<m_NCols; ++j) {
            m_Tiles[Index(j, i)].m_Neighbors[0] = &m_Tiles[Index(j, i, 0, 1)];
            m_Tiles[Index(j, i)].m_Neighbors[1] = &m_Tiles[Index(j, i, 1, 0)];
            m_Tiles[Index(j, i)].m_Neighbors[2] = &m_Tiles[Index(j, i, 0, m_NRows - 1)];
            m_Tiles[Index(j, i)].m_Neighbors[3] = &m_Tiles[Index(j, i, m_NCols - 1, m_NRows - 1)];
            m_Tiles[Index(j, i)].m_Neighbors[4] = &m_Tiles[Index(j, i, m_NCols - 1, 0)];
            m_Tiles[Index(j, i)].m_Neighbors[5] = &m_Tiles[Index(j, i, m_NCols - 1, 1)];
        }
    }

    // Initialize Edge Complements
    for (auto& tile : m_Tiles) {
        for (size_t i=0; i<6; ++i) {
            tile.m_Edges[i].m_Comp = &tile.m_Neighbors[i]->m_Edges[i>=3 ? i-3 : i+3];
        }
    }

    // Initialize vertex positions, edge lengths and tile areas
    UpdateMorphology();

}

/*!
 *  Compute Tile index from column and row, with a displacement by (dx, dy)
 *  @param col  Tile column
 *  @param row  Tile row
 *  @param dx   displacement in columns
 *  @param dy   displacement in rows
 */
size_t Grid::Index(size_t col, size_t row, size_t dx, size_t dy) {

    // If index on L subgrid and dy is uneven, adjust dx such that dx = 0 is left
    if ((row&1) && (dy&1)) ++dx;

    col = (col + dx) % m_NCols;
    row = (row + dy) % m_NRows;

    return Index(col, row);
};

/*!
 *  Update morphology (vertices, edge lengths, area) of all Tiles in the grid.
 */
void Grid::UpdateMorphology() {

    for (auto& tile : m_Tiles) {

        // Update vertices
        for (size_t i=0; i<6; ++i) {
            tile.m_Vertices[i] = tile.X();
            tile.m_Vertices[i] += ((tile.m_Neighbors[i]->X() - tile.X()) + (tile.m_Neighbors[i>=1 ? i-1 : i+5]->X() - tile.X()))/3.;
        }

        // Update edges
        for (size_t i=0; i<6; ++i) {
            tile.m_Edges[i].m_Length = (tile.m_Vertices[i>=5 ? i-5 : i+1] - tile.m_Vertices[i]).Norm();
        }

        // Update tiles
        tile.m_Area = 0.;
        for (size_t i=0; i<6; ++i) {
            Vector x1 = tile.m_Vertices[i] - tile.X();
            Vector x2 = tile.m_Vertices[i>=5 ? i-5 : i+1] - tile.X();
            tile.m_Area += (x1[0] * x2[1] - x2[0] * x1[1]);
        }
        tile.m_Area = 0.5 * std::abs(tile.m_Area);

    }

}

/*!
 *  Imprint substrate pattern on the Grid.
 *  @param parameters Parameters for substrate properties
 *  @param pattern    Pattern to be imprinted
 */
void Grid::Imprint(Parameters* parameters, std::string pattern) {
    if(pattern == "PERIODIC") ImprintPeriodic(parameters);
    else if (pattern == "NONPERIODIC") ImprintNonPeriodic(parameters);
    else if (pattern == "CIRCULAR") ImprintCircular(parameters);
    else if (pattern == "DONUT") ImprintDonut(parameters);
    else {
        std::string custom ("CUSTOM: ");
        size_t pos = pattern.find(custom);
        if (pos != std::string::npos) {
            size_t len = custom.size();
            pattern.erase(pos, len);
            ImprintCustom(parameters, pattern);
        }
    }
};

/*!
 *  Imprint periodic pattern.
 *  @param parameters Parameters for substrate properties
 */
void Grid::ImprintPeriodic(Parameters* parameters) {
    for (auto& tile : m_Tiles) {
        tile.m_CheckSeed = true;
        tile.m_CheckDivisions = true;
        tile.m_CheckMod = true;
        tile.m_CellSubstrateAdhesionPenalty = parameters->S_CellAdhesionPenalty();
    }
}

/*!
 *  Imprint non-periodic pattern.
 *  @param parameters Parameters for substrate properties
 */
void Grid::ImprintNonPeriodic(Parameters* parameters) {

    for (auto& tile : m_Tiles) {
        if(Column(tile.m_Index)!= m_NCols && Row(tile.m_Index)!= m_NRows) {
            tile.m_CheckSeed = true;
            tile.m_CheckDivisions = true;
            tile.m_CheckMod = true;
            tile.m_CellSubstrateAdhesionPenalty = parameters->S_CellAdhesionPenalty();
        }
        else {
            tile.m_CheckSeed = false;
            tile.m_CheckDivisions = false;
            tile.m_CheckMod = false;
        }
    }

}

/*!
 *  Imprint circular pattern.
 *  @param parameters Parameters for substrate properties
 */
void Grid::ImprintCircular(Parameters* parameters) {

    for (auto& tile : m_Tiles) {
        if(tile.X().Norm2() < parameters->S_ConfinementRadiusO()) {
            tile.m_CheckSeed = true;
            tile.m_CheckDivisions = true;
            tile.m_CheckMod = true;
            tile.m_CellSubstrateAdhesionPenalty = parameters->S_CellAdhesionPenalty();
        }
        else {
            tile.m_CheckSeed = false;
            tile.m_CheckDivisions = false;
            tile.m_CheckMod = false;
        }
    }

}

/*!
 *  Imprint donut pattern.
 *  @param parameters Parameters for substrate properties
 */
void Grid::ImprintDonut(Parameters* parameters) {

    for (auto& tile : m_Tiles) {
        if(parameters->S_ConfinementRadiusI() < tile.X().Norm2() < parameters->S_ConfinementRadiusO()) {
            tile.m_CheckSeed = true;
            tile.m_CheckDivisions = true;
            tile.m_CheckMod = true;
            tile.m_CellSubstrateAdhesionPenalty = parameters->S_CellAdhesionPenalty();
        }
        else {
            tile.m_CheckSeed = false;
            tile.m_CheckDivisions = false;
            tile.m_CheckMod = false;
        }
    }

}

/*!
 *  Imprint custom pattern.
 *  @param parameters Parameters for substrate properties
 */
void Grid::ImprintCustom(Parameters* parameters, std::string pattern){

    try {

        std::cout << "Importing Pattern File CustomPattern.h5." <<std::endl;
        H5::H5File file(pattern, H5F_ACC_RDONLY);

        {
            std::cout << "Cross-Checking Number of Columns in the Pattern File and in the Simulation." <<std::endl;
            uint_fast32_t NC;
            H5::Attribute attribute = file.openAttribute("NColumns");
            attribute.read(H5::PredType::NATIVE_HSIZE, &NC);
            attribute.close();
            if(NC!=m_NCols) throw std::logic_error("Number of Columns do not match.");
        }

        {
            std::cout << "Cross-Checking Number of Rows in the Pattern File and in the Simulation." <<std::endl;
            uint_fast32_t NC;
            H5::Attribute attribute = file.openAttribute("NRows");
            attribute.read(H5::PredType::NATIVE_HSIZE, &NC);
            attribute.close();
            if(NC!=m_NRows) throw std::logic_error("Number of Rows do not match.");
        }

        {
            std::cout << "Loading /SeedCheck Dataset. This Dataset indicates which tiles can be used for cell seeding." <<std::endl;
            H5::DataSet dataset = file.openDataSet("/SeedCheck");
            std::vector<uint_fast32_t> DATA(m_Tiles.size());
            dataset.read(DATA.data(), H5::PredType::NATIVE_UINT_FAST32);
            dataset.close();
            for (size_t i=0; i<m_Tiles.size(); ++i) m_Tiles[i].m_CheckSeed = DATA[i];
        }

        {
            std::cout << "Loading /DivisionCheck Dataset. This Dataset indicates which tiles can be used for cell divisions." <<std::endl;
            H5::DataSet dataset = file.openDataSet("/DivisionCheck");
            std::vector<uint_fast32_t> DATA(m_Tiles.size());
            dataset.read(DATA.data(), H5::PredType::NATIVE_UINT_FAST32);
            dataset.close();
            for (size_t i=0; i<m_Tiles.size(); ++i) m_Tiles[i].m_CheckDivisions = DATA[i];
        }

        {
            std::cout << "Loading /ModCheck Dataset. This Dataset indicates which tiles can be altered; this includes occupation by cells." <<std::endl;
            H5::DataSet dataset = file.openDataSet("/ModCheck");
            std::vector<uint_fast32_t> DATA(m_Tiles.size());
            dataset.read(DATA.data(), H5::PredType::NATIVE_UINT_FAST32);
            dataset.close();
            for (size_t i=0; i<m_Tiles.size(); ++i) m_Tiles[i].m_CheckMod = DATA[i];
        }



        {
            std::cout << "Loading /CellSubstrateAdhesionPenalty Dataset. This Dataset indicates adhesion penalty for occupying the tile." <<std::endl;
            H5::DataSet dataset = file.openDataSet("/CellSubstrateAdhesionPenalty");
            std::vector<double_t> DATA(m_Tiles.size());
            dataset.read(DATA.data(), H5::PredType::NATIVE_DOUBLE);
            dataset.close();
            double_t SCAPenalty = parameters->S_CellAdhesionPenalty();
            for (size_t i=0; i<m_Tiles.size(); ++i) m_Tiles[i].m_CellSubstrateAdhesionPenalty = DATA[i] * SCAPenalty;
        }

        file.close();

    } catch (H5::Exception) {
        throw std::logic_error("HDF5 Error detected.");
    }

    for (auto& tile : m_Tiles) {
        double_t temperature = parameters->Temperature();
        tile.m_CellSubstrateAdhesionPenalty /= temperature;
    }

}
