#ifndef CPM_Hamiltonian_hpp
#define CPM_Hamiltonian_hpp

#include <cmath>
#include <vector>
#include <utility>

#include "CPM_Edge.hpp"
#include "CPM_Tile.hpp"
#include "CPM_Cell.hpp"

class Hamiltonian {
public:

    /*!
     *  Enums indicating whether the cell will protrude, retract or remain unchanged.
     */
    enum class Event {Protrude, Retract, Rupture, Remain};

    Hamiltonian(Parameters* parameters, gsl_rng* RNG)
    : m_CellAdhesion(parameters->C_CellAdhesion() / parameters->Temperature())
    , m_CellDissipation(parameters->C_CellDissipation() / parameters->Temperature())
    , m_C_SubstrateDissipation(parameters->C_SubstrateDissipation() / parameters->Temperature())
    , g_RNG(RNG)
    {
    };

    void SetAdhesion(Parameters* pars = nullptr) {
        if(pars) {
            m_CellAdhesion = pars->C_CellAdhesion() / pars->Temperature();
            m_CellDissipation = pars->C_CellDissipation() / pars->Temperature();
        }
        else {
            m_CellAdhesion = 0.;
            m_CellDissipation = 0.;
        }
    };

    /*!
     *  Compute probabilities of protruding, retracting or doing nothing. Then, decide what the cell does.
     *  @param edge Edge along which an event shall take place
     */
    Event MakeDecision(Edge* edge) {

        // define conquerer and target tiles and cells
        const Tile* conqueror_tile = edge->GetTile();
        const Cell* conqueror_cell = conqueror_tile->GetCell();
        const Tile* target_tile = edge->GetComp()->GetTile();
        const Cell* target_cell = target_tile->GetCell();

        std::pair<Event,double_t> attempt;
        if (gsl_rng_uniform_int(g_RNG,2)==0) {
            if(target_cell) {
                if(target_cell->GetProliferationState() == Cell::ProliferationState::Presimulation) {
                    if(target_cell->Bulk().size() == 1) return Event::Remain;
                }
            }
            attempt = EnergyDifferenceProtrusion(edge);
        }
        else {
            if(conqueror_cell->GetProliferationState() == Cell::ProliferationState::Presimulation) {
                if(conqueror_cell->Bulk().size() == 1) return Event::Remain;
            }
            attempt = EnergyDifferenceRetraction(edge);
        }

        if (gsl_rng_uniform_pos(g_RNG) < std::exp(-attempt.second)) {
            switch (attempt.first) {
                case Event::Protrude:
                if (target_tile->CheckMod() && TopologyCheck(conqueror_cell, target_tile) && (target_cell ? TopologyCheck(target_cell, target_tile) : true)) return attempt.first;
                break;

                case Event::Retract:
                if (TopologyCheck(conqueror_cell, conqueror_tile) && (target_cell ? TopologyCheck(target_cell, conqueror_tile) : true)) return attempt.first;
                break;

                case Event::Rupture:
                if (TopologyCheck(conqueror_cell, conqueror_tile)) return attempt.first;
                break;

                default:
                break;
            }
        }
        return Event::Remain;
    };

    /*!
     *  Return event and corresponding energy changes for attempting to change the configuration of the cell.
     *  @param edge                 Edge along which an event shall take place
     */
    std::pair<Event, double_t> EnergyDifferenceProtrusion(Edge* edge) {

        Tile* conqueror_tile = edge->GetTile();
        Cell* conqueror_cell = conqueror_tile->GetCell();
        Tile* target_tile = edge->GetComp()->GetTile();
        Cell* target_cell = target_tile->GetCell();

        double_t DHamiltonian = 0.;

        // calculate direction of protrusion
        Vector Direction = target_tile->X() - conqueror_tile->X();
        Direction /= Direction.Norm2();

        // compute energy change due to polarization energy for protrusion / retraction
        {
            const double_t conqueror_polarization = conqueror_tile->CellSubstrateAdhesion();
            const double_t target_polarization = target_cell ? target_tile->CellSubstrateAdhesion() : target_tile->CellSubstrateAdhesionPenalty();
            const double_t DHamiltonianTmp = target_polarization - conqueror_polarization;
            DHamiltonian += DHamiltonianTmp;
        }

        // determine how much of the target tile is connected to conqueror, target or other cells
        double_t ContactLength_Conqueror = 0.;
        double_t ContactLength_Target = 0.;
        double_t ContactLength_Other = 0.;
        double_t ContactLength_Substrate = 0.;
        for (const auto& border : target_tile->Edges()) {
            auto ncell = border.GetComp()->GetTile()->GetCell();
            if (!ncell) ContactLength_Substrate += border.Length();
            else if (ncell == conqueror_cell) ContactLength_Conqueror += border.Length();
            else if (ncell == target_cell) ContactLength_Target += border.Length();
            else ContactLength_Other += border.Length();
        }

        // compute area and perimeter energy change for protrusion
        // conqueror cell
        {
            const double_t AreaDifference = +target_tile->Area();
            const double_t PerimeterDifference = ContactLength_Target + ContactLength_Other + ContactLength_Substrate - ContactLength_Conqueror;
            const double_t DHamiltonianTmp = EnergyDifferenceArea(conqueror_cell, AreaDifference) + EnergyDifferencePerimeter(conqueror_cell, PerimeterDifference);
            DHamiltonian += DHamiltonianTmp;
        }
        // target cell
        if (target_cell) {
            const double_t AreaDifference = -target_tile->Area();
            const double_t PerimeterDifference = ContactLength_Target - ContactLength_Conqueror - ContactLength_Other - ContactLength_Substrate;
            const double_t DHamiltonianTmp = EnergyDifferenceArea(target_cell, AreaDifference) + EnergyDifferencePerimeter(target_cell, PerimeterDifference);
            DHamiltonian += DHamiltonianTmp;
        }

        // compute cell adhesion energy change for protrusion
        {
            const double_t DHamiltonianTmp = target_cell ?
            m_CellAdhesion * (ContactLength_Conqueror - ContactLength_Target) + m_CellDissipation * ContactLength_Other :
            - m_CellAdhesion * ContactLength_Other;
            DHamiltonian += DHamiltonianTmp;
        }
        
        // compute cell substrate dissipation
        {
            DHamiltonian += target_cell ? m_C_SubstrateDissipation : 0.;
        }

        // return event and corresponding energy
        return {Event::Protrude, DHamiltonian};

    };

    /*!
     *  Return event and corresponding energy changes for attempting to change the configuration of the cell.
     *  @param edge                 Edge along which an event shall take place
     */
    std::pair<Event, double_t> EnergyDifferenceRetraction(Edge* edge) {

        Tile* conqueror_tile = edge->GetTile();
        Cell* conqueror_cell = conqueror_tile->GetCell();
        Tile* target_tile = edge->GetComp()->GetTile();
        Cell* target_cell = target_tile->GetCell();

        double_t DHamiltonian = 0.;
        double_t DHamiltonianRupture = 0.;

        // calculate direction of protrusion
        Vector Direction = target_tile->X() - conqueror_tile->X();
        Direction /= Direction.Norm2();

        // compute energy change due to polarization energy for protrusion / retraction
        {
            const double_t conqueror_polarization = conqueror_tile->CellSubstrateAdhesion();
            const double_t target_polarization = target_cell ? target_tile->CellSubstrateAdhesion() : conqueror_tile->CellSubstrateAdhesionPenalty();
            const double_t DHamiltonianTmp = target_polarization - conqueror_polarization;
            DHamiltonian -= DHamiltonianTmp;
            DHamiltonianRupture += conqueror_polarization - conqueror_tile->CellSubstrateAdhesionPenalty();
        }

        // determine how much of the target tile is connected to conqueror, target or other cells
        double_t ContactLength_Conqueror = 0.;
        double_t ContactLength_Target = 0.;
        double_t ContactLength_Other = 0.;
        double_t ContactLength_Substrate = 0.;
        for (const auto& border : conqueror_tile->Edges()) {
            auto ncell = border.GetComp()->GetTile()->GetCell();
            if (!ncell) ContactLength_Substrate += border.Length();
            else if (ncell == conqueror_cell) ContactLength_Conqueror += border.Length();
            else if (ncell == target_cell) ContactLength_Target += border.Length();
            else ContactLength_Other += border.Length();
        }

        // compute area and perimeter energy change for retraction
        // conqueror cell
        {
            const double_t AreaDifference = -conqueror_tile->Area();
            const double_t PerimeterDifference = ContactLength_Conqueror - ContactLength_Target - ContactLength_Other - ContactLength_Substrate;
            const double_t DHamiltonianTmp = EnergyDifferenceArea(conqueror_cell, AreaDifference) + EnergyDifferencePerimeter(conqueror_cell, PerimeterDifference);
            DHamiltonian += DHamiltonianTmp;
            DHamiltonianRupture += DHamiltonianTmp;
        }
        // target cell
        if (target_cell) {
            const double_t AreaDifference = +conqueror_tile->Area();
            const double_t PerimeterDifference = ContactLength_Conqueror + ContactLength_Other + ContactLength_Substrate - ContactLength_Target;
            const double_t DHamiltonianTmp = EnergyDifferenceArea(target_cell, AreaDifference) + EnergyDifferencePerimeter(target_cell, PerimeterDifference);
            DHamiltonian += DHamiltonianTmp;
        }

        // compute cell adhesion energy change for retraction
        {
            const double_t DHamiltonianTmp = target_cell ?
            m_CellAdhesion * (ContactLength_Target - ContactLength_Conqueror) + m_CellDissipation * ContactLength_Other :
            (m_CellAdhesion + m_CellDissipation) * ContactLength_Other;
            DHamiltonian += DHamiltonianTmp;
        }

        // compute cell adhesion energy change for rupture
        {
            const double_t DHamiltonianTmp = (m_CellAdhesion + m_CellDissipation) * (ContactLength_Other + ContactLength_Target);
            DHamiltonianRupture += DHamiltonianTmp;
        }

        // compute cell substrate dissipation
        {
            DHamiltonian += m_C_SubstrateDissipation;
            DHamiltonianRupture += m_C_SubstrateDissipation;
        }

        // return event and corresponding energy
        if (DHamiltonianRupture < DHamiltonian) return {Event::Rupture, DHamiltonianRupture};
        else return {Event::Retract, DHamiltonian};

    };

private:

    /*!
     *  Compute energy change for changing cell area.
     *  @param cell                 corresponding Cell
     *  @param AreaDifference       area change
     */
    double_t EnergyDifferenceArea(const Cell* cell, const double_t& AreaDifference) {
        double_t EnergyDifference = cell->BulkStiffness() * AreaDifference * (2. * cell->Area() + AreaDifference);
        return EnergyDifference;
    };

    /*!
     *  Compute energy change for changing cell perimeter.
     *  @param cell                  corresponding Cell
     *  @param PerimeterDifference   perimeter change
     */
    double_t EnergyDifferencePerimeter(const Cell* cell, const double_t& PerimeterDifference) {
        double_t EnergyDifference = cell->MembraneStiffness() * PerimeterDifference * (2. * cell->Perimeter() + PerimeterDifference);
        return EnergyDifference;
    };

    /*!
     *  Perform a topology check. It returns (false), if a membership change (loss or gain) of the provided tile would create loops or interrupt the continuity of the provided cell bulk (strictly forbidden).
     *  @param tile provided Tile
     *  @param cell provided Cell
     */
    const bool TopologyCheck(const Cell* cell, const Tile* tile) const {
        size_t jumps = 0;
        auto neighbors = tile->Neighbors();
        for (size_t i=0; i<6; ++i){

            if (neighbors[i]->GetCell() == cell) {
                if (neighbors[i<5 ? i+1 : 0]->GetCell() != cell) {
                    ++jumps;
                    if (jumps > 1) return false;
                }
            }
        }
        return true;
    };

    gsl_rng* g_RNG;                                     ///< Pointer to random number generator

    const double_t m_C_SubstrateDissipation;            ///< Cell substrate dissipation
    double_t m_CellAdhesion;                            ///< Cell-cell adhesion energy
    double_t m_CellDissipation;                         ///< Cell-cell dissipation energy

};

#endif /* CPM_Hamiltonian_hpp */
