#ifndef CPM_Population_hpp
#define CPM_Population_hpp

#include "CPM_Cell.hpp"
#include "CPM_Grid.hpp"
#include "IO_Parameters.hpp"

class Population {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Population() = delete;

    /*!
     *  Costructor of Cell Population.
     *  @param  grid        Pointer to Grid that the Cells live on.
     *  @param  parameters  Pointer to Cell Parameters
     *  @param  RNG         Pointer to Random Number Generator
     */
    Population(Grid* grid, Parameters* parameters, gsl_rng* RNG)
    : g_RNG(RNG)
    {
        size_t NCells = parameters->N_Cells();
        for (size_t i=0; i< NCells; ++i) {
            Tile* tile = grid->RandomUnoccupiedTile();
            if(!tile) throw std::logic_error("No unoccupied tiles left!");
            m_Cells.emplace_back(i, tile, parameters, RNG);
        }
    };

    /*!
     *  Copy constructor is not allowed.
     */
    Population(Population& base) = delete;

    /*!
     * Update cell population (update adhesion field and morphological information on the Cells).
     */
    void Update() {
        for (auto& cell : m_Cells) {
            cell.UpdateMorphology();
            cell.UpdatePolarization();
            cell.UpdateCellCycle();
        }

        std::list<Cell> tmp;
        for (auto& cell : m_Cells) {
            if(cell.CheckProliferation() == Cell::ProliferationState::DivisionSignal) {
                tmp.emplace_back(m_Cells.size() + tmp.size(), cell);
            }
        }
        m_Cells.splice(m_Cells.end(), tmp);


    };

    //void CellInflux(double_t InfluxRate) {
    //    std::vector<Cell*> cells = SeedRegionCells();
    //    while (!cells.empty() && InfluxRate > 0) {
    //        if (InfluxRate >= 1 || gsl_rng_uniform(g_RNG) > InfluxRate) {
    //            Cell* cell = RandomCell(cells);
    //            m_Cells.emplace_back(m_Cells.size(), *cell);
    //            cells.erase(std::find(cells.begin(), cells.end(), cell));
    //        }
    //        --InfluxRate;
    //    }
    //}

    /*!
     *  Set motility state of the whole Cell Population.
     */
    void SetMotilityState(const Cell::MotilityState& state) {
        for (auto& cell : m_Cells) {
            cell.SetMotilityState(state);
        }
    }

    /*!
     *  Set proliferation state of the whole Cell Population.
     */
    void SetProliferationState(const Cell::ProliferationState& state) {
        for (auto& cell : m_Cells) {
            cell.SetProliferationState(state);
        }
    }

    /*!
     *  Choose random Cell from Population. The probability is proportional to the cell membrane length.
     */
    Cell* RandomCell() {

        // roll random number
        double_t TotalMembraneLength = 0.;
        for (const auto& cell : m_Cells) {
            TotalMembraneLength += cell.Perimeter();
        }
        double_t rnd = gsl_rng_uniform(g_RNG) * TotalMembraneLength;

        // search for cell corresponding to the rolled random number
        double_t sum = 0.;
        for (auto& cell : m_Cells) {
            sum += cell.Perimeter();
            if (sum >= rnd) {
                return &cell;
            }
        }

        // if no corresponding cell is found, throw!
        throw std::logic_error("RandomCell algorithm erroneous!");
    };

    /*!
     *  Choose random Cell from list of Cell pointers.
     */
    Cell* RandomCell(std::vector<Cell*> cells) {

        // roll random number
        double_t TotalMembraneLength = 0.;
        for (const auto& cell : cells) {
            TotalMembraneLength += cell->Perimeter();
        }
        double_t rnd = gsl_rng_uniform(g_RNG) * TotalMembraneLength;

        // search for cell corresponding to the rolled random number
        double_t sum = 0.;
        for (auto& cell : cells) {
            sum += cell->Perimeter();
            if (sum >= rnd) {
                return cell;
            }
        }

        // if no corresponding cell is found, throw!
        throw std::logic_error("RandomCell algorithm erroneous!");
    };

    /*!
     *  Get vector of Cells that are in the region where Cell seeding is allowed.
     */
    std::vector<Cell*> SeedRegionCells() {
        std::vector<Cell*> SRC;
        SRC.reserve(m_Cells.size());

        for (auto& cell : m_Cells) {
            // check if whole cell is whithin the seed region
            bool seedregion = false;
            for (auto& tile : cell.Bulk()) {
                seedregion = tile->CheckSeed();
            }
            // if cell is whithin the seed region, return cell
            if(seedregion) SRC.push_back(&cell);
        }
        return SRC;
    }

    /*!
     *  Get list of all Cells.
     */
    const std::list<Cell>& Cells() const {
        return m_Cells;
    }

    /*!
     *  Get list of all Cells.
     */
    std::list<Cell>& Cells() {
        return m_Cells;
    }


private:

    gsl_rng* g_RNG;             ///< Pointer to Random Number Generator
    std::list<Cell> m_Cells;    ///< List of all Cells

};

#endif /* CPM_Population_hpp */
