#ifndef CPM_Tile_hpp
#define CPM_Tile_hpp

#include <stdio.h>
#include <cmath>
#include <array>
#include <vector>

#include "Point.hpp"
#include "Vector.hpp"

#include "CPM_Edge.hpp"

class Cell;

class Tile {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Tile() = delete;

    /*!
     * Construct a substrate Tile
     *  @param  col     Column of the Tile
     *  @param  row     Row of the Tile
     *  @param  Ncols   Number of columns in the Grid
     *  @param  Nrows   Number of rows in the Grid
     */
    Tile(size_t col, size_t row, size_t Ncols, size_t Nrows)
    : m_Index(row*Ncols + col)
    , m_X(0.,0.,0.5 * Ncols, 0.25*std::sqrt(3.) * Nrows)
    , m_Cell(nullptr)
    , m_IndexCell(0)
    , m_Area(0.)
    , m_CellSubstrateAdhesion(0.)
    , m_CellSubstrateAdhesionPenalty(0.)
    , m_Signal(0)
    , m_CheckSeed(false)
    , m_CheckDivisions(false)
    , m_CheckMod(false)
    , m_Edges({{Edge(this, 0), Edge(this, 1), Edge(this, 2), Edge(this, 3), Edge(this, 4), Edge(this, 5)}})
    , m_Vertices({{m_X, m_X, m_X, m_X, m_X, m_X}})
    {
        // compute coordinates
        Vector dx(1.,0.);
        Vector dy(0.,0.5*std::sqrt(3.));
        m_X += (row&1) ? dx * ( col + 0.5 ) : dx*col;
        m_X += dy * row;
    };

    //Tile(Tile& base) = delete;

    /*!
     *  Get local Cell substrate adhesion. It is zero, if the Tile is unoccupied.
     */
    const double_t& CellSubstrateAdhesion() const {
        return m_CellSubstrateAdhesion;
    }

    /*!
     *  Get local Cell substrate adhesion penalty for cell occupation.
     */
    const double_t& CellSubstrateAdhesionPenalty() const {
        return m_CellSubstrateAdhesionPenalty;
    }

    /*!
     *  Get area of the Tile.
     */
    const double_t& Area() const {
        return m_Area;
    }

    /*!
     *  Get Cell currently occupying this Tile. Returns nullptr, if Tile is unoccupied.
     */
    Cell* GetCell() const {
        return m_Cell;
    };

    /*
     *  Get all vertices of this Tile.
     */
    const std::array<Point, 6>& Vertices() const {
        return m_Vertices;
    }

    /*
     *  Get specific vertex of this Tile.
     */
    const Point& Vertex(size_t i) const {
        return m_Vertices[i];
    };

    /*
     *  Get index of this Tile.
     */
    const size_t& Index() const {
        return m_Index;
    };

    /*
     *  Get position of this Tile.
     */
    const Point& X() const {
        return m_X;
    }

    /*
     *  Get pointers to all neighbors of this Tile.
     */
    const std::array<Tile*, 6>& Neighbors() const {
        return m_Neighbors;
    }

    /*
     *  Get references to all edges of this Tile.
     */
    const std::array<Edge, 6>& Edges() const {
        return m_Edges;
    }

    /*
     *  Check if this Tile may be used for seeding.
     */
    const bool& CheckSeed() const {
        return m_CheckSeed;
    }

    /*
     *  Check if this Tile may be used for dividing.
     */
    const bool& CheckDivisions() const {
        return m_CheckDivisions;
    }

    /*
     *  Check if this Tile may be modified.
     */
    const bool& CheckMod() const {
        return m_CheckMod;
    }


private:

    const size_t m_Index;                           ///< Index of the Til
    Point m_X;                                      ///< Current position of the tile
    double_t m_Area;                                ///< Area of Tile


    std::array<Point,6> m_Vertices;                 ///< List of vertices
    std::array<Edge,6> m_Edges;                     ///< List of edges
    std::array<Tile*,6> m_Neighbors;                ///< List of neighbors

    Cell* m_Cell;                                   ///< Cell owning this tile
    size_t m_IndexCell;                             ///< Index of bulk position in Cell
    double_t m_CellSubstrateAdhesion;               ///< Local polarization of occupying cell
    double_t m_CellSubstrateAdhesionPenalty;        ///< Local fibronectin level
    int_fast32_t m_Signal;                          ///< Integer indicating the strenght of past signaling events

    bool m_CheckSeed;                               ///< Indicator whether site can be used for initial cell seeding
    bool m_CheckDivisions;                          ///< Indicator whether site can be used for cell divisions
    bool m_CheckMod;                                ///< Indicator whether site can be modified (can cells go here?)

    friend class Grid;
    friend class GridSolver;
    friend class Cell;

};
#endif /* CPM_Tile_hpp */
