//
//  IO_Parameters.cpp
//  CPM_Fast
//
//  Created by Andriy Goychuk on 30.10.17.
//  Copyright © 2017 Andriy Goychuk. All rights reserved.
//

#include "IO_Parameters.hpp"

#include <boost/program_options.hpp>
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/json_parser.hpp>

using namespace boost::property_tree;
using namespace boost::program_options;

/*!
 *  Template helper function for reading parameters from ptree or variables_map
 *  @param  pt          ptree storing parameters
 *  @param  vm          variables_map storing parameters
 *  @param  group       Parameter group containing the parameter
 *  @param  fullname    Full name of the parameter
 *  @param  shortname   Short name of the parameter
 *  @param  variable    Storage of the parameter
 *  @param  value       Default value of the paramter
 */
template <typename T> void read(ptree& pt, variables_map& vm, std::string group, std::string fullname, std::string shortname, T& variable, T value){
    
    if (vm.count(shortname)) {
        // Try to read from command line and write to JSON
        variable = vm[shortname].as<T>();
        pt.put(group + std::string(".") + fullname, variable);
    }
    else try {
        // Else try to read from JSON
        variable = pt.get<T>(group + std::string(".") + fullname);
    }
    catch (const ptree_bad_data) {
        // If all fails, revert to generic value and print warning
        variable = value;
        pt.put(group + std::string(".") + fullname, variable);
        std::cout << "Bad data: " + fullname + " set to default value." << std::endl;
    }
    catch (const ptree_bad_path) {
        // If all fails, revert to generic value and print warning
        variable = value;
        pt.put(group + std::string(".") + fullname, variable);
        std::cout << "Bad path: " + fullname + " set to default value." << std::endl;
    }
    
}

/*!
 *  Tries to read configuration either from .json file or from the command line. If this fails, it rolls back to the predefined values. At the end of the function, it saves the updated parameters in the .json file.
 */
Parameters::Parameters(int argc, char* argv[]){
    
    ptree pt;
    variables_map vm;
    
    ///////////////////////////
    // Read options from JSON //
    ///////////////////////////
    
    try {
        read_json("config.json", pt);
    } catch (const json_parser_error) {
        std::cout<<"Could not find Configuration file. Reverting to standard values." << std::endl;
    }
    
    
    ///////////////////////////////////////////
    // Declare the supported options for CMD //
    ///////////////////////////////////////////
    
    options_description desc("Supported Options");
    desc.add_options()
    ("Help", "Show available parameters.")
    ("C_Polarization_Model", value<std::string>(), "Set polarization model.")
    ("C_S_Adhesion", value<double_t>(), "Set mean polarization.")
    ("C_S_Adhesion_Polarizability", value<double_t>(), "Set maximal polarizability.")
    ("C_S_Dissipation", value<double_t>(), "Set cell-substrate dissipation.")
    ("C_Update_Rate", value<double_t>(), "Set cytoskeletal update rate.")
    ("C_Signalling_Radius", value<double_t>(), "Set signaling radius.")
    ("C_Stiffness_Bulk", value<double_t>(), "Set area stiffness.")
    ("C_Stiffness_Membrane", value<double_t>(), "Set perimeter stiffness.")
    ("C_C_Adhesion", value<double_t>(), "Set energy gain per cell bond.")
    ("C_C_Dissipation", value<double_t>(), "Set energy dissipation per broken cell bond.")
    ("C_GrowthDuration", value<double_t>(), "Set average duration of growth phase.")
    ("C_DivisionDuration", value<double_t>(), "Set average duration of division phase.")
    ("C_RelativeThreshold", value<double_t>(), "Set relative Threshold area to switch to growth phase.")
    ("N_Cells", value<size_t>(), "Set number of cells.")
    ("S_Pattern_Presim", value<std::string>(), "Set substrate pattern for presimulation.")
    ("S_Pattern_Sim", value<std::string>(), "Set substrate pattern for main simulation.")
    ("S_C_Adhesion_Penalty", value<double_t>(), "Set substrate adhesion penalty. For custom substrates this is the maximal adhesion penalty. For all other substrates this is the adhesion penalty of the occupyable substrate.")
    ("S_Confinement_Radius_I", value<double_t>(), "Set inner confinement radius.")
    ("S_Confinement_Radius_O", value<double_t>(), "Set outer confinement radius.")
    ("N_Columns", value<size_t>(), "Set number of columns.")
    ("N_Rows", value<size_t>(), "Set number of rows.")
    ("Temperature", value<double_t>(), "Set temperature.")
    ("Seed", value<uint32_t>(), "Set RNG seed.")
    ("N_MCS_Presim", value<uint_fast32_t>(), "Set number of MCS for presimulation.")
    ("N_MCS_Sim", value<uint_fast32_t>(), "Set number of MCS for main simulation.")
    ("O_I_Measure", value<uint_fast32_t>(), "Set measurement interval.")
    ("O_Cell_Extended", value<bool>(), "Flag indicating whether extended Measurement of cell shall be performed.")
    ("O_Cell_Maps", value<bool>(), "Flag indicating whether 2D and angular Maps are to be measured.")
    ("O_Monolayer", value<bool>(), "Flag indicating whether Monolayer front positions and substrate strains are to be measured.")
    ("O_I_Snapshot", value<uint_fast32_t>(), "Set snapshot interval.")
    ;
    
    ////////////////////////////////////////////
    // Read options from CMD and write to JSON //
    // If no options from CMD, read from JSON  //
    ////////////////////////////////////////////
    
    store(parse_command_line(argc, argv, desc), vm);
    notify(vm);
    
    if (vm.count("Help")) {
        std::cout << desc << std::endl;
        exit(0);
    }
        
    // read parameters
    read<double_t>(pt, vm, "CELL", "SUBSTRATE_ADHESION", "C_S_Adhesion", m_C_SAdhesion, 75.);
    read<double_t>(pt, vm, "CELL", "SUBSTRATE_ADHESION_POLARIZABILITY", "C_S_Adhesion_Polarizability", m_C_SAdhesionPolarizability, 50.);
    read<double_t>(pt, vm, "CELL", "SUBSTRATE_DISSIPATION", "C_S_Dissipation", m_C_SubstrateDissipation, 0.);
    read<double_t>(pt, vm, "CELL", "CYTOSKELETAL_RATE", "C_Update_Rate", m_C_UpdateRate, 0.10);
    read<double_t>(pt, vm, "CELL", "SIGNALLING_RADIUS", "C_Signalling_Radius", m_C_SignallingRadius2, 50.);
    read<double_t>(pt, vm, "CELL", "STIFFNESS_BULK", "C_Stiffness_Bulk", m_C_BulkStiffness, 0.01);
    read<double_t>(pt, vm, "CELL", "STIFFNESS_MEMBRANE", "C_Stiffness_Membrane", m_C_MembraneStiffness, 0.01);
    read<size_t>(pt, vm, "POPULATION", "NUM_CELLS", "N_Cells", m_N_Cells, 1);
    read<double_t>(pt, vm, "POPULATION", "CELL_ADHESION", "C_C_Adhesion", m_C_CAdhesion, 0.);
    read<double_t>(pt, vm, "POPULATION", "CELL_DISSIPATION", "C_C_Dissipation", m_C_CDissipation, 0.);
    read<double_t>(pt, vm, "PROLIFERATION", "CELL_GROWTH_DURATION", "C_GrowthDuration", m_C_GrowthDuration, -1.);
    read<double_t>(pt, vm, "PROLIFERATION", "CELL_DIVISION_DURATION", "C_DivisionDuration", m_C_DivisionDuration, -1.);
    read<double_t>(pt, vm, "PROLIFERATION", "CELL_RELATIVE_THRESHOLD", "C_RelativeThreshold", m_C_RelativeThreshold, -1.);
    read<std::string>(pt, vm, "SUBSTRATE", "PATTERN_PRESIM", "S_Pattern_Presim", m_S_PatternPresim, "PERIODIC");
    read<std::string>(pt, vm, "SUBSTRATE", "PATTERN_SIM", "S_Pattern_Sim", m_S_PatternSim, "PERIODIC");
    read<double_t>(pt, vm, "SUBSTRATE", "CELL_ADHESION_PENALTY", "S_C_Adhesion_Penalty", m_S_CAdhesionPenalty, 0.);
    read<double_t>(pt, vm, "SUBSTRATE", "CONFINEMENT_RADIUS_I", "S_Confinement_Radius_I", m_S_ConfinementRadiusI, 0.);
    read<double_t>(pt, vm, "SUBSTRATE", "CONFINEMENT_RADIUS_O", "S_Confinement_Radius_O", m_S_ConfinementRadiusO, 0.);
    read<size_t>(pt, vm, "SUBSTRATE", "NUM_COLUMNS", "N_Columns", m_N_Columns, 100);
    read<size_t>(pt, vm, "SUBSTRATE", "NUM_ROWS", "N_Rows", m_N_Rows, 100);
    read<double_t>(pt, vm, "SIMULATION", "TEMPERATURE", "Temperature", m_Temperature, 1.);
    read<uint32_t>(pt, vm, "SIMULATION", "SEED", "Seed", m_Seed, 50000);
    read<uint_fast32_t>(pt, vm, "SIMULATION", "NUM_MCS_PRESIM", "N_MCS_Presim", m_N_MCSPresim, 1000);
    read<uint_fast32_t>(pt, vm, "SIMULATION", "NUM_MCS_SIM", "N_MCS_Sim", m_N_MCSSim, 1000);
    read<uint_fast32_t>(pt, vm, "OUTPUT", "INTERVAL_MEASUREMENT", "O_I_Measure", m_I_Measure, 1);
    read<bool>(pt, vm, "OUTPUT", "CELL_EXTENDED", "O_Cell_Extended", m_CellMeasureExtended, false);
    read<bool>(pt, vm, "OUTPUT", "CELL_MAP", "O_Cell_Maps", m_CellMeasureMaps, false);
    read<bool>(pt, vm, "OUTPUT", "MONOLAYER", "O_Monolayer", m_MeasureMono, false);
    read<uint_fast32_t>(pt, vm, "OUTPUT", "INTERVAL_SNAPSHOT", "O_I_Snapshot", m_I_Snapshot, 0);
    
    write_json("config.json", pt, std::locale());
    
    // Perform integrity checks
    if (m_N_Rows%2 != 0) {
        m_N_Rows += 1;
        std::cout<< "m_N_Rows must be an even integer. m_N_Rows has been adjusted to " << std::to_string(m_N_Rows) << "."<<std::endl;
    }
    
    // Square the signalling radius
    m_C_SignallingRadius2 *= m_C_SignallingRadius2;
    
}

