#ifndef Point_hpp
#define Point_hpp

#include "Vector.hpp"

class Point final : protected Vector {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Point() = delete;

    /*!
     *  Constructor of the 2D Point.
     *  @param x    x-coordinate
     *  @param y    y-coordinate
     *  @param L_x  extension of the coordinate system (positive x-direction)
     *  @param L_y  extension of the coordinate system (positive y-direction)
     */
    Point(double_t x, double_t y, double_t L_x, double_t L_y):
    Vector(x,y),
    m_L({{L_x,L_y}})
    {};

    /*!
     *  Copy constructor from a different Point.
     */
    Point (const Point& base)
    : Vector(base)
    , m_L(base.m_L)
    {};

    /*!
     *  Access elements of the Point
     *  @param n offset of the element.
     *  n=0 for x-axis. n=1 for y-axis
     */
    inline const double_t& operator[] (const size_t& n) const {
        return m_X[n];
    };

    /*!
     *  Get extension of the coordinate system
     */
    inline const double_t& Extension (const size_t& n) const {
        return m_L[n];
    };

    /*!
     *  Compute norm of the Point (R)
     */
    const double_t Norm() const {
        return Vector::Norm();
    };

    /*!
     *  Compute squared norm of the Point (R^2)
     */
    const double_t Norm2() const {
        return Vector::Norm2();
    };

    /*!
     *  Return Point that is shifted to be in the same (periodic) reference frame as the reference Point.
     *  @param ref reference Point
     */
    const Point ReferenceFrame(const Point& ref) const {

        Point val = *this;

        double_t dx = val.m_X[0] - ref.m_X[0];
        if(dx < -val.m_L[0]) val.m_X[0] += 2*val.m_L[0];
        else if (dx > val.m_L[0]) val.m_X[0] -= 2*val.m_L[0];

        double_t dy = m_X[1] - ref.m_X[1];
        if(dy < -val.m_L[1]) val.m_X[1] += 2*val.m_L[1];
        else if (dy > val.m_L[1]) val.m_X[1] -= 2*val.m_L[1];

        return val;

    };

    /*!
     *  Adds a Vector to the Point such that the result is in the reference frame of the coordinate system (0,0).
     *  @param ref Vector to be added
     */
    void operator+= (const Vector& ref) {
        m_X[0]+=ref[0];
        m_X[1]+=ref[1];

        if (m_X[0] < -m_L[0]) m_X[0] += 2*m_L[0];
        else if (m_X[0] > m_L[0]) m_X[0] -= 2*m_L[0];
        if (m_X[1] < -m_L[1]) m_X[1] += 2*m_L[1];
        else if (m_X[1] > m_L[1]) m_X[1] -= 2*m_L[1];
    };

    /*!
     *  Subtracts a Vector from the Point such that the result is in the reference frame of the coordinate system (0,0).
     *  @param ref Vector to be subtracted
     */
    void operator-= (const Vector& ref) {
        m_X[0]-=ref[0];
        m_X[1]-=ref[1];

        if (m_X[0] < -m_L[0]) m_X[0] += 2*m_L[0];
        else if (m_X[0] > m_L[0]) m_X[0] -= 2*m_L[0];
        if (m_X[1] < -m_L[1]) m_X[1] += 2*m_L[1];
        else if (m_X[1] > m_L[1]) m_X[1] -= 2*m_L[1];
    };

protected:

    std::array<double_t,2> m_L; ///< Boundaries of the measurement frame

    /*!
     *  Returns the Vector that spans the distance from the left hand side to the right hand side.
     *  @param lhs left hand side Point
     *  @param rhs right hand side Point
     */
    friend inline Vector operator- (Point lhs, const Point& rhs);

    /*!
     *  Adds a Vector to the Point such that the result is in the reference frame of the coordinate system (0,0).
     *  @param lhs Point to be modified
     *  @param rhs Vector to be added
     */
    friend inline Point operator- (Point lhs, const Vector& rhs);

    /*!
     *  Subtracts a Vector from the Point such that the result is in the reference frame of the coordinate system (0,0).
     *  @param lhs Point to be modified
     *  @param rhs Vector to be subtracted
     */
    friend inline Point operator+ (Point lhs, const Vector& rhs);

};

/*!
 *  Returns the Vector that spans the distance from the left hand side to the right hand side.
 *  @param lhs left hand side Point
 *  @param rhs right hand side Point
 */
inline Vector operator- (Point lhs, const Point& rhs) {

    lhs.m_X[0] -= rhs.m_X[0];
    lhs.m_X[1] -= rhs.m_X[1];

    if (lhs.m_X[0] < -lhs.m_L[0]) lhs.m_X[0] += 2*lhs.m_L[0];
    else if (lhs.m_X[0] > lhs.m_L[0]) lhs.m_X[0] -= 2*lhs.m_L[0];
    if (lhs.m_X[1] < -lhs.m_L[1]) lhs.m_X[1] += 2*lhs.m_L[1];
    else if (lhs.m_X[1] > lhs.m_L[1]) lhs.m_X[1] -= 2*lhs.m_L[1];

    return lhs;
}

/*!
 *  Adds a Vector to the Point such that the result is in the reference frame of the coordinate system (0,0).
 *  @param ref Vector to be added
 */
inline Point operator- (Point lhs, const Vector& rhs) {
    lhs -= rhs;
    return lhs;
}

/*!
 *  Subtracts a Vector from the Point such that the result is in the reference frame of the coordinate system (0,0).
 *  @param ref Vector to be subtracted
 */
inline Point operator+ (Point lhs, const Vector& rhs) {
    lhs += rhs;
    return lhs;
}

#endif /* Point_hpp */
