#ifndef CPM_Substrate_hpp
#define CPM_Substrate_hpp

#include <string>

#include <gsl/gsl_rng.h>

#include "CPM_Tile.hpp"
#include "IO_Parameters.hpp"

class Grid {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Grid() = delete;

    /*!
     *  Construct Grid from Parameters and the Random Number Generator
     *  @param  parameters   Parameters indicating substrate properties and size
     *  @param  RNG          Pointer to the Random Number Generator
     */
    Grid(Parameters* parameters, gsl_rng* RNG);

    /*!
     *  Copy constructor is not allowed.
     */
    Grid(Grid& base) = delete;

    /*
     *  Randomly choose a Tile.
     */
    Tile* RandomTile() {
        // roll random number
        size_t rnd = gsl_rng_uniform_int(g_RNG, m_Tiles.size());
        return &m_Tiles[rnd];
    };

    /*
     *  Randomly choose an unoccupied Tile. Returns a pointer to the corresponding Tile. Returns nullptr, if no unoccupied Tile exists.
     */
    Tile* RandomUnoccupiedTile() {
        // find unoccupied tiles
        std::vector<size_t> Unoccupied_Columns;
        std::vector<size_t> Unoccupied_Rows;
        for (auto& tile : m_Tiles) {
            if(tile.m_CheckSeed) {
                if(tile.m_Cell == nullptr) {
                    Unoccupied_Columns.push_back(Column(tile.m_Index));
                    Unoccupied_Rows.push_back(Row(tile.m_Index));
                }
            }
        }

        if(Unoccupied_Rows.empty()) return nullptr;

        size_t rnd = gsl_rng_uniform_int(g_RNG, Unoccupied_Columns.size());
        return &m_Tiles[Index(Unoccupied_Columns[rnd], Unoccupied_Rows[rnd])];

    };

    /*!
     *  Imprint substrate pattern on the Grid.
     *  @param parameters Parameters for substrate properties
     *  @param pattern    Pattern to be imprinted
     */
    void Imprint(Parameters* parameters, std::string pattern);

    /*!
     *  Update morphology (vertices, edge lengths, area) of all Tiles in the grid.
     *  This algorithm assumes that all tiles are simple hexagons, i.e. no have no edges intersect.
     */
    void UpdateMorphology();


    /*!
     *  Get vector of Tiles making up the grid.
     */
    const std::vector<Tile>& Tiles() const {
        return m_Tiles;
    };

private:

    /*!
     *  Imprint periodic pattern.
     *  @param parameters Parameters for substrate properties
     */
    void ImprintPeriodic(Parameters* parameters);

    /*!
     *  Imprint non-periodic pattern.
     *  @param parameters Parameters for substrate properties
     */
    void ImprintNonPeriodic(Parameters* parameters);

    /*!
     *  Imprint circular pattern.
     *  @param parameters Parameters for substrate properties
     */
    void ImprintCircular(Parameters* parameters);

    /*!
     *  Imprint donut pattern.
     *  @param parameters Parameters for substrate properties
     */
    void ImprintDonut(Parameters* parameters);

    /*!
     *  Imprint custom pattern.
     *  @param parameters Parameters for substrate properties
     */
    void ImprintCustom(Parameters* parameters, std::string pattern);

    /*!
     *  Compute Tile index from column and row
     *  @param col Tile column
     *  @param row Tile row
     */
    size_t Index(size_t col, size_t row) {
        return row * m_NCols + col;
    };

    /*!
     *  Compute Tile index from column and row, with a displacement by (dx, dy)
     *  @param col  Tile column
     *  @param row  Tile row
     *  @param dx   displacement in columns
     *  @param dy   displacement in rows
     */
    size_t Index(size_t col, size_t row, size_t dx, size_t dy);

    /*!
     *  Compute column from Tile index
     *  @param index Tile index
     */
    size_t Column(size_t index) {
        return index % m_NCols;
    };

    /*!
     *  Compute row from Tile index
     *  @param index Tile index
     */
    size_t Row(size_t index) {
        return index / m_NCols;
    };

    gsl_rng* g_RNG; ///< Pointer to Random Number Generator
    std::vector<Tile> m_Tiles; ///< Vector of Tiles in the grid
    const size_t m_NCols; ///< Number of columns in the grid
    const size_t m_NRows; ///< Number of rows in the grid

    friend class GridSolver;

};

#endif /* CPM_Substrate_hpp */
