#include "IO_Analysis.hpp"

Analysis::Analysis(Parameters* parameters)
{

    std::array<hsize_t,2> dims;

    H5::H5File H5Import ("Data.h5", H5F_ACC_RDONLY);
    H5::DataSet H5Dataset = H5Import.openDataSet("Raw_Data");
    H5::CompType H5Type = H5Dataset.getCompType();
    H5::DataSpace H5Dataspace = H5Dataset.getSpace();
    H5Dataspace.getSimpleExtentDims(dims.data());

    if(parameters->MeasureExtended()) {
        std::vector<DataCellExtendedImport> DataImport (dims.at(0) * dims.at(1));
        H5Dataset.read(DataImport.data(), H5Type);

        const double_t L_x = 0.5 * parameters->N_Columns();
        const double_t L_y = 0.25*std::sqrt(3.) * parameters->N_Rows();

        // Sort cell data by cell ID and group by time
        m_DataCellSort.reserve(DataImport.size());
        for (const auto& val : DataImport) {
            DataCell tmp ({
                val.Time,
                val.Cell_ID,
                val.Active,
                Point(val.Position_X, val.Position_Y, L_x, L_y),
                Vector(val.Velocity_X, val.Velocity_Y),
                Vector(val.Polarity_X, val.Polarity_Y),
                Vector(val.Gradient_X, val.Gradient_Y),
                Vector(val.PCA_EVEC_X, val.PCA_EVEC_Y),
                val.PCA_EVAL1,
                val.PCA_EVAL2,
                val.Area,
                val.Perimeter
            });
            m_DataCellSort.push_back(tmp);
        }

        // Extract slices from sorted data
        for (auto begin = m_DataCellSort.begin(), end = m_DataCellSort.end(), val = begin; val!=end; ++val) {
            if(val->Time != std::next(val)->Time || std::next(val) == end) {
                DataSlice tmp ({
                    val->Time,
                    begin,
                    std::next(val)
                });
                m_Slices.push_back(tmp);
                begin = std::next(val);
            }
        }

        // Averaging of the data
        for (auto slice = m_Slices.begin(); slice != m_Slices.end(); ++slice) {
            size_t count = 1;
            for (size_t dt = 1; dt < AveragingTime; ++dt) {
                auto slice2 = std::next(slice, dt);
                if (slice2 == m_Slices.end()) break;
                for (
                     auto cell = slice->Begin, cell2 = slice2->Begin;
                     cell != slice->End && cell2 != slice2->End;
                     ++cell, ++cell2
                     ) {
                    cell->Velocity += cell2->Velocity;
                    cell->Polarity += cell2->Polarity;
                    cell->PCA_EVEC += cell2->PCA_EVEC;
                    cell->PCA_EVAL1 += cell2->PCA_EVAL1;
                    cell->PCA_EVAL2 += cell2->PCA_EVAL2;
                    cell->Area += cell2->Area;
                    cell->Perimeter += cell2->Perimeter;
                }
                ++count;
            }

            for (
                 auto cell = slice->Begin;
                 cell != slice->End;
                 ++cell
                 ) {
                cell->Velocity /= count;
                cell->Polarity /= count;
                cell->PCA_EVEC /= count;
                cell->PCA_EVAL1 /= count;
                cell->PCA_EVAL2 /= count;
                cell->Area /= count;
                cell->Perimeter /= count;
            }
        }

        // Sort cell data by time and group by cell ID
        m_DataTimeSort.reserve(DataImport.size());
        size_t cellID = 0;
        while (m_DataTimeSort.size() != m_DataCellSort.size()) {
            for (auto& slice : m_Slices) {
                m_DataTimeSort.push_back(*std::next(slice.Begin,cellID));
            }
            ++cellID;
        }

        // Extract cell tracks from sorted data
        for (auto begin = m_DataTimeSort.begin(), end = m_DataTimeSort.end(), val = begin; val!=end; ++val) {
            if(val->Cell_ID != std::next(val)->Cell_ID || std::next(val) == end) {
                DataTrack tmp ({
                    val->Cell_ID,
                    begin,
                    std::next(val)
                });
                m_Tracks.push_back(tmp);
                begin = std::next(val);
            }
        }

        file = H5::H5File("Analysis.h5", H5F_ACC_TRUNC);
        {
            H5::Group group = file.createGroup("Simple");
            group.close();
        }

        {
            H5::Group group = file.createGroup("Detail");
            group.close();
        }

        {
            H5::Group group = file.createGroup("Averages");
            group.close();
        }
    }
    else {
        throw std::logic_error("Reduced Analysis not implemented yet!");
    }

}

/*!
 *  Compute Mean Square Displacement and save to File
 */
void Analysis::ComputeMSD() {
    std::cout<<"Computing MSD"<<std::endl;
    const size_t N = m_Slices.size() / MeasuringTime;
    std::vector<Data1D> output1D (N, Data1D({0.,0.,0}));

    for (auto track : m_Tracks) {
        for(auto begin = track.Begin, end = track.End, cellT0 = begin; cellT0 != end; ++cellT0) {

            if(!cellT0->Area) continue;

            double_t t0 = cellT0->Time;
            Vector dR (0.,0.);

            for(auto cellT = cellT0; cellT != end; ++cellT) {

                if(!cellT->Area) continue;

                size_t dN = std::distance(cellT0, cellT) / MeasuringTime;
                double_t t = cellT->Time;
                double_t dt = t - t0;
                Point cellTPos = cellT->Position;
                double_t dR2 = dR.Norm2();

                output1D[dN].Coordinate = dt;
                output1D[dN].Mean += dR2;
                ++output1D[dN].Count;

                dR += cellT->Velocity;

            }
        }
    }

    for (auto& val : output1D){
        if(!val.Count) continue;
        val.Mean /= val.Count;
    }

    H5::CompType Type1D (sizeof(Data1D));
    Type1D.insertMember("DeltaTime", HOFFSET(Data1D, Coordinate), H5::PredType::NATIVE_DOUBLE);
    Type1D.insertMember("Mean", HOFFSET(Data1D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type1D.insertMember("Count", HOFFSET(Data1D, Count), H5::PredType::NATIVE_UINT_FAST32);

    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {output1D.size()};
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Simple");
        H5::DataSet dataset = group.createDataSet("MSD", Type1D, dataspace);
        dataset.write(output1D.data(), Type1D);
        dataset.close();
        group.close();
        dataspace.close();
    }
}

/*!
 *  Compute VACF and save to File
 */
void Analysis::ComputeVACF() {
    std::cout<<"Computing VACF"<<std::endl;
    const size_t N = m_Slices.size() / MeasuringTime;
    std::vector<Data1D> output1D (N, Data1D({0.,0.,0}));

    for (auto track : m_Tracks) {
        for(auto begin = track.Begin, end = track.End, cellT0 = begin; cellT0 != end; ++cellT0) {

            if(!cellT0->Area) continue;

            double_t t0 = cellT0->Time;

            Vector cellT0Vel = cellT0->Velocity;
            double_t cellT0Spd = cellT0Vel.Norm();
            if(cellT0Spd<0.00001) continue;
            cellT0Vel /= cellT0Spd;

            for(auto cellT = cellT0; cellT != end; ++cellT) {

                if(!cellT->Area) continue;

                size_t dN = std::distance(cellT0, cellT) / MeasuringTime;
                double_t t = cellT->Time;
                double_t dt = t - t0;

                Vector cellTVel = cellT->Velocity;
                double_t cellTSpd = cellTVel.Norm();
                if(cellTSpd<0.00001) continue;
                cellTVel /= cellTSpd;

                double_t val = cellT0Vel * cellTVel;

                output1D[dN].Coordinate = dt;
                output1D[dN].Mean += val;
                ++output1D[dN].Count;

            }
        }
    }

    for (auto& val : output1D){
        if(!val.Count) continue;
        val.Mean /= val.Count;
    }

    H5::CompType Type1D (sizeof(Data1D));
    Type1D.insertMember("DeltaTime", HOFFSET(Data1D, Coordinate), H5::PredType::NATIVE_DOUBLE);
    Type1D.insertMember("Mean", HOFFSET(Data1D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type1D.insertMember("Count", HOFFSET(Data1D, Count), H5::PredType::NATIVE_UINT_FAST32);

    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {output1D.size()};
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Simple");
        H5::DataSet dataset = group.createDataSet("VACF", Type1D, dataspace);
        dataset.write(output1D.data(), Type1D);
        dataset.close();
        group.close();
        dataspace.close();
    }
};

void Analysis::ComputeAverages() {

    std::cout <<"Computing Averages" <<std::endl;

    Data0D avg_Area = {0.,0.,0};
    Data0D avg_Perimeter = {0.,0.,0};
    Data0D avg_Speed = {0.,0.,0};
    Data0D avg_AspectRatio = {0.,0.,0};

    for (const auto& cell : m_DataCellSort) {
        avg_Area.Mean += cell.Area;
        avg_Area.Variance += cell.Area * cell.Area;
        ++avg_Area.Count;

        avg_Perimeter.Mean += cell.Perimeter;
        avg_Perimeter.Variance += cell.Perimeter * cell.Perimeter;
        ++avg_Perimeter.Count;

        double_t speed = cell.Velocity.Norm();
        avg_Speed.Mean += speed;
        avg_Speed.Variance += speed * speed;
        ++avg_Speed.Count;

        double_t aspect = cell.PCA_EVAL1 / cell.PCA_EVAL2;
        avg_AspectRatio.Mean += aspect;
        avg_AspectRatio.Variance += aspect * aspect;
        ++avg_AspectRatio.Count;

    }

    H5::CompType Type0D (sizeof(Data0D));
    Type0D.insertMember("Mean", HOFFSET(Data0D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Variance", HOFFSET(Data0D, Variance), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Count", HOFFSET(Data0D, Count), H5::PredType::NATIVE_UINT_FAST32);

    avg_Area.Mean /= avg_Area.Count;
    avg_Area.Variance /= avg_Area.Count;
    avg_Area.Variance -= avg_Area.Mean * avg_Area.Mean;
    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Area);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Area", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_Perimeter.Mean /= avg_Perimeter.Count;
    avg_Perimeter.Variance /= avg_Perimeter.Count;
    avg_Perimeter.Variance -= avg_Perimeter.Mean * avg_Perimeter.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Perimeter);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Perimeter", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_Speed.Mean /= avg_Speed.Count;
    avg_Speed.Variance /= avg_Speed.Count;
    avg_Speed.Variance -= avg_Speed.Mean * avg_Speed.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Speed);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Speed", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_AspectRatio.Mean /= avg_AspectRatio.Count;
    avg_AspectRatio.Variance /= avg_AspectRatio.Count;
    avg_AspectRatio.Variance -= avg_AspectRatio.Mean * avg_AspectRatio.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_AspectRatio);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("AspectRatio", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

}

void Analysis::ComputeAngularVelocity() {

    Data0D avg_Angular_Velocity = {0.,0.,0};
    for (auto track : m_Tracks) {
        for(auto begin = track.Begin, end = track.End, cellT0 = begin; cellT0 != std::prev(end); ++cellT0) {

            auto cellT1 = std::next(cellT0);

            if(!cellT0->Area) continue;
            if(!cellT1->Area) continue;

            Vector cellT0Vel = cellT0->Velocity;
            double_t cellT0Spd = cellT0Vel.Norm();
            if(cellT0Spd<0.00001) continue;
            Vector cellT0Dir = cellT0Vel / cellT0Spd;

            Vector cellT1Vel = cellT1->Velocity;
            double_t cellT1Spd = cellT1Vel.Norm();
            if(cellT1Spd<0.00001) continue;
            Vector cellT1Dir = cellT1Vel / cellT1Spd;

            Vector dv = (cellT1Dir - cellT0Dir) / cellT0Spd;
            Vector n = Vector(cellT0Dir[1],cellT0Dir[0]);

            double_t val = std::abs(dv * n);

            avg_Angular_Velocity.Mean += val;
            avg_Angular_Velocity.Variance += val * val;
            ++avg_Angular_Velocity.Count;
        }
    }

    H5::CompType Type0D (sizeof(Data0D));
    Type0D.insertMember("Mean", HOFFSET(Data0D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Variance", HOFFSET(Data0D, Variance), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Count", HOFFSET(Data0D, Count), H5::PredType::NATIVE_UINT_FAST32);

    avg_Angular_Velocity.Mean /= avg_Angular_Velocity.Count;
    avg_Angular_Velocity.Variance /= avg_Angular_Velocity.Count;
    avg_Angular_Velocity.Variance -= avg_Angular_Velocity.Mean * avg_Angular_Velocity.Mean;
    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Angular_Velocity);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Curvature", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

}

void Analysis::ComputeAngularVelocity10() {

    Data0D avg_Angular_Velocity = {0.,0.,0};
    for (auto track : m_Tracks) {
        for(auto begin = track.Begin, end = track.End, cellT0 = begin; cellT0 != std::prev(end,20); cellT0 = std::next(cellT0,10) ) {

            Vector cellT0Vel (0.,0.);
            for (int i = 0; i<10; ++i) {
                cellT0Vel+= std::next(cellT0,i)->Velocity;
            }
            double_t cellT0Spd = cellT0Vel.Norm();
            if(cellT0Spd<0.0001) continue;
            Vector cellT0Dir = cellT0Vel / cellT0Spd;

            Vector cellT1Vel (0.,0.);
            for (int i = 10; i<20; ++i) {
                cellT1Vel+= std::next(cellT0,i)->Velocity;
            }
            double_t cellT1Spd = cellT1Vel.Norm();
            if(cellT1Spd<0.0001) continue;
            Vector cellT1Dir = cellT1Vel / cellT1Spd;

            Vector dv = (cellT1Dir - cellT0Dir) / cellT0Spd;
            Vector n = Vector(cellT0Dir[1],cellT0Dir[0]);

            double_t val = std::abs(dv * n);

            avg_Angular_Velocity.Mean += val;
            avg_Angular_Velocity.Variance += val * val;
            ++avg_Angular_Velocity.Count;
        }
    }

    H5::CompType Type0D (sizeof(Data0D));
    Type0D.insertMember("Mean", HOFFSET(Data0D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Variance", HOFFSET(Data0D, Variance), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Count", HOFFSET(Data0D, Count), H5::PredType::NATIVE_UINT_FAST32);

    avg_Angular_Velocity.Mean /= avg_Angular_Velocity.Count;
    avg_Angular_Velocity.Variance /= avg_Angular_Velocity.Count;
    avg_Angular_Velocity.Variance -= avg_Angular_Velocity.Mean * avg_Angular_Velocity.Mean;
    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Angular_Velocity);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Curvature_10", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

}
