#ifndef IO_Analysis_hpp
#define IO_Analysis_hpp

#include <stdio.h>
#include <cmath>
#include <array>
#include <vector>
#include <algorithm>

#include "IO_ParametersReduced.hpp"
#include "Point.hpp"

#include <hdf5.h>
#include <H5Cpp.h>

static const size_t AveragingTime = 1;
static const size_t MeasuringTime = 1;
//static const size_t MaximalTime = 2000;

class Analysis {

public:

    /*!
     *  Simple export struct containing cell data.
     */
    struct DataCellReducedImport {
        double_t Time;          ///< current time
        size_t Cell_ID;         ///< index of the current cell
        bool Active;            ///< flag indicating whether this cell can polarize
        double_t Position_X;    ///< position (x) of the current cell
        double_t Position_Y;    ///< position (y) of the current cell
        double_t Velocity_X;    ///< velocity (x) of the current cell
        double_t Velocity_Y;    ///< velocity (y) of the current cell
        double_t Area;          ///< area of the current cell
        double_t Perimeter;     ///< perimeter of the current cell
    };

    /*!
     *  Simple export struct containing cell data.
     */
    struct DataCellExtendedImport {
        double_t Time;          ///< current time
        size_t Cell_ID;         ///< index of the current cell
        bool Active;            ///< flag indicating whether this cell can polarize
        double_t Position_X;    ///< position (x) of the current cell
        double_t Position_Y;    ///< position (y) of the current cell
        double_t Velocity_X;    ///< velocity (x) of the current cell
        double_t Velocity_Y;    ///< velocity (y) of the current cell
        double_t Polarity_X;    ///< polarity vector (x) of the current cell
        double_t Polarity_Y;    ///< polarity vector (y) of the current cell
        double_t Gradient_X;    ///< Polarization gradient (x) of the current cell
        double_t Gradient_Y;    ///< Polarization gradient (y) of the current cell
        double_t PCA_EVEC_X;    ///< shape eigenvector (x) of the current cell
        double_t PCA_EVEC_Y;    ///< shape eigenvector (y) of the current cell
        double_t PCA_EVAL1;     ///< shape eigenMean (>) of the current cell
        double_t PCA_EVAL2;     ///< shape eigenMean (<) of the current cell
        double_t Area;          ///< area of the current cell
        double_t Perimeter;     ///< perimeter of the current cell
    };

    /*!
     *  Simple export struct containing cell data.
     */
    struct DataCell {
        const double_t Time;    ///< current time
        const size_t Cell_ID;   ///< index of the current cell
        const bool Active;      ///< flag indicating whether this cell can polarize
        Point Position;         ///< position (x) of the current cell
        Vector Velocity;        ///< velocity (x) of the current cell
        Vector Polarity;        ///< polarity (x) of the current cell
        Vector Gradient;        ///< Polarization gradient of the current cell
        Vector PCA_EVEC;        ///< shape eigenvector of the current cell
        double_t PCA_EVAL1;     ///< shape eigenMean (>) of the current cell
        double_t PCA_EVAL2;     ///< shape eigenMean (<) of the current cell
        double_t Area;          ///< area of the current cell
        double_t Perimeter;     ///< perimeter of the current cell
    };

    bool CompareByTime(const DataCell& a, const DataCell& b)
    {
        // smallest comes first
        return a.Time < b.Time;
    };

    bool CompareByCellID(const DataCell& a, const DataCell& b)
    {
        // smallest comes first
        return a.Cell_ID < b.Cell_ID;
    };

    struct DataTrack {
        size_t Cell_ID;
        std::vector<DataCell>::iterator Begin;
        std::vector<DataCell>::iterator End;
    };

    struct DataSlice {
        double_t Time;
        std::vector<DataCell>::iterator Begin;
        std::vector<DataCell>::iterator End;
    };

    /*!
     *  Simple export struct for averaged data.
     */
    struct Data0D {
        double_t Mean;
        double_t Variance;
        uint_fast32_t Count;
    };

    /*!
     *  Simple export struct for linear data.
     */
    struct Data1D {
        double_t Coordinate;
        double_t Mean;
        uint_fast32_t Count;
    };

    /*!
     *  Simple export struct for linear data.
     */
    struct Data2D {
        double_t Coordinate1;
        double_t Coordinate2;
        double_t Mean;
        uint_fast32_t Count;
    };

    Analysis(Parameters* parameters);

    ~Analysis() {
        file.close();
    };

    /*!
     *  Compute mean square displacement of single cell
     */
    void ComputeMSD();

    /*!
     *  Compute velocity correlation of single cell
     */
    void ComputeVACF();

    /*!
     *  Compute averages
     */
    void ComputeAverages();

    /*!
     *  Compute angular velocity of single cell
     */
    void ComputeAngularVelocity();

    /*!
     *  Compute angular velocity of single cell
     */
    void ComputeAngularVelocity10();

private:

    std::vector<DataCell> m_DataCellSort;             ///< Data to be analyzed
    std::vector<DataCell> m_DataTimeSort;             ///< Data to be analyzed
    std::vector<DataTrack> m_Tracks;          ///< Container storing track metadata
    std::vector<DataSlice> m_Slices;          ///< Container storing slice metadata

    H5::H5File file; ///< Output file

};

#endif /* IO_Analysis_hpp */
