#include "IO_ParametersReduced.hpp"

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/json_parser.hpp>

using namespace boost::property_tree;

/*!
 *  Template helper function for reading parameters from ptree or variables_map
 *  @param  pt          ptree storing parameters
 *  @param  group       Parameter group containing the parameter
 *  @param  fullname    Full name of the parameter
 *  @param  variable    Storage of the parameter
 */
template <typename T> void read(ptree& pt, std::string group, std::string fullname, T& variable){

    try {
        // Else try to read from JSON
        variable = pt.get<T>(group + std::string(".") + fullname);
    }
    catch (const ptree_bad_data) {
        // If all fails, revert to generic value and print warning
        std::cout << "Bad data: " + fullname + "." << std::endl;
    }
    catch (const ptree_bad_path) {
        // If all fails, revert to generic value and print warning
        std::cout << "Bad path: " + fullname + "." << std::endl;
    }

}

/*!
 *  Tries to read configuration either from .json file or from the command line. If this fails, it rolls back to the predefined values. At the end of the function, it saves the updated parameters in the .json file.
 */
Parameters::Parameters(int argc, char* argv[]){

    ptree pt;

    ///////////////////////////
    // Read options from JSON //
    ///////////////////////////

    try {
        read_json("config.json", pt);
    } catch (const json_parser_error) {
        throw std::logic_error("Could not find Configuration file.");
    }

    ////////////////////////////////////////////
    // Read options from JSON                 //
    ////////////////////////////////////////////

    // read parameters
    read<size_t>(pt, "POPULATION", "NUM_CELLS", m_N_Cells);
    read<size_t>(pt, "SUBSTRATE", "NUM_COLUMNS", m_N_Columns);
    read<size_t>(pt, "SUBSTRATE", "NUM_ROWS", m_N_Rows);
    read<uint_fast32_t>(pt, "OUTPUT", "INTERVAL_MEASUREMENT", m_I_Measure);
    read<bool>(pt, "OUTPUT", "CELL_EXTENDED", m_CellMeasureExtended);
    read<bool>(pt, "OUTPUT", "CELL_MAP", m_CellMeasureMaps);
    read<bool>(pt, "OUTPUT", "MONOLAYER", m_MeasureMono);

    // Perform integrity checks
    if (m_N_Rows%2 != 0) {
        m_N_Rows += 1;
        std::cout<< "m_N_Rows must be an even integer. m_N_Rows has been adjusted to " << std::to_string(m_N_Rows) << "."<<std::endl;
    }

}
