#include <iostream>
#include <chrono>

#include <gsl/gsl_rng.h>

#include "Vector.hpp"
#include "Point.hpp"

#include "CPM_Grid.hpp"
#include "CPM_GridSolver.hpp"
#include "CPM_Population.hpp"
#include "CPM_Hamiltonian.hpp"

#include "IO_Parameters.hpp"
#include "IO_Data.hpp"
#include "IO_Graphics.hpp"

void Simulation(Grid* grid, Population* population, Hamiltonian* hamiltonian, uint_fast32_t NSteps, Data* data = nullptr, Graphics* graphics = nullptr, GridSolver* gridsolver = nullptr) {


    // Initialize cell positions at beginning of simulation (regardless if they were previously updated or not)
    for (auto& cell : population->Cells()) {
        cell.MoveVector();
    }

    std::cout << "   Begin [==========] End" << std::endl;
    std::cout << "Progress [";

    for(uint_fast32_t step=0; step < NSteps; ++step) {

        if(step % (NSteps / 10) == 0) std::cout << "=";

        // Determine amount of move attempts
        size_t size = 0;
        for (const auto& cell : population->Cells()) size += cell.Membrane().size();

        for (size_t i = 0; i<size; ++i) {

            // Choose random cell
            Cell* ChosenCell = population->RandomCell();

            // Choose random edge
            Edge* ChosenEdge = ChosenCell->RandomEdge();

            // What is the target cell?
            Cell* TargetCell = ChosenEdge->GetComp()->GetCell();

            // Determine what the chosen cell will do
            Hamiltonian::Event Choice = hamiltonian->MakeDecision(ChosenEdge);

            switch (Choice) {
                case Hamiltonian::Event::Protrude:
                    if (TargetCell) TargetCell->Retract(ChosenEdge->GetComp());
                    ChosenCell->Protrude(ChosenEdge);
                    break;
                case Hamiltonian::Event::Retract:
                    ChosenCell->Retract(ChosenEdge);
                    if (TargetCell) TargetCell->Protrude(ChosenEdge->GetComp());
                    break;
                case Hamiltonian::Event::Rupture:
                    ChosenCell->Retract(ChosenEdge);
                    break;
                case Hamiltonian::Event::Remain:
                    break;
            }
        }


        // Update cell population
        population->Update();

        // If Data-Handler is passed, attempt to print data
        if(data) data->Write();

    }

    std::cout << "] Finished" << std::endl;

};

int main(int argc, char * argv[]) {

    try {

        auto time0 = std::chrono::system_clock::now();

        // Allocate space for parameters and RNG
        Parameters* parameters = new Parameters(argc, argv);

        std::cout
        << "Allocating Random Number Generator."
        << std::endl;
        gsl_rng* RNG = gsl_rng_alloc(gsl_rng_mt19937);
        gsl_rng_set(RNG, parameters->Seed());

        // Allocate space for grid and population
        std::cout
        << "Allocating Substrate Grid."
        << std::endl;
        Grid* grid = new Grid(parameters, RNG);

        // Imprint presimulation pattern on Grid
        std::cout
        << "Imprinting Growth Stencil on Grid."
        << std::endl;
        grid->Imprint(parameters, parameters->S_PatternPresim());

        // Seed cell population on Grid
        std::cout
        << "Seeding Cell Population."
        << std::endl;
        Population* population = new Population(grid, parameters, RNG);

        Hamiltonian* hamiltonian = new Hamiltonian(parameters, RNG);

        // Perform pre-simulation (growth)
        hamiltonian->SetAdhesion(parameters);

        std::cout
        << std::endl << "Beginning Growth Phase."
        << std::endl;
        Simulation(grid, population, hamiltonian, parameters->N_MCSPresim());

        std::cout
        << "Growth Phase ended. Elapsed Time "
        << std::chrono::duration_cast<std::chrono::seconds>(std::chrono::system_clock::now() - time0).count()
        << " Seconds."
        << std::endl
        << std::endl;

        // Imprint simulation pattern on Grid
        std::cout
        << "Imprinting Simulation Stencil on Grid."
        << std::endl;
        grid->Imprint(parameters, parameters->S_PatternSim());

        Graphics* graphics = nullptr;

        // Initialize Data Handler
        Data* data = parameters->I_Measure() > 0 ? new Data(grid, population, hamiltonian, parameters) : nullptr;

        GridSolver* gridsolver = nullptr;

        // Perform simulation
        hamiltonian->SetAdhesion(parameters);
        population->SetMotilityState(Cell::MotilityState::Motile);
        population->SetProliferationState(Cell::ProliferationState::Quiescence);

        std::cout
        << std::endl
        << "Beginning Simulation Phase."
        << std::endl;
        Simulation(grid, population, hamiltonian, parameters->N_MCSSim(), data, graphics, gridsolver);

        std::cout
        << "Simulation Phase ended. Elapsed Time "
        << std::chrono::duration_cast<std::chrono::seconds>(std::chrono::system_clock::now() - time0).count()
        << " Seconds."
        << std::endl
        << std::endl;

        // free all allocated space
        delete data;
        delete grid;
        delete population;
        delete parameters;

        gsl_rng_free(RNG);

        std::cout
        << "Deallocated memory. Elapsed Time "
        << std::chrono::duration_cast<std::chrono::seconds>(std::chrono::system_clock::now() - time0).count()
        << " Seconds."
        << std::endl;

        std::cout
        << "Simulation finished successfully."
        << std::endl;

        // return success
        return EXIT_SUCCESS;


    } catch (std::exception &e) {
        std::cerr << e.what() <<std::endl;

        std::cout
        << "Simulation aborted."
        << std::endl;

        // return failure
        return EXIT_FAILURE;
    }

}
