#include "IO_Analysis.hpp"

Analysis::Analysis(Parameters* parameters)
: L_x(0.5 * parameters->N_Columns())
, L_y(0.25*std::sqrt(3.) * parameters->N_Rows())
{

    std::array<hsize_t,2> dims;

    H5::H5File H5Import ("Data.h5", H5F_ACC_RDONLY);
    H5::DataSet H5Dataset = H5Import.openDataSet("Raw_Data");
    H5::CompType H5Type = H5Dataset.getCompType();
    H5::DataSpace H5Dataspace = H5Dataset.getSpace();
    H5Dataspace.getSimpleExtentDims(dims.data());

    if(parameters->MeasureExtended()) {
        std::vector<DataCellExtendedImport> DataImport (dims.at(0) * dims.at(1));
        H5Dataset.read(DataImport.data(), H5Type);

        // Sort cell data by cell ID and group by time
        m_DataCellSort.reserve(DataImport.size());
        for (const auto& val : DataImport) {
            DataCell tmp ({
                val.Time,
                val.Cell_ID,
                val.Active,
                Point(val.Position_X, val.Position_Y, L_x, L_y),
                Vector(val.Velocity_X, val.Velocity_Y),
                Vector(val.Polarity_X, val.Polarity_Y),
                Vector(val.Gradient_X, val.Gradient_Y),
                Vector(val.PCA_EVEC_X, val.PCA_EVEC_Y),
                val.PCA_EVAL1,
                val.PCA_EVAL2,
                val.Area,
                val.Perimeter
            });
            m_DataCellSort.push_back(tmp);
        }

        // Extract slices from sorted data
        for (auto begin = m_DataCellSort.begin(), end = m_DataCellSort.end(), val = begin; val!=end; ++val) {
            if(val->Time != std::next(val)->Time || std::next(val) == end) {
                DataSlice tmp ({
                    val->Time,
                    begin,
                    std::next(val)
                });
                m_Slices.push_back(tmp);
                begin = std::next(val);
            }
        }

        // Averaging of the data
        for (auto slice = m_Slices.begin(); slice != m_Slices.end(); ++slice) {
            size_t count = 1;
            for (size_t dt = 1; dt < AveragingTime; ++dt) {
                auto slice2 = std::next(slice, dt);
                if (slice2 == m_Slices.end()) break;
                for (
                     auto cell = slice->Begin, cell2 = slice2->Begin;
                     cell != slice->End && cell2 != slice2->End;
                     ++cell, ++cell2
                     ) {
                    cell->Velocity += cell2->Velocity;
                    cell->Polarity += cell2->Polarity;
                    cell->PCA_EVEC += cell2->PCA_EVEC;
                    cell->PCA_EVAL1 += cell2->PCA_EVAL1;
                    cell->PCA_EVAL2 += cell2->PCA_EVAL2;
                    cell->Area += cell2->Area;
                    cell->Perimeter += cell2->Perimeter;
                }
                ++count;
            }

            for (
                 auto cell = slice->Begin;
                 cell != slice->End;
                 ++cell
                 ) {
                cell->Velocity /= count;
                cell->Polarity /= count;
                cell->PCA_EVEC /= count;
                cell->PCA_EVAL1 /= count;
                cell->PCA_EVAL2 /= count;
                cell->Area /= count;
                cell->Perimeter /= count;
            }
        }

        // Sort cell data by time and group by cell ID
        m_DataTimeSort.reserve(DataImport.size());
        size_t cellID = 0;
        while (m_DataTimeSort.size() != m_DataCellSort.size()) {
            for (auto& slice : m_Slices) {
                m_DataTimeSort.push_back(*std::next(slice.Begin,cellID));
            }
            ++cellID;
        }

        // Extract cell tracks from sorted data
        for (auto begin = m_DataTimeSort.begin(), end = m_DataTimeSort.end(), val = begin; val!=end; ++val) {
            if(val->Cell_ID != std::next(val)->Cell_ID || std::next(val) == end) {
                DataTrack tmp ({
                    val->Cell_ID,
                    begin,
                    std::next(val)
                });
                m_Tracks.push_back(tmp);
                begin = std::next(val);
            }
        }

        file = H5::H5File("Analysis.h5", H5F_ACC_TRUNC);
        {
            H5::Group group = file.createGroup("Simple");
            group.close();
        }

        {
            H5::Group group = file.createGroup("Detail");
            group.close();
        }

        {
            H5::Group group = file.createGroup("Averages");
            group.close();
        }
    }
    else {
        throw std::logic_error("Reduced Analysis not implemented yet!");
    }

}

void Analysis::ComputeAverages() {

    std::cout <<"Computing Averages" <<std::endl;

    Data0D avg_Area = {0.,0.,0};
    Data0D avg_Perimeter = {0.,0.,0};
    Data0D avg_Speed = {0.,0.,0};
    Data0D avg_ShapeFactor = {0.,0.,0};
    Data0D avg_AspectRatio = {0.,0.,0};
    Data0D avg_AngularVelocity = {0.,0.,0};
    Data0D avg_NormalizedAngularVelocity = {0.,0.,0};
    Data0D avg_AngularVelocity_CoM = {0.,0.,0};
    Data0D avg_NormalizedAngularVelocity_CoM = {0.,0.,0};

    Point center (0.,0.,L_x,L_y);
    Vector center_dx (0.,0.);
    double_t tot_Area = 0.;
    size_t begin_ID = m_DataCellSort.front().Cell_ID;

    for (const auto& cell : m_DataCellSort) {

        avg_Area.Mean += cell.Area;
        avg_Area.Variance += cell.Area * cell.Area;
        ++avg_Area.Count;

        avg_Perimeter.Mean += cell.Perimeter;
        avg_Perimeter.Variance += cell.Perimeter * cell.Perimeter;
        ++avg_Perimeter.Count;

        double_t speed = cell.Velocity.Norm();
        avg_Speed.Mean += speed;
        avg_Speed.Variance += speed * speed;
        ++avg_Speed.Count;

        double_t sfactor = 1. - 4*M_PI*cell.Area / (cell.Perimeter * cell.Perimeter);
        avg_ShapeFactor.Mean += sfactor;
        avg_ShapeFactor.Variance += sfactor * sfactor;
        ++avg_ShapeFactor.Count;

        double_t aspect = cell.PCA_EVAL1 / cell.PCA_EVAL2;
        avg_AspectRatio.Mean += aspect;
        avg_AspectRatio.Variance += aspect * aspect;
        ++avg_AspectRatio.Count;

        double_t Omega = std::abs( (cell.Position[0] * cell.Velocity[1] - cell.Position[1] * cell.Velocity[0]) / cell.Position.Norm2() );
        avg_AngularVelocity.Mean += Omega;
        avg_AngularVelocity.Variance += Omega * Omega;
        ++avg_AngularVelocity.Count;

        double_t OmegaNorm = std::abs( (cell.Position[0] * cell.Velocity[1] - cell.Position[1] * cell.Velocity[0]) / (cell.Position.Norm() * cell.Velocity.Norm() ) );
        avg_NormalizedAngularVelocity.Mean += OmegaNorm;
        avg_NormalizedAngularVelocity.Variance += OmegaNorm * OmegaNorm;
        ++avg_NormalizedAngularVelocity.Count;

        if(cell.Cell_ID==begin_ID) {
            center = Point (0.,0.,L_x,L_y);
            center_dx = Vector (0.,0.);
            tot_Area = 0.;
        }
        center_dx += cell.Area * (cell.Position - center);
        tot_Area += cell.Area;

    }

    center_dx /= tot_Area;
    center += center_dx;

    for (const auto& cell : m_DataCellSort) {

        Vector dx = cell.Position - center;

        double_t Omega = std::abs( (dx[0] * cell.Velocity[1] - dx[1] * cell.Velocity[0]) / dx.Norm2() );
        avg_AngularVelocity_CoM.Mean += Omega;
        avg_AngularVelocity_CoM.Variance += Omega * Omega;
        ++avg_AngularVelocity_CoM.Count;

        double_t OmegaNorm = std::abs( (dx[0] * cell.Velocity[1] - dx[1] * cell.Velocity[0]) / (dx.Norm() * cell.Velocity.Norm() ) );
        avg_NormalizedAngularVelocity_CoM.Mean += OmegaNorm;
        avg_NormalizedAngularVelocity_CoM.Variance += OmegaNorm * OmegaNorm;
        ++avg_NormalizedAngularVelocity_CoM.Count;

    }

    H5::CompType Type0D (sizeof(Data0D));
    Type0D.insertMember("Mean", HOFFSET(Data0D, Mean), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Variance", HOFFSET(Data0D, Variance), H5::PredType::NATIVE_DOUBLE);
    Type0D.insertMember("Count", HOFFSET(Data0D, Count), H5::PredType::NATIVE_UINT_FAST32);

    avg_Area.Mean /= avg_Area.Count;
    avg_Area.Variance /= avg_Area.Count;
    avg_Area.Variance -= avg_Area.Mean * avg_Area.Mean;
    // Create the dataset with chunking enabled.
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Area);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Area", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_Perimeter.Mean /= avg_Perimeter.Count;
    avg_Perimeter.Variance /= avg_Perimeter.Count;
    avg_Perimeter.Variance -= avg_Perimeter.Mean * avg_Perimeter.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Perimeter);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Perimeter", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_Speed.Mean /= avg_Speed.Count;
    avg_Speed.Variance /= avg_Speed.Count;
    avg_Speed.Variance -= avg_Speed.Mean * avg_Speed.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_Speed);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("Speed", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_ShapeFactor.Mean /= avg_ShapeFactor.Count;
    avg_ShapeFactor.Variance /= avg_ShapeFactor.Count;
    avg_ShapeFactor.Variance -= avg_ShapeFactor.Mean * avg_ShapeFactor.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_ShapeFactor);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("ShapeFactor", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_AspectRatio.Mean /= avg_AspectRatio.Count;
    avg_AspectRatio.Variance /= avg_AspectRatio.Count;
    avg_AspectRatio.Variance -= avg_AspectRatio.Mean * avg_AspectRatio.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_AspectRatio);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("AspectRatio", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_AngularVelocity.Mean /= avg_AngularVelocity.Count;
    avg_AngularVelocity.Variance /= avg_AngularVelocity.Count;
    avg_AngularVelocity.Variance -= avg_AngularVelocity.Mean * avg_AngularVelocity.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_AngularVelocity);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("AngularVelocity", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_NormalizedAngularVelocity.Mean /= avg_NormalizedAngularVelocity.Count;
    avg_NormalizedAngularVelocity.Variance /= avg_NormalizedAngularVelocity.Count;
    avg_NormalizedAngularVelocity.Variance -= avg_NormalizedAngularVelocity.Mean * avg_NormalizedAngularVelocity.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_NormalizedAngularVelocity);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("NormalizedAngularVelocity", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_AngularVelocity_CoM.Mean /= avg_AngularVelocity_CoM.Count;
    avg_AngularVelocity_CoM.Variance /= avg_AngularVelocity_CoM.Count;
    avg_AngularVelocity_CoM.Variance -= avg_AngularVelocity_CoM.Mean * avg_AngularVelocity_CoM.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_AngularVelocity_CoM);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("AngularVelocity_CoM", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

    avg_NormalizedAngularVelocity_CoM.Mean /= avg_NormalizedAngularVelocity_CoM.Count;
    avg_NormalizedAngularVelocity_CoM.Variance /= avg_NormalizedAngularVelocity_CoM.Count;
    avg_NormalizedAngularVelocity_CoM.Variance -= avg_NormalizedAngularVelocity_CoM.Mean * avg_NormalizedAngularVelocity_CoM.Mean;
    {
        hsize_t dims[1] = {1};
        std::vector<Data0D> data;
        data.push_back(avg_NormalizedAngularVelocity_CoM);
        H5::DataSpace dataspace(1, dims);
        H5::Group group = file.openGroup("Averages");
        H5::DataSet dataset = group.createDataSet("NormalizedAngularVelocity_CoM", Type0D, dataspace);
        dataset.write(data.data(), Type0D);
        dataset.close();
        group.close();
        dataspace.close();
    }

}
