#include "CPM_Cell.hpp"
#include <iostream>

/*!
 *  Perform a protrution Event.
 *  @param input Edge along which the cell protrudes
 */
void Cell::Protrude(Edge* input) {

    Tile* target = input->m_Comp->m_Tile;

    for (size_t i = 0; i != 6 ; ++i) {

        Edge* edge = &target->m_Edges[i];
        Edge* comp = edge->m_Comp;

        if (comp->m_Cell == this) {
            // remove edge from membrane by swap-popback, adjust index and perimeter
            m_Membrane[comp->m_IndexCell] = m_Membrane.back();
            m_Membrane[comp->m_IndexCell]->m_IndexCell = comp->m_IndexCell;
            m_Membrane.pop_back();
            m_MembraneLength -= comp->m_Length;

            // Overwrite members of the removed edge
            comp->m_IndexCell = 0;
            comp->m_Cell = nullptr;

        }
        else {
            // Add new membrane elements
            // Adjust boundaries of the update routine

            // add edge to membrane, adjust index and perimeter
            edge->m_IndexCell = m_Membrane.size();
            edge->m_Cell = this;
            m_Membrane.push_back(edge);
            m_MembraneLength += edge->m_Length;

        }

    }

    // Add tile to bulk
    target->m_CellSubstrateAdhesion = input->m_Tile->m_CellSubstrateAdhesion;
    target->m_Signal = input->m_Tile->m_Signal;
    target->m_Cell = this;
    target->m_IndexCell = m_Bulk.size();
    m_Bulk.push_back(target);
    m_BulkArea += target->m_Area;

    // Update Polarization Signal
    for (auto& tile : m_Bulk) {
        if ((target->X() - tile->X()).Norm2() < m_SignallingRadius2) {
            ++tile->m_Signal;
        }
    }

}

/*!
 *  Perform a retraction Event.
 *  @param input Edge along which the cell retracts
 */
void Cell::Retract(Edge* input) {

    Tile* target = input->m_Tile;

    for (size_t i = 5; i != size_t(-1) ; --i) {
        Edge* edge = &target->m_Edges[i];
        Edge* comp = edge->m_Comp;

        if (edge->m_Cell == this) {
            // remove edge from membrane by swap-popback, adjust index and perimeter
            m_Membrane[edge->m_IndexCell] = m_Membrane.back();
            m_Membrane[edge->m_IndexCell]->m_IndexCell = edge->m_IndexCell;
            m_Membrane.pop_back();
            m_MembraneLength -= edge->m_Length;

            // Overwrite members of the removed edge
            edge->m_IndexCell = 0;
            edge->m_Cell = nullptr;

        }
        else {
            // add edge to membrane, adjust index and perimeter
            comp->m_IndexCell = m_Membrane.size();
            comp->m_Cell = this;
            m_Membrane.push_back(comp);
            m_MembraneLength += comp->m_Length;

        }

    }

    // Remove tile from bulk
    m_Bulk[target->m_IndexCell] = m_Bulk.back();
    m_Bulk[target->m_IndexCell]->m_IndexCell = target->m_IndexCell;
    m_Bulk.pop_back();
    m_BulkArea -= target->m_Area;
    target->m_CellSubstrateAdhesion = 0;
    target->m_Signal = 0;
    target->m_Cell = nullptr;
    target->m_IndexCell = 0;

    // Update Polarization Signal
    for (auto& tile : m_Bulk) {
        if ((target->X() - tile->X()).Norm2() < m_SignallingRadius2) {
            --tile->m_Signal;
        }
    }

}


/*!
 *  Find the largest continuous blob of connected subtrate tiles and initialize the Cell from it.
 */
void Cell::InitializeFromPool(std::vector<Tile*> pool) {
    ////////////////////////////////////
    // Begin parent cell
    /////////////////////////

    // Search for largest connected blob!
    size_t largestCandidate = 0;
    std::vector<Tile*> NewBulk;
    NewBulk.reserve(pool.size());
    while (!pool.empty()) {
        std::vector<Tile*> Candidate;
        std::vector<Tile*> ToExplore;
        ToExplore.push_back(pool.back());
        pool.back()->m_Cell = nullptr; ///< Remove tile occupation
        pool.back()->m_IndexCell = 0; ///< Remove tile occupation
        pool.pop_back();
        size_t size = 0;

        do {
            Tile* tile = ToExplore.back();
            ToExplore.pop_back();
            Candidate.push_back(tile); ///< Grow candidate by adding tile

            for (auto& neighbor : tile->Neighbors()) {
                if (neighbor->m_Cell == this) {
                    ToExplore.push_back(neighbor);
                    neighbor->m_Cell = nullptr; ///< Remove tile occupation
                    neighbor->m_IndexCell = 0; ///< Remove tile occupation
                    pool.erase(std::find(pool.begin(), pool.end(), neighbor));
                    ++size;
                }
            }

        } while (!ToExplore.empty());

        if (size > largestCandidate) {
            largestCandidate = NewBulk.size();
            NewBulk = Candidate;
        };
    };

    // Parent cell is new largest connected blob!
    m_Bulk.reserve(NewBulk.size());
    for (auto& tile : NewBulk) {
        tile->m_Cell = this;
        tile->m_IndexCell = m_Bulk.size();
        m_BulkArea += tile->m_Area;
        m_Bulk.push_back(tile);
    }

    // Construct new unsorted membrane of parent cell
    for (auto& tile : m_Bulk) {
        for (size_t i=0; i<6; ++i) {
            Tile* neighbor = tile->m_Neighbors[i];
            Edge* edge = &tile->m_Edges[i];
            if (neighbor->m_Cell != this) {
                edge->m_Cell = this;
                edge->m_IndexCell = m_Membrane.size();
                m_MembraneLength += edge->m_Length;
                m_Membrane.push_back(edge);
            }
        }
    }

}
