#include "IO_Data.hpp"
#include "CPM_Population.hpp"

/*!
 * Some constants
 */
static const std::string DSetName_Cell = "Raw_Data";

static const std::string DGroupName_Monolayer = "Monolayer";
static const std::string DSetName_Monolayer_FrontLeft = "Front_Left";
static const std::string DSetName_Monolayer_FrontRight = "Front_Right";
static const std::string DSetName_Monolayer_StressFull = "Stress_Full";
static const std::string DSetName_Monolayer_StressTraction = "Stress_Traction";
static const std::string DSetName_Monolayer_Strain = "Strain";
static const std::string DSetName_Adhesion = "Adhesion";
static const std::string DSetName_Density = "Density";
static const std::string DSetName_Boundary = "Boundary";
static const std::string DSetName_2DMap = "2DMap";

static const std::string DGroupName_PolarizationAxis = "ProjectionPolarizationAxis";
static const std::string DGroupName_ShortAxis = "ProjectionShortAxis";
static const std::string DGroupName_LongAxis = "ProjectionLongAxis";

/*!
 *  Constructor of the data handler class using cell population, substrate and parameters.
 *  @param  grid Pointer to Grid which is to be measured
 *  @param  population Pointer to Population which is to be measured
 *  @param  parameters Pointer to Parameters (define measurement intervals etc.)
 */
Data::Data(Grid* grid, Population* population, Hamiltonian* hamiltonian, Parameters* parameters)
: m_IMeasureCurrent(1)
, m_NStepCurrent(0)
, m_IMeasure(parameters->I_Measure())
, m_MeasureExtended(parameters->MeasureExtended())
, m_MeasureMaps(parameters->MeasureMaps())
, m_MeasureMono(parameters->MeasureMono())
, m_NSteps(parameters->N_MCSSim() / parameters->I_Measure())
, m_DataAdhesion_POLAXIS()
, m_DataDensity_POLAXIS()
, m_DataBoundary_POLAXIS()
, m_DataGrid_POLAXIS()
, m_DataAdhesion_LONGAXIS()
, m_DataDensity_LONGAXIS()
, m_DataBoundary_LONGAXIS()
, m_DataGrid_LONGAXIS()
, m_DataAdhesion_SHORTAXIS()
, m_DataDensity_SHORTAXIS()
, m_DataBoundary_SHORTAXIS()
, m_DataGrid_SHORTAXIS()
, m_Grid(grid)
, m_Population(population)
, m_Hamiltonian(hamiltonian)
{

    file = H5::H5File("Data.h5", H5F_ACC_TRUNC);

    if(m_MeasureExtended) {

        const hsize_t DimsChunk[2] = {m_NSteps, 2};
        const hsize_t DimsData[2] = {m_NSteps, m_Population->Cells().size()};
        const hsize_t DimsMax[2] = {m_NSteps, H5S_UNLIMITED};

        // Create the data space for the dataset.
        H5::DataSpace DSpaceCell (2, DimsData, DimsMax);

        H5::CompType TypeCell (sizeof(DataCellExtended));
        TypeCell.insertMember("Time", HOFFSET(DataCellExtended, Time), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Cell_ID", HOFFSET(DataCellExtended, Cell_ID), H5::PredType::NATIVE_HSIZE);
        TypeCell.insertMember("Active", HOFFSET(DataCellExtended, Active), H5::PredType::NATIVE_HBOOL);
        TypeCell.insertMember("Position_X", HOFFSET(DataCellExtended, Position_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Position_Y", HOFFSET(DataCellExtended, Position_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Velocity_X", HOFFSET(DataCellExtended, Velocity_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Velocity_Y", HOFFSET(DataCellExtended, Velocity_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Polarity_X", HOFFSET(DataCellExtended, Polarity_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Polarity_Y", HOFFSET(DataCellExtended, Polarity_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Gradient_X", HOFFSET(DataCellExtended, Gradient_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Gradient_Y", HOFFSET(DataCellExtended, Gradient_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("PCA_EVEC_X", HOFFSET(DataCellExtended, PCA_EVEC_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("PCA_EVEC_Y", HOFFSET(DataCellExtended, PCA_EVEC_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("PCA_EVAL1", HOFFSET(DataCellExtended, PCA_EVAL1), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("PCA_EVAL2", HOFFSET(DataCellExtended, PCA_EVAL2), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Area", HOFFSET(DataCellExtended, Area), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Perimeter", HOFFSET(DataCellExtended, Perimeter), H5::PredType::NATIVE_DOUBLE);

        // Create the dataset with chunking enabled.
        H5::DSetCreatPropList prop;
        prop.setChunk(2, DimsChunk);
        H5::DataSet DSetCell = file.createDataSet(DSetName_Cell, TypeCell, DSpaceCell, prop);
        DSetCell.close();
        DSpaceCell.close();
        prop.close();
    }
    else {

        const hsize_t DimsChunk[2] = {m_NSteps, 2};
        const hsize_t DimsData[2] = {m_NSteps, m_Population->Cells().size()};
        const hsize_t DimsMax[2] = {m_NSteps, H5S_UNLIMITED};

        // Create the data space for the dataset.
        H5::DataSpace DSpaceCell (2, DimsData, DimsMax);

        H5::CompType TypeCell (sizeof(DataCellReduced));
        TypeCell.insertMember("Time", HOFFSET(DataCellReduced, Time), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Cell_ID", HOFFSET(DataCellReduced, Cell_ID), H5::PredType::NATIVE_HSIZE);
        TypeCell.insertMember("Active", HOFFSET(DataCellReduced, Active), H5::PredType::NATIVE_HBOOL);
        TypeCell.insertMember("Position_X", HOFFSET(DataCellReduced, Position_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Position_Y", HOFFSET(DataCellReduced, Position_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Velocity_X", HOFFSET(DataCellReduced, Velocity_X), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Velocity_Y", HOFFSET(DataCellReduced, Velocity_Y), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Area", HOFFSET(DataCellReduced, Area), H5::PredType::NATIVE_DOUBLE);
        TypeCell.insertMember("Perimeter", HOFFSET(DataCellReduced, Perimeter), H5::PredType::NATIVE_DOUBLE);

        // Create the dataset with chunking enabled.
        H5::DSetCreatPropList prop;
        prop.setChunk(2, DimsChunk);
        H5::DataSet DSetCell = file.createDataSet(DSetName_Cell, TypeCell, DSpaceCell, prop);
        DSetCell.close();
        DSpaceCell.close();
        prop.close();
    }

    if(m_MeasureMono) {
        // create strain dataset
        /////////////////////////

        H5::Group group = file.createGroup(DGroupName_Monolayer);


        // create front dataset
        const hsize_t DimsChunkFront[2] = {m_NSteps, 2};
        const hsize_t DimsDataFront[2] = {m_NSteps, 2};
        const hsize_t DimsMaxFront[2] = {m_NSteps, H5S_UNLIMITED};

        H5::CompType TypeFront (sizeof(DataFront));
        TypeFront.insertMember("X", HOFFSET(DataFront, X), H5::PredType::NATIVE_DOUBLE);
        TypeFront.insertMember("Y", HOFFSET(DataFront, Y), H5::PredType::NATIVE_DOUBLE);

        // Create the data space for the dataset.
        H5::DataSpace DSpaceFrontLeft (2, DimsDataFront, DimsMaxFront);
        // Create the dataset with chunking enabled.
        H5::DSetCreatPropList propFrontLeft;
        propFrontLeft.setChunk(2, DimsChunkFront);
        H5::DataSet DSetFrontLeft (group.createDataSet(DSetName_Monolayer_FrontLeft, TypeFront, DSpaceFrontLeft, propFrontLeft));
        DSetFrontLeft.close();
        DSpaceFrontLeft.close();
        propFrontLeft.close();

        // Create the data space for the dataset.
        H5::DataSpace DSpaceFrontRight (2, DimsDataFront, DimsMaxFront);
        // Create the dataset with chunking enabled.
        H5::DSetCreatPropList propFrontRight;
        propFrontRight.setChunk(2, DimsChunkFront);
        H5::DataSet DSetFrontRight (group.createDataSet(DSetName_Monolayer_FrontRight, TypeFront, DSpaceFrontRight, propFrontRight));
        DSetFrontRight.close();
        DSpaceFrontLeft.close();
        propFrontRight.close();

        // create stress datasets
        const hsize_t DimsChunkStress[2] = {m_NSteps, 2};
        const hsize_t DimsDataStress[2] = {m_NSteps, m_Population->Cells().size()};
        const hsize_t DimsMaxStress[2] = {m_NSteps, H5S_UNLIMITED};

        H5::CompType TypeStressTensor (sizeof(DataStressTensor));
        TypeStressTensor.insertMember("X", HOFFSET(DataStressTensor, X), H5::PredType::NATIVE_DOUBLE);
        TypeStressTensor.insertMember("Y", HOFFSET(DataStressTensor, Y), H5::PredType::NATIVE_DOUBLE);
        TypeStressTensor.insertMember("Stress_XX", HOFFSET(DataStressTensor, Stress_XX), H5::PredType::NATIVE_DOUBLE);
        TypeStressTensor.insertMember("Stress_XY", HOFFSET(DataStressTensor, Stress_XY), H5::PredType::NATIVE_DOUBLE);
        TypeStressTensor.insertMember("Stress_YX", HOFFSET(DataStressTensor, Stress_YX), H5::PredType::NATIVE_DOUBLE);
        TypeStressTensor.insertMember("Stress_YY", HOFFSET(DataStressTensor, Stress_YY), H5::PredType::NATIVE_DOUBLE);

        // Create the data space for the dataset.
        H5::DataSpace DSpaceStressFull (2, DimsDataStress, DimsMaxStress);
        // Create the dataset with chunking enabled.
        H5::DSetCreatPropList propStressFull;
        propStressFull.setChunk(2, DimsChunkStress);
        H5::DataSet DSetStressFull (group.createDataSet(DSetName_Monolayer_StressFull, TypeStressTensor, DSpaceStressFull, propStressFull));
        DSetStressFull.close();
        DSpaceStressFull.close();
        propStressFull.close();

        H5::DataSpace DSpaceStressTraction (2, DimsDataStress, DimsMaxStress);
        H5::DSetCreatPropList propStressTraction;
        propStressTraction.setChunk(2, DimsChunkStress);
        H5::DataSet DSetStressTraction (group.createDataSet(DSetName_Monolayer_StressTraction, TypeStressTensor, DSpaceStressTraction, propStressTraction) );
        DSetStressTraction.close();
        DSpaceStressTraction.close();
        propStressTraction.close();

        group.close();

    }

}

/*!
 *  Destructor of the data handler class. This frees all assigned resources.
 */
Data::~Data() {

    if(m_MeasureMaps) {
        H5::Group group;

        group = file.createGroup(DGroupName_PolarizationAxis);
        group.close();
        SaveGrid(DGroupName_PolarizationAxis, DSetName_2DMap, m_DataGrid_POLAXIS);
        SaveAngluar(DGroupName_PolarizationAxis, DSetName_Adhesion, m_DataAdhesion_POLAXIS);
        SaveAngluar(DGroupName_PolarizationAxis, DSetName_Density, m_DataDensity_POLAXIS);
        SaveAngluar(DGroupName_PolarizationAxis, DSetName_Boundary, m_DataBoundary_POLAXIS);

        group = file.createGroup(DGroupName_LongAxis);
        group.close();
        SaveGrid(DGroupName_LongAxis, DSetName_2DMap, m_DataGrid_LONGAXIS);
        SaveAngluar(DGroupName_LongAxis, DSetName_Adhesion, m_DataAdhesion_LONGAXIS);
        SaveAngluar(DGroupName_LongAxis, DSetName_Density, m_DataDensity_LONGAXIS);
        SaveAngluar(DGroupName_LongAxis, DSetName_Boundary, m_DataBoundary_LONGAXIS);

        group = file.createGroup(DGroupName_ShortAxis);
        group.close();
        SaveGrid(DGroupName_ShortAxis, DSetName_2DMap, m_DataGrid_SHORTAXIS);
        SaveAngluar(DGroupName_ShortAxis, DSetName_Adhesion, m_DataAdhesion_SHORTAXIS);
        SaveAngluar(DGroupName_ShortAxis, DSetName_Density, m_DataDensity_SHORTAXIS);
        SaveAngluar(DGroupName_ShortAxis, DSetName_Boundary, m_DataBoundary_SHORTAXIS);
    }

    file.close();

}

/*!
 *  Try to write current configuration to HDF5 file. It is successfull, if m_IMeasureCurrent == m_IMeasure.
 */
void Data::Write() {

    if(m_IMeasureCurrent < m_IMeasure) {
        ++m_IMeasureCurrent;
        return;
    }
    else m_IMeasureCurrent = 1;

    if(m_MeasureExtended) {

        // Write data to temporary array
        std::vector<DataCellExtended> data;
        data.reserve(m_Population->Cells().size());

        for(auto& cell : m_Population->Cells()) {

            const Vector V = cell.MoveVector() / m_IMeasure;
            const Point X = cell.X();
            const Vector P = cell.PolarizationVector();
            const Vector Grad = MeasureGradient(&cell);

            DataCellExtended container;
            container.Time = m_NStepCurrent * m_IMeasure;
            container.Cell_ID = cell.Index();
            container.Active = cell.CheckMotility() == Cell::MotilityState::Motile ? true : false;
            container.Position_X = X[0];
            container.Position_Y = X[1];
            container.Velocity_X = V[0];
            container.Velocity_Y = V[1];
            container.Polarity_X = P[0];
            container.Polarity_Y = P[1];
            container.Gradient_X = Grad[0];
            container.Gradient_Y = Grad[1];
            container.Area = cell.Area();
            container.Perimeter = cell.Perimeter();

            if(m_MeasureMaps) {
                MeasureAdhesion(&cell, P, m_DataAdhesion_POLAXIS);
                MeasureBoundary(&cell, P, m_DataBoundary_POLAXIS);
            }

            Cell::PrincipalComponents PCA = cell.PrincipalComponentAnalysis();
            container.PCA_EVAL1 = PCA.Eigenvalue1;
            container.PCA_EVAL2 = PCA.Eigenvalue2;

            Vector EV1 = PCA.Eigenvector1;
            if (EV1*P<0) EV1 *= -1;
            container.PCA_EVEC_X = EV1[0];
            container.PCA_EVEC_Y = EV1[1];

            if(m_MeasureMaps) {
                MeasureAdhesion(&cell, EV1, m_DataAdhesion_LONGAXIS);
                MeasureBoundary(&cell, EV1, m_DataBoundary_LONGAXIS);
            }

            Vector EV2 = PCA.Eigenvector2;
            if (EV2*P<0) EV2 *= -1;

            if(m_MeasureMaps) {
                MeasureAdhesion(&cell, EV2, m_DataAdhesion_SHORTAXIS);
                MeasureBoundary(&cell, EV2, m_DataBoundary_SHORTAXIS);
            }

            data.push_back(container);
        }

        const hsize_t DimsData[2] = {m_NSteps, m_Population->Cells().size()};
        const hsize_t DimsSlab[2] = {1, m_Population->Cells().size()};
        const hsize_t Offset[2] = {m_NStepCurrent, 0};

        // Extend Dataset
        H5::DataSet DSetCell (file.openDataSet(DSetName_Cell));
        DSetCell.extend(DimsData);
        H5::DataSpace DSpaceCell (DSetCell.getSpace());
        H5::CompType TypeCell (DSetCell.getCompType());

        // Define Memory Dataspace. Get file dataspace and select
        // a subset from the file dataspace.
        H5::DataSpace MemSpace(2, DimsSlab);
        DSpaceCell.selectHyperslab(H5S_SELECT_SET, DimsSlab, Offset);
        DSetCell.write(data.data(), TypeCell, MemSpace, DSpaceCell);
        DSetCell.close();
        MemSpace.close();
        DSpaceCell.close();

    }
    else {

        // Write data to temporary array
        std::vector<DataCellReduced> data;
        data.reserve(m_Population->Cells().size());

        for(auto& cell : m_Population->Cells()) {

            const Vector V = cell.MoveVector() / m_IMeasure;
            const Point X = cell.X();

            DataCellReduced container;
            container.Time = m_NStepCurrent * m_IMeasure;
            container.Cell_ID = cell.Index();
            container.Active = cell.CheckMotility() == Cell::MotilityState::Motile ? true : false;
            container.Position_X = X[0];
            container.Position_Y = X[1];
            container.Velocity_X = V[0];
            container.Velocity_Y = V[1];
            container.Area = cell.Area();
            container.Perimeter = cell.Perimeter();

            if(m_MeasureMaps) {
                const Vector P = cell.PolarizationVector();
                Measure2DMap(m_Grid, X, P, m_DataGrid_POLAXIS);
                MeasureDensity(&cell, P, m_DataDensity_POLAXIS);
                MeasureAdhesion(&cell, P, m_DataAdhesion_POLAXIS);
                MeasureBoundary(&cell, P, m_DataBoundary_POLAXIS);
            }

            if(m_MeasureMaps) {
                Cell::PrincipalComponents PCA = cell.PrincipalComponentAnalysis();
                Vector P = cell.PolarizationVector();

                Vector EV1 = PCA.Eigenvector1;
                if (EV1*P<0) EV1 *= -1;

                Measure2DMap(m_Grid, X, EV1, m_DataGrid_LONGAXIS);
                MeasureDensity(&cell, EV1, m_DataDensity_LONGAXIS);
                MeasureAdhesion(&cell, EV1, m_DataAdhesion_LONGAXIS);
                MeasureBoundary(&cell, EV1, m_DataBoundary_LONGAXIS);

                Vector EV2 = PCA.Eigenvector2;
                if (EV2*P<0) EV2 *= -1;

                Measure2DMap(m_Grid, X, EV2, m_DataGrid_SHORTAXIS);
                MeasureDensity(&cell, EV2, m_DataDensity_SHORTAXIS);
                MeasureAdhesion(&cell, EV2, m_DataAdhesion_SHORTAXIS);
                MeasureBoundary(&cell, EV2, m_DataBoundary_SHORTAXIS);
            }

            data.push_back(container);
        }

        const hsize_t DimsData[2] = {m_NSteps, m_Population->Cells().size()};
        const hsize_t DimsSlab[2] = {1, m_Population->Cells().size()};
        const hsize_t Offset[2] = {m_NStepCurrent, 0};

        // Extend Dataset
        H5::DataSet DSetCell (file.openDataSet(DSetName_Cell));
        DSetCell.extend(DimsData);
        H5::DataSpace DSpaceCell (DSetCell.getSpace());
        H5::CompType TypeCell (DSetCell.getCompType());

        // Define Memory Dataspace. Get file dataspace and select
        // a subset from the file dataspace.
        H5::DataSpace MemSpace(2, DimsSlab);
        DSpaceCell.selectHyperslab(H5S_SELECT_SET, DimsSlab, Offset);
        DSetCell.write(data.data(), TypeCell, MemSpace, DSpaceCell);
        DSetCell.close();
        MemSpace.close();
        DSpaceCell.close();

    }

    if(m_MeasureMono) {
        auto LeftFront = MeasureFrontLeft(m_Grid);
        SaveFront(DGroupName_Monolayer, DSetName_Monolayer_FrontLeft, LeftFront);
        auto RightFront = MeasureFrontRight(m_Grid);
        SaveFront(DGroupName_Monolayer, DSetName_Monolayer_FrontRight, RightFront);
        SaveStress(DGroupName_Monolayer, DSetName_Monolayer_StressFull, MeasureStressFull(m_Population));
        SaveStress(DGroupName_Monolayer, DSetName_Monolayer_StressTraction, MeasureStressTraction(m_Population));
    }

    ++m_NStepCurrent;

}

/*!
 *  Measure 2D-Map of substrate properties centered around an achoring point and a given principal axis.
 *  @param  grid    Pointer to Grid which is to be measured
 *  @param  center  Point, around which the measurement is performed
 *  @param  axis    Vector indicating the principal axis for the measurement
 *  @param  storage vector of DataGrid structures that stores the measurement and is updated by this function
 */
void Data::Measure2DMap(const Grid* grid, const Point& center, const Vector& axis, std::vector<DataGrid>& storage) {

    const size_t Size_Theta = 100;
    const size_t Size_R = 50;
    const double_t dr = 0.5;
    const double_t dtheta = M_PI / Size_Theta;

    if(storage.empty()) {
        storage.reserve(Size_R * Size_Theta);
        for (size_t i=0 ; i<Size_R; ++i) {
            for (size_t j=0 ; j<Size_Theta; ++j) {
                storage.push_back({(i+0.5) * dr, (j+0.5) * dtheta, 0., 0., 0, 0., 0., 0});
            }
        }
    }

    for (const auto& tile : grid->Tiles()) {
        Vector vec = tile.X() - center;
        double_t r = vec.Norm();
        size_t radiusIndex = (size_t) (r / dr);

        double_t theta = std::acos( (axis * vec) / (axis.Norm() * vec.Norm()) );
        size_t thetaIndex = (size_t) (theta / dtheta);
        if(thetaIndex == Size_Theta) --thetaIndex;

        if (radiusIndex < Size_R) {

            Vector strain (0.,0.);
            for (const auto& neighbor : tile.Neighbors()) {
                strain += neighbor->X() - tile.X();
            }

            storage[radiusIndex*Size_Theta + thetaIndex].Area += tile.Area();
            storage[radiusIndex*Size_Theta + thetaIndex].Adhesion += tile.CellSubstrateAdhesion();
            if(tile.CellSubstrateAdhesion()>0) ++storage[radiusIndex*Size_Theta + thetaIndex].OccupiedCount;
            storage[radiusIndex*Size_Theta + thetaIndex].Strain_X += strain[0];
            storage[radiusIndex*Size_Theta + thetaIndex].Strain_Y += strain[1];
            ++storage[radiusIndex*Size_Theta + thetaIndex].TotalCount;
        }
    }
};

/*!
 *  Save 2D-Map of substrate properties in the given HDF5 group.
 *  @param  groupname name of the H5::Group that will be written to
 *  @param  dsetname name of the H5::DataSet that will be written to
 *  @param  storage vector of DataGrid structures to be saved
 */
void Data::SaveGrid (const std::string& groupname, const std::string& dsetname, const std::vector<DataGrid>& storage) {
    if(storage.empty()) return;

    H5::CompType TypeGrid (sizeof(DataGrid));
    TypeGrid.insertMember("Radius", HOFFSET(DataGrid, Radius), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("Theta", HOFFSET(DataGrid, Theta), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("Area", HOFFSET(DataGrid, Area), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("Adhesion", HOFFSET(DataGrid, Adhesion), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("OccupiedCount", HOFFSET(DataGrid, OccupiedCount), H5::PredType::NATIVE_UINT_FAST32);
    TypeGrid.insertMember("Strain_X", HOFFSET(DataGrid, Strain_X), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("Strain_Y", HOFFSET(DataGrid, Strain_Y), H5::PredType::NATIVE_DOUBLE);
    TypeGrid.insertMember("TotalCount", HOFFSET(DataGrid, TotalCount), H5::PredType::NATIVE_UINT_FAST32);

    const hsize_t Dims [1] = {storage.size()};
    H5::DataSpace dataspace (1, Dims);
    H5::Group group (file.openGroup(groupname));
    H5::DataSet dataset (group.createDataSet(dsetname, TypeGrid, dataspace));
    dataset.write(storage.data(), TypeGrid);
    dataset.close();
    group.close();
    dataspace.close();

}

/*!
 *  Measure average cell adhesion at the cell boundary around a given principal axis.
 *  @param  cell    Pointer to Cell which is to be measured
 *  @param  axis    Vector indicating the principal axis for the measurement
 *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
 */
void Data::MeasureAdhesion(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage) {

    const size_t Size = 100;
    const double_t dtheta = M_PI / Size;

    if(storage.empty()) {
        storage.reserve(Size);
        for (size_t i=0 ; i<Size; ++i) {
            storage.push_back({(i+0.5) * dtheta, 0., 0});
        }
    }

    for (const auto& edge : cell->Membrane()) {
        const Tile* tile = edge->GetTile();
        Vector vec = (tile->X()) - (cell->X());

        double_t theta = std::acos( (axis * vec) / (axis.Norm() * vec.Norm()) );
        size_t index = (size_t) (theta / dtheta);

        if (index == Size) --index;
        storage[index].Value += tile->CellSubstrateAdhesion();
        ++storage[index].TotalCount;
    }
};

/*!
 *  Save average cell adhesion at the cell boundary in the given HDF5 group.
 *  @param  groupname name of the H5::Group that will be written to
 *  @param  dsetname name of the H5::DataSet that will be written to
 *  @param  storage vector of DataAngular structures to be saved
 */
void Data::SaveAngluar (const std::string& groupname, const std::string& dsetname, const std::vector<DataAngular>& storage) {
    if(storage.empty()) return;

    H5::CompType TypeAngular (sizeof(DataAngular));
    TypeAngular.insertMember("Theta", HOFFSET(DataAngular, Theta), H5::PredType::NATIVE_DOUBLE);
    TypeAngular.insertMember("Value", HOFFSET(DataAngular, Value), H5::PredType::NATIVE_DOUBLE);
    TypeAngular.insertMember("TotalCount", HOFFSET(DataAngular, TotalCount), H5::PredType::NATIVE_UINT_FAST32);

    const hsize_t Dims [1] = {storage.size()};
    H5::DataSpace dataspace (1, Dims);
    H5::Group group (file.openGroup(groupname));
    H5::DataSet dataset = group.createDataSet(dsetname, TypeAngular, dataspace);
    dataset.write(storage.data(), TypeAngular);
    dataset.close();
    group.close();
    dataspace.close();

}

/*!
 *  Measure average substrate density at the cell boundary and update storage.
 *  @param  cell    Pointer to Cell which is to be measured
 *  @param  axis    Vector indicating the principal axis for the measurement
 *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
 */
void Data::MeasureDensity(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage) {

    const size_t Size = 100;
    const double_t dtheta = M_PI / Size;

    if(storage.empty()) {
        storage.reserve(Size);
        for (size_t i=0 ; i<Size; ++i) {
            storage.push_back({(i+0.5) * dtheta, 0., 0});
        }
    }

    for (const auto& edge : cell->Membrane()) {

        // Measure density inside of the cell
        {
            const Tile* tile = edge->GetTile();
            Vector vec = (tile->X()) - (cell->X());

            double_t theta = std::acos( (axis * vec) / (axis.Norm() * vec.Norm()) );
            size_t index = (size_t) (theta / dtheta);

            if (index == Size) --index;
            storage[index].Value += tile->Area();
            ++storage[index].TotalCount;
        }

        // Measure density outside of the cell
        {
            const Tile* tile = edge->GetComp()->GetTile();
            Vector vec = (tile->X()) - (cell->X());

            double_t theta = std::acos( (axis * vec) / (axis.Norm() * vec.Norm()) );
            size_t index = (size_t) (theta / dtheta);

            if (index == Size) --index;
            storage[index].Value += tile->Area();
            ++storage[index].TotalCount;
        }

    }
};


/*!
 *  Measure average distance of the cell boundary from the cell center and update storage.
 *  @param  cell    Pointer to Cell which is to be measured
 *  @param  axis    Vector indicating the principal axis for the measurement
 *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
 */
void Data::MeasureBoundary(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage) {

    const size_t Size = 100;
    const double_t dtheta = M_PI / Size;

    if(storage.empty()) {
        storage.reserve(Size);
        for (size_t i=0 ; i<Size; ++i) {
            storage.push_back({(i+0.5) * dtheta, 0., 0});
        }
    }

    for (const auto& edge : cell->Membrane()) {
        const Tile* tile = edge->GetTile();
        size_t edgeIndex = edge->Index();
        Point X1 = tile->Vertex(edgeIndex);
        Point X2 = tile->Vertex(edgeIndex<5 ? edgeIndex + 1 : 0);

        Vector vec = 0.5 * ( (X1 - cell->X()) + (X2 - cell->X()) );
        double_t dist = vec.Norm();

        double_t theta = std::acos( (axis * vec) / (axis.Norm() * dist) );
        size_t index = (size_t) (theta / dtheta);

        if (index == Size) --index;
        storage[index].Value += dist;
        ++storage[index].TotalCount;

    }
};

/*!
 *  Measure positions of the fronts of confluent tissues
 *  @param  grid    Pointer to Grid which is to be measured
 */
std::vector<Data::DataFront> Data::MeasureFrontLeft(const Grid* grid) {

    std::vector<DataFront> storage;

    for (const auto& tile : grid->Tiles()) {
        Tile* right = tile.Neighbors()[1];
        if(tile.GetCell() && !right->GetCell()) storage.push_back({tile.X()[0], tile.X()[1]});
    }

    return storage;

}

/*!
 *  Measure positions of the fronts of confluent tissues
 *  @param  grid    Pointer to Grid which is to be measured
 */
std::vector<Data::DataFront> Data::MeasureFrontRight(const Grid* grid) {

    std::vector<DataFront> storage;

    for (const auto& tile : grid->Tiles()) {
        Tile* left = tile.Neighbors()[4];
        if(tile.GetCell() && !left->GetCell()) storage.push_back({tile.X()[0], tile.X()[1]});
    }

    return storage;

}

/*!
 *  Save positions of the fronts of confluent tissues.
 *  @param  groupname name of the H5::Group that will be written to
 *  @param  dsetname name of the H5::DataSet that will be written to
 *  @param  storage vector of DataStrain structures to be saved
 */
void Data::SaveFront (const std::string& groupname, const std::string& dsetname, const std::vector<DataFront>& storage) {
    if(storage.empty()) return;

    const hsize_t DimsData[2] = {m_NSteps, storage.size()};
    const hsize_t DimsSlab[2] = {1, storage.size()};
    const hsize_t Offset[2] = {m_NStepCurrent, 0};

    // Extend Dataset
    H5::Group group (file.openGroup(groupname));
    H5::DataSet dataset (group.openDataSet(dsetname));
    dataset.extend(DimsData);
    H5::DataSpace DSpaceFront (dataset.getSpace());
    H5::CompType TypeFront (dataset.getCompType());

    // Define Memory Dataspace. Get file dataspace and select
    // a subset from the file dataspace.
    H5::DataSpace MemSpace(2, DimsSlab);
    DSpaceFront.selectHyperslab(H5S_SELECT_SET, DimsSlab, Offset);
    dataset.write(storage.data(), TypeFront, MemSpace, DSpaceFront);
    MemSpace.close();
    DSpaceFront.close();
}


/*!
 *  Measure polarization gradient in cell using Gauss's formula.
 *  @param  cell    Pointer to Cell which is to be measured
 */
Vector Data::MeasureGradient(const Cell* cell) {
    Vector gradient(0.,0.);

    for (const auto& edge : cell->Membrane()) {
        Tile* inner = edge->GetTile();
        Tile* outer = edge->GetComp()->GetTile();
        Vector normal = outer->X() - inner->X();
        normal /= normal.Norm();

        gradient += edge->Length() * inner->CellSubstrateAdhesion() * normal;

    }

    gradient /= cell->Area();
    return gradient;
}

/*!
 *  Save stress field of the population.
 *  @param  groupname name of the H5::Group that will be written to
 *  @param  dsetname name of the H5::DataSet that will be written to
 *  @param  storage vector of DataStrain structures to be saved
 */
void Data::SaveStress(const std::string& groupname, const std::string& dsetname, const std::vector<DataStressTensor>& storage) {

    const hsize_t DimsData[2] = {m_NSteps, m_Population->Cells().size()};
    const hsize_t DimsSlab[2] = {1, m_Population->Cells().size()};
    const hsize_t Offset[2] = {m_NStepCurrent, 0};

    // Extend Dataset
    H5::Group group (file.openGroup(groupname));
    H5::DataSet dataset (group.openDataSet(dsetname));
    dataset.extend(DimsData);
    H5::DataSpace DSpace (dataset.getSpace());
    H5::CompType Type (dataset.getCompType());

    // Define Memory Dataspace. Get file dataspace and select
    // a subset from the file dataspace.
    H5::DataSpace MemSpace(2, DimsSlab);
    DSpace.selectHyperslab(H5S_SELECT_SET, DimsSlab, Offset);
    dataset.write(storage.data(), Type, MemSpace, DSpace);
    MemSpace.close();
    DSpace.close();
    dataset.close();
    group.close();

}

/*!
 *  Measure stress tensor of the population
 *  @param  population    Pointer to Population which is to be measured
 */
std::vector<Data::DataStressTensor> Data::MeasureStressFull(const Population* population) {
    // initialize vector of stress tensors
    std::vector<DataStressTensor> result;
    result.reserve(population->Cells().size());

    for (const auto& cell : population->Cells()) {
        DataStressTensor stresstensor ({cell.X()[0], cell.X()[1], 0., 0., 0., 0.});
        for (const auto& edge : cell.Membrane()) {
            double_t energy_protrusion = m_Hamiltonian->EnergyDifferenceProtrusion(edge).second;
            double_t energy_retraction = m_Hamiltonian->EnergyDifferenceRetraction(edge).second;
            Tile* target = edge->GetComp()->GetTile();
            Tile* conqueror = edge->GetTile();
            Vector dx = target->X() - conqueror->X();
            Vector force = -(energy_protrusion - energy_retraction) * dx / dx.Norm2();
            Vector pos = conqueror->X() - cell.X();
            stresstensor.Stress_XX += 2 * force[0] * pos[0];
            stresstensor.Stress_YY += 2 * force[1] * pos[1];
            stresstensor.Stress_XY += force[0] * pos[1] + force[1] * pos[0];
            stresstensor.Stress_YX += force[0] * pos[1] + force[1] * pos[0];
        }
        stresstensor.Stress_XX /= (2*cell.Area());
        stresstensor.Stress_XY /= (2*cell.Area());
        stresstensor.Stress_YX /= (2*cell.Area());
        stresstensor.Stress_YY /= (2*cell.Area());
        result.push_back(stresstensor);
    }

    return result;
}

/*!
 *  Measure stress tensor of the population
 *  @param  population    Pointer to Population which is to be measured
 */
std::vector<Data::DataStressTensor> Data::MeasureStressTraction(const Population* population) {
    // initialize vector of stress tensors
    std::vector<DataStressTensor> result;
    result.reserve(population->Cells().size());

    for (const auto& cell : population->Cells()) {
        DataStressTensor stresstensor ({cell.X()[0], cell.X()[1], 0., 0., 0., 0.});
        for (const auto& edge : cell.Membrane()) {
            Tile* target = edge->GetComp()->GetTile();
            Tile* conqueror = edge->GetTile();
            double_t energy_protrusion = -conqueror->CellSubstrateAdhesion() + target->CellSubstrateAdhesion();
            double_t energy_retraction = +conqueror->CellSubstrateAdhesion() - target->CellSubstrateAdhesion();
            Vector dx = target->X() - conqueror->X();
            Vector force = -(energy_protrusion - energy_retraction)/2. * dx / dx.Norm2();
            Vector pos = conqueror->X() - cell.X();
            stresstensor.Stress_XX += 2 * force[0] * pos[0];
            stresstensor.Stress_YY += 2 * force[1] * pos[1];
            stresstensor.Stress_XY += force[0] * pos[1] + force[1] * pos[0];
            stresstensor.Stress_YX += force[0] * pos[1] + force[1] * pos[0];
        }
        stresstensor.Stress_XX /= (2*cell.Area());
        stresstensor.Stress_XY /= (2*cell.Area());
        stresstensor.Stress_YX /= (2*cell.Area());
        stresstensor.Stress_YY /= (2*cell.Area());
        result.push_back(stresstensor);
    }

    return result;
}
