#ifndef IO_Data_hpp
#define IO_Data_hpp

#include "CPM_Grid.hpp"
#include "CPM_Population.hpp"
#include "CPM_Hamiltonian.hpp"

#include <hdf5.h>
#include <H5Cpp.h>

class Data {

    /*!
     *  Simple export struct containing cell data.
     */
    struct DataCellReduced {
        double_t Time;          ///< current time
        size_t Cell_ID;         ///< index of the current cell
        bool Active;            ///< flag indicating whether this cell can polarize
        double_t Position_X;    ///< position (x) of the current cell
        double_t Position_Y;    ///< position (y) of the current cell
        double_t Velocity_X;    ///< velocity (x) of the current cell
        double_t Velocity_Y;    ///< velocity (y) of the current cell
        double_t Area;          ///< area of the current cell
        double_t Perimeter;     ///< perimeter of the current cell
    };

    /*!
     *  Simple export struct containing cell data.
     */
    struct DataCellExtended {
        double_t Time;          ///< current time
        size_t Cell_ID;         ///< index of the current cell
        bool Active;            ///< flag indicating whether this cell can polarize
        double_t Position_X;    ///< position (x) of the current cell
        double_t Position_Y;    ///< position (y) of the current cell
        double_t Velocity_X;    ///< velocity (x) of the current cell
        double_t Velocity_Y;    ///< velocity (y) of the current cell
        double_t Polarity_X;    ///< polarity vector (x) of the current cell
        double_t Polarity_Y;    ///< polarity vector (y) of the current cell
        double_t Gradient_X;    ///< Polarization gradient (x) of the current cell
        double_t Gradient_Y;    ///< Polarization gradient (y) of the current cell
        double_t PCA_EVEC_X;    ///< shape eigenvector (x) of the current cell
        double_t PCA_EVEC_Y;    ///< shape eigenvector (y) of the current cell
        double_t PCA_EVAL1;     ///< shape eigenvalue (>) of the current cell
        double_t PCA_EVAL2;     ///< shape eigenvalue (<) of the current cell
        double_t Area;          ///< area of the current cell
        double_t Perimeter;     ///< perimeter of the current cell
    };

    /*!
     *  Simple export struct containing grid data at the position (R, Theta).
     */
    struct DataGrid {
        const double_t Radius;          ///< substrate position (R)
        const double_t Theta;           ///< substrate position (Theta)
        double_t Area;                  ///< cumulative area
        double_t Adhesion;              ///< cumulative adhesion
        uint_fast32_t OccupiedCount;    ///< counter indicating whether substrate at this position is occupied
        double_t Strain_X;              ///< local strain (x)
        double_t Strain_Y;              ///< local strain (y)
        uint_fast32_t TotalCount;       ///< total number of measurements
    };

    /*!
     *  Simple export struct containing strain data at the position (X, Y).
     */
    struct DataStrain {
        double_t X;
        double_t Y;
        double_t Strain_X;
        double_t Strain_Y;
    };

    /*!
     *  Simple export struct containing stress tensor at position (X, Y).
     */
    struct DataStressTensor {
        const double_t X;
        const double_t Y;
        double_t Stress_XX;
        double_t Stress_XY;
        double_t Stress_YX;
        double_t Stress_YY;
    };

    /*!
     *  Simple export struct containing the front positions (X, Y).
     */
    struct DataFront {
        double_t X;
        double_t Y;
    };

    /*!
     *  Simple export struct containing cell boundary data at the position (Theta).
     */
    struct DataAngular {
        const double_t Theta;           ///< position (Theta)
        double_t Value;                 ///< cumulative value
        uint_fast32_t TotalCount;       ///< total number of measurements
    };

    /*!
     *  Measure positions of the left front of confluent tissues
     *  @param  grid    Pointer to Grid which is to be measured
     */
    std::vector<DataFront> MeasureFrontLeft(const Grid* grid);

    /*!
     *  Measure positions of the right front of confluent tissues
     *  @param  grid    Pointer to Grid which is to be measured
     */
    std::vector<DataFront> MeasureFrontRight(const Grid* grid);

    /*!
     *  Measure 2D-Map of substrate properties centered around an achoring point and a given principal axis.
     *  @param  grid    Pointer to Grid which is to be measured
     *  @param  center  Point, around which the measurement is performed
     *  @param  axis    Vector indicating the principal axis for the measurement
     *  @param  storage vector of DataGrid structures that stores the measurement and is updated by this function
     */
    void Measure2DMap(const Grid* grid, const Point& center, const Vector& axis, std::vector<DataGrid>& storage);

    /*!
     *  Measure polarization gradient in cell using Gauss's formula.
     *  @param  cell    Pointer to Cell which is to be measured
     */
    Vector MeasureGradient(const Cell* cell);

    /*!
     *  Measure average cell adhesion at the cell boundary around a given principal axis.
     *  @param  cell    Pointer to Cell which is to be measured
     *  @param  axis    Vector indicating the principal axis for the measurement
     *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
     */
    void MeasureAdhesion(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage);

    /*!
     *  Measure average substrate density at the cell boundary and update storage.
     *  @param  cell    Pointer to Cell which is to be measured
     *  @param  axis    Vector indicating the principal axis for the measurement
     *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
     */
    void MeasureDensity(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage);

    /*!
     *  Measure average distance of the cell boundary from the cell center and update storage.
     *  @param  cell    Pointer to Cell which is to be measured
     *  @param  axis    Vector indicating the principal axis for the measurement
     *  @param  storage vector of DataAngular structures that stores the measurement and is updated by this function
     */
    void MeasureBoundary(const Cell* cell, const Vector& axis, std::vector<DataAngular>& storage);

    /*!
     *  Measure stress tensor of the population
     *  @param  population    Pointer to Population which is to be measured
     */
    std::vector<DataStressTensor> MeasureStressFull(const Population* population);

    /*!
     *  Measure stress tensor of the population
     *  @param  population    Pointer to Population which is to be measured
     */
    std::vector<DataStressTensor> MeasureStressTraction(const Population* population);

    /*!
     *  Save 2D-Map of substrate properties in the given HDF5 group.
     *  @param  groupname name of the H5::Group that will be written to
     *  @param  dsetname name of the H5::DataSet that will be written to
     *  @param  storage vector of DataGrid structures to be saved
     */
    void SaveGrid (const std::string& groupname, const std::string& dsetname, const std::vector<DataGrid>& storage);

    /*!
     *  Save average cell adhesion at the cell boundary in the given HDF5 group.
     *  @param  groupname name of the H5::Group that will be written to
     *  @param  dsetname name of the H5::DataSet that will be written to
     *  @param  storage vector of DataAngular structures to be saved
     */
    void SaveAngluar (const std::string& groupname, const std::string& dsetname, const std::vector<DataAngular>& storage);

    /*!
     *  Procedurally save stress field of the population.
     *  @param  groupname name of the H5::Group that will be written to
     *  @param  dsetname name of the H5::DataSet that will be written to
     *  @param  storage vector of DataStrain structures to be saved
     */
    void SaveStress(const std::string& groupname, const std::string& dsetname, const std::vector<DataStressTensor>& storage);

    /*!
     *  Procedurally save positions of the fronts of confluent tissues.
     *  @param  groupname name of the H5::Group that will be written to
     *  @param  dsetname name of the H5::DataSet that will be written to
     *  @param  storage vector of DataStrain structures to be saved
     */
    void SaveFront (const std::string& groupname, const std::string& dsetname, const std::vector<DataFront>& storage);


public:

    /*!
     *  Default constructor is not allowed.
     */
    Data() = delete;

    /*!
     *  Constructor of the data handler class using cell population, substrate and parameters.
     *  @param  grid Pointer to Grid which is to be measured
     *  @param  population Pointer to Population which is to be measured
     *  @param  parameters Pointer to Parameters (define measurement intervals etc.)
     */
    Data(Grid* grid, Population* population, Hamiltonian* hamiltonian, Parameters* parameters);

    /*!
     *  Copy constructor is not allowed.
     */
    Data(const Data& base) = delete;

    /*!
     *  Destructor of the data handler class. This frees all assigned resources.
     */
    ~Data();

    /*!
     *  Try to write current configuration to HDF5 file. It is successfull, if m_IMeasureCurrent == m_IMeasure.
     */
    void Write();

private:

    Population* m_Population;                               ///< Pointer to Population which is to be measured
    Hamiltonian* m_Hamiltonian;                             ///< Pointer to Hamiltonian
    Grid* m_Grid;                                           ///< Pointer to Grid which is to be measured

    H5::H5File file;                                        ///< Output file

    uint_fast32_t m_IMeasureCurrent;                        ///< Current poisition in the measurement interval
    uint_fast32_t m_NStepCurrent;                           ///< Current data handler step
    const uint_fast32_t m_IMeasure;                         ///< Measurement interval
    const uint_fast32_t m_NSteps;                           ///< Total number of data handler steps
    const bool m_MeasureExtended;                           ///< Indicator whether extended cell measurements shall be performed
    const bool m_MeasureMaps;                               ///< Indicator whether coordinate-centered angular maps of the data shall be created
    const bool m_MeasureMono;                               ///< Indicator whether monolayer measurements shall be performed

    /*
     * Values that are averaged over the whole simulation
     */
    std::vector<DataAngular> m_DataAdhesion_POLAXIS;        ///< Cell and polarization axis centered angular map of the cell adhesion profile
    std::vector<DataAngular> m_DataDensity_POLAXIS;         ///< Cell and polarization axis centered angular map of the substrate density
    std::vector<DataAngular> m_DataBoundary_POLAXIS;        ///< Cell and polarization axis centered angular map of the membrane distance from the cell center
    std::vector<DataGrid> m_DataGrid_POLAXIS;               ///< Cell and polarization axis centered substrate measurement

    std::vector<DataAngular> m_DataAdhesion_LONGAXIS;       ///< Cell and long principal axis centered angular map of the cell adhesion profile
    std::vector<DataAngular> m_DataDensity_LONGAXIS;        ///< Cell and long principal axis centered angular map of the substrate density
    std::vector<DataAngular> m_DataBoundary_LONGAXIS;       ///< Cell and long principal axis centered angular map of the membrane distance from the cell center
    std::vector<DataGrid> m_DataGrid_LONGAXIS;              ///< Cell and long principal axis centered substrate measurement

    std::vector<DataAngular> m_DataAdhesion_SHORTAXIS;      ///< Cell and short principal axis centered angular map of the cell adhesion profile
    std::vector<DataAngular> m_DataDensity_SHORTAXIS;       ///< Cell and short principal axis centered angular map of the substrate density
    std::vector<DataAngular> m_DataBoundary_SHORTAXIS;      ///< Cell and short principal axis centered angular map of the membrane distance from the cell center
    std::vector<DataGrid> m_DataGrid_SHORTAXIS;             ///< Cell and short principal axis centered substrate measurement

};

#endif /* IO_Data_hpp */
