#ifndef Vector_hpp
#define Vector_hpp

#include <stdio.h>
#include <cmath>
#include <array>
#include <vector>

class Vector {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Vector() = delete;

    /*!
     *  Constructor of the 2D Vector.
     *  @param x    x-coordinate
     *  @param y    y-coordinate
     */
    Vector(double_t x, double_t y):
    m_X({{x,y}})
    {};

    /*!
     *  Copy constructor from a different Vector.
     */
    Vector (const Vector& base)
    : m_X(base.m_X)
    {};

    /*!
     *  Access elements of the Vector
     *  @param n offset of the element.
     *  n=0 for x-axis. n=1 for y-axis
     */
    inline const double_t& operator[] (const size_t& n) const {
        return m_X[n];
    };

    /*!
     *  Access elements of the Vector
     *  @param n offset of the element
     *  n=0 for x-axis. n=1 for y-axis
     */
    inline double_t& operator[] (const size_t& n) {
        return m_X[n];
    };

    /*!
     *  Compute norm of the Vector (R)
     */
    const double_t Norm() const {
        return std::sqrt(m_X[0] * m_X[0] + m_X[1] * m_X[1]);
    };

    /*!
     *  Compute squared norm of the Vector (R^2)
     */
    const double_t Norm2() const {
        return (m_X[0] * m_X[0] + m_X[1] * m_X[1]);
    };

    /*!
     *  Add reference Vector.
     *  @param ref reference Vector
     */
    void operator+= (const Vector& ref) {
        m_X[0] += ref.m_X[0];
        m_X[1] += ref.m_X[1];
    };

    /*!
     *  Subtract reference Vector.
     *  @param ref reference Vector
     */
    void operator-= (const Vector& ref) {
        m_X[0] -= ref.m_X[0];
        m_X[1] -= ref.m_X[1];
    };

    /*!
     *  Multiply with scalar.
     *  @param ref reference scalar
     */
    void operator*= (const double_t& ref) {
        m_X[0] *= ref;
        m_X[1] *= ref;
    };

    /*!
     *  Divide by scalar.
     *  @param ref reference scalar
     */
    void operator/= (const double_t& ref) {
        m_X[0] /= ref;
        m_X[1] /= ref;
    };

protected:

    std::array<double_t,2> m_X; ///< coordinates of the Vector

private:

    // arithmetic operators

    /*!
     *  Addition of Vectors on the right hand side and left hand side.
     *  @param lhs  left hand side Vector
     *  @param rhs  right hand side Vector
     */
    friend inline Vector operator+ (Vector lhs, const Vector& rhs);

    /*!
     *  Subtraction of right hand side Vector from left hand side Vector.
     *  @param lhs  left hand side Vector
     *  @param rhs  right hand side Vector
     */
    friend inline Vector operator- (Vector lhs, const Vector& rhs);

    /*!
     *  Multiplication of Vector with scalar.
     *  @param lhs  left hand side Vector
     *  @param rhs  right hand side scalar
     */
    friend inline Vector operator* (Vector lhs, const double_t& rhs);

    /*!
     *  Multiplication of Vector with scalar.
     *  @param lhs  left hand side scalar
     *  @param rhs  right hand side Vector
     */
    friend inline Vector operator* (const double_t& lhs, Vector rhs);

    /*!
     *  Division of Vector by scalar.
     *  @param lhs  left hand side Vector
     *  @param rhs  right hand side scalar
     */
    friend inline Vector operator/ (Vector lhs, const double_t& rhs);

    /*!
     *  Scalar product between Vectors on the right hand side and left hand side.
     *  @param lhs  left hand side Vector
     *  @param rhs  right hand side Vector
     */
    friend inline double_t operator* (Vector lhs, const Vector& rhs);

};

/*!
 *  Addition of Vectors on the right hand side and left hand side.
 *  @param lhs  left hand side Vector
 *  @param rhs  right hand side Vector
 */
inline Vector operator+ (Vector lhs, const Vector& rhs) {
    lhs += rhs;
    return lhs;
}

/*!
 *  Subtraction of right hand side Vector from left hand side Vector.
 *  @param lhs  left hand side Vector
 *  @param rhs  right hand side Vector
 */
inline Vector operator- (Vector lhs, const Vector& rhs) {
    lhs -= rhs;
    return lhs;
}

/*!
 *  Multiplication of Vector with scalar.
 *  @param lhs  left hand side Vector
 *  @param rhs  right hand side scalar
 */
inline Vector operator* (Vector lhs, const double_t& rhs) {
    lhs *= rhs;
    return lhs;
}

/*!
 *  Multiplication of Vector with scalar.
 *  @param lhs  left hand side scalar
 *  @param rhs  right hand side Vector
 */
inline Vector operator* (const double_t& lhs, Vector rhs) {
    rhs *= lhs;
    return rhs;

}

/*!
 *  Division of Vector by scalar.
 *  @param lhs  left hand side Vector
 *  @param rhs  right hand side scalar
 */
inline Vector operator/ (Vector lhs, const double_t& rhs) {
    lhs /= rhs;
    return lhs;
}

/*!
 *  Scalar product between Vectors on the right hand side and left hand side.
 *  @param lhs  left hand side Vector
 *  @param rhs  right hand side Vector
 */
inline double_t operator* (const Vector lhs, const Vector& rhs) {
    return (lhs.m_X[0]*rhs.m_X[0] + lhs.m_X[1]*rhs.m_X[1]);
}

#endif /* Vector_hpp */
