#ifndef IO_Parameters_hpp
#define IO_Parameters_hpp

#include <iostream>
#include <cmath>
#include <stdio.h>
#include <string>

class Parameters {

public:

    /*!
     *  Default constructor is not allowed.
     */
    Parameters() = delete;

    /*!
     *  Tries to read configuration either from .json file or from the command line. If this fails, it rolls back to the predefined values. At the end of the function, it saves the updated parameters in the .json file.
     */
    Parameters(int argc, char* argv[]);

    /*!
     *  Copy constructor is not allowed.
     */
    Parameters(Parameters& base) = delete;

    ///////////////////////////////////////////////
    ///// Getter Functions
    ////////////////////////

    /*!
     *  Get cell substrate adhesion energy.
     */
    const double_t& C_SubstrateAdhesion() const {
        return m_C_SAdhesion;
    };

    /*!
     *  Get cell substrate dissipation.
     */
    const double_t& C_SubstrateDissipation() const {
        return m_C_SubstrateDissipation;
    };

    /*!
     *  Get cell substrate adhesion energy range.
     */
    const double_t& C_SubstrateAdhesionPolarizability() const {
        return m_C_SAdhesionPolarizability;
    };

    /*!
     *  Get cell polarization update rate.
     */
    const double_t& C_PolarizationUpdateRate() const {
        return m_C_UpdateRate;
    };

    /*!
     *  Get cell internal signalling radius squared.
     */
    const double_t& C_SignallingRadius2() const {
        return m_C_SignallingRadius2;
    };

    /*!
     *  Get cell bulk stiffness.
     */
    const double_t& C_BulkStiffness() const {
        return m_C_BulkStiffness;
    };

    /*!
     *  Get cell membrane stiffness.
     */
    const double_t& C_MembraneStiffness() const {
        return m_C_MembraneStiffness;
    };

    /*!
     *  Get cell cell adhesion energy.
     */
    const double_t& C_CellAdhesion() const {
        return m_C_CAdhesion;
    };

    /*!
     *  Get cell cell dissipation.
     */
    const double_t& C_CellDissipation() const {
        return m_C_CDissipation;
    };

    /*!
     *  Get substrate presimulation pattern.
     */
    const std::string& S_PatternPresim() const {
        return m_S_PatternPresim;
    };

    /*!
     *  Get substrate simulation pattern.
     */
    const std::string& S_PatternSim() const {
        return m_S_PatternSim;
    };

    /*!
     *  Get cell substrate adhesion penalty.
     */
    const double_t& S_CellAdhesionPenalty() const {
        return m_S_CAdhesionPenalty;
    };

    /*!
     *  Get inner confinement radius.
     */
    const double_t& S_ConfinementRadiusI() const {
        return m_S_ConfinementRadiusI;
    };

    /*!
     *  Get outer confinement radius.
     */
    const double_t& S_ConfinementRadiusO() const {
        return m_S_ConfinementRadiusO;
    };

    /*!
     *  Get effective temperature.
     */
    const double_t& Temperature() const {
        return m_Temperature;
    };

    /*!
     *  Get seed for random number generator.
     */
    const uint32_t& Seed() const {
        return m_Seed;
    };

    /*!
     *  Get number of cells.
     */
    const size_t& N_Cells() const {
        return m_N_Cells;
    };

    /*!
     *  Get number of columns.
     */
    const size_t& N_Columns() const {
        return m_N_Columns;
    };

    /*!
     *  Get number of rows.
     */
    const size_t& N_Rows() const {
        return m_N_Rows;
    };

    /*!
     *  Get number of presimulation steps.
     */
    const uint_fast32_t& N_MCSPresim() const {
        return m_N_MCSPresim;
    };

    /*!
     *  Get number of simulation steps.
     */
    const uint_fast32_t& N_MCSSim() const {
        return m_N_MCSSim;
    };

    /*!
     *  Get measurement interval.
     */
    const uint_fast32_t& I_Measure() const {
        return m_I_Measure;
    };

    /*!
     *  Get flag indicating whether extended measurement of cell shall be performed.
     */
    const bool& MeasureExtended() const {
        return m_CellMeasureExtended;
    };

    /*!
     *  Get flag indicating whether Maps are to be measured.
     */
    const bool& MeasureMaps() const {
        return m_CellMeasureMaps;
    };

    /*!
     *  Get flag indicating whether Monolayers are to be measured.
     */
    const bool& MeasureMono() const {
        return m_MeasureMono;
    };

    /*!
     *  Get snapshot interval.
     */
    const uint_fast32_t& I_Snapshot() const {
        return m_I_Snapshot;
    };



private:

    double_t m_C_SAdhesion; ///< Cell substrate adhesion energy
    double_t m_C_SAdhesionPolarizability; ///< Cell substrate adhesion energy range
    double_t m_C_SubstrateDissipation; ///< Cell substrate dissipation
    double_t m_C_UpdateRate; ///< Cell polarization update rate
    double_t m_C_SignallingRadius2; ///< Cell internal signalling radius squared
    double_t m_C_BulkStiffness; ///< Cell bulk stiffness
    double_t m_C_MembraneStiffness; ///< Cell membrane stiffness
    double_t m_C_CAdhesion; ///< Cell cell adhesion energy
    double_t m_C_CDissipation; ///< Cell cell dissipation


    std::string m_S_PatternPresim; ///< Substrate presimulation pattern
    std::string m_S_PatternSim; ///< Substrate simulation pattern
    double_t m_S_CAdhesionPenalty; ///< Cell substrate adhesion penalty (this only affects the custom substrate patterns!)
    double_t m_S_ConfinementRadiusI; ///< Inner confinement radius
    double_t m_S_ConfinementRadiusO; ///< Outer confinement radius


    double_t m_Temperature; ///< Effective temperature
    uint32_t m_Seed; ///< Seed for random number generator

    size_t m_N_Cells; ///< Number of cells
    size_t m_N_Columns; ///< Number of columns
    size_t m_N_Rows; ///< Number of rows
    uint_fast32_t m_N_MCSPresim; ///< Number of presimulation steps
    uint_fast32_t m_N_MCSSim; ///< Number of simulation steps

    uint_fast32_t m_I_Measure; ///< Measurement interval
    bool m_CellMeasureExtended; ///< Perform extended measurement of the cell?
    bool m_CellMeasureMaps; ///< Flag indicating whether Maps are to be measured
    bool m_MeasureMono; ///< Flag indicating whether Monolayer measurements are to be done
    uint_fast32_t m_I_Snapshot; ///< Snapshot interval

};

#endif /* IO_Parameters_hpp */
