#include "CPM_Cell.hpp"
#include <iostream>

/*!
 *  Perform a protrution Event.
 *  @param input Edge along which the cell protrudes
 */
void Cell::Protrude(Edge* input) {

    Tile* target = input->m_Comp->m_Tile;

    for (size_t i = 0; i != 6 ; ++i) {

        Edge* edge = &target->m_Edges[i];
        Edge* comp = edge->m_Comp;

        if (comp->m_Cell == this) {
            // remove edge from membrane by swap-popback, adjust index and perimeter
            m_Membrane[comp->m_IndexCell] = m_Membrane.back();
            m_Membrane[comp->m_IndexCell]->m_IndexCell = comp->m_IndexCell;
            m_Membrane.pop_back();
            m_MembraneLength -= comp->m_Length;

            // Overwrite members of the removed edge
            comp->m_IndexCell = 0;
            comp->m_Cell = nullptr;

        }
        else {
            // Add new membrane elements
            // Adjust boundaries of the update routine

            // add edge to membrane, adjust index and perimeter
            edge->m_IndexCell = m_Membrane.size();
            edge->m_Cell = this;
            m_Membrane.push_back(edge);
            m_MembraneLength += edge->m_Length;

        }

    }

    // Add tile to bulk
    target->m_CellSubstrateAdhesion = input->m_Tile->m_CellSubstrateAdhesion;
    target->m_Signal = input->m_Tile->m_Signal;
    target->m_Cell = this;
    target->m_IndexCell = m_Bulk.size();
    m_Bulk.push_back(target);
    m_BulkArea += target->m_Area;

    // Update Polarization Signal
    for (auto& tile : m_Bulk) {
        if ((target->X() - tile->X()).Norm2() < m_SignallingRadius2) {
            ++tile->m_Signal;
        }
    }

}

/*!
 *  Perform a retraction Event.
 *  @param input Edge along which the cell retracts
 */
void Cell::Retract(Edge* input) {

    Tile* target = input->m_Tile;

    for (size_t i = 5; i != size_t(-1) ; --i) {
        Edge* edge = &target->m_Edges[i];
        Edge* comp = edge->m_Comp;

        if (edge->m_Cell == this) {
            // remove edge from membrane by swap-popback, adjust index and perimeter
            m_Membrane[edge->m_IndexCell] = m_Membrane.back();
            m_Membrane[edge->m_IndexCell]->m_IndexCell = edge->m_IndexCell;
            m_Membrane.pop_back();
            m_MembraneLength -= edge->m_Length;

            // Overwrite members of the removed edge
            edge->m_IndexCell = 0;
            edge->m_Cell = nullptr;

        }
        else {
            // add edge to membrane, adjust index and perimeter
            comp->m_IndexCell = m_Membrane.size();
            comp->m_Cell = this;
            m_Membrane.push_back(comp);
            m_MembraneLength += comp->m_Length;

        }

    }

    // Remove tile from bulk
    m_Bulk[target->m_IndexCell] = m_Bulk.back();
    m_Bulk[target->m_IndexCell]->m_IndexCell = target->m_IndexCell;
    m_Bulk.pop_back();
    m_BulkArea -= target->m_Area;
    target->m_CellSubstrateAdhesion = 0;
    target->m_Signal = 0;
    target->m_Cell = nullptr;
    target->m_IndexCell = 0;

    // Update Polarization Signal
    for (auto& tile : m_Bulk) {
        if ((target->X() - tile->X()).Norm2() < m_SignallingRadius2) {
            --tile->m_Signal;
        }
    }

}

/*!
 *  Cell division operator by explicit copy construction. It divides the Cell along its short axis and distributes the substrate tiles left and right of the division axis to the corresponding daughter Cells.
 *  @param identifier index of the new Cell
 *  @param base Cell to be divided
 */
Cell::Cell(size_t identifier, Cell& base)
: m_X(base.m_X)
, m_MotilityState(base.m_MotilityState)
, m_ProliferationState(base.m_ProliferationState)
, m_Index(identifier)
, m_BulkArea(0.)
, m_BulkStiffness(base.m_BulkStiffness)
, m_MembraneLength(0.)
, m_MembraneStiffness(base.m_MembraneStiffness)
, m_SubstrateAdhesionMax(base.m_SubstrateAdhesionMax)
, m_SubstrateAdhesionMin(base.m_SubstrateAdhesionMin)
, m_BulkStiffnessQuiescence(base.m_BulkStiffnessQuiescence)
, m_MembraneStiffnessQuiescence(base.m_MembraneStiffnessQuiescence)
, m_UpdateRate(base.m_UpdateRate)
, m_SignallingRadius2(base.m_SignallingRadius2)
, m_GrowthDuration(base.m_GrowthDuration)
, m_DivisionDuration(base.m_DivisionDuration)
, m_AreaThresholdGrowth(base.m_AreaThresholdGrowth)
, m_AreaThresholdDivision(base.m_AreaThresholdDivision)
, g_RNG(base.g_RNG)
, m_Timer(base.m_Timer)
{

    // Begin division by updating coordinates of parent cell
    base.MoveVector();

    // Clear membrane of parent cell!
    for (auto& edge : base.m_Membrane) {
        edge->m_Cell = nullptr;
        edge->m_IndexCell = 0;
    }
    base.m_Membrane.clear();

    // Clear shape information of parent cell!
    base.m_MembraneLength = 0.;
    base.m_BulkArea = 0.;

    // Determine division axis: perpendicular to the long axis of the cell
    Cell::PrincipalComponents PCA = base.PrincipalComponentAnalysis();
    Vector axis = PCA.Eigenvector1; ///< Long axis

    // Create respective pools to pick new bulk elements of parent and daughter cells
    std::vector<Tile*> parentPool;
    parentPool.reserve(base.m_Bulk.size());
    std::vector<Tile*> daughterPool;
    daughterPool.reserve(base.m_Bulk.size());
    for (auto& tile : base.m_Bulk) {
        Vector vec = tile->X() - base.X();
        double_t correlator = vec * axis;
        if(correlator >= 0) {
            parentPool.push_back(tile);
        }
        else {
            tile->m_Cell = this;
            daughterPool.push_back(tile);
        }
    }
    base.m_Bulk.clear();

    // Use respective pools to initialize cell bulk and membrane
    base.InitializeFromPool(parentPool);
    InitializeFromPool(daughterPool);

    // End division by updating coordinates of parent and daughter cell
    base.MoveVector();
    MoveVector();

}

/*!
 *  Find the largest continuous blob of connected subtrate tiles and initialize the Cell from it.
 */
void Cell::InitializeFromPool(std::vector<Tile*> pool) {
    ////////////////////////////////////
    // Begin parent cell
    /////////////////////////

    // Search for largest connected blob!
    size_t largestCandidate = 0;
    std::vector<Tile*> NewBulk;
    NewBulk.reserve(pool.size());
    while (!pool.empty()) {
        std::vector<Tile*> Candidate;
        std::vector<Tile*> ToExplore;
        ToExplore.push_back(pool.back());
        pool.back()->m_Cell = nullptr; ///< Remove tile occupation
        pool.back()->m_IndexCell = 0; ///< Remove tile occupation
        pool.pop_back();
        size_t size = 0;

        do {
            Tile* tile = ToExplore.back();
            ToExplore.pop_back();
            Candidate.push_back(tile); ///< Grow candidate by adding tile

            for (auto& neighbor : tile->Neighbors()) {
                if (neighbor->m_Cell == this) {
                    ToExplore.push_back(neighbor);
                    neighbor->m_Cell = nullptr; ///< Remove tile occupation
                    neighbor->m_IndexCell = 0; ///< Remove tile occupation
                    pool.erase(std::find(pool.begin(), pool.end(), neighbor));
                    ++size;
                }
            }

        } while (!ToExplore.empty());

        if (size > largestCandidate) {
            largestCandidate = NewBulk.size();
            NewBulk = Candidate;
        };
    };

    // Parent cell is new largest connected blob!
    m_Bulk.reserve(NewBulk.size());
    for (auto& tile : NewBulk) {
        tile->m_Cell = this;
        tile->m_IndexCell = m_Bulk.size();
        m_BulkArea += tile->m_Area;
        m_Bulk.push_back(tile);
    }

    // Construct new unsorted membrane of parent cell
    for (auto& tile : m_Bulk) {
        for (size_t i=0; i<6; ++i) {
            Tile* neighbor = tile->m_Neighbors[i];
            Edge* edge = &tile->m_Edges[i];
            if (neighbor->m_Cell != this) {
                edge->m_Cell = this;
                edge->m_IndexCell = m_Membrane.size();
                m_MembraneLength += edge->m_Length;
                m_Membrane.push_back(edge);
            }
        }
    }

}
