% This script creates Figure 6B-E
%  Samora Okujeni 5/22/2019
%
%% load the data
load('data_calcium.mat');

% fields:
% cda:          1: Goe6976, 2: Control (chronic drug application) 
% ca_amplitude: amplitude of Ca-transient for an individual SBE and cell
% pfr:          peak firing rate in the network assigned to every ca_amplitude (redundant accross cells of the same network)
% cellid:       cell index per network for every ca_amplitude
% networkid:    network index for every ca_amplitude
% sbeid:        burst index for a given network assigned to every ca_amplitude (redundant accross cells of the same network)
% stats.exponents:  exponent calculated for evey cell 
% stats.R:          R2 value for the fit of the exponential function for every cell
% stats.cda:        treatment of the cell 1: Goe6976, 2: Control (chronic drug application)
%% Plot Ca-amplitudes against peak firing rate for all SBEs and cells
xscaling = 10;
col = [[1 0 0];[0 0 0]];
figure,
for cc=1:2 % loop across PKC condition (1: PKC- Goe6976, 2: Control)
	subplot(1,2,cc); hold on;
    n = ceil(max(data.pfr(data.cda==cc)))*xscaling;
    plotbins = 0:n;
    xdata = zeros(1,n);
    ydata = zeros(1,n);
    ydata_std = zeros(1,n);
    for ii=1:n,
        id = find(round(data.pfr*xscaling)==ii & data.cda==cc);
        if ~isempty(id), % no data for this pfr bin
            ydata(ii) = median(data.ca_amplitude(id));
            ydata_std(ii) = std(data.ca_amplitude(id));
            xdata(ii) = ii/xscaling;
        end
    end
    valid = find(xdata.*ydata>0); % id specifying bins with values
    xdata = xdata(valid);
    ydata = ydata(valid);
    ydata_std =ydata_std(valid);
    errorbar(xdata,ydata,ydata_std,ydata_std,'k.','Color',[.7 .7 .7],'Marker','none'); %,'Color',col(ii,:));
    plot(xdata,ydata,'ks','MarkerFaceColor',col(cc,:),'MarkerSize',5); %,'Color',col(ii,:));
    
    % plot exponential fit
    [xdata,id] = sort(xdata,'ascend');
    ydata = ydata(id);
    exponent = fit_exponent(xdata,ydata);
    xrange = min(xdata):.1:max(xdata);
    exp_fit = exp(exponent*xrange)-1;
    hold on; plot(xrange,exp_fit,'m');
    
    set(gca,'XLim',[0 60],'YLim',[0 ceil(max(exp_fit)/100)*100]);
    axis square
    title(['exponent: ' num2str(exponent,3)]);
end

%% retrieve exponents of exponential fit for every cell
ucid = unique(data.cellid);
exponents = nan(size(ucid));
R = nan(size(ucid));
for ii=1:length(ucid),
    id = find(data.cellid==ucid(ii));
    xdata = data.pfr(id);
    ydata = data.ca_amplitude(id);
    [xdata,id] = sort(xdata,'ascend');
    ydata = ydata(id);
    exponents(ii) = fit_exponent(xdata,ydata);
    
    % calculate R2 for quality of fit
    yfit = exp(exponents(ii)*xdata)-1;
    SSres = sum((ydata-yfit).^2);
    SStot = sum((ydata-mean(ydata)).^2);
    R(ii) = 1-(SSres/SStot);
end
isequal(data.stats.exponents,exponents); % check if new fitting produces the same result
%% plot exponent distributions
col = [[1 0 0];[0 0 0]];
figure, hold on;
for ii=1:2
    id = find(data.stats.cda==ii);
    exp_distribution = histc(data.stats.exponents(id),[0:.005:.2]); 
    exp_distribution = exp_distribution/sum(exp_distribution); % normalize distributions
    plot([0:.005:.2],exp_distribution,'Color',col(ii,:));
end
xlabel('exponent'); 
ylabel('fraction'); 
legend('PKC^-','PKC^N');
axis square
plot([.11 .11],[0 .25],'b');
%% plot data for individual cells
ii = 522; % chose a cell manually
col = [[1 0 0];[0 0 0]];
ucid = unique(data.cellid);
id = find(data.cellid==ucid(ii)); % data index for chosen cell
xdata = data.pfr(id);
ydata = data.ca_amplitude(id);
figure, 
plot(xdata,ydata,'.','Color',col(unique(data.cda(id)),:)); 
xrange = min(xdata):.1:max(xdata);
exponent = fit_exponent(xdata,ydata);
exp_fit = exp(exponent*xrange)-1;
hold on; plot(xrange,exp_fit,'-','Color',col(unique(data.cda(id)),:));
xlabel('peak firing rate [Hz]');
ylabel(['dF/F']);
title(['exponent: ' num2str(data.stats.exponents(ucid(ii)),3) ', R^2: ' num2str(data.stats.R(ucid(ii)))]);
set(gca,'XLim',[0 60]);
%% statistics
[~,p]=ttest2(data.stats.exponents(data.stats.cda==1),data.stats.exponents(data.stats.cda==2));
[~,id] = unique(data.cellid,'first'); % index for table lookup
[~,id2]=unique(data.networkid,'last'); % index for table lookup
T = array2table([...
    [length(unique(data.networkid(data.cda==1))),length(unique(data.networkid(data.cda==2)))];... % N_networks
    [sum(data.cda(id)==1),sum(data.cda(id)==2)];... % N_neurons
    [sum(data.sbeid(id2(data.cda(id2)==1))),sum(data.sbeid(id2(data.cda(id2)==2)))];... % N_sbe
    [mean(data.stats.exponents(data.stats.cda==1)),mean(data.stats.exponents(data.stats.cda==2))];... % exp_mean
    [std(data.stats.exponents(data.stats.cda==1)),std(data.stats.exponents(data.stats.cda==2))];... %exp_std
    [p 1]],... % p-value (unpaired t-test PKC^- against PKC^N)
    'VariableNames',{'Goe6976','Control'},...
    'RowNames',{'# networks','# neurons','# sbes','exponent mean','exponent std','p-value (t-test against Control)'});
disp(T);
T2 = array2table(...
    [mean(data.stats.exponents);... % exp_mean_all (mean of all exponents)
    std(data.stats.exponents)],... % exp_std_all (std of all exponents)
    'VariableNames',{'all'},'RowNames',{'exponent mean','exponent std'});
disp(T2);





