% This script creates Figure 6B-K and table 1
%  Samora Okujeni 5/22/2019
%
%% load the data
load('data_morphology_development.mat')
%
% Fields in data structure:
%
% cda:              1: Goe6976, 2: Control (chronic drug application) 
% div:              days in vitro
% cts_dendrites     dendrite pixel counts within each image
% cts_neurons       number of neurons within each image
% cts_synapses      number of synapses within each image
% area              image area in mm2
% ci                clustering index for each image
% micronsperpixel   micrometer per pixel

%% manually choose parameter to plot
p = 2; % 1: dendrites/neuron, 2: clustering index, 3: synapses/dendrite, 4: neuron density, 5: synapses/neuron, 6: maximal connectivity
param_names = {'dendrite size (dendrites/neuron)','clustering index','synapses/dendrite','neuron density','synapses/neuron','maximal connectivity'};
%
switch p
    case 1        % dendrites/neuron
        param = data.cts_dendrites*data.micronsperpixel./data.cts_neurons;ymax = 2e3; ylab = 'dendrites/neuron [\mum]'; sv = 0;
    case 2        % clustering index
        param = data.ci;ymax = 1.5; ylab = 'ci'; sv = 1;
    case 3        % synapses/dendrite
        param = data.cts_synapses./(data.cts_dendrites*data.micronsperpixel);ymax = 2; ylab = 'synapses/dendrite [\mum^{-1}]'; sv = 0;
    case 4        % neuron density
        param = data.cts_neurons./data.area; ymax = .5e3; ylab = 'neuron density [mm^{-2}]'; sv = (6e6/8*.1)/(pi*(6)^2)/2;
    case 5        % synapses/neuron
        param = data.cts_synapses./data.cts_neurons;ymax = 2.5e3; ylab = 'synapses/neuron'; sv = 0;
    case 6        % maximal connectivity
        param = (data.cts_synapses./data.cts_neurons)./(data.cts_neurons./data.area*(pi*(6)^2));ymax = .3; ylab = 'maximal connectivity'; sv = 0;
end
%% Plot parameter
udiv = unique(data.div);
ucda = unique(data.cda);
parameter_mean = nan(length(ucda),length(udiv));
parameter_std = nan(length(ucda),length(udiv));
parameter_sem = nan(length(ucda),length(udiv));
parameter_p = nan(length(ucda),length(udiv));
n = nan(length(ucda),length(udiv));
for dd=1:length(udiv),
    for cc=1:length(ucda),
      id_con = find((data.cda==2).*(data.div==udiv(dd)));    % index for control data within developmental time window
      id = find((data.cda==ucda(cc)).*(data.div==udiv(dd))); % index for specified treatment and developmental time window
      
      parameter_mean(cc,dd)=mean(param(id));                  % mean 
      parameter_std(cc,dd)=std(param(id));                    % STD
      parameter_sem(cc,dd)=std(param(id))/sqrt(length(id));   % SEM 
      [~,parameter_p(cc,dd)]=ttest2(param(id),param(id_con)); % significance test
      n(cc,dd) = length(id);                                  % number of images
    end
end
figure, hold on;
col = [[1 0 0];[0 0 0];[0 0.5 0]];
sig = {'','*','**','***'};
for cc=1:length(ucda),
    errorbar(udiv,parameter_mean(cc,:),parameter_sem(cc,:),parameter_sem(cc,:),'o:','Color',col(cc,:));
    if cc~=2,
        text(udiv,parameter_mean(cc,:)+parameter_sem(cc,:),sig(1+(parameter_p(cc,:)<0.05)+(parameter_p(cc,:)<0.01)+(parameter_p(cc,:)<0.001)),'HorizontalAlignment','center')
    end
    plot([0 udiv(1)],[sv parameter_mean(cc,1)],':','Color',col(cc,:));    
end
set(gca,'XLim',[0 35],'YLim',[0 ymax]);
xlabel('DIV');
ylabel(ylab);
set(gca,'TickDir','out')
%% Statistics for chosen parameter
clear T
for ii=1:size(parameter_mean,1)
    for jj=1:size(parameter_mean,2),
        if ii~=2
            T{jj,ii} = [num2str(parameter_mean(ii,jj),'%10.2e\n') ' ' char(177) ' ' num2str(parameter_sem(ii,jj),'%10.2e\n') ' (p=' num2str(parameter_p(ii,jj),'%10.2e\n') ', n=' num2str(n(ii,jj)) ')'];
        else
            T{jj,ii} = [num2str(parameter_mean(ii,jj),'%10.2e\n') ' ' char(177) ' ' num2str(parameter_sem(ii,jj),'%10.2e\n') ' (n=' num2str(n(ii,jj)) ')'];
        end
    end
end
T = array2table(T,'VariableNames',{'Goe6976','Control','PMA'},'RowNames',{'8 DIV','15 DIV','22 DIV','29 DIV'});

disp('---------------------------');
disp(param_names{p});
disp(T);
disp('---------------------------');