%% Segment foci and fibers in files in a folder, record foci numbers in each cell

% folder containing TIFF image streams
pathImages = '/Volumes/lab-tolarp/working/Sophie Roper/180109/DMSO_1/';

readframes = 1:200; % analyze these frames in each file

results = table(); % to store results
counter = 1;

% get filenames
fnames = dir(pathImages);
fnames = struct2table(fnames);
fnames = fnames.name;
fnames = fnames(3:end);

% go through files
for f = 1: numel(fnames)
    disp (['reading image ' fnames{f}])
    
    % load image stream
    imageFull = read_image_sequence([pathImages fnames{f}], readframes);
    imageFull = squeeze(imageFull);
    
    % find cells
    cellSize = 50; % for primary cells, use 150 for Ramos
    mask = bpass(imageFull(:,:,1), 15, cellSize);
    cells = pkfnd(mask, 100, cellSize); % adjust threshold for Lifeact brightness
    
    % analyze each cell
    for c = 1:size(cells, 1)
        
        %% segmentation
        disp (['processing cell ', num2str(c)])
        
        image = imageFull(cells(c,2)-cellSize:cells(c,2)+cellSize, cells(c,1)-cellSize:cells(c,1)+cellSize, :);
        
        image = image-100; % background subtraction
        
        % segment cell and estimate brightness
        imageBorder = image;
        imageBorder(20:size(image,1)-20, 20:size(image,1), :) = nan;
        imageBorder = imageBorder(~isnan(imageBorder));
        
        th = 2 * quantile(imageBorder, 0.15) + (quantile(image(:), 0.9) - quantile(imageBorder, 0.15))./5;
        meanIntensity = mean(image(image>th));
        cellAreaImage = image>th;
        cellArea = 0.0042*sum(sum(image>th)); % 0.065*0.065=0.0042 um^2 per pixel for Orca4.0
        
        % filter foci and fibers: primary B cells
        [maskF, maskF2] = bpassLine(image,4,4); % use 7, 7 for Ramos
        maskS = bpassseq(image,5); % use 7 for Ramos
        maskS = maskS-maskF2;
        
        % threshold spots or fibers masks using fraction of Lifeact intensity (adjust as needed)
        % primary B cells
        fiberThreshold = meanIntensity/15; % use 20 for Ramos
        fociThreshold = meanIntensity/5;   % use 10 for Ramos
        
        maskFthed = maskF>fiberThreshold;
        maskSthed = maskS>fociThreshold;
        
        % process spots and fibers
        foci = maskS;
        foci(maskSthed==0) = 0;
        fibers = maskF;
        fibers(maskFthed==0) = 0;
        
        % remove elongated spots
        fociCleaned = foci;
        for m = 1:size(image,3)
            fociCleaned(:,:,m) = bwpropfilt(foci(:,:,m)>0, 'Eccentricity', [0, 0.9]);
        end
        
        % take out small stuff
        se = strel('sphere', 1);
        fociCleaned = imopen(fociCleaned, se);
        
        fociCleaned = fociCleaned.*maskS;
        
        % remove foci not inside of cell
        se = strel('disk', 2, 4);
        for m = 1:size(image,3)
            cellAreaImage(:,:,m) = imerode(cellAreaImage(:,:,m), se);
        end
        fociCleaned(cellAreaImage<1) = 0;
        
        % track foci
        coordinates = zeros (1,3);
        for frame = 1:size(image,3)
            fociPeaks = pkfnd(fociCleaned(:,:,frame), fociThreshold, 3);
            fociPeaks(:,3) = frame;
            coordinates = cat(1, coordinates, fociPeaks);
        end
        coordinates = coordinates(2:end,:);
        coordinates = coordinates(coordinates(:,1)~=0, :);
        param.mem = 2; % how many frames could miss
        param.dim = 2;
        param.good = 2; %shortest track allowed
        param.quiet = 0;
        maxdisplacement = 4;
        if ~isempty(coordinates) && size(coordinates,1)>1
            tracks = track(coordinates, maxdisplacement, param);
        else
            tracks = [];
        end
        
        % get tracked foci from the original foci shapes
        if ~isempty(tracks)
            for frame = 1:size(image,3)
                foci(:,:,frame) = bwselect(foci(:,:,frame), tracks(tracks(:,3)==frame, 1), tracks(tracks(:,3)==frame, 2));
            end
            foci = foci.*maskS;
        else
            foci = foci.*0;
        end
        
        % final foci structures are now in the foci variable
        fibers(foci>0) = 0;
        
        %% data
        results.file{counter,1} = fnames{f};
        results.cell(counter,1) = c;
        
        % calculate and store foci numbers per frame per 10 um^2
        results.numberper10u(counter,1) = 10 * size(tracks, 1) / size(image,3) / mean(cellArea);
        
        % increment cell counter
        counter = counter+1;
    end
    
end

% save results
writetable(results, ['results_', strrep(pathImages, '/', '_'), '.txt'], 'Delimiter', 'tab')



