#!/usr/bin/env python
# In case of poor (Sh***y) commenting contact adam.lamson@colorado.edu
# YOLO edelmaie@colorado.edu (too)
# Basic
import sys, os, pdb
import gc
## Analysis
import numpy as np

'''
Name: quaternion.py
Description: Quaternoin utilities based on CJE C++ version used in newagebob
Input:
Output:
'''

# Create the axis angle representation for rotation between two vectors a and b
def create_axisangle(a, b):
    theta = np.arccos(np.dot(a, b))
    eaxis = np.zeros(3)

    if np.fabs(theta) > 1e-5:
        eaxis = np.cross(a, b)
        eaxis = eaxis / np.linalg.norm(eaxis)

    return [theta, eaxis]

# Create a quaternion from an axisangle representation
def quaternion_from_axisangle(theta, eaxis):
    q = np.zeros(4)
    q[0] = np.cos(theta/2.)
    q[1] = eaxis[0] * np.sin(theta/2.)
    q[2] = eaxis[1] * np.sin(theta/2.)
    q[3] = eaxis[2] * np.sin(theta/2.)
    return q

# Get the axisangle from a quaternion
def axisangle_from_quaternion(q):
    theta = 2. * np.arctan2(np.sqrt(np.dot(q[1:],q[1:])), q[0])
    eaxis = np.zeros(3)
    if theta != 0.0:
        eaxis[0] = q[1] / np.sin(theta/2.)
        eaxis[1] = q[2] / np.sin(theta/2.)
        eaxis[2] = q[3] / np.sin(theta/2.)
    return [theta, eaxis]

# Rodrigues rotation formula to rotate v to vrot (return) by e and theta
def rodrigues_axisangle(v, theta, e):
    sintheta = np.sin(theta)
    costheta = np.cos(theta)
    e_dot_v = np.dot(e, v)
    e_cross_v = np.cross(e, v)
    return costheta * v + sintheta * e_cross_v + (1. - costheta) * e_dot_v * e

# Generates the quaternion between two vectors a and b, and returns the 4vec
def quaternion_between_vectors(a, b):
    # Normalize the vectors
    na = a / np.linalg.norm(a)
    nb = b / np.linalg.norm(b)

    # Create the axisangle representation, then create quaternion from that
    theta, e = create_axisangle(na, nb)

    # Get the resultant quaternion
    q = quaternion_from_axisangle(theta, e)

    return q / np.linalg.norm(q)

# Multiply  quaternions
def quaternion_multiply(q, r):
    t = np.zeros(4)
    t[0] = r[0]*q[0] - r[1]*q[1] - r[2]*q[2] - r[3]*q[3]
    t[1] = r[0]*q[1] + r[1]*q[0] - r[2]*q[3] + r[3]*q[2]
    t[2] = r[0]*q[2] + r[1]*q[3] + r[2]*q[0] - r[3]*q[1]
    t[3] = r[0]*q[3] - r[1]*q[2] + r[2]*q[1] + r[3]*q[0]
    return t / np.linalg.norm(t)

# Getting a unit quaternion into a rotation matrix
# Assuming a unit quaternion
def rotation_matrix_from_quaternion(q):
    R = np.zeros((3,3))

    R[0][0] = 1. - 2.*(q[2] * q[2] + q[3] * q[3])
    R[0][1] = 2. * (q[1] * q[2] - q[3] * q[0])
    R[0][2] = 2. * (q[1] * q[3] + q[2] * q[0])

    R[1][0] = 2. * (q[1] * q[2] + q[3] * q[0])
    R[1][1] = 1 - 2. * (q[1] * q[1] + q[3] * q[3])
    R[1][2] = 2. * (q[2] * q[3] - q[1] * q[0])

    R[2][0] = 2. * (q[1] * q[3] - q[2] * q[0])
    R[2][1] = 2. * (q[2] * q[3] + q[1] * q[0])
    R[2][2] = 1 - 2. * (q[1] * q[1] + q[2] * q[2])

    return R

# Geberate a max distance (angular) between successive frames!
def rotate_towards(q1, q2, maxAngle):
    q = np.zeros(4)
    if maxAngle < 0.001:
        q = q1
        return q
    
    cosTheta = np.dot(q1, q2)

    if cosTheta > 0.9999:
        q = q2
        return q

    if cosTheta < 0.:
        q1 = -1.0 * q1
        cosTheta *= -1.0

    angle = np.arccos(cosTheta)

    if angle < maxAngle:
        q = q2
        return q

    fT = maxAngle / angle
    angle = maxAngle

    q[0] = (np.sin((1.0 - fT) * angle) * q1[0] + np.sin(fT * angle) * q2[0]) / np.sin(angle)
    q[1] = (np.sin((1.0 - fT) * angle) * q1[1] + np.sin(fT * angle) * q2[1]) / np.sin(angle)
    q[2] = (np.sin((1.0 - fT) * angle) * q1[2] + np.sin(fT * angle) * q2[2]) / np.sin(angle)
    q[3] = (np.sin((1.0 - fT) * angle) * q1[3] + np.sin(fT * angle) * q2[3]) / np.sin(angle)

    q = q / np.linalg.norm(q)
    return q
