#!/usr/bin/env python
# In case of poor (Sh**y) commenting contact adam.lamson@colorado.edu
# Basic
import sys, os, pdb
import gc
from collections import OrderedDict
## Analysis
sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Lib'))
from run_base import RunBase
from spindle_sim import SpindleSim
from spindle_seed import SpindleSeed
from spindle_unit_dict import SpindleUnitDict, ModifyXLabel
from base_funcs import *
from bin_functions import *
from run_graph_funcs import *
from stylelib.ase1_styles import ase1_runs_stl
from stylelib.ase1_styles import cp_spindle_stl

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from mpl_toolkits.axes_grid1 import make_axes_locatable
import itertools
# from math import *

'''
Name: spindle_run.py
Description:
'''

uc = SpindleUnitDict()

def MakeColorBar(ax, label = r'Number of Seeds',
                 cmap=mpl.cm.rainbow, v_min=0, v_max=1):
    divider = make_axes_locatable(ax)
    cax = divider.append_axes('right', size='5%', pad=0.1)
    cb = mpl.colorbar.ColorbarBase(cax, cmap=cmap,
            norm = mpl.colors.Normalize(vmin=v_min, vmax=v_max))
    cb.set_label(label)

def BinomialScatterPlot(ax, xticks, p_vals, binary_arr,
                        color='k', percent_fac=100, label = None, 
                        lin_reg=False, **kwargs):
    # Sort successes into bins based on parameter value
    bins, bin_err, bin_edges, bin_n, bin_width, bin_max = \
        BinomialBarArray(p_vals, binary_arr)
    
    y = bins*percent_fac
    yerr = bin_err*percent_fac

    # Graph fraction success percentage in a scatter plot vs their parameter value.
    # Also graph the error based on their bin_err.
    ax.scatter(xticks, y, zorder=100,
               s=100, marker='s', color=color, label=label)

    # Make error bar graph using param values (xticks),
    # percent successful (bins*percent_fac), and calculated
    # error of percentage (bin_err*percent_fac).
    # All other variables are for formating error bars.
    ax.errorbar(xticks, y, yerr=yerr,
        ecolor=color, elinewidth=2, capsize=7, capthick=1, zorder=0,
        fmt='none', marker='none')

    if lin_reg: add_linear_regression_lines(ax, xticks, y)

    # Set x-axis limits to make data readable
    ax.set_xlim(left=xticks[0]-bin_width*1.05, right=xticks[-1]+bin_width*1.05)
    # ax.set_xticks(xticks)

    return

def BinScatterPlot(ax, xticks, p_vals, arr, label = None, marker = 's'):
    # xticks = np.unique(p_vals)
    Abs_bin_max = p_vals.size/len(xticks)
    # Sort successes into bins based on parameter value
    bins, bin_err, bin_edges, bin_n, bin_width, bin_max = \
        BarArray(p_vals, arr, use_zeros=False)

    # Graph average spindle separation(bins) vs param values(xticks).
    # Color is determined by successful spindle fraction of that
    # particular parameter value.
    ax.scatter(xticks, bins, color=mpl.cm.rainbow(bin_n/Abs_bin_max),
               zorder=100, s=100, label=label, marker=marker)

    # Graph error bars
    ax.errorbar(xticks, bins, yerr=bin_err,
        ecolor='k', elinewidth=2, capsize=7, capthick=1, zorder=0,
        fmt='none', marker='none')
    # Set x and y limits for readability
    ax.set_xlim(left=xticks[0]-bin_width*1.05, right=xticks[-1]+bin_width*1.05)
    return

class ParamError(Exception):
    def __init__(self, param=''):
        if param:
            err_str = "*** {} not in criteria data file. ***".format(param)
        else:
            err_str = "*** No param given to graph. ***"

        Exception.__init__(self, err_str)

#Class definition
class SpindleRun(RunBase):
    def __init__(self, opts, simdir_path="simulations",
            simType=SpindleSim, seedType=SpindleSeed):
        RunBase.__init__(self, opts, simType=SpindleSim, seedType=SpindleSeed)

        self.sim_crit_df = pd.DataFrame() # Sim Criteria Data Frame

    def MakeCriteriaDataFrame(self):
        # Default criteria for spindle success
        crit_dict = { 'thresh': 1.0,
                      'break_length': .2,
                      'duration': 2.0,
                      'end_flag': True
                    }
        for sim in self.sims:
            sim.CalcSimSuccess()
            for sd in sim.seeds:
                d = sd.MakeSeedSuccessDict(**crit_dict)
                for p, v in sim.params.iteritems():
                    d[p] = v
                # TODO Comment what is going on in this nasty function
                self.sim_crit_df = self.sim_crit_df.append(pd.DataFrame(d, columns=d, index=[0]), ignore_index=True)
            print "-- Criteria Sim: {}/{} -- ".format(self.sims.index(sim)+1, len(self.sims))
        return

    def WriteCriteriaDataFrame(self):
        with open("{}_Seed_Criteria.dat".format(os.path.join(self.opts.datadir, self.run_name)), 'w') as df:
            self.sim_crit_df.to_csv(df, sep=" ", index=False)

    def ReadCriteriaDataFrame(self, crit_df_file=None):
        if not self.sim_crit_df.empty:
            self.p_names = list(set(list(self.sim_crit_df)).intersection(uc.ud.keys()))
            return

        # If Run has not created a criteria data frame find it in the data path
        elif not crit_df_file:
            dat_files = []
            dat_files += [ f for f in os.listdir(self.opts.datadir) if f.endswith('.dat')]
            # If there only one .dat file exists try to use that as your data frame
            if len(dat_files) == 1:
                crit_df_file = os.path.join(self.opts.datadir, dat_files[0])

        try: crit_df_file = os.path.abspath(crit_df_file)
        except:
            "Could not find path to {}".format(crit_df_file)
            raise

        self.sim_crit_df = pd.read_csv(crit_df_file, delimiter=' ', header=0)

        self.p_names = list(set(list(self.sim_crit_df)).intersection(uc.ud.keys()))
        return

    def GraphSuccFracScatter(self, ax, param=''):
        """ Method that makes a scatter plot with relative successes with
        respect to a single parameter from a criteria DataFrame. DataFrame
        must be either created or read in.
        """

        # Find the param you will be graphing
        param = self.CheckGraphParam(param)

        # Set graphing details
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        # Determine if param variable is a string or list which will
        # tell you if this is a scan or a slice run
        if isinstance(param, str): self.GraphSuccScan(ax, param, lin_reg=self.opts.lin_reg)
        elif isinstance(param, list):
            self.GraphSuccSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])
        # TODO Make another statement for when its more than a
        # two parameter slice.

        # Set y-axis limits
        yticks = ax.get_yticks()
        ax.set_yticks(filter(lambda y: y<=100 and y>=0, yticks))
        if ax.get_ylim()[0] < 0: ax.set_ylim(bottom=0)
        if ax.get_ylim()[1] > 100: ax.set_ylim(top=100)
        ax.set_ylim(top=ax.get_ylim()[1]*1.1)

        # Trying to make x-axis and title prettier some times
        if self.opts.xlog:
            ax.xaxis.set_major_locator(mpl.ticker.LogLocator(base=2))
        else: ax.xaxis.set_major_locator(mpl.ticker.MaxNLocator())
        # ax.set_xticklabels(ax.get_xticks(), rotation=-30, horizontalalignment='left')
        ax.set_xticklabels(ax.get_xticks(), rotation=-30,
                           horizontalalignment='left')#,
                           # fontproperties=mpl.font_manager.FontProperties(font))
        ax.xaxis.set_major_formatter(mpl.ticker.FormatStrFormatter('%.3f'))

        # Axes and sup title
        ax.set_ylabel(r'Bipolar Spindle Frequency ($\%$)')
        title = r'{}'.format(self.run_name.replace('_',' '))

        plt.draw()

        # Name of the file where graph will be saved
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_succ_frac'.format(self.run_name, param))

        if self.opts.xlog:
            file_name += '_xlog'
            title  += ' xlog'
        if self.opts.ylog:
            file_name += '_ylog'
            title  += ' ylog'
        # TODO add option what kind of picture format you would like
        file_name += '.pdf'
        return file_name, title

    def GraphSuccScan(self, ax, param, **kwargs):
        p_vals = self.sim_crit_df[param].astype(uc[param][2])*uc[param][1]
        xticks = np.unique(p_vals)

        if self.opts.xlog: ax.set_xscale("log", nonposx='clip')

        BinomialScatterPlot(ax, xticks, p_vals, self.sim_crit_df['succ'], **kwargs)
        ModifyXLabel(ax, param)
        # if self.opts.xlog: ax.set_xticks(xticks)
        return

    def GraphSuccSlice(self, ax, param_1, param_2):
        p2_unique = np.sort(self.sim_crit_df[param_2].unique())

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        for p2v, color in zip(np.nditer(p2_unique), colors):
            # Make a dataframe that only contains rows with p2v value entries
            p2v_df = self.sim_crit_df.loc[ (self.sim_crit_df[param_2].values == p2v ) ]
            p1_vals = p2v_df[param_1].astype(uc[param_1][2]).values*uc[param_1][1]
            xticks = np.unique(p1_vals)
            label = r'{} = {}'.format(param_2,
                                     uc[param_2][2](p2v)*uc[param_2][1])
            BinomialScatterPlot(ax, xticks, p1_vals, p2v_df['succ'],
                                label = label, color=color)

        ModifyXLabel(ax, param_1)

        ax.legend()
        return

    def GraphSpindleSepScatter(self, ax, param=''):
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)
        # font ={'size' : 15}
        # #         # 'family' : 'Times'}
        #       }
        # mpl.rc('font', **font)
        # mpl.rc('text', usetex=True)
        # fig, ax = plt.subplots()

        # Determine if param variable is a string or list which will
        # tell you if this is a scan or a slice run
        if isinstance(param, str): self.GraphSpindleSepScan(ax, param)
        elif isinstance(param, list):
            self.GraphSpindleSepSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])
        # TODO Make another statement for when its more than a
        # two parameter slice.

        if self.opts.xlog:
            ax.xaxis.set_major_locator(mpl.ticker.LogLocator(base=2))
        else: ax.xaxis.set_major_locator(mpl.ticker.MaxNLocator())
        # ax.xaxis.set_major_locator(mpl.ticker.MaxNLocator())
        # ax.set_xticklabels(ax.get_xticks(), rotation=-30, horizontalalignment='left', fontproperties=mpl.font_manager.FontProperties(font))
        ax.set_xticklabels(ax.get_xticks(), rotation=-30, horizontalalignment='left' )
        # ax.xaxis.set_major_formatter(mpl.ticker.FormatStrFormatter('%.3f'))
        ax.set_ylim(bottom=0)
        ax.set_ylim(top=ax.get_ylim()[1]*1.1)


        # Make x, y, and color bar labels
        ax.set_ylabel(r'Late-time SPB Seperation ($\mu$m)')
        MakeColorBar(ax, v_max=100, label=r'Bipolar Spindle Frequency ($\%$)')

        title = r'{}'.format(self.run_name.replace('_',' '))

        # Create graph
        plt.draw()
        # ax.xaxis.set_minor_locator(mpl.ticker.FixedLocator(xticks))
        # plt.tight_layout()
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_SpindleSep.pdf'.format(self.run_name, param))

        return file_name, title

    def GraphSpindleSepScan(self, ax, param):
        p_vals = self.sim_crit_df[param].astype(uc[param][2])*uc[param][1]
        xticks = np.unique(p_vals)
        if self.opts.xlog: ax.set_xscale("log")
        BinScatterPlot(ax, xticks, p_vals, self.sim_crit_df['SpindleSepAvg']*uc['um'][1])
        ModifyXLabel(ax, param)

    def GraphSpindleSepSlice(self, ax, param_1, param_2 ):
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        p2_unique = np.sort(self.sim_crit_df[param_2].unique())

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        for p2v, color in zip(np.nditer(p2_unique), colors):
            # Make a dataframe that only contains rows with p2v value entries
            p2v_df = self.sim_crit_df.loc[ (self.sim_crit_df[param_2].values == p2v ) ]
            p1_vals = p2v_df[param_1].astype(uc[param_1][2]).values*uc[param_1][1]
            xticks = np.unique(p1_vals)
            label = r'{} = {}'.format(param_2,
                                     uc[param_2][2](p2v)*uc[param_2][1])
            BinScatterPlot(ax, xticks, p1_vals, p2v_df['SpindleSepAvg']*uc['um'][1], label = label, marker=marker.next())
        ax.legend(loc='best')
        ModifyXLabel(ax, param_1)

        return

    # Graph the final spindle length, regardless of success
    def GraphFinalSpindleLength(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        # Determine if this is a parameter scan or slice
        if isinstance(param, str): self.GraphFinalSpindleLengthScan(ax, param)
        elif isinstance(param, list):
            self.GraphFinalSpindleLengthSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])

        ax.set_ylabel(r'Avg. late spindle length ($\mu$m)')

        # Create the graph
        plt.draw()
        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_FinalSpindleLength.pdf'.format(self.run_name, param))

        return file_name, title

    # Graph the final spindle length for a simple scan
    def GraphFinalSpindleLengthScan(self, ax, param):
        final_length = {}
        final_length_stddev= {}
        nseeds = {}
        x_vals = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param:
                    if (v * uc[param][1]) not in final_length:
                        final_length[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in final_length_stddev:
                        final_length_stddev[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in nseeds:
                        nseeds[v*uc[param][1]] = 0
                    final_length[v*uc[param][1]] += [sim.final_length_mean]
                    final_length_stddev[v*uc[param][1]] += [sim.final_length_std]
                    nseeds[v*uc[param][1]] += len(sim.seeds)
                    x_vals += [v * uc[param][1]]

        x_vals = np.unique(x_vals)
        length_vals = np.zeros(len(x_vals))
        length_errs = np.zeros(len(x_vals))
        length_devm = np.zeros(len(x_vals)) # Standard deviation of the mean
        # For each unique entry in the list, do a weighted average for the error and mean chromosome seconds
        for i in xrange(len(x_vals)):
            xval = x_vals[i]
            length_vals[i], length_errs[i] = WeightedMeanError(final_length[xval], final_length_stddev[xval])
            length_devm[i] = length_errs[i]/np.sqrt(nseeds[xval])
            #length_devm[i] = length_errs[i]

        ax.scatter(x_vals, length_vals, zorder=100, s=100, marker='s', color='k', label=None)
        ax.errorbar(x_vals, length_vals, length_devm, ecolor='k', elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='s')
        if self.opts.xlog: ax.set_xscale("log")
        ModifyXLabel(ax, param)
        ax.set_ylim(0.0, 3.0)

    # Slice of the spindle lengths
    def GraphFinalSpindleLengthSlice(self, ax, param_1, param_2):
        # Set up the same way as before...
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        #print "cs scan: {}, {}".format(param_1, param_2)
        final_length = {}
        final_length_stddev = {}
        nseeds = {}
        # Find the different values that param2 can hold...
        p1_unique = []
        p2_unique = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_unique += [v]
                if p == param_2:
                    p2_unique += [v]

        p1_unique = np.sort(np.unique(p1_unique))
        p2_unique = np.sort(np.unique(p2_unique))
        #print "p1_unique = {}".format(p1_unique)
        #print "p2_unique = {}".format(p2_unique)

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        # Build up a 2D lookup table of the combinations of the paramters that we have access to
        # Reverse the order for ease later, with param2 being the first key
        for p2 in p2_unique:
            if (p2 * uc[param_2][1]) not in final_length:
                final_length[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in final_length_stddev:
                final_length_stddev[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in nseeds:
                nseeds[p2 * uc[param_2][1]] = {}
            for p1 in p1_unique:
                if (p1 * uc[param_1][1]) not in final_length[p2 * uc[param_2][1]]:
                    final_length[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in final_length_stddev[p2 * uc[param_2][1]]:
                    final_length_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in nseeds[p2 * uc[param_2][1]]:
                    nseeds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = 0

        #print "final_length: {}".format(final_length)

        # Loop over simulations again, adding the actual values to the LUT
        for sim in self.sims:
            p1_val = 0
            p2_val = 0
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_val = v
                if p == param_2:
                    p2_val = v
            p1_val = p1_val * uc[param_1][1]
            p2_val = p2_val * uc[param_2][1]
            #print "current value: {}, {}".format(p1_val, p2_val)
            final_length[p2_val][p1_val] += [sim.final_length_mean]
            final_length_stddev[p2_val][p1_val] += [sim.final_length_std]
            nseeds[p2_val][p1_val] += len(sim.seeds)

        #print "final_length: {}".format(final_length)

        # Plot the different values based on p2v, the second parameter
        for p2v, color in zip(np.nditer(p2_unique), colors):
            p2v_real = p2v * uc[param_2][1]
            x_vals = np.zeros(len(p1_unique))
            c_vals = np.zeros(len(p1_unique))
            c_errs = np.zeros(len(p1_unique))
            c_devm = np.zeros(len(p1_unique))

            for i in xrange(len(x_vals)):
                x_vals[i] = p1_unique[i] * uc[param_1][1]
                c_vals[i], c_errs[i] = WeightedMeanError(final_length[p2v_real][x_vals[i]], final_length_stddev[p2v_real][x_vals[i]])
                c_devm[i] = c_errs[i]/np.sqrt(nseeds[p2v_real][x_vals[i]])
                #c_devm[i] = c_errs[i]

            #print "x_vals = {}".format(x_vals)
            #print "c_vals = {}".format(c_vals)
            #print "c_devm = {}".format(c_devm)


            label = r'{0} = {1:3.4f}'.format(param_2,
                                      uc[param_2][2](p2v)*uc[param_2][1])
            ax.scatter(x_vals, c_vals, zorder=100, s=100, color=color, label=label, marker = marker.next())
            ax.errorbar(x_vals, c_vals, c_devm, ecolor=color, elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='none')
        ax.legend(loc='best')
        ModifyXLabel(ax, param_1)
        ax.set_ylim(0.0, 3.0)

        return


    # Graph the number of chromosome seconds in either a scan or a slice
    def GraphChromosomeSeconds(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.style.use(cp_spindle_stl)
        plt.figure(ax.get_figure().number)
        #plt.style.use(ase1_runs_stl)
        plt.style.use(cp_spindle_stl)

        # Determine if this is a parameter scan or slice
        if isinstance(param, str): self.GraphChromosomeSecondsScan(ax, param)
        elif isinstance(param, list):
            self.GraphChromosomeSecondsSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])

        ax.set_ylabel(r'Fraction simultaneous biorientation')

        ## Hopefully do the logx right
        #if self.opts.xlog:
        #    ax.xaxis.set_major_locator(mpl.ticker.LogLocator(base=2))
        #else: ax.xaxis.set_major_locator(mpl.ticker.MaxNLocator())
        #ax.set_xticklabels(ax.get_xticks(), rotation=-30, horizontalalignment='left' )
        #ax.set_ylim(bottom=0)
        #ax.set_ylim(top=ax.get_ylim()[1]*1.1)

        # Create the graph
        plt.draw()
        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_ChromosomeSeconds.pdf'.format(self.run_name, param))

        return file_name, title

    # Graph the number of chromosome seconds for this run, including multiple sims!
    def GraphChromosomeSecondsScan(self, ax, param = ''):
        #param = self.CheckGraphParam(param)
        #plt.figure(ax.get_figure().number)
        #plt.style.use(ase1_runs_stl)
        # We have to create the chromosome seconds for the crit dataframe
        chromosome_seconds = {}
        chromosome_seconds_stddev = {}
        nseeds = {}
        x_vals = []
        # Keep track of the write_data for writing out a CSV of the chromosome sections (really fraction integrated biorientation time) for
        # the simulations
        write_data = {}
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param:
                    if (v * uc[param][1]) not in chromosome_seconds:
                        chromosome_seconds[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in chromosome_seconds_stddev:
                        chromosome_seconds_stddev[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in nseeds:
                        nseeds[v*uc[param][1]] = 0
                    chromosome_seconds[v*uc[param][1]] += [sim.fraction_integrated_biorientation_time_mean]
                    chromosome_seconds_stddev[v*uc[param][1]] += [sim.fraction_integrated_biorientation_time_std]
                    nseeds[v*uc[param][1]] += len(sim.seeds)
                    x_vals += [v * uc[param][1]]

                    # Grab the information about what all went into the calculation for the fraction biorientation time
                    fsb = []
                    for sd in sim.seeds:
                        fsb.append(sd.fitness.fraction_integrated_biorientation_time)
                        #fsb += sd.fitness.fraction_integrated_biorientation_time
                    write_data[v * uc[param][1]] = fsb

        print write_data
        for key,value in write_data.iteritems():
            allvals = ", ".join(repr(e) for e in value)
            print("{}, {}".format(key, allvals))

        x_vals = np.unique(x_vals)
        chromosome_vals = np.zeros(len(x_vals))
        chromosome_errs = np.zeros(len(x_vals))
        chromosome_devm = np.zeros(len(x_vals)) # Standard deviation of the mean
        # For each unique entry in the list, do a weighted average for the error and mean chromosome seconds
        for i in xrange(len(x_vals)):
            xval = x_vals[i]
            chromosome_vals[i], chromosome_errs[i] = WeightedMeanError(chromosome_seconds[xval], chromosome_seconds_stddev[xval])
            chromosome_devm[i] = chromosome_errs[i]/np.sqrt(nseeds[xval])
            #chromosome_devm[i] = chromosome_errs[i]

        ax.scatter(x_vals, chromosome_vals, zorder=100, s=50, marker='s', color='k', label=None, facecolors='none')
        ax.errorbar(x_vals, chromosome_vals, chromosome_devm, ecolor='k', elinewidth=2, capsize=5, capthick=1, zorder=0, fmt='none', marker='s')
        if self.opts.xlog: ax.set_xscale("log")
        ModifyXLabel(ax, param)
        print "Current label is {}".format(param)
        if param == 'abk':
            print "Specializing axis for {}".format(param)
            ax.set_xlim(-15.0, 105.0)
            ax.set_ylim(-0.1, 1.0)
        else:
            ax.set_ylim(0.0, 1.0)

    # Chromosome seconds slice functionality
    def GraphChromosomeSecondsSlice(self, ax, param_1, param_2):
        # Set up the same way as before...
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        #print "cs scan: {}, {}".format(param_1, param_2)
        chromosome_seconds = {}
        chromosome_seconds_stddev = {}
        nseeds = {}
        # Find the different values that param2 can hold...
        p1_unique = []
        p2_unique = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_unique += [v]
                if p == param_2:
                    p2_unique += [v]

        p1_unique = np.sort(np.unique(p1_unique))
        p2_unique = np.sort(np.unique(p2_unique))
        #print "p1_unique = {}".format(p1_unique)
        #print "p2_unique = {}".format(p2_unique)

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        # Build up a 2D lookup table of the combinations of the paramters that we have access to
        # Reverse the order for ease later, with param2 being the first key
        for p2 in p2_unique:
            if (p2 * uc[param_2][1]) not in chromosome_seconds:
                chromosome_seconds[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in chromosome_seconds_stddev:
                chromosome_seconds_stddev[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in nseeds:
                nseeds[p2 * uc[param_2][1]] = {}
            for p1 in p1_unique:
                if (p1 * uc[param_1][1]) not in chromosome_seconds[p2 * uc[param_2][1]]:
                    chromosome_seconds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in chromosome_seconds_stddev[p2 * uc[param_2][1]]:
                    chromosome_seconds_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in nseeds[p2 * uc[param_2][1]]:
                    nseeds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = 0

        #print "chromosome_seconds: {}".format(chromosome_seconds)

        # Loop over simulations again, adding the actual values to the LUT
        for sim in self.sims:
            p1_val = 0
            p2_val = 0
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_val = v
                if p == param_2:
                    p2_val = v
            p1_val = p1_val * uc[param_1][1]
            p2_val = p2_val * uc[param_2][1]
            #print "current value: {}, {}".format(p1_val, p2_val)
            chromosome_seconds[p2_val][p1_val] += [sim.fraction_integrated_biorientation_time_mean]
            chromosome_seconds_stddev[p2_val][p1_val] += [sim.fraction_integrated_biorientation_time_std]
            nseeds[p2_val][p1_val] += len(sim.seeds)

        #print "chromosome_seconds: {}".format(chromosome_seconds)
        # Write out the chromosome seconds from the map into a format that we can then read
        with open("chromosomeseconds.csv","w") as mfile:
            for key,value in chromosome_seconds.iteritems():
                for key2,value2 in value.iteritems():
                    mfile.write('{}, {}, {}\n'.format(key, key2, value2))

        # Plot the different values based on p2v, the second parameter
        for p2v, color in zip(np.nditer(p2_unique), colors):
            p2v_real = p2v * uc[param_2][1]
            x_vals = np.zeros(len(p1_unique))
            c_vals = np.zeros(len(p1_unique))
            c_errs = np.zeros(len(p1_unique))
            c_devm = np.zeros(len(p1_unique))

            for i in xrange(len(x_vals)):
                x_vals[i] = p1_unique[i] * uc[param_1][1]
                c_vals[i], c_errs[i] = WeightedMeanError(chromosome_seconds[p2v_real][x_vals[i]], chromosome_seconds_stddev[p2v_real][x_vals[i]])
                #c_devm[i] = c_errs[i]/np.sqrt(nseeds[p2v_real][x_vals[i]])
                c_devm[i] = c_errs[i]

            #print "x_vals = {}".format(x_vals)
            #print "c_vals = {}".format(c_vals)
            #print "c_devm = {}".format(c_devm)


            label = r'{0} = {1:3.4f}'.format(param_2,
                                      uc[param_2][2](p2v)*uc[param_2][1])
            ax.scatter(x_vals, c_vals, zorder=100, s=100, color=color, label=label, marker = marker.next())
            ax.errorbar(x_vals, c_vals, c_devm, ecolor=color, elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='none')
        ax.legend(loc='best')
        ModifyXLabel(ax, param_1)
        ax.set_ylim(0.0, 1.1)

        return

    # Graph the late time IPF of the run
    def GraphFinalIPF(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        # Determine if this is a parameter scan or slice
        if isinstance(param, str): self.GraphIPFScan(ax, param)
        elif isinstance(param, list):
            self.GraphIPFSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])

        ax.set_ylabel(r'Avg. late interpolar fraction')

        # Create the graph
        plt.draw()
        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_FinalIPF.pdf'.format(self.run_name, param))

        return file_name, title

    # Graph the final spindle length for a simple scan
    def GraphIPFScan(self, ax, param):
        final_ipf = {}
        final_ipf_stddev= {}
        nseeds = {}
        x_vals = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param:
                    if (v * uc[param][1]) not in final_ipf:
                        final_ipf[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in final_ipf_stddev:
                        final_ipf_stddev[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in nseeds:
                        nseeds[v*uc[param][1]] = 0
                    final_ipf[v*uc[param][1]] += [sim.final_ipf_mean]
                    final_ipf_stddev[v*uc[param][1]] += [sim.final_ipf_std]
                    nseeds[v*uc[param][1]] += len(sim.seeds)
                    x_vals += [v * uc[param][1]]

        x_vals = np.unique(x_vals)
        ipf_vals = np.zeros(len(x_vals))
        ipf_errs = np.zeros(len(x_vals))
        ipf_devm = np.zeros(len(x_vals)) # Standard deviation of the mean
        # For each unique entry in the list, do a weighted average for the error and mean chromosome seconds
        for i in xrange(len(x_vals)):
            xval = x_vals[i]
            ipf_vals[i], ipf_errs[i] = WeightedMeanError(final_ipf[xval], final_ipf_stddev[xval])
            ipf_devm[i] = ipf_errs[i]/np.sqrt(nseeds[xval])
            #ipf_devm[i] = ipf_errs[i]

        ax.scatter(x_vals, ipf_vals, zorder=100, s=100, marker='s', color='k', label=None)
        ax.errorbar(x_vals, ipf_vals, ipf_devm, ecolor='k', elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='s')
        if self.opts.xlog: ax.set_xscale("log")
        ModifyXLabel(ax, param)
        ax.set_ylim(0.0, 1.1)

    # Slice of the late spindle IPFs
    def GraphIPFSlice(self, ax, param_1, param_2):
        # Set up the same way as before...
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        #print "cs scan: {}, {}".format(param_1, param_2)
        final_ipf = {}
        final_ipf_stddev = {}
        nseeds = {}
        # Find the different values that param2 can hold...
        p1_unique = []
        p2_unique = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_unique += [v]
                if p == param_2:
                    p2_unique += [v]

        p1_unique = np.sort(np.unique(p1_unique))
        p2_unique = np.sort(np.unique(p2_unique))
        #print "p1_unique = {}".format(p1_unique)
        #print "p2_unique = {}".format(p2_unique)

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        # Build up a 2D lookup table of the combinations of the paramters that we have access to
        # Reverse the order for ease later, with param2 being the first key
        for p2 in p2_unique:
            if (p2 * uc[param_2][1]) not in final_ipf:
                final_ipf[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in final_ipf_stddev:
                final_ipf_stddev[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in nseeds:
                nseeds[p2 * uc[param_2][1]] = {}
            for p1 in p1_unique:
                if (p1 * uc[param_1][1]) not in final_ipf[p2 * uc[param_2][1]]:
                    final_ipf[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in final_ipf_stddev[p2 * uc[param_2][1]]:
                    final_ipf_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in nseeds[p2 * uc[param_2][1]]:
                    nseeds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = 0

        #print "final_ipf: {}".format(final_ipf)

        # Loop over simulations again, adding the actual values to the LUT
        for sim in self.sims:
            p1_val = 0
            p2_val = 0
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_val = v
                if p == param_2:
                    p2_val = v
            p1_val = p1_val * uc[param_1][1]
            p2_val = p2_val * uc[param_2][1]
            #print "current value: {}, {}".format(p1_val, p2_val)
            final_ipf[p2_val][p1_val] += [sim.final_ipf_mean]
            final_ipf_stddev[p2_val][p1_val] += [sim.final_ipf_std]
            nseeds[p2_val][p1_val] += len(sim.seeds)

        #print "final_ipf: {}".format(final_ipf)

        # Plot the different values based on p2v, the second parameter
        for p2v, color in zip(np.nditer(p2_unique), colors):
            p2v_real = p2v * uc[param_2][1]
            x_vals = np.zeros(len(p1_unique))
            c_vals = np.zeros(len(p1_unique))
            c_errs = np.zeros(len(p1_unique))
            c_devm = np.zeros(len(p1_unique))

            for i in xrange(len(x_vals)):
                x_vals[i] = p1_unique[i] * uc[param_1][1]
                c_vals[i], c_errs[i] = WeightedMeanError(final_ipf[p2v_real][x_vals[i]], final_ipf_stddev[p2v_real][x_vals[i]])
                c_devm[i] = c_errs[i]/np.sqrt(nseeds[p2v_real][x_vals[i]])

            #print "x_vals = {}".format(x_vals)
            #print "c_vals = {}".format(c_vals)
            #print "c_devm = {}".format(c_devm)


            label = r'{0} = {1:3.4f}'.format(param_2,
                                      uc[param_2][2](p2v)*uc[param_2][1])
            ax.scatter(x_vals, c_vals, zorder=100, s=100, color=color, label=label, marker = marker.next())
            ax.errorbar(x_vals, c_vals, c_devm, ecolor=color, elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='none')
        ax.legend(loc='best')
        ModifyXLabel(ax, param_1)
        ax.set_ylim(0.0, 3.0)

        return

    # Graph the final spindle length, regardless of success
    def GraphFinalOccupancy(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        # Determine if this is a parameter scan or slice
        if isinstance(param, str): self.GraphFinalOccupancyScan(ax, param)
        elif isinstance(param, list):
            self.GraphFinalOccupancySlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])

        ax.set_ylabel(r'Fraction avg. late KC-MT occupancy')

        # Create the graph
        plt.draw()
        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_FinalOccupancy.pdf'.format(self.run_name, param))

        return file_name, title

    # Graph the final spindle length for a simple scan
    def GraphFinalOccupancyScan(self, ax, param):
        final_occupancy = {}
        final_occupancy_stddev= {}
        nseeds = {}
        x_vals = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param:
                    if (v * uc[param][1]) not in final_occupancy:
                        final_occupancy[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in final_occupancy_stddev:
                        final_occupancy_stddev[v*uc[param][1]] = []
                    if (v * uc[param][1]) not in nseeds:
                        nseeds[v*uc[param][1]] = 0
                    final_occupancy[v*uc[param][1]] += [sim.final_occupancy_mean/sim.total_occupancy_max]
                    final_occupancy_stddev[v*uc[param][1]] += [sim.final_occupancy_std/sim.total_occupancy_max]
                    nseeds[v*uc[param][1]] += len(sim.seeds)
                    x_vals += [v * uc[param][1]]

        x_vals = np.unique(x_vals)
        length_vals = np.zeros(len(x_vals))
        length_errs = np.zeros(len(x_vals))
        length_devm = np.zeros(len(x_vals)) # Standard deviation of the mean
        # For each unique entry in the list, do a weighted average for the error and mean chromosome seconds
        for i in xrange(len(x_vals)):
            xval = x_vals[i]
            length_vals[i], length_errs[i] = WeightedMeanError(final_occupancy[xval], final_occupancy_stddev[xval])
            length_devm[i] = length_errs[i]/np.sqrt(nseeds[xval])
            #length_devm[i] = length_errs[i]

        ax.scatter(x_vals, length_vals, zorder=100, s=100, marker='s', color='k', label=None)
        ax.errorbar(x_vals, length_vals, length_devm, ecolor='k', elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='s')
        if self.opts.xlog: ax.set_xscale("log")
        ModifyXLabel(ax, param)
        ax.set_ylim(0.0, 1.1)

    # Slice of the spindle lengths
    def GraphFinalOccupancySlice(self, ax, param_1, param_2):
        # Set up the same way as before...
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        #print "cs scan: {}, {}".format(param_1, param_2)
        final_occupancy = {}
        final_occupancy_stddev = {}
        nseeds = {}
        # Find the different values that param2 can hold...
        p1_unique = []
        p2_unique = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_unique += [v]
                if p == param_2:
                    p2_unique += [v]

        p1_unique = np.sort(np.unique(p1_unique))
        p2_unique = np.sort(np.unique(p2_unique))
        #print "p1_unique = {}".format(p1_unique)
        #print "p2_unique = {}".format(p2_unique)

        colors = mpl.cm.rainbow(np.linspace(0,1,p2_unique.size))

        # Build up a 2D lookup table of the combinations of the paramters that we have access to
        # Reverse the order for ease later, with param2 being the first key
        for p2 in p2_unique:
            if (p2 * uc[param_2][1]) not in final_occupancy:
                final_occupancy[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in final_occupancy_stddev:
                final_occupancy_stddev[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in nseeds:
                nseeds[p2 * uc[param_2][1]] = {}
            for p1 in p1_unique:
                if (p1 * uc[param_1][1]) not in final_occupancy[p2 * uc[param_2][1]]:
                    final_occupancy[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in final_occupancy_stddev[p2 * uc[param_2][1]]:
                    final_occupancy_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in nseeds[p2 * uc[param_2][1]]:
                    nseeds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = 0

        #print "final_occupancy: {}".format(final_occupancy)

        # Loop over simulations again, adding the actual values to the LUT
        for sim in self.sims:
            p1_val = 0
            p2_val = 0
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_val = v
                if p == param_2:
                    p2_val = v
            p1_val = p1_val * uc[param_1][1]
            p2_val = p2_val * uc[param_2][1]
            #print "current value: {}, {}".format(p1_val, p2_val)
            final_occupancy[p2_val][p1_val] += [sim.final_occupancy_mean]
            final_occupancy_stddev[p2_val][p1_val] += [sim.final_occupancy_std]
            nseeds[p2_val][p1_val] += len(sim.seeds)

        #print "final_occupancy: {}".format(final_occupancy)

        # Plot the different values based on p2v, the second parameter
        for p2v, color in zip(np.nditer(p2_unique), colors):
            p2v_real = p2v * uc[param_2][1]
            x_vals = np.zeros(len(p1_unique))
            c_vals = np.zeros(len(p1_unique))
            c_errs = np.zeros(len(p1_unique))
            c_devm = np.zeros(len(p1_unique))

            for i in xrange(len(x_vals)):
                x_vals[i] = p1_unique[i] * uc[param_1][1]
                c_vals[i], c_errs[i] = WeightedMeanError(final_occupancy[p2v_real][x_vals[i]], final_occupancy_stddev[p2v_real][x_vals[i]])
                #c_devm[i] = c_errs[i]/np.sqrt(nseeds[p2v_real][x_vals[i]])
                c_devm[i] = c_errs[i]

            #print "x_vals = {}".format(x_vals)
            #print "c_vals = {}".format(c_vals)
            #print "c_devm = {}".format(c_devm)


            label = r'{0} = {1:3.4f}'.format(param_2,
                                      uc[param_2][2](p2v)*uc[param_2][1])
            ax.scatter(x_vals, c_vals, zorder=100, s=100, color=color, label=label, marker = marker.next())
            ax.errorbar(x_vals, c_vals, c_devm, ecolor=color, elinewidth=2, capsize=7, capthick=1, zorder=0, fmt='none', marker='none')
        ax.legend(loc='best')
        ModifyXLabel(ax, param_1)
        ax.set_ylim(0.0, 1.1)

        return

    # Graph the final forces from each pole (just do pole0)
    def GraphFinalForces(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        # Determine if this is a parameter scan or slice
        if isinstance(param, str): self.GraphFinalForcesScan(ax, param)
        elif isinstance(param, list):
            self.GraphFinalForcesSlice(ax, param[0], param[1])
            param = "{}_{}".format(param[0], param[1])

        ax.set_ylabel(r'Avg. late forces ($\mu$m)')

        # Create the graph
        plt.draw()
        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_FinalSpindleLength.pdf'.format(self.run_name, param))

        return file_name, title

    # A special box plot for the motor ratios
    def GraphSpecialLegos(self, ax, param = ''):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        plt.figure(ax.get_figure().number)
        plt.style.use(ase1_runs_stl)

        title = r'{}'.format(self.run_name.replace('_',' '))
        file_name = os.path.join(self.opts.datadir,
                '{}_{}_SpecialLego.pdf'.format(self.run_name, param))

        # Make sure if we are going to plot this that we have multiple params
        if not isinstance(param, list):
            return file_name, title

        param_1 = param[0]
        param_2 = param[1]

        # Now combine everything form the spindle length and chromosome seconds
        # DO spindle length first
        # Set up the same way as before...
        marker = itertools.cycle(('s', 'o', 'v', '^', 'H', 'D', '8', 'p', '<','>','*','h','d'))
        #print "cs scan: {}, {}".format(param_1, param_2)
        final_length = {}
        final_length_stddev = {}
        chromosome_seconds = {}
        chromosome_seconds_stddev = {}
        nseeds = {}
        # Find the different values that param2 can hold...
        p1_unique = []
        p2_unique = []
        for sim in self.sims:
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_unique += [v]
                if p == param_2:
                    p2_unique += [v]

        p1_unique = np.sort(np.unique(p1_unique))
        p2_unique = np.sort(np.unique(p2_unique))
        #print "p1_unique = {}".format(p1_unique)
        #print "p2_unique = {}".format(p2_unique)

        # Build up a 2D lookup table of the combinations of the paramters that we have access to
        # Reverse the order for ease later, with param2 being the first key
        for p2 in p2_unique:
            if (p2 * uc[param_2][1]) not in final_length:
                final_length[p2 * uc[param_2][1]] = {}
                chromosome_seconds[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in final_length_stddev:
                final_length_stddev[p2 * uc[param_2][1]] = {}
                chromosome_seconds_stddev[p2 * uc[param_2][1]] = {}
            if (p2 * uc[param_2][1]) not in nseeds:
                nseeds[p2 * uc[param_2][1]] = {}
            for p1 in p1_unique:
                if (p1 * uc[param_1][1]) not in final_length[p2 * uc[param_2][1]]:
                    final_length[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                    chromosome_seconds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in final_length_stddev[p2 * uc[param_2][1]]:
                    final_length_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                    chromosome_seconds_stddev[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = []
                if (p1 * uc[param_1][1]) not in nseeds[p2 * uc[param_2][1]]:
                    nseeds[p2 * uc[param_2][1]][p1 * uc[param_1][1]] = 0

        #print "final_length: {}".format(final_length)

        # Loop over simulations again, adding the actual values to the LUT
        for sim in self.sims:
            p1_val = 0
            p2_val = 0
            for p,v in sim.params.iteritems():
                if p == param_1:
                    p1_val = v
                if p == param_2:
                    p2_val = v
            p1_val = p1_val * uc[param_1][1]
            p2_val = p2_val * uc[param_2][1]
            #print "current value: {}, {}".format(p1_val, p2_val)
            final_length[p2_val][p1_val] += [sim.final_length_mean]
            final_length_stddev[p2_val][p1_val] += [sim.final_length_std]
            chromosome_seconds[p2_val][p1_val] += [sim.fraction_integrated_biorientation_time_mean]
            chromosome_seconds_stddev[p2_val][p1_val] += [sim.fraction_integrated_biorientation_time_std]
            nseeds[p2_val][p1_val] += len(sim.seeds)

        #print "final_length: {}".format(final_length)
        # We need things in an X, Y, Z array type methods
        #_x = np.arange(len(p1_unique))
        #_y = np.arange(len(p2_unique))
        _x = p1_unique
        _y = p2_unique
        _xx, _yy = np.meshgrid(_x, _y)
        x, y = _xx.ravel(), _yy.ravel()
        #top = x + y
        # Load the top with the information on the spindle length
        top = np.zeros(x.shape[0])
        cs = np.zeros(x.shape[0])
        # Loop over X, then Y, do this as an indent
        for idx in xrange(len(p1_unique)):
            for jdx in xrange(len(p2_unique)):
                p1v = p1_unique[idx]
                p2v = p2_unique[jdx]
                p1v_real = p1v * uc[param_1][1]
                p2v_real = p2v * uc[param_2][1]
                # We have the index, and the damn way of getting the final length
                c_val, c_err = WeightedMeanError(final_length[p2v][p1v], final_length_stddev[p2v][p1v])
                if np.isnan(c_val):
                    c_val = 0.0
                    c_err = 0.0
                top[idx + len(p1_unique)*jdx] = c_val
                c_val, c_err = WeightedMeanError(chromosome_seconds[p2v][p1v], chromosome_seconds_stddev[p2v][p1v])
                if np.isnan(c_val):
                    c_val = 0.0
                    c_err = 0.0
                cs[idx + len(p1_unique)*jdx] = c_val

        bottom = np.zeros_like(top)
        #width = depth = 100
        width = p1_unique[1] - p1_unique[0]
        depth = p2_unique[1] - p2_unique[0]

        # Try to do the colormap conversion for the color
        norm = mpl.colors.Normalize(vmin=0, vmax=1)
        cmap = mpl.cm.viridis
        m = mpl.cm.ScalarMappable(norm=norm, cmap=cmap)
        print m.to_rgba(cs)

        print x
        print y
        print top
        print cs

        #ax.bar3d(x, y, bottom, width, depth, top, color=cs, shade=True)
        ax.bar3d(x, y, bottom, width, depth, top, color=m.to_rgba(cs), shade=True)

        return file_name, title

    def MakeDistributionGraphs(self, **kwargs):
        self.MakeXlinkSPBDistributionGraph(**kwargs)
        self.MakeXlinkSpindleDistanceGraph(**kwargs)
        self.MakeMTDistributionGraph(**kwargs)
        self.MakeKCDistributionsGraph(**kwargs)
        return

    def MakeXlinkSPBDistributionGraph( self, figsize=(12,8), param = '',
                                        xstate=r'integrated', **kwargs):
        param = self.CheckGraphParam(param)
        fig, axarr = plt.subplots(2,2, figsize=figsize)
        try: graph_run_xlink_distr_succ_compare(self, axarr, opts=self.opts, xstate=xstate, xlabel=True)
        except:
            print " *   No xlink data was found or was corrupted. Run could not make xlink_distr_succ_compare graph.   *"
            return
        file_name = os.path.join(self.opts.datadir,
                                 '{}_{}_xlink_distr.pdf'.format(self.run_name, param))
        fig.savefig(file_name, dpi=fig.dpi)
        return

    def MakeXlinkSpindleDistanceGraph( self, figsize=(12,10), param = '', **kwargs ):
        param = self.CheckGraphParam(param)
        from stylelib.ase1_styles import ase1_xl_spindle_stl
        with plt.style.context(ase1_xl_spindle_stl):
            fig, axarr = plt.subplots(3,1, figsize=figsize)
            # graph_run_spindle_xlink_distance_error_all(self, axarr, **kwargs)
            try: graph_run_spindle_xlink_distance_error_all(self, axarr, **kwargs)
            except:
                print " *   No xlink spindle data was found or was corrupted. Run could not make xlink_spindle_distance graph.   *"
                return
            file_name = os.path.join(self.opts.datadir,
                                     '{}_{}_spindle_xlink_dist.pdf'.format(self.run_name, param))
        fig.savefig(file_name, dpi=fig.dpi)
        return

    def MakeMTDistributionGraph( self, figsize=(12,10), param = '', **kwargs):
        param = self.CheckGraphParam(param)
        fig, axarr = plt.subplots(4,1, figsize=figsize)
        try: graph_run_mt_length_distr_error(self, axarr)
        except:
            print " *   No mt data was found or was corrupted. Run could not make mt_length_distr_error graph.   *"
            return
        file_name = os.path.join(self.opts.datadir,
                                 '{}_{}_mt_distr.pdf'.format(self.run_name, param))

    def MakeKCDistributionsGraph(self, figsize=(10,8), param = '', xstate=r'integrated', **kwargs):
        if not self.sims[0].analyze_chromosomes:
            return
        param = self.CheckGraphParam(param)
        fig, axarr = plt.subplots(2,2, figsize=(10,8))
        graph_run_kc_spindle1d(self, axarr, opts=self.opts, xstate=r'integrated', xlabel=True)
        file_name = os.path.join(self.opts.datadir,
                                 '{}_{}_kc_spindle1d_distr.pdf'.format(self.run_name, param))
        fig.savefig(file_name, dpi=fig.dpi)
        return

    def ReadDataFiles(self):
        raise NotImplementedError("ReadDataFiles")

    def CheckGraphParam(self, param=''):
        if not param and self.opts.params:
            # Scan graph for single param
            if len(self.opts.params) == 1:
                param = self.opts.params[0]
                if param not in self.p_names:
                    ParamError(param)

            # Slice run for two params
            elif len(self.opts.params) == 2:
                param_1, param_2 = self.opts.params
                if param_1 not in self.p_names: ParamError(param_1)
                if param_2 not in self.p_names: ParamError(param_2)

            # Too many parameters were given to opts
            else:
                print " *** Too many params given in argument. ***"
                sys.exit(1)

        elif not self.opts.params:
            if len(self.p_names) == 1: param = self.p_names[0]
            elif len(self.p_names) == 2: param_1, param_2 = self.p_names
            else:
                print " *** Too many parameters in dataframe. ***"
                sys.exit(1)

        else: ParamError()

        if 'param_1' in locals():
            return [param_1, param_2]
        elif 'param' in locals():
            self.OrderSimList(param)
            return param

        else:
            raise ValueError('Params are not being assigned properly')

    def OrderSimList(self, param=''):
        """ Method to order self.sims list by param value """
        if not param: param = self.p_names[0]
        self.sims.sort(key=lambda sim: sim.params[param])
        return param

    def MakeSuccFailMovieDict(self):
        """ Method that returns a dictionary of seed paths. 
            Dictionary keys are parameters first and then
            whether it was a success "succ" or failure "fail".
            Each entry contains a list of seed paths at the end.
        """
        mov_dict = {}
        # TODO Exception if sim_crit_df is empty
        # print self.sim_crit_df
        df = self.sim_crit_df
        for sim in self.sims:
            # Set up movie dictionary to hold seed paths
            mov_dict[sim.name] = { 'sim': sim,
                                   'succ': [],
                                   'fail': []
                                 }
            for sd in sim.seeds:
                state = df.loc[(df['sd_num'] == sd.seed_num)]
                # Iteratively check to find seed criteria value with 
                # right param values.
                for p, v in sim.params.iteritems():
                    state = state.loc[(state[p] == v)]
                # Add path to correct dictionary entry
                if state.empty: continue
                elif state['succ'].item(): mov_dict[sim.name]['succ'] += [sd.path]
                else: mov_dict[sim.name]['fail'] += [sd.path]
        return mov_dict

    ### Fitness information
    def Fitness(self, wt_mat, lstream_mat):
        for sim in self.sims:
            sim.Fitness(wt_mat, lstream_mat)

        # Now print the sim and the fitness information (Hack for now)
        if self.sims[0].analyze_chromosomes:
            print "- Fitness Information Chromosomes -"
            for sim in self.sims:
                print "-- Sim {} -- ".format(sim.title)
                print "   Total fitness: {}".format(sim.final_fitness)
                print "   Chromosome Seconds: {}".format(sim.integrated_biorientation_time_mean)
                print "   Chromosome Seconds Std. Dev: {}".format(sim.integrated_biorientation_time_std)
                print "   Chromosome Seconds Fraction: {}".format(sim.fraction_integrated_biorientation_time_mean)
                print "   Chromosome Seconds Fraction Std. Dev: {}".format(sim.fraction_integrated_biorientation_time_std)
                print "   Final Spindle Length: {}".format(sim.final_length_mean)
                print "   Final Spindle Length Std. Dev: {}".format(sim.final_length_std)
                print "   Final IPF: {}".format(sim.final_ipf_mean)
                print "   Final IPF Std. Dev: {}".format(sim.final_ipf_std)
            # Maybe print in a form that is more CSV friendly
            with open(os.path.join(self.opts.datadir, 'fitness.csv'), 'w') as csvfile:
                csvfile.write('{},{},{},{},{},{},{},{},'.format('final_fitness',
                    'emfitness_log',
                    'avg_correlation_combined_mean',
                    'fbiorient_mean',
                    'fraction_integrated_biorientation_time_mean',
                    'avg_length_correlation_mean',
                    'avg_kc_distance_correlation_mean',
                    'avg_kc_sep_correlation_mean'))
                for param,value in self.sims[0].params.iteritems():
                    csvfile.write('{},'.format(param))
                csvfile.write('\n')
                for sim in self.sims:
                    csvfile.write('{},{},{},{},{},{},{},{},'.format(sim.final_fitness,
                        sim.emfitness_log,
                        sim.avg_correlation_combined_mean,
                        sim.fbiorient_mean,
                        sim.fraction_integrated_biorientation_time_mean,
                        sim.avg_length_correlation_mean,
                        sim.avg_kc_distance_correlation_mean,
                        sim.avg_kc_sep_correlation_mean))
                    for param,value in sim.params.iteritems():
                        csvfile.write('{},'.format(value))
                    csvfile.write('\n')


def make_dict_recurse(data_dict, p_names, p_val_dict, index=0):
    #XXX this is where I left off
    if index == len(p_names): data_dict = []
    else:
        data_dict[p_val_dict[p_names[index]]] = {}
        make_dict_recurse( data_dict[p_val_dict[p_names[index]]], 
                           p_names, p_val_dict, index+1)

    return 

##########################################
if __name__ == "__main__":
    print "Not implemented yet"



# TODO Full bar graph run
        # fig = plt.figure(figsize=(20,25))

        # # How to separate subplots into grids
        # gs1 = gridspec.GridSpec(5,2, width_ratios=[20,1])

        # # Get rid of excess white spaces around image
        # gs1.update(top=.99,bottom=.05, left=.08, right=.92, hspace=0.01, wspace=.01)

        # # The color bar axis
        # cb1_ax = plt.subplot(gs1[1:4,1])

        # # Make all graph axis objects and where they should be placed
        # axarr = [None]*5
        # axarr[4] = fig.add_subplot(gs1[4,0])
        # axarr[0] = fig.add_subplot(gs1[0,0], sharex=axarr[4])
        # axarr[1] = fig.add_subplot(gs1[1,0], sharex=axarr[4])
        # axarr[2] = fig.add_subplot(gs1[2,0], sharex=axarr[4])
        # axarr[3] = fig.add_subplot(gs1[3,0], sharex=axarr[4])

        # # First graph returns the largest number of seeds in a bar
        # # self.v_max = self.GraphSuccessFracBar(axarr[0], param, cbflag=False, xlflag=False)
        # self.v_max = self.GraphSuccessPercentBar(axarr[0], param, cbflag=False, xlflag=False)
        # self.GraphAvgSSBar(axarr[1], param, cbflag=False, xlflag=False)
        # self.GraphAvgStartTimeBar(axarr[2], param, cbflag=False, xlflag=False)
        # self.GraphAvgMTLengthBar(axarr[3], param, cbflag=False, xlflag=False)
        # self.GraphAvgIFBar(axarr[4], param, cbflag=False)
        # self.ModifyXLabel(axarr[4], param)

        # # Initialize colorbar
        # cmap=mpl.cm.rainbow
        # v_min=0
        # label = r'Number of Simulations'

        # cb1 = mpl.colorbar.ColorbarBase(cb1_ax, cmap=cmap,
        #         norm = mpl.colors.Normalize(vmin=v_min, vmax=self.v_max))
        # cb1.set_label(label)

        # # Format font of graph
        # for ax in axarr:
        #     ax.set_yticklabels(ax.get_yticks(), font)

        # axarr[4].set_xticklabels(axarr[4].get_xticks(), font)
        # if param == 'N': #Make sure that only integers are shown for N
        #     axarr[4].xaxis.set_major_formatter(mpl.ticker.FormatStrFormatter('%d'))

        # # plt.tight_layout()
        # # plt.suptitle('{} Param Statistics {} '.format(self.run_name.replace('_',' '), param))
        # plt.draw()
        # plt.savefig(self.seed_dat_file.replace('.dat', '') +
        #         '_{}_all.pdf'.format(param))
        # fig.clf()
        # plt.close()
        # gc.collect()


