#!/usr/bin/env python
# In case of poor (Sh**y) commenting contact adam.lamson@colorado.edu
# Basic
import sys, os, pdb
import gc
## Analysis
# import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from scipy.stats import ks_2samp
# from math import *
sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Lib'))
from sim_base import SimBase
from spindle_seed import SpindleSeed
from bin_functions import *
from base_funcs import *
from seed_graph_funcs import *
from sim_graph_funcs import *
from stylelib.ase1_styles import ase1_sims_stl

# CJE christopher.edelmaier@colorado.edu
from sim_graph_funcs import graph_sim_kc_attachment_types
from seed_graph_funcs import graph_kc_attachment_type

'''
Name: spindle_sim.py
Description:
Input:
Output:
'''

#Class definition
class SpindleSim(SimBase):
    def __init__(self, path, opts, seedType=SpindleSeed ):
        SimBase.__init__(self, path, opts, seedType=seedType)

        # Flags for success criteria
        self.calc_flag = False
        self.succ_calc_flag = False
        self.succ_num = 0

        self.graph_time_funcs = [ graph_spb_sep,
                                  graph_interpolar_fraction,
                                  graph_interpolar_length_fraction,
                                  graph_avg_mt_splay
                                 # graph_avg_mt_length
                                ]

        self.graph_distr_funcs = [ (graph_sim_spindle_xlink_distance_final, 3, 2,(12,8)),
                                   (graph_sim_start_time_hist, 1, 1, (6,4)),
                                   (graph_sim_kc_spindle1d, 2, 2, (10,8)),
                                   (graph_sim_kc_interkc, 1, 1, (6, 4)),
                                   (graph_sim_anaphase_onset, 1, 1, (6, 4)),
                                    
                # TODO Add in the rest here
                                   # graph_sim_mt_length_distr_succ_compare,
                                   # graph_sim_mt_length_distr_error,
                                   # (graph_sim_spb_xlink_distr_final,
                                 ]


        # self.graph_funcs_post = [ #seedType.GraphSPBXlinkDistance,
        #                           seedType.GraphXlinkSpindle1D
        #                         ]
                                  # seedType.GraphKCSpindle1D ]

        self.sim_crit_info = {}
        self.sim_datadir = None

    def Analyze(self, raw_dat_list=[]):
        for sd in self.seeds: sd.AnalyzeAll()
        print "- Sim {} analyzed -".format(self.name)
        self.CalcSimSuccess()
        self.calc_flag = True

    def GraphSimulation(self):
        if not self.opts.datadir: self.opts.datadir = create_datadir(self.opts.workdir)
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print "Don't know successful fraction. Criteria analysis might not have been run on sim {}".format(self.name)
            title = self.title
        if self.opts.workdir != self.sim_path:
            self.sim_datadir = create_datadir( self.opts.datadir, 
                                 datadir_name = "{}_data".format(self.name) )
        else:
            self.sim_datadir = self.opts.datadir
        with plt.style.context(ase1_sims_stl):
            self.GraphTimeData(title)
        self.GraphDistrData(title)

    def GraphTimeData(self, title):
        # font ={#'weight' : 'bold',
        #         'size' : 20}
        # mpl.rc('font', **font)
        # mpl.rc('text', usetex=True)
        fig, axarr = plt.subplots(4, 2, figsize=(25,16))
        colors = mpl.cm.rainbow(np.linspace(0,1,len(self.seeds)))

        min_size = 0
        for sd in self.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        time = self.seeds[-1].time[:min_size]

        for graph, axr in zip(self.graph_time_funcs, axarr):
            avg = np.zeros(min_size)
            num_seeds = 0
            for sd, col in zip(self.seeds, colors):
                yarr = graph(sd, axr[0], color=col, xlabel=False)
                if not self.opts.write or sd.succ_info_dict['succ'] == 1:
                    num_seeds += 1
                    avg = np.add(avg, yarr[:min_size])

            avg /= float(num_seeds)
            axr[1].plot(time, avg)

        axarr[1, 0].legend(loc='center left', bbox_to_anchor=(2.2,-.19))

        axarr[-1,1].set_xlabel(r'Time (min)')
        axarr[-1,0].set_xlabel(r'Time (min)')

        plt.suptitle(title)
        fig.tight_layout()
        plt.subplots_adjust(hspace = .38, right=.85, top=.87)
        plt.savefig("{}_{}.png".format(os.path.join(self.sim_datadir, 'timedata'), self.name))
        fig.clf()
        plt.close()
        gc.collect()
        mpl.rcdefaults()

        # Determine if we have chromosomes enabled or not...
        self.analyze_chromosomes = self.seeds[0].PostAnalysis.analyze_chromosomes

        if self.analyze_chromosomes:
            self.GraphChromosomes(colors)
            self.GraphCombinedAttachments(colors)

    def GraphChromosomes(self, colors):
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print 'Criteria analysis has not been run on sim {}'.format(self.name)
            title = self.title

        font ={#'weight' : 'bold',
                'size' : 20}
        mpl.rc('font', **font)
        # mpl.rc('text', usetex=True)

        # Before we run the simulation graph code, go analyze the forces from different seeds, etc
        [xlink_force, af_force, interkcforce] = self.AnalyzeSpindleForces()

        self.interkcforce = interkcforce
        fig, axarr = plt.subplots(6, 2, figsize=(30,20))
        graph_sim_kc_attachment_types(self, axarr, opts = self.opts, colors = colors)


        plt.subplots_adjust(hspace = .38)
        plt.suptitle(title)
        fig.tight_layout()
        plt.savefig("{}_{}.png".format(os.path.join(self.sim_datadir, 'kcattachments'), self.name))
        fig.clf()
        plt.close()
        gc.collect()
        mpl.rcdefaults()

    def AnalyzeSpindleForces(self):
        min_size = 0
        for sd in self.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        measure_time = 8.0/60.0
        time = self.seeds[-1].time[:min_size]
        nmeasure = np.int(measure_time / ((time[1] - time[0])))
        nmeasure = 4 * nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(self.seeds)))
        time = self.seeds[-1].time[:min_size]
        time_resampled = time[0::nmeasure]

        avg_std_xlink = np.zeros((len(self.seeds), time_resampled.size))
        avg_std_af_0 = np.zeros((len(self.seeds), time_resampled.size))
        avg_std_af_1 = np.zeros((len(self.seeds), time_resampled.size))

        avg_xlink_axis = np.zeros((len(self.seeds), time_resampled.size))
        avg_xlink_0_axis = np.zeros((len(self.seeds), time_resampled.size))
        avg_xlink_1_axis = np.zeros((len(self.seeds), time_resampled.size))
        avg_xlink_2_axis = np.zeros((len(self.seeds), time_resampled.size))
        avg_af_axis = np.zeros((len(self.seeds), time_resampled.size))

        # Try to get the perpendicular information too
        avg_xlink_0_perp = np.zeros((len(self.seeds), time_resampled.size))
        avg_xlink_1_perp = np.zeros((len(self.seeds), time_resampled.size))
        avg_xlink_2_perp = np.zeros((len(self.seeds), time_resampled.size))
        avg_af_perp = np.zeros((len(self.seeds), time_resampled.size))

        avg_interkinetochore = np.zeros((len(self.seeds), time_resampled.size))

        num_seeds = 0

        # Long time behavior
        average_xlink_force = []
        average_af_force = []
        average_af0_force = []
        average_af1_force = []
        average_x_force = []

        # Long time chromosome force magnitude
        average_chromosome_force = []

        isd = 0

        for sd in self.seeds:
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::nmeasure]

            # We want the spindle lengths for binning the appropriate force distribution
            spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
            spindle_length_arr = np.array([spbsep_dict[ts] for ts in time])

            # The forces are stored as a dictionary, so get each contribution
            forces_dict_0 = sd.PostAnalysis.timedata['pole_forces'][0]

            # The first entry is the time, then it is a dictionary of the different forces
            forces_arr_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][0] for ts in time])
            forces_arr_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][1] for ts in time])
            forces_arr_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][2] for ts in time])
            forces_axis_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][0] for ts in time])
            forces_axis_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][1] for ts in time])
            forces_axis_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][2] for ts in time])
            # and the total Xlink force
            forces_axis_0_xlink = np.array([forces_dict_0[ts]['xlink_axis'] for ts in time])

            # Grab the AF forces too
            forces_arr_0_af = np.array([forces_dict_0[ts]['af_mag'] for ts in time])
            forces_axis_0_af = np.array([forces_dict_0[ts]['af_axis'] for ts in time])

            # What about the inter-kinetochore force?
            forces_interkinetochore = np.array([forces_dict_0[ts]['chromosome_mag'] for ts in time])

            # Redo the mean average stuff
            forces_arr_0_subtype_0 = np.mean(forces_arr_0_subtype_0.reshape(-1, nmeasure), axis=1)
            forces_arr_0_subtype_1 = np.mean(forces_arr_0_subtype_1.reshape(-1, nmeasure), axis=1)
            forces_arr_0_subtype_2 = np.mean(forces_arr_0_subtype_2.reshape(-1, nmeasure), axis=1)
            forces_arr_0_af = np.mean(forces_arr_0_af.reshape(-1, nmeasure), axis=1)

            forces_axis_0_subtype_0 = np.mean(forces_axis_0_subtype_0.reshape(-1, nmeasure), axis=1)
            forces_axis_0_subtype_1 = np.mean(forces_axis_0_subtype_1.reshape(-1, nmeasure), axis=1)
            forces_axis_0_subtype_2 = np.mean(forces_axis_0_subtype_2.reshape(-1, nmeasure), axis=1)
            forces_axis_0_xlink = np.mean(forces_axis_0_xlink.reshape(-1, nmeasure), axis=1)
            forces_axis_0_af = np.mean(forces_axis_0_af.reshape(-1, nmeasure), axis=1)

            forces_interkinetochore = np.mean(forces_interkinetochore.reshape(-1, nmeasure), axis=1)

            # We should be able to construct the perpendicular force by taking the sqrt(total2 - axis2)
            forces_perp_0_subtype_0 = np.sqrt(np.square(forces_arr_0_subtype_0) - np.square(forces_axis_0_subtype_0))
            forces_perp_0_subtype_1 = np.sqrt(np.square(forces_arr_0_subtype_1) - np.square(forces_axis_0_subtype_1))
            forces_perp_0_subtype_2 = np.sqrt(np.square(forces_arr_0_subtype_2) - np.square(forces_axis_0_subtype_2))
            forces_perp_0_af = np.sqrt(np.square(forces_arr_0_af) - np.square(forces_axis_0_af))

            spindle_length_arr = np.mean(spindle_length_arr.reshape(-1, nmeasure), axis=1) * uc['um'][1]

            avg_xlink_axis[isd,:] = forces_axis_0_xlink[:min_size]
            avg_xlink_0_axis[isd,:] = forces_axis_0_subtype_0[:min_size]
            avg_xlink_1_axis[isd,:] = forces_axis_0_subtype_1[:min_size]
            avg_xlink_2_axis[isd,:] = forces_axis_0_subtype_2[:min_size]
            avg_af_axis[isd,:] = forces_axis_0_af[:min_size]

            avg_interkinetochore[isd,:] = forces_interkinetochore[:min_size]

            # Perpendicular
            avg_xlink_0_perp[isd,:] = forces_perp_0_subtype_0[:min_size]
            avg_xlink_1_perp[isd,:] = forces_perp_0_subtype_1[:min_size]
            avg_xlink_2_perp[isd,:] = forces_perp_0_subtype_2[:min_size]
            avg_af_perp[isd,:] = forces_perp_0_af[:min_size]

            # Grab the longtime behavior of these, make sure to take absolute values, as they are along the spindle axis
            long_time_xlink = np.absolute(forces_axis_0_xlink[time_resampled > 10.0])
            long_time_af = np.absolute(forces_axis_0_af[time_resampled > 10.0])
            long_time_interkinetochore = np.absolute(forces_interkinetochore[time_resampled > 10.0])

            average_xlink_force += [long_time_xlink[:]]
            average_af_force += [long_time_af[:]]
            average_chromosome_force += [long_time_interkinetochore[:]]

            # Load the information of the spindle length and forces into a self consistent dictionary
            if spindle_length_arr.shape[0] != forces_axis_0_af.shape[0]:
                print "moo moo bad"
                sys.exit(1)

            # Create a dataframe of the information to bin later
            # Convert with the minus sign here!
            data = {'spindle_length': spindle_length_arr, 'xlink_type0_axis_force': -1.0 * forces_axis_0_subtype_0,
                    'xlink_type1_axis_force': -1.0 * forces_axis_0_subtype_1, 'xlink_type2_axis_force': -1.0 * forces_axis_0_subtype_2,
                    'af_axis_force': -1.0 * forces_axis_0_af, 'time': time_resampled}
            df = pd.DataFrame(data)
            #print df

            # Load the first one
            if num_seeds == 0:
                self.length_force_dataframe = df
            else:
                self.length_force_dataframe = self.length_force_dataframe.append(df, ignore_index = True)

            # Make the seed save its own force information for later (we will need this to make graphing easier)
            sd.SetForceData(time_resampled, forces_interkinetochore[:min_size])

            num_seeds += 1
            isd += 1

        avg_axis_xlink = np.mean(avg_xlink_axis, axis=0)
        avg_axis_xlink_0 = np.mean(avg_xlink_0_axis, axis=0)
        avg_axis_xlink_1 = np.mean(avg_xlink_1_axis, axis=0)
        avg_axis_xlink_2 = np.mean(avg_xlink_2_axis, axis=0)
        avg_axis_af = np.mean(avg_af_axis, axis=0)
        avg_interkinetochore_force = np.mean(avg_interkinetochore, axis=0)

        stddev_axis_xlink = np.std(avg_xlink_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_0 = np.std(avg_xlink_0_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_1 = np.std(avg_xlink_1_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_2 = np.std(avg_xlink_2_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_af = np.std(avg_af_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_interkinetochore_force = np.std(avg_interkinetochore, axis=0, ddof=1) / np.sqrt(num_seeds)

        avg_perp_xlink_0 = np.mean(avg_xlink_0_perp, axis=0)
        avg_perp_xlink_1 = np.mean(avg_xlink_1_perp, axis=0)
        avg_perp_xlink_2 = np.mean(avg_xlink_2_perp, axis=0)
        avg_perp_af = np.mean(avg_af_perp, axis=0)
        stddev_perp_xlink_0 = np.std(avg_xlink_0_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_xlink_1 = np.std(avg_xlink_1_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_xlink_2 = np.std(avg_xlink_2_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_af = np.std(avg_af_perp, axis=0, ddof=1) / np.sqrt(num_seeds)

        # Write the df to CSV
        export_csv = self.length_force_dataframe.to_csv(r'sim_spindleforcemeasurements.csv', index=None, header=True)

        # Load our own data
        self.time_resampled_forces = time_resampled
        self.avg_interkinetochore_force = avg_interkinetochore_force
        self.stddev_interkinetochore_force = stddev_interkinetochore_force

        return [np.asarray(average_xlink_force), np.asarray(average_af_force), np.asarray(average_chromosome_force)]


    def GraphDistrData(self, title):
        # Loop over graph functions and their options
        # graph function, row_number, column_number, figsize
        # font ={#'weight' : 'bold',
                # 'size' : 15}
        # mpl.rc('font', **font)
        for gf, rn, cn, fs in self.graph_distr_funcs:
            fig, axarr = plt.subplots(rn, cn, figsize=fs)
            gf(self, axarr)
            plt.figure(fig.number)
            plt.suptitle(title)
            fig.tight_layout()
            plt.subplots_adjust(top=.87)
            # Truncate path length to work properly on stupid OSX
            pruned_name = self.name[0:127]
            fig.savefig("{}_{}.pdf".format(os.path.join(self.sim_datadir, gf.__name__),pruned_name), dpi=fig.dpi) 
            fig.clf()
            plt.close()
            gc.collect()
        mpl.rcdefaults()

    def GraphCombinedAttachments(self, colors):
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print 'Criteria analysis has not been run on sim {}'.format(self.name)
            title = self.title

        font ={#'weight' : 'bold',
                'size' : 20}
        mpl.rc('font', **font)
        # mpl.rc('text', usetex=True)

        fig, axarr = plt.subplots(4, 2, figsize=(25,16))
        min_size = 0
        for sd in self.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        time = self.seeds[-1].time[:min_size]

        # Replac ethe last graph with the amphitelic attachment one
        graph_arr = self.graph_time_funcs
        graph_arr[-1] = graph_kc_attachment_type

        for graph, axr in zip(graph_arr, axarr):
            avg = np.zeros(min_size)
            num_seeds = 0
            for sd, col in zip(self.seeds, colors):
                yarr = graph(sd, axr[0], color=col, xlabel=False)
                if not self.opts.write or sd.succ_info_dict['succ'] == 1:
                    num_seeds += 1
                    avg = np.add(avg, yarr[:min_size])

            avg /= float(num_seeds)
            axr[1].plot(time, avg)

        axarr[1, 0].legend(loc='center left', bbox_to_anchor=(2.2,-.19))

        axarr[-1,1].set_xlabel(r'Time (min)')
        axarr[-1,0].set_xlabel(r'Time (min)')

        plt.subplots_adjust(hspace = .38)
        plt.suptitle(title)
        plt.subplots_adjust(hspace = .38, right=.85, top=.87)
        plt.savefig("{}_{}.png".format(os.path.join(self.sim_datadir, 'combined'), self.name))
        fig.clf()
        plt.close()
        gc.collect()
        mpl.rcdefaults()


    # TODO Depricated function, change to be more functional
    def GraphSimulationPost(self):
        try:
            title = (self.title + ' Post ' +
                    '\n Successful Fraction: {}'.format(self.sf))
        except:
            print "Criteria analysis has not been run on sim {}".format(self.name)
            title = self.title

        font ={#'weight' : 'bold',
                'size' : 20}
        mpl.rc('font', **font)
        # mpl.rc('text', usetex=True)
        fig, axarr = plt.subplots(3, 2, figsize=(25,16))
        colors = mpl.cm.rainbow(np.linspace(0,1,len(self.seeds)))

        # don't care about times, just graph this
        # Kinda specialized, as we can just combine the histograms and plot them then!
        for graph, axr in zip(self.graph_funcs_post, axarr):
            datastream = np.array([])
            weightstream = np.array([])
            for sd, col in zip(self.seeds, colors):
                yarr = graph(sd, axr[0], col, xstate = r'integrated') # Gives back the xdata (data, weight)
                datastream = np.append(datastream, yarr[0])
                weightstream = np.append(weightstream, yarr[1])
                print "datastream: {}".format(datastream)
                print "weightstream: {}".format(weightstream)
                print "len: {}, {}".format(datastream.shape, weightstream.shape)
            # Run the override version of the plot!
                graph(sd, axr[1], xdata_in = [datastream, weightstream])

        axarr[1, 0].legend(loc='center left', bbox_to_anchor=(2.2,-.19))
        plt.subplots_adjust(hspace = .38)
        plt.suptitle(title)
        plt.savefig("{}_post.png".format(os.path.join(self.sim_datadir, self.name)))
        fig.clf()
        plt.close()
        gc.collect()

    def CalcSimSuccess(self, calc_avg=None, **kwargs):
        if self.succ_calc_flag: return
        start_list = []
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed:
                continue
            if sd.succ_info_dict['succ']:
                self.succ_num += 1
                start_list += [sd.succ_info_dict['start_time']]

        s = float(self.succ_num) # Number of successes
        n = float(len(self.seeds)) # Number of seeds used
        if self.succ_num != 0:
            self.sf = s/n
            self.sf_err= BinomialError(s, n)
        else:
            self.sf = 0.0
            self.sf_err = 0.0
        self.succ_calc_flag = True
        return

    def GetSimCritInfo(self):
        raise NotImplementedError("GetSimCritInfo")

    def SetSimDataDir(self):
        # If sim data directory has already been set, skip 
        if self.sim_datadir: pass
        # If no data directory is specified, make it
        elif not self.opts.datadir: self.opts.datadir = create_datadir(self.opts.workdir)
        # If a data directory does exist but it is not in the sim directory,
        # make a subdirectory to store sim specific information
        elif self.opts.workdir != self.sim_path:
            poss_path = os.path.join(self.opts.datadir, "{}_data".format(self.name))
            if os.path.exists( poss_path ): self.sim_datadir = poss_path
            else: self.sim_datadir = create_datadir( self.opts.datadir, 
                                        datadir_name = "{}_data".format(self.name))
        # If all else fails, just make the data directory the sim data directory
        else: self.sim_datadir = self.opts.datadir
        return self.sim_datadir
    
    # Do the combined EM fitness, where we pool the results from all of the seeds
    # into one locaiton, then do the KS tests
    def CombinedEMFitness(self):
        varnames = [['interpolar', 'interpolar'],
                    ['pairing_length_max', 'pair'],
                    ['total', 'lengths'],
                    ['angles', 'angles']]
        # Convert everything from sim units to real units
        conversions = {}
        conversions['interpolar'] = uc['um'][1]
        conversions['pairing_length_max'] = uc['um'][1]
        conversions['total'] = uc['um'][1]
        conversions['angles'] = 1.0

        # Combine all of our current seeds into one set of data
        ngoodseeds = 0
        total_base = 0.0
        total_log = 0.0
        for varname in varnames:
            combined = {}
            for sd in self.seeds:
                # Make sure that we have a good seed first
                if sd.PostAnalysis.bad_seed:
                    continue
                ngoodseeds += 1

                distrdata = sd.PostAnalysis.distrdata['fitness']
                emdata_exp = sd.fitness.experiment # Grab the experiment values

                # Do the standard thing of combining information into the total array
                if 'short' not in combined:
                    if varname[0] in distrdata['1.05um']:
                        combined['short'] = np.array(distrdata['1.05um'][varname[0]])*conversions[varname[0]]
                else:
                    if varname[0] in distrdata['1.05um']:
                        combined['short'] = np.append(combined['short'], np.array(distrdata['1.05um'][varname[0]])*conversions[varname[0]])

                if 'medium' not in combined:
                    if varname[0] in distrdata['1.825um']:
                        combined['medium'] = np.array(distrdata['1.825um'][varname[0]])*conversions[varname[0]]
                else:
                    if varname[0] in distrdata['1.825um']:
                        combined['medium'] = np.append(combined['medium'], np.array(distrdata['1.825um'][varname[0]])*conversions[varname[0]])

                if 'long' not in combined:
                    if varname[0] in distrdata['2.15um']:
                        combined['long'] = np.array(distrdata['2.15um'][varname[0]])*conversions[varname[0]]
                else:
                    if varname[0] in distrdata['2.15um']:
                        combined['long'] = np.append(combined['long'], np.array(distrdata['2.15um'][varname[0]])*conversions[varname[0]])

            # Now that we have the combined stream for this type of measurement, use it!
            matlabname = varname[1]
            inres = {}
            if 'short' in combined:
                inres['short'] = (ks_2samp(combined['short'], np.array(emdata_exp[matlabname]['short'])*conversions[varname[0]])).pvalue
            else:
                inres['short'] = 0.0

            if 'medium' in combined:
                inres['medium'] = (ks_2samp(combined['medium'], np.array(emdata_exp[matlabname]['med'])*conversions[varname[0]])).pvalue
            else:
                inres['medium'] = 0.0

            if 'long' in combined:
                inres['long'] = (ks_2samp(combined['long'], np.array(emdata_exp[matlabname]['long'])*conversions[varname[0]])).pvalue
            else:
                inres['long'] = 0.0
            #print "Short  {} KS: {}".format(varname[0], inres['short'])
            #print "Medium {} KS: {}".format(varname[0], inres['medium'])
            #print "Long   {} KS: {}".format(varname[0], inres['long'])
            combined_val = (inres['short'] + inres['long'] + inres['medium'])/3.
            # Take the log of each, and then if the pvalue is 0.0, set to -10.0
            logres_short = -10.0
            logres_medium = -10.0
            logres_long = -10.0
            if inres['short'] > 0.0:
                logres_short = np.log10(inres['short'])/100.
            if inres['medium'] > 0.0:
                logres_medium = np.log10(inres['medium'])/100.
            if inres['long'] > 0.0:
                logres_long = np.log10(inres['long'])/100.
            combined_log = (logres_short + logres_medium + logres_long)/3.
            if np.isnan(combined_log) or np.isinf(combined_log):
                combined_log = -10.0 # Set a minimum value...
            total_base += combined_val
            total_log += combined_log
            #print "{} Combined {} ({})".format(varname[0], combined_val, combined_log)

        self.emfitness_base = total_base/4.
        self.emfitness_log = total_log/4.
        #print "Total Base EM Fitness {}".format(total_base/4.)
        #print "Total Log EM Fitness {}".format(total_log/4.)

    def Fitness(self, wt_mat, lstream_mat):
        print "- Sim {} Fitness -".format(self.name)
        # First go through seeds and make them generate the fitness
        self.analyze_chromosomes = False
        strain_name = ''
        for sd in self.seeds:
            strain_name = sd.Fitness(wt_mat, lstream_mat)
            self.analyze_chromosomes = sd.PostAnalysis.analyze_chromosomes

        if strain_name == 'WT_wt':
            self.Mis12GFP(wt_mat, lstream_mat)
        elif strain_name == 'WT_Cen2':
            self.Cen2GFP()
        elif strain_name == 'Cerevisiae_basic':
            self.CerevisiaeBasic()
        elif strain_name == 'Cerevisiae_length':
            self.CerevisiaeBasic()

    # Old Mis12GFP way of doing things without the Cen2 marker
    def Mis12GFP(self, wt_mat, lstream_mat): 
        # Go through them and find the max fitness for the EM tomography data
        # Also grab the spindle length data on the way
        lstream_model = []
        chromosome_seconds = []
        chromosome_seconds_fraction = []
        lstream_exp = []
        self.length_correlation_avg = 0.0
        fbiorient = []

        ngoodseeds = 0
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed:
                continue
            ngoodseeds += 1
            fitness = sd.fitness
            lstream_model = np.append(lstream_model, fitness.lstream_model)
            lstream_exp = fitness.flstream_exp
            # Get the length correlation average
            self.length_correlation_avg += fitness.length_correlation_avg
            if self.analyze_chromosomes:
                chromosome_seconds += [fitness.chromosome_seconds]
                chromosome_seconds_fraction += [fitness.chromosome_seconds_fraction]
                fbiorient += [fitness.fbiorient]
                self.experiment_data_adj = fitness.experiment_data_adj
                self.best_tracks = fitness.best_tracks
        if ngoodseeds > 0:
            self.length_correlation_avg /= ngoodseeds

        # Run the EM tomogram fitness function
        self.CombinedEMFitness()
      
        # Create defaults for these values
        self.mean_chromosome_seconds = 0.0
        self.stddev_chromosome_seconds = 0.0
        self.mean_chromosome_seconds_fraction = 0.0
        self.stddev_chromosome_seconds_fraction = 0.0
        self.mean_fbiorient = 0.0
        self.stddev_fbiorient = 0.0
        self.kc_spb_dist_pval = -10.0
        self.kc_spindle1d_pval = -10.0
        if self.analyze_chromosomes and ngoodseeds > 0:
            chromosome_seconds = np.asarray(chromosome_seconds)
            self.mean_chromosome_seconds = np.mean(chromosome_seconds)
            self.stddev_chromosome_seconds = np.std(chromosome_seconds, ddof=1)
            chromosome_seconds_fraction = np.asarray(chromosome_seconds_fraction)
            self.mean_chromosome_seconds_fraction = np.mean(chromosome_seconds_fraction)
            self.stddev_chromosome_seconds_fraction = np.std(chromosome_seconds_fraction, ddof=1)

            # New fbiorientation information
            self.mean_fbiorient = np.mean(fbiorient)
            self.stddev_fbiorient = np.std(fbiorient, ddof=1)

            # Also do the correct fitness for chromosome information
            self.KCDistributionFitness()
        if len(lstream_exp) > 0:
            length_res = ks_2samp(lstream_model, lstream_exp)
            self.length_fitness = length_res.pvalue
        else:
            self.length_fitness = 0.0

        if not self.opts.datadir: self.opts.datadir = create_datadir(self.opts.workdir)
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print "Don't know successful fraction. Criteria analysis might not have been run on sim {}".format(self.name)
            title = self.title
        if self.opts.workdir != self.sim_path:
            self.sim_datadir = create_datadir( self.opts.datadir, 
                                 datadir_name = "{}_data".format(self.name) )
        else:
            self.sim_datadir = self.opts.datadir

        with open(os.path.join(self.sim_datadir, 'fitness_final.yaml'), 'w') as stream:
            stream.write('length_fitness: {}\n'.format(self.length_fitness))
            stream.write('length_correlation_avg: {}\n'.format(self.length_correlation_avg))
            stream.write('em_fitness_log: {}\n'.format(self.emfitness_log))
            stream.write('em_fitness_base: {}\n'.format(self.emfitness_base))
            stream.write('success_fraction: {}\n'.format(self.sf))

            # Common fitness information (without even chromosomes)
            self.final_fitness = self.emfitness_log + 2.0*self.length_correlation_avg

            if self.analyze_chromosomes:
                stream.write('chromosome_seconds: {}\n'.format(self.mean_chromosome_seconds))
                stream.write('chromosome_seconds_stddev: {}\n'.format(self.stddev_chromosome_seconds))
                stream.write('chromosome_seconds_fraction: {}\n'.format(self.mean_chromosome_seconds_fraction))
                stream.write('chromosome_seconds_fraction_stddev: {}\n'.format(self.stddev_chromosome_seconds_fraction))
                stream.write('fbiorientation: {}\n'.format(self.mean_fbiorient))
                stream.write('fbiorientation_stddev: {}\n'.format(self.stddev_fbiorient))
                stream.write('chromosome_kc_spb_distance: {}\n'.format(self.kc_spb_dist_pval))
                stream.write('chromosome_kc_spindle1d: {}\n'.format(self.kc_spindle1d_pval))

                # Add to the new fitness
                # Just added in new success fraction definition for chromosomes, with amphitelic attachments and interpolar fraction
                self.final_fitness += 10.0*self.mean_fbiorient + np.mean([self.kc_spb_dist_pval, self.kc_spindle1d_pval]) + 2.0*self.mean_chromosome_seconds_fraction

            stream.write('FINAL_FITNESS: {}\n'.format(self.final_fitness))
            print 'FINAL_FITNESS: {}'.format(self.final_fitness)


        #outfilename = os.path.join(self.opts.datadir, 'fitness.dat')
        #with open(outfilename, 'w') as outstream:
        #    outstream.write("length_fitness: {}\n".format(length_fitness))
        #    for k,v in em_fitness.iteritems():
        #        outstream.write("{}: {}\n".format(k,v))

    def Cen2GFP(self):
        print "Cen2GFP simulation fitness sim {}".format(self.name)

        # Empty arrays to do averaging over
        max_correlation_combined = []
        max_length_correlation = []
        max_kc_distance_correlation = []
        max_kc_sep_correlation = []
        avg_correlation_combined = []
        avg_length_correlation = []
        avg_kc_distance_correlation = []
        avg_kc_sep_correlation = []
        integrated_biorientation_time = []
        fraction_integrated_biorientation_time = []
        fbiorient = []
        segregation_success = []
        final_length = []
        final_ipf = []
        final_occupancy = []
        kmt_lifetimes = []
        self.max_correlation_combined_mean                  = 0.0
        self.max_correlation_combined_std                   = 0.0
        self.max_length_correlation_mean                    = 0.0
        self.max_length_correlation_std                     = 0.0
        self.max_kc_distance_correlation_mean               = 0.0
        self.max_kc_distance_correlation_std                = 0.0
        self.max_kc_sep_correlation_mean                    = 0.0
        self.max_kc_sep_correlation_std                     = 0.0
        self.avg_correlation_combined_mean                  = 0.0
        self.avg_correlation_combined_std                   = 0.0
        self.avg_length_correlation_mean                    = 0.0
        self.avg_length_correlation_std                     = 0.0
        self.avg_kc_distance_correlation_mean               = 0.0
        self.avg_kc_distance_correlation_std                = 0.0
        self.avg_kc_sep_correlation_mean                    = 0.0
        self.avg_kc_sep_correlation_std                     = 0.0
        self.integrated_biorientation_time_mean             = 0.0
        self.integrated_biorientation_time_std              = 0.0
        self.fraction_integrated_biorientation_time_mean    = 0.0
        self.fraction_integrated_biorientation_time_std     = 0.0
        self.fbiorient_mean                                 = 0.0
        self.fbiorient_std                                  = 0.0

        # We might be doing anaphase
        self.segregation_success_mean                       = 0.0
        self.segregation_success_std                        = 0.0

        # Because its easy, grab the final length...
        self.final_length_mean                              = 0.0
        self.final_length_std                               = 0.0
        self.final_ipf_mean                                 = 0.0
        self.final_ipf_std                                  = 0.0
        self.final_occupancy_mean                           = 0.0
        self.final_occupancy_std                            = 0.0
        self.kmt_lifetime_mean                              = 0.0
        self.kmt_lifetime_std                               = 0.0

        ngoodseeds = 0
        naf = 1
        nx = 0
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed or not sd.PostAnalysis.analyze_chromosomes:
                continue
            ngoodseeds += 1
            fitness = sd.fitness
            
            # Add to the arrays for averaging
            max_correlation_combined                += [fitness.max_correlation_combined]
            max_length_correlation                  += [fitness.max_length_correlation]
            max_kc_distance_correlation             += [fitness.max_kc_distance_correlation]
            max_kc_sep_correlation                  += [fitness.max_kc_sep_correlation]
            avg_correlation_combined                += [fitness.avg_correlation_combined]
            avg_length_correlation                  += [fitness.avg_length_correlation]
            avg_kc_distance_correlation             += [fitness.avg_kc_distance_correlation]
            avg_kc_sep_correlation                  += [fitness.avg_kc_sep_correlation]
            integrated_biorientation_time           += [fitness.integrated_biorientation_time]
            fraction_integrated_biorientation_time  += [fitness.fraction_integrated_biorientation_time]
            fbiorient                               += [fitness.fbiorient]

            # We also want to check the anaphase fitness if it really is going on
            if sd.PostAnalysis.do_anaphase:
                segregation_success                     += [sd.PostAnalysis.segregation_success]

            # Grab the final spindle length, regardless of success
            # Do this for more than just the last time point - do it for the last 
            # 10 minutes of simulation
            # Time
            time = np.asarray(sd.PostAnalysis.timedata['time'])
            # Spindle length
            spbsep_arr = sd.PostAnalysis.timedata['spb_separation']
            spbsep_arr = np.array([spbsep_arr[ts] for ts in time])
            spbsep_arr = np.array(spbsep_arr)*uc['um'][1]
            # Interpolar fraction
            ipf_dict = sd.PostAnalysis.timedata['interpolar_fraction']
            ipf_arr = np.array([ipf_dict[ts] for ts in time])
            # Occupancy
            occ_dict = sd.PostAnalysis.timedata['kc_occupancy']
            occ_arr = np.array([occ_dict[ts] for ts in time])
            # kMT lifetimes
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                kmt_lifetimes_dict = sd.PostAnalysis.timedata['attachment_lifetimes']
                kmt_lifetimes_arr = [kmt_lifetimes_dict[ts] for ts in time]
            else:
                kmt_lifetimes_arr = []

            # Grab the time points after 10 minutes
            late_length = spbsep_arr[time > 10.0]
            late_ipf = ipf_arr[time > 10.0]
            late_occupancy = occ_arr[time > 10.0]
            measure_time = 8.0/60.0
            nmeasure = np.int(measure_time / ((time[1] - time[0])))

            # Recalibrate for the measurement frequency
            late_length = late_length[0::nmeasure]
            late_ipf = late_ipf[0::nmeasure]
            late_occupancy = late_occupancy[0::nmeasure]
            final_length                            += [late_length[:]]
            final_ipf                               += [late_ipf[:]]
            final_occupancy                         += [late_occupancy[:]]
            naf = sd.PostAnalysis.naf
            nxsome = sd.PostAnalysis.nxsome

            # Grab the kMT lifetimes and compile them into some reasonable form (and change to minutes)
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                kmt_lifetimes += [kmt_lifetimes_arr[-1] * (time[1] - time[0])]
                print "number of lifetime measurements = {}".format(len(np.concatenate(kmt_lifetimes).ravel()))
            else:
                kmt_lifetimes += [0.0]
            #kmt_lifetimes_mean += [np.mean(kmt_lifetimes_arr[-1]*(time[1] - time[0]))]
            #kmt_lifetimes_std += [np.std(kmt_lifetimes_arr[-1]*(time[1] - time[0]), ddof=1)]
            

        # Run the same EM fitness
        self.CombinedEMFitness()

        # Find the averages and std deviation
        self.ngoodseeds = ngoodseeds
        if ngoodseeds > 0:
            self.max_correlation_combined_mean                  = np.mean(np.asarray(max_correlation_combined))
            self.max_correlation_combined_std                   = np.std(np.asarray(max_correlation_combined), ddof=1)

            self.max_length_correlation_mean                    = np.mean(np.asarray(max_length_correlation))
            self.max_length_correlation_std                     = np.std(np.asarray(max_length_correlation), ddof=1)

            self.max_kc_distance_correlation_mean               = np.mean(np.asarray(max_kc_distance_correlation))
            self.max_kc_distance_correlation_std                = np.std(np.asarray(max_kc_distance_correlation), ddof=1)

            self.max_kc_sep_correlation_mean                    = np.mean(np.asarray(max_kc_sep_correlation))
            self.max_kc_sep_correlation_std                     = np.std(np.asarray(max_kc_sep_correlation), ddof=1)

            self.avg_correlation_combined_mean                  = np.mean(np.asarray(avg_correlation_combined))
            self.avg_correlation_combined_std                   = np.std(np.asarray(avg_correlation_combined), ddof=1)

            self.avg_length_correlation_mean                    = np.mean(np.asarray(avg_length_correlation))
            self.avg_length_correlation_std                     = np.std(np.asarray(avg_length_correlation), ddof=1)

            self.avg_kc_distance_correlation_mean               = np.mean(np.asarray(avg_kc_distance_correlation))
            self.avg_kc_distance_correlation_std                = np.std(np.asarray(avg_kc_distance_correlation), ddof=1)

            self.avg_kc_sep_correlation_mean                    = np.mean(np.asarray(avg_kc_sep_correlation))
            self.avg_kc_sep_correlation_std                     = np.std(np.asarray(avg_kc_sep_correlation), ddof=1)

            self.integrated_biorientation_time_mean             = np.mean(np.asarray(integrated_biorientation_time))
            self.integrated_biorientation_time_std              = np.std(np.asarray(integrated_biorientation_time), ddof=1)

            self.fraction_integrated_biorientation_time_mean    = np.mean(np.asarray(fraction_integrated_biorientation_time))
            self.fraction_integrated_biorientation_time_std     = np.std(np.asarray(fraction_integrated_biorientation_time), ddof=1)
            print "integrated biorientation time = {}".format(fraction_integrated_biorientation_time)

            self.fbiorient_mean                                 = np.mean(np.asarray(fbiorient))
            self.fbiorient_std                                  = np.std(np.asarray(fbiorient), ddof=1)

            self.segregation_success_mean                       = np.mean(np.asarray(segregation_success))
            self.segregation_success_std                        = np.std(np.asarray(segregation_success), ddof=1)

            #self.final_length_mean                              = np.mean(np.asarray(final_length).flatten())
            #self.final_length_std                               = np.std(np.asarray(final_length).flatten(), ddof=1)
            self.final_length_mean                              = np.mean(np.concatenate(final_length).ravel())
            self.final_length_std                               = np.std(np.concatenate(final_length).ravel(), ddof=1)

            #self.final_ipf_mean                                 = np.mean(np.asarray(final_ipf))
            #self.final_ipf_std                                  = np.std(np.asarray(final_ipf), ddof=1)
            self.final_ipf_mean                                 = np.mean(np.concatenate(final_ipf).ravel())
            self.final_ipf_std                                  = np.std(np.concatenate(final_ipf).ravel(), ddof=1)

            #self.final_occupancy_mean                           = np.mean(np.asarray(final_occupancy))
            #self.final_occupancy_std                            = np.std(np.asarray(final_ipf), ddof=1)
            self.final_occupancy_mean                           = np.mean(np.concatenate(final_occupancy).ravel())
            self.final_occupancy_std                            = np.std(np.concatenate(final_ipf).ravel(), ddof=1)
            self.total_occupancy_max                            = naf * nxsome * 2

            # Do the kmt lifetimes
            #self.kmt_lifetime_mean                              = np.average(kmt_lifetimes_mean, weights = kmt_lifetimes_std)
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                self.kmt_lifetime_mean                              = np.mean(np.concatenate(kmt_lifetimes).ravel())
                self.kmt_lifetime_std                               = np.std(np.concatenate(kmt_lifetimes).ravel(), ddof=1)
                print "number of lifetime measurements = {}".format(len(np.concatenate(kmt_lifetimes).ravel()))
                print "mean kmt lifetime = {}".format(self.kmt_lifetime_mean)
                print "std  kmt lifetime = {}".format(self.kmt_lifetime_std)
                print "std  err lifetime = {}".format(self.kmt_lifetime_std/np.sqrt(len(np.concatenate(kmt_lifetimes).ravel())))
                #print "std  err lifetime = {}".format(self.kmt_lifetime_std/np.sqrt(ngoodseeds))
                allvals = ", ".join(repr(e) for e in np.concatenate(kmt_lifetimes).ravel())
                #print 'attachment_lifetime, {}'.format(allvals)
                with open("moomoo.csv","a") as mfile:
                    mfile.write('attachment_lifetime, {}\n'.format(allvals))
            else:
                self.kmt_lifetime_mean                              = 0.0
                self.kmt_lifetime_std                               = 0.0
            #print "kmt_lifetimes_mean = {}".format(kmt_lifetimes_mean)
            #print "kmt_lifetimes_std  = {}".format(kmt_lifetimes_std)

        if not self.opts.datadir: self.opts.datadir = create_datadir(self.opts.workdir)
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print "Don't know successful fraction. Criteria analysis might not have been run on sim {}".format(self.name)
            title = self.title
        if self.opts.workdir != self.sim_path:
            self.sim_datadir = create_datadir( self.opts.datadir, 
                                 datadir_name = "{}_data".format(self.name) )
        else:
            self.sim_datadir = self.opts.datadir

        with open(os.path.join(self.sim_datadir, 'fitness_final.yaml'), 'w') as stream:
            # Write all the correlation averages
            stream.write('max_correlation_combined_mean: {}\n'.format(self.max_correlation_combined_mean))
            stream.write('max_length_correlation_mean: {}\n'.format(self.max_length_correlation_mean))
            stream.write('max_kc_distance_correlation_mean: {}\n'.format(self.max_kc_distance_correlation_mean))
            stream.write('max_kc_sep_correlation_mean: {}\n'.format(self.max_kc_sep_correlation_mean))
            stream.write('avg_correlation_combined_mean: {}\n'.format(self.avg_correlation_combined_mean))
            stream.write('avg_length_correlation_mean: {}\n'.format(self.avg_length_correlation_mean))
            stream.write('avg_kc_distance_correlation_mean: {}\n'.format(self.avg_kc_distance_correlation_mean))
            stream.write('avg_kc_sep_correlation_mean: {}\n'.format(self.avg_kc_sep_correlation_mean))
            stream.write('integrated_biorientation_time_mean: {}\n'.format(self.integrated_biorientation_time_mean))
            stream.write('fraction_integrated_biorientation_time_mean: {}\n'.format(self.fraction_integrated_biorientation_time_mean))
            stream.write('fraction_normalized_biorientation_time_mean: {}\n'.format(self.fbiorient_mean))

            # All the standard deviations
            stream.write('max_correlation_combined_std: {}\n'.format(self.max_correlation_combined_std))
            stream.write('max_length_correlation_std: {}\n'.format(self.max_length_correlation_std))
            stream.write('max_kc_distance_correlation_std: {}\n'.format(self.max_kc_distance_correlation_std))
            stream.write('max_kc_sep_correlation_std: {}\n'.format(self.max_kc_sep_correlation_std))
            stream.write('avg_correlation_combined_std: {}\n'.format(self.avg_correlation_combined_std))
            stream.write('avg_length_correlation_std: {}\n'.format(self.avg_length_correlation_std))
            stream.write('avg_kc_distance_correlation_std: {}\n'.format(self.avg_kc_distance_correlation_std))
            stream.write('avg_kc_sep_correlation_std: {}\n'.format(self.avg_kc_sep_correlation_std))
            stream.write('integrated_biorientation_time_std: {}\n'.format(self.integrated_biorientation_time_std))
            stream.write('fraction_integrated_biorientation_time_std: {}\n'.format(self.fraction_integrated_biorientation_time_std))

            # EM measures and success
            stream.write('em_fitness_log: {}\n'.format(self.emfitness_log))
            stream.write('em_fitness_base: {}\n'.format(self.emfitness_base))
            stream.write('success_fraction: {}\n'.format(self.sf))

            # Anaphase success measures
            stream.write('anaphase_segregation_success: {}\n'.format(self.segregation_success_mean))

            self.final_fitness = self.emfitness_log + self.avg_correlation_combined_mean + 2.0*self.fbiorient_mean + 2.0*self.fraction_integrated_biorientation_time_mean

            stream.write('FINAL_FITNESS: {}\n'.format(self.final_fitness))
            print 'FINAL_FITNESS: {}'.format(self.final_fitness)

    # TODO implement __str__ function
    # def __str__(self):
        # sf, ast, sf_err, ast_err = self.GetSimSuccess()
        # return ("Sim dir: {}\n" +
                # "    Success fraction = {} +-{}\n" +
                # "    Avg start time = {} +-").format(self.sim_name, sf,  ast)

    # TODO implement __getitem__ function
    # def __getitem__(self,i):
        # return self.GetSimSuccess()[i]

    # Cerevisiae basic simulation fitness
    def CerevisiaeBasic(self):
        print "Cerevisiae basic/length fitness sim {}".format(self.name)

        # Empty arrays to do averaging over
        max_correlation_combined = []
        max_length_correlation = []
        #max_kc_distance_correlation = []
        #max_kc_sep_correlation = []
        avg_correlation_combined = []
        avg_length_correlation = []
        #avg_kc_distance_correlation = []
        #avg_kc_sep_correlation = []
        integrated_biorientation_time = []
        fraction_integrated_biorientation_time = []
        fbiorient = []
        segregation_success = []
        final_length = []
        final_ipf = []
        final_occupancy = []
        kmt_lifetimes = []
        self.max_correlation_combined_mean                  = 0.0
        self.max_correlation_combined_std                   = 0.0
        self.max_length_correlation_mean                    = 0.0
        self.max_length_correlation_std                     = 0.0
        #self.max_kc_distance_correlation_mean               = 0.0
        #self.max_kc_distance_correlation_std                = 0.0
        #self.max_kc_sep_correlation_mean                    = 0.0
        #self.max_kc_sep_correlation_std                     = 0.0
        self.avg_correlation_combined_mean                  = 0.0
        self.avg_correlation_combined_std                   = 0.0
        self.avg_length_correlation_mean                    = 0.0
        self.avg_length_correlation_std                     = 0.0
        #self.avg_kc_distance_correlation_mean               = 0.0
        #self.avg_kc_distance_correlation_std                = 0.0
        #self.avg_kc_sep_correlation_mean                    = 0.0
        #self.avg_kc_sep_correlation_std                     = 0.0
        self.integrated_biorientation_time_mean             = 0.0
        self.integrated_biorientation_time_std              = 0.0
        self.fraction_integrated_biorientation_time_mean    = 0.0
        self.fraction_integrated_biorientation_time_std     = 0.0
        self.fbiorient_mean                                 = 0.0
        self.fbiorient_std                                  = 0.0

        # We might be doing anaphase
        self.segregation_success_mean                       = 0.0
        self.segregation_success_std                        = 0.0

        # Because its easy, grab the final length...
        self.final_length_mean                              = 0.0
        self.final_length_std                               = 0.0
        self.final_ipf_mean                                 = 0.0
        self.final_ipf_std                                  = 0.0
        self.final_occupancy_mean                           = 0.0
        self.final_occupancy_std                            = 0.0
        self.kmt_lifetime_mean                              = 0.0
        self.kmt_lifetime_std                               = 0.0

        ngoodseeds = 0
        naf = 1
        nx = 0
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed or not sd.PostAnalysis.analyze_chromosomes:
                continue
            ngoodseeds += 1
            fitness = sd.fitness
            
            # Add to the arrays for averaging
            max_correlation_combined                += [fitness.max_correlation_combined]
            max_length_correlation                  += [fitness.max_length_correlation]
            #max_kc_distance_correlation             += [fitness.max_kc_distance_correlation]
            #max_kc_sep_correlation                  += [fitness.max_kc_sep_correlation]
            avg_correlation_combined                += [fitness.avg_correlation_combined]
            avg_length_correlation                  += [fitness.avg_length_correlation]
            #avg_kc_distance_correlation             += [fitness.avg_kc_distance_correlation]
            #avg_kc_sep_correlation                  += [fitness.avg_kc_sep_correlation]
            integrated_biorientation_time           += [fitness.integrated_biorientation_time]
            fraction_integrated_biorientation_time  += [fitness.fraction_integrated_biorientation_time]
            fbiorient                               += [fitness.fbiorient]

            # We also want to check the anaphase fitness if it really is going on
            if sd.PostAnalysis.do_anaphase:
                segregation_success                     += [sd.PostAnalysis.segregation_success]

            # Grab the final spindle length, regardless of success
            # Do this for more than just the last time point - do it for the last 
            # 10 minutes of simulation
            # Time
            time = np.asarray(sd.PostAnalysis.timedata['time'])
            # Spindle length
            spbsep_arr = sd.PostAnalysis.timedata['spb_separation']
            spbsep_arr = np.array([spbsep_arr[ts] for ts in time])
            spbsep_arr = np.array(spbsep_arr)*uc['um'][1]
            # Interpolar fraction
            ipf_dict = sd.PostAnalysis.timedata['interpolar_fraction']
            ipf_arr = np.array([ipf_dict[ts] for ts in time])
            # Occupancy
            occ_dict = sd.PostAnalysis.timedata['kc_occupancy']
            occ_arr = np.array([occ_dict[ts] for ts in time])
            # kMT lifetimes
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                kmt_lifetimes_dict = sd.PostAnalysis.timedata['attachment_lifetimes']
                kmt_lifetimes_arr = [kmt_lifetimes_dict[ts] for ts in time]
            else:
                kmt_lifetimes_arr = []

            # Grab the time points after 10 minutes
            late_length = spbsep_arr[time > 10.0]
            late_ipf = ipf_arr[time > 10.0]
            late_occupancy = occ_arr[time > 10.0]
            measure_time = 12.0/60.0
            nmeasure = np.int(measure_time / ((time[1] - time[0])))

            # Recalibrate for the measurement frequency
            late_length = late_length[0::nmeasure]
            late_ipf = late_ipf[0::nmeasure]
            late_occupancy = late_occupancy[0::nmeasure]
            final_length                            += [late_length[:]]
            final_ipf                               += [late_ipf[:]]
            final_occupancy                         += [late_occupancy[:]]
            naf = sd.PostAnalysis.naf
            nxsome = sd.PostAnalysis.nxsome

            # Grab the kMT lifetimes and compile them into some reasonable form (and change to minutes)
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                kmt_lifetimes += [kmt_lifetimes_arr[-1] * (time[1] - time[0])]
                print "number of lifetime measurements = {}".format(len(np.concatenate(kmt_lifetimes).ravel()))
            else:
                kmt_lifetimes += [0.0]
            #kmt_lifetimes_mean += [np.mean(kmt_lifetimes_arr[-1]*(time[1] - time[0]))]
            #kmt_lifetimes_std += [np.std(kmt_lifetimes_arr[-1]*(time[1] - time[0]), ddof=1)]
            

        # Run the same EM fitness
        #self.CombinedEMFitness()

        # Find the averages and std deviation
        self.ngoodseeds = ngoodseeds
        if ngoodseeds > 0:
            self.max_correlation_combined_mean                  = np.mean(np.asarray(max_correlation_combined))
            self.max_correlation_combined_std                   = np.std(np.asarray(max_correlation_combined), ddof=1)

            self.max_length_correlation_mean                    = np.mean(np.asarray(max_length_correlation))
            self.max_length_correlation_std                     = np.std(np.asarray(max_length_correlation), ddof=1)

            #self.max_kc_distance_correlation_mean               = np.mean(np.asarray(max_kc_distance_correlation))
            #self.max_kc_distance_correlation_std                = np.std(np.asarray(max_kc_distance_correlation), ddof=1)

            #self.max_kc_sep_correlation_mean                    = np.mean(np.asarray(max_kc_sep_correlation))
            #self.max_kc_sep_correlation_std                     = np.std(np.asarray(max_kc_sep_correlation), ddof=1)

            self.avg_correlation_combined_mean                  = np.mean(np.asarray(avg_correlation_combined))
            self.avg_correlation_combined_std                   = np.std(np.asarray(avg_correlation_combined), ddof=1)

            self.avg_length_correlation_mean                    = np.mean(np.asarray(avg_length_correlation))
            self.avg_length_correlation_std                     = np.std(np.asarray(avg_length_correlation), ddof=1)

            #self.avg_kc_distance_correlation_mean               = np.mean(np.asarray(avg_kc_distance_correlation))
            #self.avg_kc_distance_correlation_std                = np.std(np.asarray(avg_kc_distance_correlation), ddof=1)

            #self.avg_kc_sep_correlation_mean                    = np.mean(np.asarray(avg_kc_sep_correlation))
            #self.avg_kc_sep_correlation_std                     = np.std(np.asarray(avg_kc_sep_correlation), ddof=1)

            self.integrated_biorientation_time_mean             = np.mean(np.asarray(integrated_biorientation_time))
            self.integrated_biorientation_time_std              = np.std(np.asarray(integrated_biorientation_time), ddof=1)

            self.fraction_integrated_biorientation_time_mean    = np.mean(np.asarray(fraction_integrated_biorientation_time))
            self.fraction_integrated_biorientation_time_std     = np.std(np.asarray(fraction_integrated_biorientation_time), ddof=1)

            self.fbiorient_mean                                 = np.mean(np.asarray(fbiorient))
            self.fbiorient_std                                  = np.std(np.asarray(fbiorient), ddof=1)

            self.segregation_success_mean                       = np.mean(np.asarray(segregation_success))
            self.segregation_success_std                        = np.std(np.asarray(segregation_success), ddof=1)

            #self.final_length_mean                              = np.mean(np.asarray(final_length).flatten())
            #self.final_length_std                               = np.std(np.asarray(final_length).flatten(), ddof=1)
            self.final_length_mean                              = np.mean(np.concatenate(final_length).ravel())
            self.final_length_std                               = np.std(np.concatenate(final_length).ravel(), ddof=1)

            #self.final_ipf_mean                                 = np.mean(np.asarray(final_ipf))
            #self.final_ipf_std                                  = np.std(np.asarray(final_ipf), ddof=1)
            self.final_ipf_mean                                 = np.mean(np.concatenate(final_ipf).ravel())
            self.final_ipf_std                                  = np.std(np.concatenate(final_ipf).ravel(), ddof=1)

            #self.final_occupancy_mean                           = np.mean(np.asarray(final_occupancy))
            #self.final_occupancy_std                            = np.std(np.asarray(final_ipf), ddof=1)
            self.final_occupancy_mean                           = np.mean(np.concatenate(final_occupancy).ravel())
            self.final_occupancy_std                            = np.std(np.concatenate(final_ipf).ravel(), ddof=1)
            self.total_occupancy_max                            = naf * nxsome * 2

            # Do the kmt lifetimes
            #self.kmt_lifetime_mean                              = np.average(kmt_lifetimes_mean, weights = kmt_lifetimes_std)
            if 'attachment_lifetimes' in sd.PostAnalysis.timedata:
                self.kmt_lifetime_mean                              = np.mean(np.concatenate(kmt_lifetimes).ravel())
                self.kmt_lifetime_std                               = np.std(np.concatenate(kmt_lifetimes).ravel(), ddof=1)
                print "number of lifetime measurements = {}".format(len(np.concatenate(kmt_lifetimes).ravel()))
                print "mean kmt lifetime = {}".format(self.kmt_lifetime_mean)
                print "std  kmt lifetime = {}".format(self.kmt_lifetime_std)
                print "std  err lifetime = {}".format(self.kmt_lifetime_std/np.sqrt(len(np.concatenate(kmt_lifetimes).ravel())))
                #print "std  err lifetime = {}".format(self.kmt_lifetime_std/np.sqrt(ngoodseeds))
            else:
                self.kmt_lifetime_mean                              = 0.0
                self.kmt_lifetime_std                               = 0.0
            #print "kmt_lifetimes_mean = {}".format(kmt_lifetimes_mean)
            #print "kmt_lifetimes_std  = {}".format(kmt_lifetimes_std)

        if not self.opts.datadir: self.opts.datadir = create_datadir(self.opts.workdir)
        try:
            title = (self.title +
                '\n Successful Fraction: {}'.format(self.sf))
        except:
            print "Don't know successful fraction. Criteria analysis might not have been run on sim {}".format(self.name)
            title = self.title
        if self.opts.workdir != self.sim_path:
            self.sim_datadir = create_datadir( self.opts.datadir, 
                                 datadir_name = "{}_data".format(self.name) )
        else:
            self.sim_datadir = self.opts.datadir

        with open(os.path.join(self.sim_datadir, 'fitness_final.yaml'), 'w') as stream:
            # Write all the correlation averages
            stream.write('max_correlation_combined_mean: {}\n'.format(self.max_correlation_combined_mean))
            stream.write('max_length_correlation_mean: {}\n'.format(self.max_length_correlation_mean))
            #stream.write('max_kc_distance_correlation_mean: {}\n'.format(self.max_kc_distance_correlation_mean))
            #stream.write('max_kc_sep_correlation_mean: {}\n'.format(self.max_kc_sep_correlation_mean))
            stream.write('avg_correlation_combined_mean: {}\n'.format(self.avg_correlation_combined_mean))
            stream.write('avg_length_correlation_mean: {}\n'.format(self.avg_length_correlation_mean))
            #stream.write('avg_kc_distance_correlation_mean: {}\n'.format(self.avg_kc_distance_correlation_mean))
            #stream.write('avg_kc_sep_correlation_mean: {}\n'.format(self.avg_kc_sep_correlation_mean))
            stream.write('integrated_biorientation_time_mean: {}\n'.format(self.integrated_biorientation_time_mean))
            stream.write('fraction_integrated_biorientation_time_mean: {}\n'.format(self.fraction_integrated_biorientation_time_mean))
            stream.write('fraction_normalized_biorientation_time_mean: {}\n'.format(self.fbiorient_mean))

            # All the standard deviations
            stream.write('max_correlation_combined_std: {}\n'.format(self.max_correlation_combined_std))
            stream.write('max_length_correlation_std: {}\n'.format(self.max_length_correlation_std))
            #stream.write('max_kc_distance_correlation_std: {}\n'.format(self.max_kc_distance_correlation_std))
            #stream.write('max_kc_sep_correlation_std: {}\n'.format(self.max_kc_sep_correlation_std))
            stream.write('avg_correlation_combined_std: {}\n'.format(self.avg_correlation_combined_std))
            stream.write('avg_length_correlation_std: {}\n'.format(self.avg_length_correlation_std))
            #stream.write('avg_kc_distance_correlation_std: {}\n'.format(self.avg_kc_distance_correlation_std))
            #stream.write('avg_kc_sep_correlation_std: {}\n'.format(self.avg_kc_sep_correlation_std))
            stream.write('integrated_biorientation_time_std: {}\n'.format(self.integrated_biorientation_time_std))
            stream.write('fraction_integrated_biorientation_time_std: {}\n'.format(self.fraction_integrated_biorientation_time_std))

            # EM measures and success
            #stream.write('em_fitness_log: {}\n'.format(self.emfitness_log))
            #stream.write('em_fitness_base: {}\n'.format(self.emfitness_base))
            #stream.write('success_fraction: {}\n'.format(self.sf))

            # Anaphase success measures
            stream.write('anaphase_segregation_success: {}\n'.format(self.segregation_success_mean))

            #self.final_fitness = self.emfitness_log + self.avg_correlation_combined_mean + 2.0*self.fbiorient_mean + 2.0*self.fraction_integrated_biorientation_time_mean
            self.final_fitness = self.avg_correlation_combined_mean + 2.0*self.fbiorient_mean + 2.0*self.fraction_integrated_biorientation_time_mean

            stream.write('FINAL_FITNESS: {}\n'.format(self.final_fitness))
            print 'FINAL_FITNESS: {}'.format(self.final_fitness)

    # Calculate the kinetochore distribution fitness
    def KCDistributionFitness(self):
        # Two calls, first, build the KC-SPB distance histograms
        self.SPBKCDistanceFitness()
        self.KCSpindle1DFitness()

    def SPBKCDistanceFitness(self):
        dist_stream_experiment = []
        # Grab the adjusted experiments
        for track in self.best_tracks:
            df = self.experiment_data_adj[track]
            kc0 = df['kc0_dist']
            kc1 = df['kc1_dist']
            kc2 = df['kc2_dist']
            kc3 = df['kc3_dist']
            kc4 = df['kc4_dist']
            kc5 = df['kc5_dist']
            dist_stream_experiment = np.append(dist_stream_experiment, [kc0, kc1, kc2, kc3, kc4, kc5])

        # Now get all of the model data
        dist_stream_model = []
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed:
                continue
            dist_model = sd.fitness.kc_dist_model
            dist_stream_model = np.append(dist_stream_model, dist_model)

        # Check the KS test for this setup!
        if len(dist_stream_experiment) > 0 and len(dist_stream_model) > 0:
            length_res = ks_2samp(dist_stream_model, dist_stream_experiment)
            print "KC Dist pvalue: {}".format(length_res)
            if length_res.pvalue <= 0.0:
                self.kc_spb_dist_pval = -10.0
            else:
                self.kc_spb_dist_pval = np.log10(length_res.pvalue)/100.0
        else:
            print "length pvalue: 0.0 (lengths not correct)"
            self.kc_spb_dist_pval = -10.0

    # 1d KC spindle histogram test
    def KCSpindle1DFitness(self):
        # Graph the 1d spindle location and ks test of the kinetochores
        kc_spindle_experiment = []
        for track in self.best_tracks:
            df = self.experiment_data_adj[track]
            lengths = df['length']
            kc0 = df['kc0_dist']
            kc1 = df['kc1_dist']
            kc2 = df['kc2_dist']
            kc3 = df['kc3_dist']
            kc4 = df['kc4_dist']
            kc5 = df['kc5_dist']

            # For experiments, assume that the KC is on the spindle, and that it is always the distance away from the SPB
            # That it is tagged to, so that we can generate a 1d length distribution
            for i in xrange(len(lengths)):
                kc0_1d = kc0[i] / lengths[i]
                kc1_1d = kc1[i] / lengths[i]
                kc2_1d = kc2[i] / lengths[i]
                kc3_1d = kc3[i] / lengths[i]
                kc4_1d = kc4[i] / lengths[i]
                kc5_1d = kc5[i] / lengths[i]

                kc_spindle_experiment = np.append(kc_spindle_experiment, [kc0_1d, kc1_1d, kc2_1d, kc3_1d, kc4_1d, kc5_1d])

        # Now do the model versions of this calculation
        kc_spindle_model = []
        for sd in self.seeds:
            if sd.PostAnalysis.bad_seed:
                continue
            kc_spindle_model = np.append(kc_spindle_model, sd.fitness.kc_spindle_model)

        # Check the KS test for this setup!
        if len(kc_spindle_experiment) > 0 and len(kc_spindle_model) > 0:
            length_res = ks_2samp(kc_spindle_model, kc_spindle_experiment)
            print "KC 1D pvalue: {}".format(length_res)
            if length_res.pvalue <= 0.0:
                self.kc_spindle1d_pval = -10.0
            else:
                self.kc_spindle1d_pval = np.log10(length_res.pvalue)/100.0
        else:
            print "length pvalue: 0.0 (lengths not correct)"
            self.kc_spindle1d_pval = -10.0



##########################################
if __name__ == "__main__":
    print "Not implemented yet"




