#!/usr/bin/env python
# In case of poor (Sh**y) commenting contact christopher.edelmaier@colorado.edu
# Basic
import sys, os, pdb
import gc
import argparse
import fnmatch
## Analysis
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib as mpl

sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'Lib'))
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'Spindle'))

from spindle_sim import SpindleSim
from sim_graph_funcs import *
from stylelib.ase1_styles import cp_spindle_stl

from scipy.stats import ks_2samp
from scipy.spatial.distance import euclidean
#from fastdtw import fastdtw
import scipy.io as sio

'''
Name: SingleRunChromosome.py
Description: Takes in a good WT plot, and a bad plot, and then does
the spindle length, interpolar fraction, and all attachment states
average over time for them.
Input:
Output:
'''

def parse_args():
    parser = argparse.ArgumentParser(prog='SingleRunChromosomes.py')
    # General options that are actually required
    parser.add_argument('-w', '--wildtype', required=True, type=str,
            help='WT sim')

    # Minimum of extra options to make this work
    parser.add_argument('--nopost', action='store_true',
            help="Do not use post analyis program to decrease time of analysis.")
    parser.add_argument('-F', '--fitness', type=str, default='WT_Cen2', nargs='?', const='WT_Cen2',
            help='Create fitness for spindle simulations.')
    parser.add_argument('-A', '--analyze', action='store_true',
            help='Analyze data from multiple simulations')
    parser.add_argument('--datadir', type=str,
            help='Name of the data directory in which all analyzed data files will be read/written. \
                    Also the directory where all the graphs will be placed when saved. \
                    Default is set to {workdir}/data/.')
    parser.add_argument('-d', '--workdir', type=str,
            help='Name of the working directory where simulation will be run.')

    opts = parser.parse_args()
    return opts


# Class definition
class SingleRunChromosomes(object):
    def __init__(self, opts):
        self.opts = opts
        self.cwd = os.getcwd()
        self.fig_pretty_size = (2,2)

        print "opts: {}".format(opts)

        self.ReadOpts()
        self.AnalyzeSims()

    def ReadOpts(self):
        if not self.opts.workdir:
            self.opts.workdir = os.path.abspath(self.cwd)
        elif not os.path.exists(self.opts.workdir):
            raise IOError( "Working directory {} does not exist.".format(
                self.opts.workdir) )
        else:
            self.opts.workdir = os.path.abspath(self.opts.workdir)

        self.wt_dir = os.path.abspath(self.opts.wildtype)
        self.wt_sim = SpindleSim(self.wt_dir, opts)

    def AnalyzeSims(self):
        self.wt_sim.Analyze()
        self.wt_sim.CalcSimSuccess()
        if self.opts.fitness:
            wt_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', 'Data', 'wt.mat')
            lstream_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', 'Data', 'lstream.mat')
            self.wt_sim.Fitness(wt_path, lstream_path)

    # Graph the avg spindle length for various delete strains (or just multiple strains)
    def GraphSpindleLength(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots(figsize=self.fig_pretty_size) 

        # Graph the specific versions of the input data averages
        self.GraphSpindleLengthSim(fig, ax, self.wt_sim)

        #ax.set_ylim(0.0, 2.1)
        ax.set_ylim(0.0, 3.1)

        ax.axhline(2.75, color = 'k', linestyle = '--', linewidth = 1.0)
        #ax.axhline(2.00, color = 'k', linestyle = '--', linewidth = 1.0)
        ax.set_xlabel('Time (min)')
        ax.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
        #ax.legend()
        fig.tight_layout() 
        plt.savefig('spindle_length_vs_t.pdf', dpi=fig.dpi)

    def GraphSpindleLengthSim(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[-1].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
        self.nmeasure = 4 * self.nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(sim.seeds)))
        time = sim.seeds[-1].time[:min_size]
        time_resampled = time[0::self.nmeasure]
        avg_std = np.zeros((len(sim.seeds), time_resampled.size))
        num_seeds = 0

        # Write the results to CSV files for the length vs time
        write_data = {'time': time_resampled}
        new_write_data = {}

        isd = 0
        for sd in sim.seeds:
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::self.nmeasure]
            spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
            spbsep_arr = [spbsep_dict[ts] for ts in time ]
            spbsep_arr = np.array(spbsep_arr)*uc['um'][1]
            spbsep_arr = np.mean(spbsep_arr.reshape(-1, self.nmeasure), axis=1)
            avg_std[isd,:] = spbsep_arr[:min_size]
            ax.plot(time_resampled, spbsep_arr, color = colors[isd], alpha=0.7)

            write_data['s{0:d}_spindlelength'.format(num_seeds)] = spbsep_arr
            new_write_data['s{0:d}_fractionsimultaneousbiorientation'.format(num_seeds)] = [sd.fitness.fraction_integrated_biorientation_time ]

            num_seeds += 1
            isd += 1

        avg = np.mean(avg_std, axis=0)
        avg_stddev = np.std(avg_std, axis=0, ddof=1)
        #ax.errorbar(x = time_resampled, y = avg, yerr = avg_stddev, color = 'k')

        dfw = pd.DataFrame(write_data)
        export_csv = dfw.to_csv(r'combined_spindlelength_vs_time.csv', index=None, header=True)

        dfw2 = pd.DataFrame(new_write_data)
        export_csv = dfw2.to_csv(r'fractionsimultaneousbiorientation.csv', index=None, header=True)


    # Graph the avg spindle length for various delete strains (or just multiple strains)
    def GraphSpindleIPF(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots() 

        # Graph the specific versions of the input data averages
        self.GraphSpindleIPFSim(fig, ax, self.wt_sim)

        ax.set_ylim(0.0, 1.1)

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel(r'Interpolar Fraction')
        fig.tight_layout() 
        plt.savefig('spindle_ipf_vs_t.pdf', dpi=fig.dpi)

    def GraphSpindleIPFSim(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[-1].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
        self.nmeasure = 4 * self.nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(sim.seeds)))
        time = sim.seeds[-1].time[:min_size]
        time_resampled = time[0::self.nmeasure]
        avg_std = np.zeros((len(sim.seeds), time_resampled.size))
        num_seeds = 0

        isd = 0

        for sd in sim.seeds:
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::self.nmeasure]
            ipf_dict = sd.PostAnalysis.timedata['interpolar_fraction']
            ipf_arr = [ipf_dict[ts] for ts in time ]
            ipf_arr = np.array(ipf_arr)
            ipf_arr = np.mean(ipf_arr.reshape(-1, self.nmeasure), axis=1)
            avg_std[isd,:] = ipf_arr[:min_size]
            ax.plot(time_resampled, ipf_arr, color = colors[isd])
            num_seeds += 1
            isd += 1

        avg = np.mean(avg_std, axis=0)
        avg_stddev = np.std(avg_std, axis=0, ddof=1)
        ax.errorbar(x = time_resampled, y = avg, yerr = avg_stddev, color = 'k')

    # Graph the merotelic attachment state for the simulation over time
    def GraphSpindleAttachStates(self):
        plt.style.use(cp_spindle_stl)
        attachment_types = ['unattached',
                            'monotelic',
                            'merotelic',
                            'syntelic',
                            'amphitelic']
        for atype in attachment_types:
            fig, ax = plt.subplots()

            # Graph the specific versions of the input data averages
            self.GraphSpindleStateSim(fig, ax, self.wt_sim, atype) 

            ax.set_ylim(0.0, self.nchromo + 0.1)

            ax.set_xlabel(r'Time (min)')
            ax.set_ylabel('N {}'.format(atype))
            fig.tight_layout() 
            plt.savefig('spindle_{}_vs_t.pdf'.format(atype), dpi=fig.dpi)
            plt.close()

    def GraphSpindleStateSim(self, fig, ax, sim, atype):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[-1].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
        self.nmeasure = 4 * self.nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(sim.seeds)))
        print colors
        time = sim.seeds[-1].time[:min_size]
        time_resampled = time[0::self.nmeasure]
        avg_std = np.zeros((len(sim.seeds), time_resampled.size))
        num_seeds = 0

        isd = 0

        for sd in sim.seeds:
            #print sd.succ_info_dict
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::self.nmeasure]
            attach_dict = sd.PostAnalysis.timedata['kc_atypes']
            attach_arr = [attach_dict[ts] for ts in time]
            nchromo = len(attach_arr[0])
            #atype = 'merotelic'
            if atype == 'unattached':
                atypeint = 0
            elif atype == 'monotelic':
                atypeint = 1
            elif atype == 'merotelic':
                atypeint = 2
            elif atype == 'syntelic':
                atypeint = 3
            elif atype == 'amphitelic':
                atypeint = 4

            plot_arr = np.zeros(len(attach_arr))
            self.nchromo = nchromo
            
            # Regenerate the merotelic numbers
            for x in xrange(len(attach_arr)):
                for ic in xrange(nchromo):
                    attach = attach_arr[x][ic]
                    if attach == atypeint:
                        plot_arr[x] += 1.0
            plot_arr = np.mean(plot_arr.reshape(-1, self.nmeasure), axis=1)
            avg_std[isd,:] = plot_arr[:min_size]
            ax.plot(time_resampled, plot_arr, color = colors[isd])
            num_seeds += 1
            isd += 1

        avg = np.mean(avg_std, axis=0)
        avg_stddev = np.std(avg_std, axis=0, ddof=1)
        ax.errorbar(x = time_resampled, y = avg, yerr = avg_stddev, color = 'k')

    # Graph the spindle occupany of KC-MT binding sites
    def GraphOccupancyBindingSites(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots()

        # Graph the specific versions of the input data averages
        self.GraphOccupancySim(fig, ax, self.wt_sim)

        ax.set_ylim(0.0, 19.0)

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel('Occupancy')
        fig.tight_layout() 
        plt.savefig('spindle_occupancy_vs_t.pdf', dpi=fig.dpi)
        plt.close()

    def GraphOccupancySim(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[-1].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
        self.nmeasure = 4 * self.nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(sim.seeds)))
        time = sim.seeds[-1].time[:min_size]
        time_resampled = time[0::self.nmeasure]
        avg_std = np.zeros((len(sim.seeds), time_resampled.size))
        num_seeds = 0

        isd = 0

        for sd in sim.seeds:
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::self.nmeasure]
            occupancy_dict = sd.PostAnalysis.timedata['kc_occupancy']
            occupancy_arr = np.array([occupancy_dict[ts] for ts in time])
            occupancy_arr = np.mean(occupancy_arr.reshape(-1, self.nmeasure), axis=1)
            avg_std[isd,:] = occupancy_arr[:min_size]
            ax.plot(time_resampled, occupancy_arr, color = colors[isd])
            num_seeds += 1
            isd += 1

        avg = np.mean(avg_std, axis=0)
        avg_stddev = np.std(avg_std, axis=0, ddof=1)
        ax.errorbar(x = time_resampled, y = avg, yerr = avg_stddev, color = 'k')

    def GraphSimpleGaussians(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots()
        def my_gaussian(x, mean, amplitude, standard_deviation):
            return amplitude * np.exp( -0.5 * ((x-mean) / standard_deviation) **2)

        x = np.linspace(-50, 50, 1000)
        y = my_gaussian(x, 0.0, 0.5, 25.0)

        plt.plot(x, y, linewidth = 10.0)
        #plt.axis('off')
        fig.tight_layout() 
        plt.savefig('simple_gaussian.pdf', dpi=fig.dpi)
        plt.close()
    
    # Grab the final spindle forces for this simulation point
    def GraphSpindleForces(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots(figsize=(2.5,2))
        fig2, ax2 = plt.subplots(figsize=(2.5,2))

        # Graph the points
        #[xlink_force, af0_force, af1_force, x_force] = self.GraphSpindleForcesSim(fig, ax, self.wt_sim)
        [xlink_force, af_force, interkcforce] = self.GraphSpindleForcesSim(fig, ax, fig2, ax2, self.wt_sim)

        ax.set_ylim(-100.0, 100.0)

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel('Force (pN)')
        #ax.legend()
        fig.tight_layout() 
        fig.savefig('spindle_forces_vs_t.pdf', dpi=fig.dpi)

        ax2.set_xlabel(r'Time (min)')
        ax2.set_ylabel('Force (pN)')
        fig2.tight_layout()
        fig2.savefig('spindle_forces_perp_vs_t.pdf', dpi=fig2.dpi)

        plt.close()

        # Set up the next plot
        fig, ax = plt.subplots()

        avg_xlink_force = np.mean(xlink_force)
        std_xlink_force = np.std(xlink_force, ddof=1)/np.sqrt(12)
        avg_af_force = np.mean(af_force)
        std_af_force = np.std(af_force, ddof=1)/np.sqrt(12)
        avg_interkc_force = np.mean(interkcforce)
        std_interkc_force = np.std(interkcforce, ddof=1)/np.sqrt(12)

        ind = np.arange(2)
        print "Average xlink force = {} +- {}".format(avg_xlink_force, std_xlink_force)
        print "Average af force = {} +- {}".format(avg_af_force, std_af_force)
        print "Average interkc force = {} +- {}".format(avg_interkc_force, std_interkc_force)
        avg_forces = [avg_xlink_force, avg_af_force]
        err_forces = [std_xlink_force, std_af_force]
        ax.bar(ind, avg_forces, 0.35, yerr=err_forces)
        ax.set_ylim(0.0, 100.0)
        plt.xticks(ind, ('Motors/Crosslinkers', 'AFs'))
        ax.set_ylabel('Avg. Pole 1 Axis Force (pN)')

        fig.tight_layout()
        plt.savefig('spindle_force_late.pdf', dpi=fig.dpi)
        plt.close()

        # Now, do the force on the spindle as a binned spindle length plot
        print self.length_force_dataframe
        fig, ax = plt.subplots(figsize=(2.5, 2))

        self.GraphSpindleLengthForce(fig, ax, self.wt_sim)

        ax.set_ylim(-100.0, 100.0)
        ax.set_xlabel('Spindle length ($\mu$m)')
        ax.set_ylabel('Force (pN)')
        #ax.legend()

        fig.tight_layout()
        plt.savefig('spindle_force_vs_length.pdf', dpi=fig.dpi)
        plt.close()

    # Do the axis forces for the simulation, but take the absoluate value to make all of them positive.
    def GraphSpindleForcesSim(self, fig, ax, fig2, ax2, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[-1].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
        self.nmeasure = 4 * self.nmeasure

        colors = mpl.cm.rainbow(np.linspace(0,1,len(sim.seeds)))
        time = sim.seeds[-1].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        avg_std_xlink = np.zeros((len(sim.seeds), time_resampled.size))
        avg_std_af_0 = np.zeros((len(sim.seeds), time_resampled.size))
        avg_std_af_1 = np.zeros((len(sim.seeds), time_resampled.size))

        avg_xlink_axis = np.zeros((len(sim.seeds), time_resampled.size))
        avg_xlink_0_axis = np.zeros((len(sim.seeds), time_resampled.size))
        avg_xlink_1_axis = np.zeros((len(sim.seeds), time_resampled.size))
        avg_xlink_2_axis = np.zeros((len(sim.seeds), time_resampled.size))
        avg_af_axis = np.zeros((len(sim.seeds), time_resampled.size))

        # Try to get the perpendicular information too
        avg_xlink_0_perp = np.zeros((len(sim.seeds), time_resampled.size))
        avg_xlink_1_perp = np.zeros((len(sim.seeds), time_resampled.size))
        avg_xlink_2_perp = np.zeros((len(sim.seeds), time_resampled.size))
        avg_af_perp = np.zeros((len(sim.seeds), time_resampled.size))

        avg_interkinetochore = np.zeros((len(sim.seeds), time_resampled.size))

        num_seeds = 0

        # Long time behavior
        average_xlink_force = []
        average_af_force = []
        average_af0_force = []
        average_af1_force = []
        average_x_force = []

        # Long time chromosome force magnitude
        average_chromosome_force = []

        isd = 0

        for sd in sim.seeds:
            time = sd.PostAnalysis.timedata['time']
            time_resampled = time[0::self.nmeasure]

            # We want the spindle lengths for binning the appropriate force distribution
            spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
            spindle_length_arr = np.array([spbsep_dict[ts] for ts in time])

            # The forces are stored as a dictionary, so get each contribution
            forces_dict_0 = sd.PostAnalysis.timedata['pole_forces'][0]

            # The first entry is the time, then it is a dictionary of the different forces
            forces_arr_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][0] for ts in time])
            forces_arr_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][1] for ts in time])
            forces_arr_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][2] for ts in time])
            forces_axis_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][0] for ts in time])
            forces_axis_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][1] for ts in time])
            forces_axis_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][2] for ts in time])
            # and the total Xlink force
            forces_axis_0_xlink = np.array([forces_dict_0[ts]['xlink_axis'] for ts in time])

            # Grab the AF forces too
            forces_arr_0_af = np.array([forces_dict_0[ts]['af_mag'] for ts in time])
            forces_axis_0_af = np.array([forces_dict_0[ts]['af_axis'] for ts in time])

            # What about the inter-kinetochore force?
            forces_interkinetochore = np.array([forces_dict_0[ts]['chromosome_mag'] for ts in time])

            # Redo the mean average stuff
            forces_arr_0_subtype_0 = np.mean(forces_arr_0_subtype_0.reshape(-1, self.nmeasure), axis=1)
            forces_arr_0_subtype_1 = np.mean(forces_arr_0_subtype_1.reshape(-1, self.nmeasure), axis=1)
            forces_arr_0_subtype_2 = np.mean(forces_arr_0_subtype_2.reshape(-1, self.nmeasure), axis=1)
            forces_arr_0_af = np.mean(forces_arr_0_af.reshape(-1, self.nmeasure), axis=1)

            forces_axis_0_subtype_0 = np.mean(forces_axis_0_subtype_0.reshape(-1, self.nmeasure), axis=1)
            forces_axis_0_subtype_1 = np.mean(forces_axis_0_subtype_1.reshape(-1, self.nmeasure), axis=1)
            forces_axis_0_subtype_2 = np.mean(forces_axis_0_subtype_2.reshape(-1, self.nmeasure), axis=1)
            forces_axis_0_xlink = np.mean(forces_axis_0_xlink.reshape(-1, self.nmeasure), axis=1)
            forces_axis_0_af = np.mean(forces_axis_0_af.reshape(-1, self.nmeasure), axis=1)

            forces_interkinetochore = np.mean(forces_interkinetochore.reshape(-1, self.nmeasure), axis=1)

            # We should be able to construct the perpendicular force by taking the sqrt(total2 - axis2)
            forces_perp_0_subtype_0 = np.sqrt(np.square(forces_arr_0_subtype_0) - np.square(forces_axis_0_subtype_0))
            forces_perp_0_subtype_1 = np.sqrt(np.square(forces_arr_0_subtype_1) - np.square(forces_axis_0_subtype_1))
            forces_perp_0_subtype_2 = np.sqrt(np.square(forces_arr_0_subtype_2) - np.square(forces_axis_0_subtype_2))
            forces_perp_0_af = np.sqrt(np.square(forces_arr_0_af) - np.square(forces_axis_0_af))

            spindle_length_arr = np.mean(spindle_length_arr.reshape(-1, self.nmeasure), axis=1) * uc['um'][1]

            avg_xlink_axis[isd,:] = forces_axis_0_xlink[:min_size]
            avg_xlink_0_axis[isd,:] = forces_axis_0_subtype_0[:min_size]
            avg_xlink_1_axis[isd,:] = forces_axis_0_subtype_1[:min_size]
            avg_xlink_2_axis[isd,:] = forces_axis_0_subtype_2[:min_size]
            avg_af_axis[isd,:] = forces_axis_0_af[:min_size]

            avg_interkinetochore[isd,:] = forces_interkinetochore[:min_size]

            # Perpendicular
            avg_xlink_0_perp[isd,:] = forces_perp_0_subtype_0[:min_size]
            avg_xlink_1_perp[isd,:] = forces_perp_0_subtype_1[:min_size]
            avg_xlink_2_perp[isd,:] = forces_perp_0_subtype_2[:min_size]
            avg_af_perp[isd,:] = forces_perp_0_af[:min_size]

            # Grab the longtime behavior of these, make sure to take absolute values, as they are along the spindle axis
            long_time_xlink = np.absolute(forces_axis_0_xlink[time_resampled > 10.0])
            long_time_af = np.absolute(forces_axis_0_af[time_resampled > 10.0])
            long_time_interkinetochore = np.absolute(forces_interkinetochore[time_resampled > 10.0])

            average_xlink_force += [long_time_xlink[:]]
            average_af_force += [long_time_af[:]]
            average_chromosome_force += [long_time_interkinetochore[:]]

            # Load the information of the spindle length and forces into a self consistent dictionary
            if spindle_length_arr.shape[0] != forces_axis_0_af.shape[0]:
                print "moo moo bad"
                sys.exit(1)

            # Create a dataframe of the information to bin later
            # Convert with the minus sign here!
            data = {'spindle_length': spindle_length_arr, 'xlink_type0_axis_force': -1.0 * forces_axis_0_subtype_0,
                    'xlink_type1_axis_force': -1.0 * forces_axis_0_subtype_1, 'xlink_type2_axis_force': -1.0 * forces_axis_0_subtype_2,
                    'af_axis_force': -1.0 * forces_axis_0_af, 'time': time_resampled}
            df = pd.DataFrame(data)
            #print df

            # Load the first one
            if num_seeds == 0:
                self.length_force_dataframe = df
            else:
                self.length_force_dataframe = self.length_force_dataframe.append(df, ignore_index = True)

            num_seeds += 1
            isd += 1

        avg_axis_xlink = np.mean(avg_xlink_axis, axis=0)
        avg_axis_xlink_0 = np.mean(avg_xlink_0_axis, axis=0)
        avg_axis_xlink_1 = np.mean(avg_xlink_1_axis, axis=0)
        avg_axis_xlink_2 = np.mean(avg_xlink_2_axis, axis=0)
        avg_axis_af = np.mean(avg_af_axis, axis=0)

        stddev_axis_xlink = np.std(avg_xlink_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_0 = np.std(avg_xlink_0_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_1 = np.std(avg_xlink_1_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_xlink_2 = np.std(avg_xlink_2_axis, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_axis_af = np.std(avg_af_axis, axis=0, ddof=1) / np.sqrt(num_seeds)

        # Reverse the signs for pole0, make the inward force negative
        ax.errorbar(x = time_resampled[::2], y = -1.0 * avg_axis_xlink_0[::2], yerr = stddev_axis_xlink_0[::2], color = 'r', label = 'K5')
        ax.errorbar(x = time_resampled[::2], y = -1.0 * avg_axis_xlink_1[::2], yerr = stddev_axis_xlink_1[::2], color = 'b', label = 'K14')
        ax.errorbar(x = time_resampled[::2], y = -1.0 * avg_axis_xlink_2[::2], yerr = stddev_axis_xlink_2[::2], color = 'k', label = 'XL')
        ax.errorbar(x = time_resampled[::2], y = -1.0 * avg_axis_af[::2], yerr = stddev_axis_af[::2], color = 'c', label = 'AFs')

        avg_perp_xlink_0 = np.mean(avg_xlink_0_perp, axis=0)
        avg_perp_xlink_1 = np.mean(avg_xlink_1_perp, axis=0)
        avg_perp_xlink_2 = np.mean(avg_xlink_2_perp, axis=0)
        avg_perp_af = np.mean(avg_af_perp, axis=0)
        stddev_perp_xlink_0 = np.std(avg_xlink_0_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_xlink_1 = np.std(avg_xlink_1_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_xlink_2 = np.std(avg_xlink_2_perp, axis=0, ddof=1) / np.sqrt(num_seeds)
        stddev_perp_af = np.std(avg_af_perp, axis=0, ddof=1) / np.sqrt(num_seeds)

        # Plot the absolute value of this
        ax2.errorbar(x = time_resampled[::2], y = avg_perp_xlink_0[::2], yerr = stddev_perp_xlink_0[::2], color = 'r', label = 'K5')
        ax2.errorbar(x = time_resampled[::2], y = avg_perp_xlink_1[::2], yerr = stddev_perp_xlink_1[::2], color = 'b', label = 'K14')
        ax2.errorbar(x = time_resampled[::2], y = avg_perp_xlink_2[::2], yerr = stddev_perp_xlink_2[::2], color = 'k', label = 'XL')
        ax2.errorbar(x = time_resampled[::2], y = avg_perp_af[::2], yerr = stddev_perp_af[::2], color = 'c', label = 'AFs')

        # Write the df to CSV
        export_csv = self.length_force_dataframe.to_csv(r'spindleforcemeasurements.csv', index=None, header=True)

        return [np.asarray(average_xlink_force), np.asarray(average_af_force), np.asarray(average_chromosome_force)]

    # Graph the spindle length and force binned
    def GraphSpindleLengthForce(self, fig, ax, sim):
        bins = np.linspace(0.0, 2.75, 11)
        bin_mids = np.linspace(0.0, 2.475, 10) + 0.275/2. # FIXME: Off by 1
        df = self.length_force_dataframe
        df['bins'] = pd.cut(df['spindle_length'], bins)
        df_means = df.groupby('bins').mean()
        df_std = df.groupby('bins').std() / np.sqrt(12)

        # Plot this!
        ax.scatter(x = bin_mids, y = df_means['af_axis_force'], zorder=100, s=50, color = 'c', label = 'AFs', marker = 's', facecolors='none')
        ax.scatter(x = bin_mids, y = df_means['xlink_type0_axis_force'], zorder=100, s=50, color = 'r', label = 'K5', marker = 's', facecolors='none')
        ax.scatter(x = bin_mids, y = df_means['xlink_type1_axis_force'], zorder=100, s=50, color = 'b', label = 'K14', marker = 's', facecolors='none')
        ax.scatter(x = bin_mids, y = df_means['xlink_type2_axis_force'], zorder=100, s=50, color = 'k', label = 'XL', marker = 's', facecolors='none')

        ax.errorbar(x = bin_mids, y = df_means['af_axis_force'], yerr = df_std['af_axis_force'], ecolor = 'c', elinewidth = 2,
                capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        ax.errorbar(x = bin_mids, y = df_means['xlink_type0_axis_force'], yerr = df_std['xlink_type0_axis_force'], ecolor = 'r', elinewidth = 2,
                capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        ax.errorbar(x = bin_mids, y = df_means['xlink_type1_axis_force'], yerr = df_std['xlink_type1_axis_force'], ecolor = 'b', elinewidth = 2,
                capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        ax.errorbar(x = bin_mids, y = df_means['xlink_type2_axis_force'], yerr = df_std['xlink_type2_axis_force'], ecolor = 'k', elinewidth = 2,
                capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)



    def GraphKinetochoreInformation(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots()
        # Just make a hsitogram of all of the distances of the SPB-KC distances
        sim = self.wt_sim
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        all_distances = np.zeros(1)
        for sd in sim.seeds:
            time = sd.time[:min_size]
            dist_dict = sd.PostAnalysis.timedata['kc_distance']
            dist_arr = np.array([dist_dict[ts] for ts in time])

            # Exclude the first 1 minute of time taking
            dist_arr = dist_arr[time > 1.0]
            print "excluding first times = {}".format(len(dist_arr))
            dist0 = np.zeros(len(dist_arr))
            dist1 = np.zeros(len(dist_arr))

            for ic in xrange(3):
                for x in xrange(len(dist_arr)):
                    dist0[x] = dist_arr[x][2*ic]
                    dist1[x] = dist_arr[x][2*ic+1]

            all_distances = np.append(all_distances, dist0)
            all_distances = np.append(all_distances, dist1)
        print "all distances = {}".format(all_distances)
        print "len = {}".format(len(all_distances))

        hist, bin_edges = np.histogram(all_distances, bins=20, range=(0.0, 2.75), density=True)
        print "hist = {}".format(hist)
        print "bin_edges = {}".format(bin_edges)

        plt.hist(all_distances, bins=20, range=(0.0, 2.75), density=True)
        ax.set_xlabel('SPB-KC distance ($\mu$m)')
        ax.set_ylabel('Probability density')
        ax.set_ylim(0.0, 3.0)
        fig.tight_layout()
        plt.savefig('spb_kc_dist_histogram.pdf', dpi=fig.dpi)
        plt.close()



################
if __name__ == "__main__":
    opts = parse_args()
    x = SingleRunChromosomes(opts)
    x.GraphSpindleLength()
    x.GraphSpindleIPF()
    x.GraphSpindleAttachStates()
    x.GraphOccupancyBindingSites()

    # Grab the final forces on the spindle
    x.GraphSpindleForces()

    # Because I can use it here, draw some simple gaussians
    x.GraphSimpleGaussians()

    x.GraphKinetochoreInformation()
