#!/usr/bin/env python
# In case of poor (Sh**y) commenting contact christopher.edelmaier@colorado.edu
# Basic
import sys, os, pdb
import gc
import argparse
import fnmatch
## Analysis
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib as mpl

sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'Lib'))
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'Spindle'))

from spindle_sim import SpindleSim
from sim_graph_funcs import *
from stylelib.ase1_styles import ase1_runs_stl
from stylelib.ase1_styles import cp_spindle_stl

from scipy.stats import ks_2samp
from scipy.spatial.distance import euclidean
from fastdtw import fastdtw
import scipy.io as sio

'''
Name: SingleSeedChromosomes.py
Description: Takes in a good WT plot, and a bad plot, and then does
the spindle length, interpolar fraction, and all attachment states
average over time for them.
Input:
Output:
'''

def parse_args():
    parser = argparse.ArgumentParser(prog='MotorDeleteGraph.py')
    # General options that are actually required
    parser.add_argument('-w', '--wildtype', required=True, type=str,
            help='WT sim')
    parser.add_argument('-s', '--seed', required=True, type=int,
            help='Seed')

    # Minimum of extra options to make this work
    parser.add_argument('--nopost', action='store_true',
            help="Do not use post analyis program to decrease time of analysis.")
    parser.add_argument('-F', '--fitness', action='store_true',
            help='Create fitness for spindle simulations.')
    parser.add_argument('-A', '--analyze', action='store_true',
            help='Analyze data from multiple simulations')
    parser.add_argument('--datadir', type=str,
            help='Name of the data directory in which all analyzed data files will be read/written. \
                    Also the directory where all the graphs will be placed when saved. \
                    Default is set to {workdir}/data/.')
    parser.add_argument('-d', '--workdir', type=str,
            help='Name of the working directory where simulation will be run.')
    parser.add_argument('--timecutafter', type=float,
            help='Time after which to cut the graphs off')
    parser.add_argument('--timecutbefore', type=float,
            help='Time to cut on the beginning')

    opts = parser.parse_args()
    return opts


# Class definition
class SingleSeedChromosomes(object):
    def __init__(self, opts):
        self.opts = opts
        self.cwd = os.getcwd()
        self.fig_pretty_size = (2,2.5)

        print "opts: {}".format(opts)

        self.ReadOpts()
        self.AnalyzeSims()

    def ReadOpts(self):
        if not self.opts.workdir:
            self.opts.workdir = os.path.abspath(self.cwd)
        elif not os.path.exists(self.opts.workdir):
            raise IOError( "Working directory {} does not exist.".format(
                self.opts.workdir) )
        else:
            self.opts.workdir = os.path.abspath(self.opts.workdir)

        self.wt_dir = os.path.abspath(self.opts.wildtype)
        self.wt_sim = SpindleSim(self.wt_dir, opts)
        self.s = np.int(self.opts.seed)

    def AnalyzeSims(self):
        self.wt_sim.Analyze()
        self.wt_sim.CalcSimSuccess()
        if self.opts.fitness:
            wt_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', 'Data', 'wt.mat')
            lstream_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', 'Data', 'lstream.mat')
            self.wt_sim.Fitness(wt_path, lstream_path)

        # If the seed we are doing includes anaphase, then do it properly
        sd = self.wt_sim.seeds[self.s]
        if sd.PostAnalysis.do_anaphase:
            self.anaphase_b_time = sd.PostAnalysis.anaphase_onset + 2.0

    # Graph the avg spindle length for various delete strains (or just multiple strains)
    def GraphSpindleLength(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots(figsize=self.fig_pretty_size) 

        # Graph the specific versions of the input data averages
        self.GraphSpindleLengthSeed(fig, ax, self.wt_sim)

        ax.set_ylim(0.0, 3.1)
        if self.opts.timecutafter:
            ax.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)

        # Draw horizontal line for the maximum value of the nucleus
        ax.axhline(2.75, color = 'k', linestyle = '--', linewidth = 1.0)
        ax.set_xlabel('Time (min)')
        ax.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
        #ax.legend()
        fig.tight_layout() 
        plt.savefig('spindle_length_vs_t_s{}.pdf'.format(self.s), dpi=fig.dpi)

    # Overwrite the seeds for just one
    def GraphSpindleLengthSeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
        spbsep_arr = [spbsep_dict[ts] for ts in time ]
        spbsep_arr = np.array(spbsep_arr)*uc['um'][1]
        if sd.PostAnalysis.do_anaphase:
            spbsep_arr_adj = spbsep_arr[time < self.anaphase_b_time]
            adjusted_end = spbsep_arr_adj.shape[0] + (self.nmeasure - spbsep_arr_adj.shape[0] % self.nmeasure)
            spbsep_arr = spbsep_arr[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        # Do a check to make sure that nmeasure goes evenly into the array length,
        # otherwise, increment until we can do it.
        spbsep_arr = np.mean(spbsep_arr.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, spbsep_arr, color = 'b')

        # Write the results to CSV files for the length vs time
        write_data = {'time': time_resampled,
                      'spindle_length': spbsep_arr}
        dfw = pd.DataFrame(write_data, columns = ['time', 'spindle_length'])
        export_csv = dfw.to_csv(r'spindlelength_vs_time.csv', index=None, header=True)

        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

    # Graph the avg spindle length for various delete strains (or just multiple strains)
    def GraphSpindleIPF(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots() 

        # Graph the specific versions of the input data averages
        self.GraphSpindleIPFSeed(fig, ax, self.wt_sim)

        ax.set_ylim(0.0, 1.1)
        if self.opts.timecutafter:
            ax.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel(r'Interpolar Fraction')
        fig.tight_layout() 
        plt.savefig('spindle_ipf_vs_t_s{}.pdf'.format(self.s), dpi=fig.dpi)

    def GraphSpindleIPFSeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        time = sd.PostAnalysis.timedata['time']
        time_resampled = time[0::self.nmeasure]
        ipf_dict = sd.PostAnalysis.timedata['interpolar_fraction']
        ipf_arr = [ipf_dict[ts] for ts in time ]
        ipf_arr = np.array(ipf_arr)
        if sd.PostAnalysis.do_anaphase:
            ipf_arr_adj = ipf_arr[time < self.anaphase_b_time]
            adjusted_end = ipf_arr_adj.shape[0] + (self.nmeasure - ipf_arr_adj.shape[0] % self.nmeasure)
            ipf_arr = ipf_arr[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        ipf_arr = np.mean(ipf_arr.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, ipf_arr, color = 'b')

        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

    # Graph the merotelic attachment state for the simulation over time
    def GraphSpindleAttachStates(self):
        plt.style.use(cp_spindle_stl)
        attachment_types = ['unattached',
                            'monotelic',
                            'merotelic',
                            'syntelic',
                            'amphitelic',
                            'combined']
        for atype in attachment_types:
            fig, ax = plt.subplots()

            # Graph the specific versions of the input data averages
            self.GraphSpindleStateSeed(fig, ax, self.wt_sim, atype) 

            ax.set_ylim(0.0, 3.1)
            if self.opts.timecutafter:
                ax.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)

            ax.set_xlabel(r'Time (min)')
            ax.set_ylabel('N {}'.format(atype))
            fig.tight_layout() 
            plt.savefig('spindle_{}_vs_t_s{}.pdf'.format(atype, self.s), dpi=fig.dpi)
            plt.close()

    def GraphSpindleStateSeed(self, fig, ax, sim, atype):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]
        avg_std = np.zeros((len(sim.seeds), time_resampled.size))

        sd = sim.seeds[self.s]
        time = sd.PostAnalysis.timedata['time']
        time_resampled = time[0::self.nmeasure]
        attach_dict = sd.PostAnalysis.timedata['kc_atypes']
        attach_arr = [attach_dict[ts] for ts in time]
        nchromo = len(attach_arr[0])
        #atype = 'merotelic'
        if atype == 'unattached':
            atypeint = [0]
        elif atype == 'monotelic':
            atypeint = [1]
        elif atype == 'merotelic':
            atypeint = [2]
        elif atype == 'syntelic':
            atypeint = [3]
        elif atype == 'amphitelic':
            atypeint = [4]
        elif atype == 'combined':
            atypeint = [0,1,3]

        plot_arr = np.zeros(len(attach_arr))
        
        # Regenerate the merotelic numbers
        for x in xrange(len(attach_arr)):
            for ic in xrange(nchromo):
                attach = attach_arr[x][ic]
                if attach in atypeint:
                    plot_arr[x] += 1.0
        if sd.PostAnalysis.do_anaphase:
            plot_arr_adj = plot_arr[time < self.anaphase_b_time]
            adjusted_end = plot_arr_adj.shape[0] + (self.nmeasure - plot_arr_adj.shape[0] % self.nmeasure)
            plot_arr = plot_arr[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        plot_arr = np.mean(plot_arr.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, plot_arr, color = 'b')
        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

    # Graph the spindle occupany of KC-MT binding sites
    def GraphOccupancyBindingSites(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots()

        # Graph the specific versions of the input data averages
        self.GraphOccupancySeed(fig, ax, self.wt_sim)

        ax.set_ylim(0.0, 19.0)
        if self.opts.timecutafter:
            ax.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel('Occupancy')
        fig.tight_layout() 
        plt.savefig('spindle_occupancy_vs_t_s{}.pdf'.format(self.s), dpi=fig.dpi)
        plt.close()

    def GraphOccupancySeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        time = sd.PostAnalysis.timedata['time']
        time_resampled = time[0::self.nmeasure]
        occupancy_dict = sd.PostAnalysis.timedata['kc_occupancy']
        occupancy_arr = np.array([occupancy_dict[ts] for ts in time])
        if sd.PostAnalysis.do_anaphase:
            occupancy_arr_adj = occupancy_arr[time < self.anaphase_b_time]
            adjusted_end = occupancy_arr_adj.shape[0] + (self.nmeasure - occupancy_arr_adj.shape[0] % self.nmeasure)
            occupancy_arr = occupancy_arr[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        occupancy_arr = np.mean(occupancy_arr.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, occupancy_arr, color = 'b')
        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

    def GraphKinetochorePositions(self):
        plt.style.use(cp_spindle_stl)
        fig1, ax1 = plt.subplots(figsize=(2.5, 2.5))

        # Graph the specific versions of the input data averages
        self.GraphSPBKinetochoresSeed(fig1, ax1, self.wt_sim)
        ax1.set_ylim(0.0, 3.1)
        if self.opts.timecutafter:
            ax1.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)
        ax1.set_xlabel('Time (min)')
        ax1.set_ylabel('Spindle pole body-kinetochore\ndistance ($\mu$m)')
        fig1.tight_layout() 
        plt.savefig('spindle_spbkc_vs_t_s{}.pdf'.format(self.s), dpi=fig1.dpi)
        plt.close()

        fig2, ax2 = plt.subplots(figsize=(2.5, 2.5))
        self.GraphKCKCSeed(fig2, ax2, self.wt_sim)
        if self.wt_sim.seeds[self.s].PostAnalysis.do_anaphase:
            ax2.set_ylim(0.0, 3.1)
        else:
            ax2.set_ylim(0.0, 1.0)
        if self.opts.timecutafter:
            ax2.set_xlim(self.opts.timecutbefore, self.opts.timecutafter)
        ax2.set_xlabel(r'Time (min)')
        ax2.set_ylabel('Kinetochore separation ($\mu$m)')
        fig2.tight_layout() 
        plt.savefig('spindle_kckc_vs_t_s{}.pdf'.format(self.s), dpi=fig2.dpi)
        plt.close()

    # Pass in both of the axes, since we need 2 graphs
    def GraphSPBKinetochoresSeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        # Generate the images from the kinetochores, do it by anaphase as well
        dist_dict = sd.PostAnalysis.timedata['kc_distance']
        dist_arr = np.array([dist_dict[ts] for ts in time])
        nchromo = len(dist_arr[0]) / 2

        # To replicate Cen2 type results, just take chromosome 1
        dist0 = np.zeros(len(dist_arr))
        dist1 = np.zeros(len(dist_arr))

        ic = 1
        for x in xrange(len(dist_arr)):
            dist0[x] = dist_arr[x][2*ic]
            dist1[x] = dist_arr[x][2*ic+1]
        if sd.PostAnalysis.do_anaphase:
            dist0_arr_adj = dist0[time < self.anaphase_b_time]
            adjusted_end = dist0_arr_adj.shape[0] + (self.nmeasure - dist0_arr_adj.shape[0] % self.nmeasure)
            dist0 = dist0[0:adjusted_end]
            dist1 = dist1[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        dist0_resampled = np.mean(dist0.reshape(-1, self.nmeasure), axis=1)
        dist1_resampled = np.mean(dist1.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, dist0_resampled, color = 'b', linestyle = '-')
        ax.plot(time_resampled, dist1_resampled, color = 'c', linestyle = '--')
        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

        # Write the results to CSV files for the length vs time
        write_data = {'time': time_resampled,
                      'spb0kc_distance': dist0_resampled,
                      'spb1kc_distance': dist1_resampled}
        dfw = pd.DataFrame(write_data, columns = ['time', 'spb0kc_distance', 'spb1kc_distance'])
        export_csv = dfw.to_csv(r'spbkc_distance_vs_time.csv', index=None, header=True)

    # Pass in both of the axes, since we need 2 graphs
    def GraphKCKCSeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        # Generate the images from the kinetochores, do it by anaphase as well
        dist_dict = sd.PostAnalysis.timedata['kc_distance']
        dist_arr = np.array([dist_dict[ts] for ts in time])
        nchromo = len(dist_arr[0]) / 2

        # To replicate Cen2 type results, just take chromosome 1
        dist0 = np.zeros(len(dist_arr))
        dist1 = np.zeros(len(dist_arr))

        ic = 1
        for x in xrange(len(dist_arr)):
            dist0[x] = dist_arr[x][2*ic]
            dist1[x] = dist_arr[x][2*ic+1]
        if sd.PostAnalysis.do_anaphase:
            dist0_arr_adj = dist0[time < self.anaphase_b_time]
            adjusted_end = dist0_arr_adj.shape[0] + (self.nmeasure - dist0_arr_adj.shape[0] % self.nmeasure)
            dist0 = dist0[0:adjusted_end]
            dist1 = dist1[0:adjusted_end]
            time_resampled = time[0:adjusted_end:self.nmeasure]
        kc_sep = np.fabs(dist1 - dist0)
        kc_sep_resampled = np.mean(kc_sep.reshape(-1, self.nmeasure), axis=1)
        ax.plot(time_resampled, kc_sep_resampled, color = 'b', linestyle = '-')
        if sd.PostAnalysis.do_anaphase:
            ax.axvline(sd.PostAnalysis.anaphase_onset, color = 'k', linestyle = '--')

        # Write the results to CSV files for the length vs time
        write_data = {'time': time_resampled,
                      'kckc_distance': kc_sep_resampled}
        dfw = pd.DataFrame(write_data, columns = ['time', 'kckc_distance'])
        export_csv = dfw.to_csv(r'kckc_distance_vs_time.csv', index=None, header=True)

    # Graph the pole0 forces on the seed
    def GraphPoleForces(self):
        plt.style.use(cp_spindle_stl)
        fig, ax = plt.subplots()

        # Graph the pole forces
        self.GraphPoleForcesSeed(fig, ax, self.wt_sim)

        ax.set_ylim(-150.0, 150.0)
        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel('Pole axis force (pN)')
        ax.legend()
        fig.tight_layout() 
        plt.savefig('spindle_force_s{}.pdf'.format(self.s), dpi=fig.dpi)
        plt.close()

    def GraphPoleForcesSeed(self, fig, ax, sim):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        # Set the measurement time for everybody
        self.measure_time = 8.0/60.0
        time = sim.seeds[self.s].time[:min_size]
        self.nmeasure = np.int(self.measure_time / ((time[1] - time[0])))

        time = sim.seeds[self.s].time[:min_size]
        time_resampled = time[0::self.nmeasure]

        sd = sim.seeds[self.s]
        time = sd.PostAnalysis.timedata['time']
        time_resampled = time[0::self.nmeasure]

        # The forces are stored as a dictionary, so get each contribution
        forces_dict_0 = sd.PostAnalysis.timedata['pole_forces'][0]
        # The first entry is the time, then it is a dictionary of the different forces
        forces_arr_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][0] for ts in time])
        forces_arr_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][1] for ts in time])
        forces_arr_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_mag_subtype'][2] for ts in time])
        forces_axis_0_subtype_0 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][0] for ts in time])
        forces_axis_0_subtype_1 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][1] for ts in time])
        forces_axis_0_subtype_2 = np.array([forces_dict_0[ts]['xlink_axis_subtype'][2] for ts in time])

        forces_arr_0_af = np.array([forces_dict_0[ts]['af_mag'] for ts in time])
        forces_axis_0_af = np.array([forces_dict_0[ts]['af_axis'] for ts in time])

        # Redo the mean average stuff
        forces_arr_0_subtype_0 = np.mean(forces_arr_0_subtype_0.reshape(-1, self.nmeasure), axis=1)
        forces_arr_0_subtype_1 = np.mean(forces_arr_0_subtype_1.reshape(-1, self.nmeasure), axis=1)
        forces_arr_0_subtype_2 = np.mean(forces_arr_0_subtype_2.reshape(-1, self.nmeasure), axis=1)
        forces_arr_0_af = np.mean(forces_arr_0_af.reshape(-1, self.nmeasure), axis=1)
        forces_axis_0_subtype_0 = np.mean(forces_axis_0_subtype_0.reshape(-1, self.nmeasure), axis=1)
        forces_axis_0_subtype_1 = np.mean(forces_axis_0_subtype_1.reshape(-1, self.nmeasure), axis=1)
        forces_axis_0_subtype_2 = np.mean(forces_axis_0_subtype_2.reshape(-1, self.nmeasure), axis=1)
        forces_axis_0_af = np.mean(forces_axis_0_af.reshape(-1, self.nmeasure), axis=1)

        ax.plot(time_resampled, forces_axis_0_subtype_0, color = 'r', label = 'K5')
        ax.plot(time_resampled, forces_axis_0_subtype_1, color = 'g', label = 'K14')
        ax.plot(time_resampled, forces_axis_0_subtype_2, color = 'b', label = 'XL')
        ax.plot(time_resampled, forces_axis_0_af, color = 'k', label = 'AF')


        #forces_dict_0 = sd.PostAnalysis.timedata['pole_forces'][0]
        #forces_arr_0 = np.array([forces_dict_0[ts] for ts in time])
        #forces_dict_1 = sd.PostAnalysis.timedata['pole_forces'][1]
        #forces_arr_1 = np.array([forces_dict_1[ts] for ts in time])

        ## Grab the arrays individually
        #xlink_mag_arr_0 = np.array([forces_arr_0[:,0]])
        #af_mag_arr_0 = np.array([forces_arr_0[:,1]])
        #xlink_mag_arr_1 = np.array([forces_arr_1[:,0]])
        #af_mag_arr_1 = np.array([forces_arr_1[:,1]])
       
        #xlink_mag_arr_0 = np.mean(xlink_mag_arr_0.reshape(-1, self.nmeasure), axis=1)
        #af_mag_arr_0 = np.mean(af_mag_arr_0.reshape(-1, self.nmeasure), axis=1)
        #xlink_mag_arr_1 = np.mean(xlink_mag_arr_1.reshape(-1, self.nmeasure), axis=1)
        #af_mag_arr_1 = np.mean(af_mag_arr_1.reshape(-1, self.nmeasure), axis=1)

        #ax.plot(time_resampled, xlink_mag_arr_0, color = 'r', label = "Crosslinkers/Motors")
        ##ax.plot(time_resampled, xlink_mag_arr_1, color = 'r', label = "Crosslinkers Pole 2", linestyle = '--')
        #ax.plot(time_resampled, af_mag_arr_0, color = 'b', label = "AFs")
        ##ax.plot(time_resampled, af_mag_arr_1, color = 'b', label = "AFs Pole 2", linestyle = '--')

################
if __name__ == "__main__":
    opts = parse_args()
    x = SingleSeedChromosomes(opts)
    x.GraphSpindleLength()
    x.GraphSpindleIPF()
    x.GraphSpindleAttachStates()
    x.GraphOccupancyBindingSites()

    # KC specific information
    x.GraphKinetochorePositions()

    # Force information
    x.GraphPoleForces()
