#!/usr/bin/env python
# In case of poor (Sh**y) commenting contact christopher.edelmaier@colorado.edu
# Basic
import sys, os, pdb
import gc
import argparse
import fnmatch
## Analysis
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib as mpl

sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Lib'))
sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Spindle'))

from spindle_sim import SpindleSim
from sim_graph_funcs import *
from stylelib.ase1_styles import ase1_sims_stl

from scipy.stats import ks_2samp
from scipy.spatial.distance import euclidean
from fastdtw import fastdtw
import scipy.io as sio

'''
Name: MotorDeleteGraph.py
Description:
Input:
Output:
'''

def parse_args():
    parser = argparse.ArgumentParser(prog='MotorDeleteGraph.py')
    # General options that are actually required
    parser.add_argument('-w', '--wildtype', required=True, type=str,
            help='WT sim')
    parser.add_argument('--cut7del', required=True, type=str,
            help='K5 delete sim')
    parser.add_argument('--pkl1del', required=True, type=str,
            help='K14 delete sim')
    parser.add_argument('--ase1del', required=True, type=str,
            help='Ase1 delete sim')

    # Minimum of extra options to make this work
    parser.add_argument('--nopost', action='store_true',
            help="Do not use post analyis program to decrease time of analysis.")
    parser.add_argument('-F', '--fitness', action='store_true',
            help='Create fitness for spindle simulations.')
    parser.add_argument('-A', '--analyze', action='store_true',
            help='Analyze data from multiple simulations')
    parser.add_argument('--datadir', type=str,
            help='Name of the data directory in which all analyzed data files will be read/written. \
                    Also the directory where all the graphs will be placed when saved. \
                    Default is set to {workdir}/data/.')
    parser.add_argument('-d', '--workdir', type=str,
            help='Name of the working directory where simulation will be run.')

    opts = parser.parse_args()
    return opts


# Class definition
class MotorDeleteGraph(object):
    def __init__(self, opts):
        self.opts = opts
        self.cwd = os.getcwd()

        print "opts: {}".format(opts)

        self.ReadOpts()
        self.AnalyzeSims()

    def ReadOpts(self):
        if not self.opts.workdir:
            self.opts.workdir = os.path.abspath(self.cwd)
        elif not os.path.exists(self.opts.workdir):
            raise IOError( "Working directory {} does not exist.".format(
                self.opts.workdir) )
        else:
            self.opts.workdir = os.path.abspath(self.opts.workdir)

        self.wt_dir = os.path.abspath(self.opts.wildtype)
        self.wt_sim = SpindleSim(self.wt_dir, opts)
        self.dcut7_dir = os.path.abspath(self.opts.cut7del)
        self.dcut7_sim = SpindleSim(self.dcut7_dir, opts)
        self.dpkl1_dir = os.path.abspath(self.opts.pkl1del)
        self.dpkl1_sim = SpindleSim(self.dpkl1_dir, opts)
        self.dase1_dir = os.path.abspath(self.opts.ase1del)
        self.dase1_sim = SpindleSim(self.dase1_dir, opts)

    def AnalyzeSims(self):
        self.wt_sim.Analyze()
        self.dcut7_sim.Analyze()
        self.dpkl1_sim.Analyze()
        self.dase1_sim.Analyze()
        self.wt_sim.CalcSimSuccess()
        self.dcut7_sim.CalcSimSuccess()
        self.dpkl1_sim.CalcSimSuccess()
        self.dase1_sim.CalcSimSuccess()
        if self.opts.fitness:
            wt_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', 'Data', 'wt.mat')
            lstream_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', 'Data', 'lstream.mat')
            self.wt_sim.Fitness(wt_path, lstream_path)

    # Graph the avg spindle length for various delete strains (or just multiple strains)
    def GraphSpindleLength(self):
        fig, ax = plt.subplots() 

        nsims = 4
        colors = mpl.cm.rainbow(np.linspace(0,1,nsims))

        # Graph the specific versions of the input data averages
        self.GraphSpindleLengthSim(fig, ax, self.wt_sim, colors, 0, r'WT')
        self.GraphSpindleLengthSim(fig, ax, self.dcut7_sim, colors, 1, r'K5$\Delta$')
        self.GraphSpindleLengthSim(fig, ax, self.dpkl1_sim, colors, 2, r'K14$\Delta$')
        self.GraphSpindleLengthSim(fig, ax, self.dase1_sim, colors, 3, r'XL$\Delta$')

        ax.set_xlabel(r'Time (min)')
        ax.set_ylabel(r'SPB Separation ($\mu$m)')
        ax.legend()
        fig.tight_layout() 
        plt.savefig('motor_delete_spindle_length.pdf', dpi=fig.dpi)

    def GraphSpindleLengthSim(self, fig, ax, sim, colors, num, label):
        min_size = 0
        for sd in sim.seeds:
            if (min_size == 0 or min_size > sd.time.size):
                min_size = sd.time.size

        avg = np.zeros(min_size)
        time = sim.seeds[-1].time[:min_size]
        num_seeds = 0

        for sd in sim.seeds:
            #print sd.succ_info_dict
            if sd.succ_info_dict['succ'] == 1:
                time = sd.PostAnalysis.timedata['time']
                spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
                spbsep_arr = [spbsep_dict[ts] for ts in time ]
                spbsep_arr = np.array(spbsep_arr)*uc['um'][1]
                avg = np.add(avg, spbsep_arr[:min_size])
                num_seeds += 1

        avg /= float(num_seeds)
        ax.plot(time, avg, color=colors[num], label=label)


################
if __name__ == "__main__":
    opts = parse_args()
    x = MotorDeleteGraph(opts)
    x.GraphSpindleLength()
