#!/usr/bin/env python
# In case of poor commenting, contact christopher.edelmaier@colorado.edy
# Cen2 specific fitness and graphing information

import sys, os, pdb
import gc
import argparse
import fnmatch
## Analysis
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib as mpl

sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Lib'))
sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'Spindle'))

from spindle_sim import SpindleSim
from sim_graph_funcs import *
from stylelib.ase1_styles import ase1_sims_stl
from stylelib.ase1_styles import cp_spindle_stl

from scipy.stats import ks_2samp
from scipy.spatial.distance import euclidean
from fastdtw import fastdtw
import scipy.io as sio
from scipy.signal import medfilt
from stylelib.ase1_styles import ase1_runs_stl
from stylelib.ase1_styles import cp_spindle_stl

'''
Name: Cen2Fitness.py
Description: Does the Cen2 fitness information for the validation, wrapping it all in a different script
Input:
Output:
'''

def cje_moving_average(a, n=3):
    ret = np.cumsum(a, dtype=float)
    ret[n:] = ret[n:] - ret[:-n]
    return ret[n-1:]/n

class Cen2Fitness(object):
    def __init__(self, rft):
        self.best_tracks = rft.best_tracks
        self.model_sim = rft.model_sim
        self.experiment_data = rft.experiment_data
        self.anaphase = rft.anaphase
        self.measure_time = rft.measure_time

        # Specific data that we know about
        self.exclude_frames = {}
        self.exclude_frames['001_2'] = np.array([192]) - 105
        self.exclude_frames['002_1'] = np.array([80, 97, 168, 225]) - 52
        self.exclude_frames['002_2'] = np.array([180, 146, 219, 222]) - 54
        self.exclude_frames['002_3'] = np.array([136]) - 13
        self.exclude_frames['003_3'] = np.array([14, 15, 16]) - 1

    def Run(self):
        self.GraphCen2Experiment()
        self.CombinedCen2Fitness()

    def GraphCen2Experiment(self):
        # Graph the KC time series for both experiment and model, using the cen2 marker data
        # Do the experiments first....
        colors = mpl.cm.rainbow(np.linspace(0,1,len(self.best_tracks)))

        fig_all, axarr_all = plt.subplots(3, 1, figsize=(15,10))
        plt.style.use(ase1_runs_stl)
        axarr_all[0].set_ylabel(r'SPB-SPB distance($\mu$m)')
        axarr_all[1].set_ylabel(r'Spindle pole body-kinetochore distance($\mu$m)')
        axarr_all[2].set_ylabel(r'KC-sep ($\mu$m)')
        axarr_all[2].set_xlabel(r'Time (min)')

        # Get the maximum time of the simulations for later use
        self.max_time = 1000.0
        self.experiment_data_adj = {}
        self.experiment_data_sanitized = {}
        self.experiment_data_adj_sanitized = {}
        self.experiment_data_adj_postanaphase = {}
        for sd in self.model_sim.seeds:
            time = sd.PostAnalysis.timedata['time']
            self.max_time = max(self.max_time, time[-1])

        # Prepare the tracks
        for track in self.best_tracks:
            df = self.experiment_data[track]
            dfnew = df

            #pd.options.display.max_rows = 999
            #print "track {}".format(track)
            #print dfnew

            # Sanitize the inputs for large errors...
            # Try excluding the proper frames
            if track in self.exclude_frames:
                dfnew = dfnew.drop(dfnew.index[self.exclude_frames[track]])

            # Prevent the kinetochores from swapping paths with each other
            kc0_dist = np.asarray(dfnew['kc0_dist'])
            kc1_dist = np.asarray(dfnew['kc1_dist'])
            if kc0_dist[0] > kc1_dist[0]:
                kc0_nocross = np.fmax(kc0_dist, kc1_dist)
                kc1_nocross = np.fmin(kc0_dist, kc1_dist)
            else:
                kc0_nocross = np.fmin(kc0_dist, kc1_dist)
                kc1_nocross = np.fmax(kc0_dist, kc1_dist)
            dfnew['kc0_dist'] = kc0_nocross
            dfnew['kc1_dist'] = kc1_nocross

            # Use a 3sigma cutoff (including the point in question) on the spindle length
            dfnew = dfnew[dfnew['length_error'] <= 3.*dfnew['length_error'].mean()]

            # Do absolute cutoffs on the error for the two kinetochores
            # - this must lie within the nuclear envelope region at worst, 
            # so cutoff the error above 10
            dfnew = dfnew[np.fabs(dfnew['kc0_dist_error']) <= 10.0]
            dfnew = dfnew[np.fabs(dfnew['kc1_dist_error']) <= 10.0]

            # Use a 3sigma cutoff on the distances
            dfnew = dfnew[dfnew['kc0_dist_error'] <= 3.*dfnew['kc0_dist_error'].mean()]
            dfnew = dfnew[dfnew['kc1_dist_error'] <= 3.*dfnew['kc1_dist_error'].mean()]

            ## Pad to be the proper length - we lost 2 from the first/last moving average
            #moving_average_length = np.pad(np.asarray(dfnew['length']), (2,2), 'edge')
            #moving_average_kc0 = np.pad(np.asarray(dfnew['kc0_dist']), (2,2), 'edge')
            #moving_average_kc1 = np.pad(np.asarray(dfnew['kc1_dist']), (2,2), 'edge')
            ## Try to generate a moving average on the points/series we care about...
            #moving_average_length = cje_moving_average(moving_average_length, 5)
            #moving_average_kc0 = cje_moving_average(moving_average_kc0, 5)
            #moving_average_kc1 = cje_moving_average(moving_average_kc1, 5)
            ### Yank the spindle length down to its minimum value
            ##moving_average_length = moving_average_length - np.amin(moving_average_length)

            #dfnew['length'] = moving_average_length
            #dfnew['kc0_dist'] = moving_average_kc0
            #dfnew['kc1_dist'] = moving_average_kc1

            ## Try running this dfnew through a rolling average now
            #dfnew = dfnew.rolling(5, center=True).mean()

            ### Moving average to smooth the functions
            ##dfnew = df.rolling(5).mean()

            # Save off the sanitized track
            self.experiment_data_sanitized[track] = dfnew

            # Correct for anaphase behavior and the max_time behavior
            dfnew_adj = dfnew
            dfnew_post_anaphase = dfnew
            if self.anaphase[track] != 0:
                df = df[df.time_length <= self.anaphase[track]]
                dfnew_adj = dfnew[dfnew.time_length <= self.anaphase[track]]
                dfnew_post_anaphase = dfnew[dfnew.time_length > self.anaphase[track]]
            # Correct for max time
            df = df[df.time_length <= self.max_time]
            dfnew_adj = dfnew_adj[dfnew_adj.time_length <= self.max_time]
            self.experiment_data_adj[track] = df
            self.experiment_data_adj_sanitized[track] = dfnew_adj
            self.experiment_data_adj_postanaphase[track] = dfnew_post_anaphase


        for ic, track in enumerate(self.best_tracks):
            fig, axarr = plt.subplots(5,1, figsize=(20,10))
            #df = self.experiment_data_adj[track]
            #df = self.experiment_data[track]
            df = self.experiment_data_adj_sanitized[track]
            times = df['time_kc']
            dist0 = df['kc0_dist']
            dist1 = df['kc1_dist']
            length = df['length']
            time_length = df['time_length']
            length_error = df['length_error']
            dist0_error = df['kc0_dist_error']
            dist1_error = df['kc1_dist_error']
            kc_sep = np.fabs(dist0 - dist1)
            kc_sep_err = np.sqrt(np.square(dist0_error) + np.square(dist1_error))
            axarr[0].plot(time_length, length, color = colors[ic], linestyle = '-')
            spb_velocity = np.gradient(length, time_length)
            axarr[1].plot(time_length, spb_velocity, color = colors[ic], linestyle = '-')
            axarr[2].plot(times, dist0, color = colors[ic], linestyle = '-')
            axarr[2].plot(times, dist1, color = colors[ic], linestyle = '--')
            axarr[3].plot(times, kc_sep, color = colors[ic], linestyle = '-')
            # Look at the kc_velocity....
            kc_velocity = np.gradient(kc_sep, times)
            axarr[4].plot(times, kc_velocity, color = colors[ic], linestyle = '-')
            print "Cen2 {} (track {}) avg/max spb_sep velocity: {} ({})".format(ic, track, np.mean(spb_velocity), np.amax(spb_velocity))
            print "                     avg/max  kc_sep velocity: {} ({})".format(np.mean(kc_velocity), np.amax(kc_velocity))
            # Duplicate plotting for the overall one
            axarr_all[0].plot(time_length, length, color = colors[ic], linestyle = '-')
            axarr_all[1].plot(times, dist0, color = colors[ic], linestyle = '-')
            axarr_all[1].plot(times, dist1, color = colors[ic], linestyle = '--')
            axarr_all[2].plot(times, kc_sep, color = colors[ic], linestyle = '-')

            axarr[0].set_ylabel('SPB-SPB distance($\mu$m)')
            axarr[1].set_ylabel('SPB-SPB velocity($\mu$m/min)')
            axarr[2].set_ylabel('Spindle pole body-kinetochore distance($\mu$m)')
            axarr[3].set_ylabel('KC-sep ($\mu$m)')
            axarr[4].set_ylabel('KC-sep velocity ($\mu$m/min)')
            axarr[4].set_xlabel('Time (min)')
            fig.tight_layout()
            fig.savefig('cen2_{}_kinetochores.pdf'.format(ic), dpi=fig.dpi)
            plt.close(fig)

            # Do one that is just the reduced version in blue
            fig, axarr = plt.subplots(3,1, figsize=(15, 10))
            #axarr[0].plot(time_length, length, color = 'b', linestyle = '-')
            #axarr[1].plot(times, dist0, color = 'b', linestyle = '-')
            #axarr[1].plot(times, dist1, color = 'b', linestyle = '--')
            #axarr[2].plot(times, kc_sep, color = 'b', linestyle = '-')
            axarr[0].errorbar(time_length, length, yerr=length_error, color = 'b', linestyle = 'none', marker = 's', mfc = 'none')
            axarr[1].errorbar(times, dist0, yerr=dist0_error, color = 'b', linestyle = 'none', marker = 's', mfc = 'none')
            axarr[1].errorbar(times, dist1, yerr=dist1_error, color = 'b', linestyle = 'none', marker = 'o', mfc = 'none')
            axarr[2].errorbar(times, kc_sep, color = 'b', linestyle = 'none', marker = 's', mfc = 'none')
        
            axarr[0].axvline(self.anaphase[track], color = 'k', linestyle = '--')
            axarr[1].axvline(self.anaphase[track], color = 'k', linestyle = '--')
            axarr[2].axvline(self.anaphase[track], color = 'k', linestyle = '--')

            axarr[0].set_ylabel('SPB-SPB distance($\mu$m)')
            axarr[1].set_ylabel('Spindle pole body-kinetochore distance($\mu$m)')
            axarr[2].set_ylabel('KC-sep ($\mu$m)')
            axarr[2].set_xlabel('Time (min)')

            fig.tight_layout()
            fig.savefig('cen2_{}_kinetochores_blue.pdf'.format(track), dpi=fig.dpi)
            plt.close(fig)

            # Do the blue one on separate plots to use in the same way as the model sims
            #plt.style.use(ase1_runs_stl)
            plt.style.use(cp_spindle_stl)
            # Spindle length
            fig_pretty_size = (2,2)
            fig_length, ax_length = plt.subplots(figsize=fig_pretty_size)
            ax_length.scatter(x = time_length[::4], y = length[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_length.errorbar(x = time_length[::4], y = length[::4], yerr = length_error[::4], ecolor = 'b', elinewidth = 2,
                    capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_length.set_ylim(0.0, 3.1)
            ax_length.set_xlabel('Time (min)')
            ax_length.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
            fig_length.tight_layout()
            fig_length.savefig('cen2_{}_length_pretty.pdf'.format(track), dpi=fig_length.dpi)
            print "strain {} xlim: {}".format(track, ax_length.get_xlim())
            plt.close(fig_length)

            # SPB KC distance
            fig_kc, ax_kc = plt.subplots(figsize=fig_pretty_size)
            ax_kc.scatter(x = times[::4], y = dist0[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_kc.scatter(x = times[::4], y = dist1[::4], zorder = 100, s = 10, color = 'c', label = None, marker = 'o', facecolor = 'none')
            ax_kc.errorbar(x = times[::4], y = dist0[::4], yerr = dist0_error[::4], ecolor = 'b', elinewidth = 2,
                    capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
            ax_kc.errorbar(x = times[::4], y = dist1[::4], yerr = dist1_error[::4], ecolor = 'c', elinewidth = 2,
                    capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_kc.set_ylim(0.0, 3.1)
            ax_kc.set_xlabel('Time (min)')
            ax_kc.set_ylabel('Spindle pole body-kinetochore\ndistance ($\mu$m)')
            fig_kc.tight_layout()
            fig_kc.savefig('cen2_{}_spbkc_pretty.pdf'.format(track), dpi=fig_kc.dpi)
            plt.close(fig_kc)

            # Interkc distance
            fig_ikc, ax_ikc = plt.subplots(figsize=fig_pretty_size)
            ax_ikc.scatter(x = times[::4], y = kc_sep[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_ikc.errorbar(x = times[::4], y = kc_sep[::4], yerr = kc_sep_err[::4], ecolor = 'b', elinewidth = 2,
                    capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_ikc.set_ylim(0.0, 1.0)
            ax_ikc.set_xlabel('Time (min)')
            ax_ikc.set_ylabel('Kinetochore separation ($\mu$m)')
            fig_ikc.tight_layout()
            fig_ikc.savefig('cen2_{}_kckc_pretty.pdf'.format(track), dpi=fig_ikc.dpi)
            plt.close(fig_ikc)

        # Do some anaphase and after information on the tracks...
        for ic, track in enumerate(self.best_tracks):
            df = self.experiment_data_adj_postanaphase[track]
            times = df['time_kc']
            dist0 = df['kc0_dist']
            dist1 = df['kc1_dist']
            length = df['length']
            time_length = df['time_length']
            length_error = df['length_error']
            dist0_error = df['kc0_dist_error']
            dist1_error = df['kc1_dist_error']
            kc_sep = np.fabs(dist0 - dist1)
            kc_sep_err = np.sqrt(np.square(dist0_error) + np.square(dist1_error))
            # Do the blue one on separate plots to use in the same way as the model sims
            plt.style.use(cp_spindle_stl)
            # Spindle length
            fig_length, ax_length = plt.subplots(figsize=(2.5,2))
            ax_length.scatter(x = time_length, y = length, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_length.errorbar(x = time_length, y = length, yerr = length_error, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_length.set_ylim(0.0, 3.1)
            ax_length.set_xlabel('Time (min)')
            ax_length.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
            fig_length.tight_layout()
            fig_length.savefig('cen2_{}_length_pretty_postanaphase.pdf'.format(track), dpi=fig_length.dpi)
            plt.close(fig_length)

            # SPB KC distance
            fig_kc, ax_kc = plt.subplots(figsize=(2.5,2))
            ax_kc.scatter(x = times, y = dist0, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_kc.scatter(x = times, y = dist1, zorder = 100, s = 50, color = 'c', label = None, marker = 'o', facecolor = 'none')
            ax_kc.errorbar(x = times, y = dist0, yerr = dist0_error, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
            ax_kc.errorbar(x = times, y = dist1, yerr = dist1_error, ecolor = 'c', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_kc.set_ylim(0.0, 3.1)
            ax_kc.set_xlabel(r'Time (min)')
            ax_kc.set_ylabel('Spindle pole body-kinetochore\ndistance ($\mu$m)')
            fig_kc.tight_layout()
            fig_kc.savefig('cen2_{}_spbkc_pretty_postanaphase.pdf'.format(track), dpi=fig_kc.dpi)
            plt.close(fig_kc)

            # Interkc distance
            fig_ikc, ax_ikc = plt.subplots(figsize=(2.5,2))
            ax_ikc.scatter(x = times, y = kc_sep, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_ikc.errorbar(x = times, y = kc_sep, yerr = kc_sep_err, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_ikc.set_ylim(0.0, 3.1)
            ax_ikc.set_xlabel('Time (min)')
            ax_ikc.set_ylabel('Kinetochore separation ($\mu$m)')
            fig_ikc.tight_layout()
            fig_ikc.savefig('cen2_{}_kckc_pretty_postanaphase.pdf'.format(track), dpi=fig_ikc.dpi)
            plt.close(fig_ikc)

        # Do some anaphase bounded around the track time
        for ic, track in enumerate(self.best_tracks):
            if self.anaphase[track] == 0:
                continue
            df = self.experiment_data_sanitized[track]
            df = df[df.time_length >= self.anaphase[track]-1.0]
            df = df[df.time_length <= self.anaphase[track]+1.0]
            times = df['time_kc']
            dist0 = df['kc0_dist']
            dist1 = df['kc1_dist']
            length = df['length']
            time_length = df['time_length']
            length_error = df['length_error']
            dist0_error = df['kc0_dist_error']
            dist1_error = df['kc1_dist_error']
            kc_sep = np.fabs(dist0 - dist1)
            kc_sep_err = np.sqrt(np.square(dist0_error) + np.square(dist1_error))
            # Do the blue one on separate plots to use in the same way as the model sims
            plt.style.use(cp_spindle_stl)
            # Spindle length
            fig_length, ax_length = plt.subplots(figsize=(2.5,2.5))
            ax_length.scatter(x = time_length, y = length, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_length.errorbar(x = time_length, y = length, yerr = length_error, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_length.set_ylim(0.0, 3.1)
            ax_length.set_xlabel('Time (min)')
            ax_length.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
            fig_length.tight_layout()
            fig_length.savefig('cen2_{}_length_pretty_boundanaphase.pdf'.format(track), dpi=fig_length.dpi)
            plt.close(fig_length)

            # SPB KC distance
            fig_kc, ax_kc = plt.subplots(figsize=(2.5,2.5))
            ax_kc.scatter(x = times, y = dist0, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_kc.scatter(x = times, y = dist1, zorder = 100, s = 50, color = 'c', label = None, marker = 'o', facecolor = 'none')
            ax_kc.errorbar(x = times, y = dist0, yerr = dist0_error, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
            ax_kc.errorbar(x = times, y = dist1, yerr = dist1_error, ecolor = 'c', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_kc.set_ylim(0.0, 3.1)
            ax_kc.set_xlabel(r'Time (min)')
            ax_kc.set_ylabel('Spindle pole body-kinetochore\ndistance ($\mu$m)')
            ax_kc.axvline(self.anaphase[track], color = 'k', linestyle = '--')
            fig_kc.tight_layout()
            fig_kc.savefig('cen2_{}_spbkc_pretty_boundanaphase.pdf'.format(track), dpi=fig_kc.dpi)
            plt.close(fig_kc)

            # Interkc distance
            fig_ikc, ax_ikc = plt.subplots(figsize=(2.5,2.5))
            ax_ikc.scatter(x = times, y = kc_sep, zorder = 100, s = 50, color = 'b', label = None, marker = 's', facecolor = 'none')
            ax_ikc.errorbar(x = times, y = kc_sep, yerr = kc_sep_err, ecolor = 'b', elinewidth = 2,
                    capsize = 5, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)

            ax_ikc.set_ylim(0.0, 3.1)
            ax_ikc.set_xlabel('Time (min)')
            ax_ikc.set_ylabel('Kinetochore separation ($\mu$m)')
            ax_ikc.axvline(self.anaphase[track], color = 'k', linestyle = '--')
            fig_ikc.tight_layout()
            fig_ikc.savefig('cen2_{}_kckc_pretty_boundanaphase.pdf'.format(track), dpi=fig_ikc.dpi)
            plt.close(fig_ikc)
        

        fig_all.tight_layout()
        fig_all.savefig('cen2_combined_kinetochores.pdf', dpi=fig_all.dpi)
        plt.close()
        gc.collect()
        mpl.rcdefaults()

    def CombinedCen2Fitness(self):
        # Do the combined cen2 fitness for everything
        # Spindle length vs. time
        # Spindle pole body-kinetochore distance vs. time
        # KC stretch vs. time
        # Loop over the model seeds to start with
        self.model_nseeds = len(self.model_sim.seeds)

        max_avg_correlation_combined = 0.0
        max_avg_length_correlation = 0.0
        max_avg_kc_distance_correlation = 0.0
        max_avg_kc_sep_correlation = 0.0
        avg_avg_correlation_combined = 0.0
        avg_avg_length_correlation = 0.0
        avg_avg_kc_distance_correlation = 0.0
        avg_avg_kc_sep_correlation = 0.0
        

        for mseed in xrange(self.model_nseeds):
            #print "Model seed {}".format(mseed)
            sd = self.model_sim.seeds[mseed]

            # What is my maximum correlation?
            max_correlation_combined = 0.0
            max_length_correlation = 0.0
            max_kc_distance_correlation = 0.0
            max_kc_sep_correlation = 0.0
            max_correlation_track = -1

            # What about the averages?
            avg_correlation_combined = 0.0
            avg_length_correlation = 0.0
            avg_kc_distance_correlation = 0.0
            avg_kc_sep_correlation = 0.0

            # Get the time, spindle length, kinetochore position, and interkc stretch as separate objects
            time = np.array(sd.PostAnalysis.timedata['time'])

            spbsep_dict = sd.PostAnalysis.timedata['spb_separation']
            dist_dict = sd.PostAnalysis.timedata['kc_distance']

            spindle_length = np.array([spbsep_dict[ts] for ts in time])*uc['um'][1]
            kc_distance = np.array([dist_dict[ts] for ts in time]) # KC doesn't need unit conversion, already done!!!!!
            
            # Generate the actual time traces, matching sampling
            self.nchromo = len(kc_distance[0])/2
            nmeasure = np.int(self.measure_time / ((time[1] - time[0])))
            time = time[0::nmeasure]
            spindle_length = spindle_length[0::nmeasure]
            kc_distance = kc_distance[0::nmeasure]

            max_spb_speed = self.PlotModel(mseed, time, spindle_length, kc_distance)

            # FITNESS
            # Now we do the magic of figuring out the proper fitness for this seed...
            # Loop over the tracks
            model_ts_length = pd.Series(spindle_length)
            ntracks = len(self.best_tracks)
            for it, track in enumerate(self.best_tracks):
                #df = self.experiment_data_adj[track]
                df = self.experiment_data_adj_sanitized[track]
                exp_times = df['time_kc']
                exp_dist0 = df['kc0_dist']
                exp_dist1 = df['kc1_dist']
                exp_length = df['length']
                exp_time_length = df['time_length']

                corr_length = model_ts_length.corr(exp_time_length)
                #print "Length Pearson correlation {} : {} = {}".format(mseed, track, corr_length)

                # Now do the complicated switching of chromosome and kinetochore movements
                corr0 = np.zeros(self.nchromo)
                corr1 = np.zeros(self.nchromo)
                corr_kc_sep = np.zeros(self.nchromo)
                for ic in xrange(self.nchromo):
                    dist0 = pd.Series(kc_distance[:,2*ic])
                    dist1 = pd.Series(kc_distance[:,2*ic+1])
                    # Do all 4 correlations that cross
                    corr_0_0 = dist0.corr(exp_dist0)
                    corr_1_1 = dist1.corr(exp_dist1)
                    corr_0_1 = dist0.corr(exp_dist1)
                    corr_1_0 = dist1.corr(exp_dist0)

                    # Pick the best set
                    if (corr_0_0 + corr_1_1) > (corr_0_1 + corr_1_0):
                        corr0[ic] = corr_0_0
                        corr1[ic] = corr_1_1
                    else:
                        corr0[ic] = corr_0_1
                        corr1[ic] = corr_1_0

                    #print "  Chromosome {} Dist Pearson correlation {} : {} = {}, {}".format(ic, mseed, track, corr0[ic], corr1[ic])

                    # Kinetochore separation
                    kc_sep = pd.Series(np.fabs(kc_distance[:,2*ic] - kc_distance[:,2*ic+1]))
                    kc_sep_exp = pd.Series(np.fabs(exp_dist0 - exp_dist1))
                    corr_kc_sep[ic] = kc_sep.corr(kc_sep_exp)

                    #print "  Chromosome {} Sep Pearson correlation {} : {} = {}".format(ic, mseed, track, corr_kc_sep[ic])

                # Average the kinetochore correlation measures
                corr_kc_combined = (np.mean(corr0) + np.mean(corr1))/2.0
                corr_kc_sep_combined = np.mean(corr_kc_sep)

                #print "KC Pearson correlation {} : {} = {}".format(mseed, track, corr_kc_combined)
                #print "KC Separation Pearson correlation {} : {} = {}".format(mseed, track, corr_kc_sep_combined)

                if (corr_length + corr_kc_combined + corr_kc_sep_combined) > max_correlation_combined:
                    max_correlation_combined = corr_length + corr_kc_combined + corr_kc_sep_combined
                    max_length_correlation = corr_length
                    max_kc_distance_correlation = corr_kc_combined
                    max_kc_sep_correlation = corr_kc_sep_combined
                    max_correlation_track = track

                # Averages
                avg_correlation_combined += (corr_length + corr_kc_combined + corr_kc_sep_combined)
                avg_length_correlation += corr_length
                avg_kc_distance_correlation += corr_kc_combined
                avg_kc_sep_correlation += corr_kc_sep_combined
           
            print "Seed {} max correlation with experimental track {} = {} (length: {}, kc: {}, kc_sep: {})".format(mseed, max_correlation_track, max_correlation_combined, max_length_correlation, max_kc_distance_correlation, max_kc_sep_correlation)
            avg_correlation_combined /= np.float_(ntracks)
            avg_length_correlation /= np.float_(ntracks)
            avg_kc_distance_correlation /= np.float_(ntracks)
            avg_kc_sep_correlation /= np.float_(ntracks)
            print "   avg correlation {} (length: {}, kc: {}, kc_sep: {})".format(avg_correlation_combined, avg_length_correlation, avg_kc_distance_correlation, avg_kc_sep_correlation)
            print "   max spb separation speed: {}".format(max_spb_speed)
            self.PlotModelExperiment(mseed, time, spindle_length, kc_distance, max_correlation_track)

            max_avg_correlation_combined += max_correlation_combined
            max_avg_length_correlation += max_length_correlation
            max_avg_kc_distance_correlation += max_kc_distance_correlation
            max_avg_kc_sep_correlation += max_kc_sep_correlation
            avg_avg_correlation_combined += avg_correlation_combined
            avg_avg_length_correlation += avg_length_correlation
            avg_avg_kc_distance_correlation += avg_kc_distance_correlation
            avg_avg_kc_sep_correlation += avg_kc_sep_correlation

        max_avg_correlation_combined    /= self.model_nseeds 
        max_avg_length_correlation      /= self.model_nseeds  
        max_avg_kc_distance_correlation /= self.model_nseeds 
        max_avg_kc_sep_correlation      /= self.model_nseeds 
        avg_avg_correlation_combined    /= self.model_nseeds 
        avg_avg_length_correlation      /= self.model_nseeds  
        avg_avg_kc_distance_correlation /= self.model_nseeds   
        avg_avg_kc_sep_correlation      /= self.model_nseeds  
        print "Maximum average correlation: {} (length: {}, kc_dist: {}, kc_sep: {})".format(max_avg_correlation_combined, max_avg_length_correlation, max_avg_kc_distance_correlation, max_avg_kc_sep_correlation)
        print "Average average correlation: {} (length: {}, kc_dist: {}, kc_sep: {})".format(avg_avg_correlation_combined, avg_avg_length_correlation, avg_avg_kc_distance_correlation, avg_avg_kc_sep_correlation)

        self.max_fitness_combined = max_avg_correlation_combined
        self.max_fitness_length   = max_avg_length_correlation
        self.max_fitness_kc_dist  = max_avg_kc_distance_correlation
        self.max_fitness_kc_sep   = max_avg_kc_sep_correlation
        self.avg_fitness_combined = avg_avg_correlation_combined
        self.avg_fitness_length   = avg_avg_length_correlation
        self.avg_fitness_kc_dist  = avg_avg_kc_distance_correlation
        self.avg_fitness_kc_sep   = avg_avg_kc_sep_correlation

    def PlotModel(self, iseed, time, spindle_length, kc_distance):
        plt.style.use(ase1_runs_stl)
        fig, axarr = plt.subplots(3, 1, figsize=(15,10))
        colors = ['c', 'y', 'm']

        # First, plot the spindle length vs. time
        axarr[0].plot(time, spindle_length, color = 'r', linestyle = '-')
        spb_velocity = np.gradient(spindle_length, time)

        # Find out when we've reached the spindle max length for this simulation first...
        max_spb_velocity = np.amax(spb_velocity)
        #print("max velocity = {}".format(np.amax(spb_velocity)))

        # Figure out how to plot all nchromosomes
        for ic in xrange(self.nchromo):
            dist0 = kc_distance[:,2*ic]
            dist1 = kc_distance[:,2*ic+1]
            axarr[1].plot(time, dist0, color = colors[ic], linestyle = '-')
            axarr[1].plot(time, dist1, color = colors[ic], linestyle = '--')
            kc_sep = np.fabs(dist0 - dist1)
            axarr[2].plot(time, kc_sep, color = colors[ic], linestyle = '-')

        axarr[0].set_ylabel(r'SPB-SPB distance($\mu$m)')
        axarr[1].set_ylabel(r'Spindle pole body-kinetochore distance($\mu$m)')
        axarr[2].set_ylabel(r'KC-sep ($\mu$m)')
        axarr[2].set_xlabel(r'Time (min)')

        fig.tight_layout()
        fig.savefig('cen2_model_{}_single.pdf'.format(iseed), dpi=fig.dpi)
        plt.close(fig)

        return max_spb_velocity

    # Plot a specific seed and track on the same axes
    def PlotModelExperiment(self,
                            iseed,
                            model_time,
                            model_spindle_length,
                            model_kc_distance,
                            track):
        colors = ['c', 'y', 'm']

        # Do the experimental side first
        df = self.experiment_data_adj_sanitized[track]
        times = df['time_kc']
        dist0 = df['kc0_dist']
        dist1 = df['kc1_dist']
        length = df['length']
        time_length = df['time_length']
        length_error = df['length_error']
        dist0_error = df['kc0_dist_error']
        dist1_error = df['kc1_dist_error']
        kc_sep = np.fabs(dist0 - dist1)
        kc_sep_err = np.sqrt(np.square(dist0_error) + np.square(dist1_error))

        plt.style.use(cp_spindle_stl)

        # Spindle length
        fig_pretty_size = (2,2)
        fig_length, ax_length = plt.subplots(figsize=fig_pretty_size)
        # Experimental track
        ax_length.scatter(x = time_length[::4], y = length[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
        ax_length.errorbar(x = time_length[::4], y = length[::4], yerr = length_error[::4], ecolor = 'b', elinewidth = 2,
                capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        # Model track
        ax_length.plot(model_time, model_spindle_length, color = 'r')

        ax_length.axhline(2.75, color = 'k', linestyle = '--', linewidth = 1.0)

        ax_length.set_ylim(0.0, 3.1)
        ax_length.set_xlabel('Time (min)')
        ax_length.set_ylabel('Spindle pole body\nseparation ($\mu$m)')
        fig_length.tight_layout()
        fig_length.savefig('cen2_model{}_track{}_length_comparison.pdf'.format(iseed, track), dpi=fig_length.dpi)
        plt.close(fig_length)

        # SPB KC distance
        fig_kc, ax_kc = plt.subplots(figsize=fig_pretty_size)
        # Experimental track
        ax_kc.scatter(x = times[::4], y = dist0[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
        ax_kc.scatter(x = times[::4], y = dist1[::4], zorder = 100, s = 10, color = 'c', label = None, marker = 'o', facecolor = 'none')
        ax_kc.errorbar(x = times[::4], y = dist0[::4], yerr = dist0_error[::4], ecolor = 'b', elinewidth = 2,
                capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        ax_kc.errorbar(x = times[::4], y = dist1[::4], yerr = dist1_error[::4], ecolor = 'c', elinewidth = 2,
                capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        # Model track, only do 'Cen2'
        ic = 1
        model_dist0 = model_kc_distance[:,2*ic]
        model_dist1 = model_kc_distance[:,2*ic+1]
        #dist0_resampled = np.mean(dist0.reshape(-1, self.nmeasure), axis=1)
        #dist1_resampled = np.mean(dist1.reshape(-1, self.nmeasure), axis=1)
        ax_kc.plot(model_time, model_dist0, color = 'r', linestyle = '-')
        ax_kc.plot(model_time, model_dist1, color = 'm', linestyle = '--')

        ax_kc.set_ylim(0.0, 3.1)
        ax_kc.set_xlabel('Time (min)')
        ax_kc.set_ylabel('Spindle pole body-kinetochore\ndistance ($\mu$m)')
        fig_kc.tight_layout()
        fig_kc.savefig('cen2_model{}_track{}_spbkc_comparison.pdf'.format(iseed, track), dpi=fig_kc.dpi)
        plt.close(fig_kc)

        # KC KC distance
        fig_ikc, ax_ikc = plt.subplots(figsize=fig_pretty_size)
        # Experiment
        ax_ikc.scatter(x = times[::4], y = kc_sep[::4], zorder = 100, s = 10, color = 'b', label = None, marker = 's', facecolor = 'none')
        ax_ikc.errorbar(x = times[::4], y = kc_sep[::4], yerr = kc_sep_err[::4], ecolor = 'b', elinewidth = 2,
                capsize = 2, capthick = 1, zorder = 0, fmt = 'none', marker = 'none', label = None)
        # Model
        kc_sep = np.fabs(model_dist0 - model_dist1)
        ax_ikc.plot(model_time, kc_sep, color = 'r')

        ax_ikc.set_ylim(0.0, 1.0)
        ax_ikc.set_xlabel('Time (min)')
        ax_ikc.set_ylabel('Kinetochore separation ($\mu$m)')
        fig_ikc.tight_layout()
        fig_ikc.savefig('cen2_model{}_track{}_kckc_comparison.pdf'.format(iseed, track), dpi=fig_ikc.dpi)
        plt.close(fig_ikc)





        #axarr[0].plot(exp_time_length, exp_length, color = 'b', linestyle = '-')
        #axarr[1].plot(exp_times, exp_dist0, color = 'b', linestyle = '-')
        #axarr[1].plot(exp_times, exp_dist1, color = 'b', linestyle = '--')
        #axarr[2].plot(exp_times, exp_kc_sep, color = 'b', linestyle = '-')

        ## Now plot the particular seed information
        #axarr[0].plot(model_time, model_spindle_length, color = 'r', linestyle = '-')

        ## Figure out how to plot all nchromosomes
        #for ic in xrange(self.nchromo):
        #    dist0 = model_kc_distance[:,2*ic]
        #    dist1 = model_kc_distance[:,2*ic+1]
        #    axarr[1].plot(model_time, dist0, color = colors[ic], linestyle = '-')
        #    axarr[1].plot(model_time, dist1, color = colors[ic], linestyle = '--')
        #    kc_sep = np.fabs(dist0 - dist1)
        #    axarr[2].plot(model_time, kc_sep, color = colors[ic], linestyle = '-')

        #axarr[0].set_ylabel(r'SPB-SPB distance($\mu$m)')
        #axarr[1].set_ylabel(r'Spindle pole body-kinetochore distance($\mu$m)')
        #axarr[2].set_ylabel(r'KC-sep ($\mu$m)')
        #axarr[2].set_xlabel(r'Time (min)')

        #fig.tight_layout()
        #fig.savefig('cen2_model{}_track{}_comparison.pdf'.format(iseed, track), dpi=fig.dpi)
        #plt.close(fig)

########################
if __name__ == "__main__":
    print "Cen2Fitness not available as a standalone!"
