#include "bob.h"

#include <iostream>

double anchor_potential_bd(system_parameters *parameters, system_properties *properties,
                           double **f_bond, double **virial, double **t_bond, int *calc_matrix) {
    int i, i_bond, i_anchor, n_anchors, n_dim, n_bonds;
    double u, k, dr[3], *pos;
    double *origin;
    double **r_bond, **r_anchor, **f_anchor, **tau_local, *length, **u_bond;

    n_dim = parameters->n_dim;
    n_bonds = properties->bonds.n_bonds;
    r_bond = properties->bonds.r_bond;
    length = properties->bonds.length;
    u_bond = properties->bonds.u_bond;
    n_anchors = properties->anchors.n_anchors;
    f_anchor = properties->anchors.f_anchor;
    r_anchor = properties->anchors.r_anchor;
    tau_local = properties->anchors.tau_local;
    al_list *anchor_list = properties->anchors.anchor_list;
    
    if (n_dim != 3) {
        error_exit("anchor_potential_bd only supports 3 dimensions");
    }

    /* Zero vdw forces, potential energy and virial. */
    u = 0.0;
    if (properties->control.virial_flag)
        memset(virial[0], 0, n_dim * n_dim * sizeof(double));
    memset(f_bond[0], 0, n_bonds * n_dim * sizeof(double));
    memset(t_bond[0], 0, n_bonds * 3 * sizeof(double));
    
    for (i_anchor = 0; i_anchor < n_anchors; ++i_anchor) {
        k = properties->anchors.k[i_anchor];
        double r0 = properties->anchors.r0[i_anchor];
        origin = r_anchor[i_anchor];
       
        // Zero the total forces on the anchors, should probably move this somewhere else
        // XXX FIXME
        for (i = 0; i < n_dim; ++i) {
            f_anchor[i_anchor][i] = 0.0;
        }
        for (i = 0; i < 3; ++i) {
            tau_local[i_anchor][i] = 0.0;
        }
       
        for (al_list::iterator p = anchor_list[i_anchor].begin();
             p < anchor_list[i_anchor].end();
             p++) {

            i_bond = p->label;
            pos = p->pos;
            //dr is the distance from the end of the bond to the attachment site
            //r_bond is CM coordinates of the rod, pos is the attach site of the rod
            //and u_bond is the orientation vector of the rod
            for (i = 0; i < n_dim; ++i)
                dr[i] = r_bond[i_bond][i] - pos[i] - 
                    0.5 * length[i_bond] * u_bond[i_bond][i];

            double dr_mag2 = dot_product(n_dim, dr, dr);
            double factor = k * (1.0 - r0 / sqrt(dr_mag2));

            //Newton's laws for a spring
            for (i = 0; i < n_dim; ++i)
                f_bond[i_bond][i] = -factor * dr[i];
            for (i = 0; i < n_dim; ++i)
                f_anchor[i_anchor][i] += factor * dr[i];

            /* Update the potential energy from a spring */
            u += 0.5 * dot_product(n_dim, f_bond[i_bond], f_bond[i_bond]) / k;

            //std::cout << "  uanchor: " << i_anchor << ", bond: " << i_bond << std::endl;
            //std::cout << "    fbond("
            //    << f_bond[i_bond][0] << ", "
            //    << f_bond[i_bond][1] << ", "
            //    << f_bond[i_bond][2] << ")\n";
            //std::cout << "    u: " << 0.5 *  dot_product(n_dim, f_bond[i_bond], f_bond[i_bond]) / k << std::endl;

            // Update the virial forces
            // Use the center-to-center separation to do this, not the contact area
            // Distance is rij = rj - ri
            if (properties->control.virial_flag) {
                double r_cm[3] = {0.0};
                for (int i = 0; i < n_dim; ++i) {
                    r_cm[i] = origin[i] - r_bond[i_bond][i];
                }
                for (int i = 0; i < n_dim; ++i) {
                    for (int j = 0; j < n_dim; ++j) {
                        //virial[i][j] += dr[i] * factor * dr[j];
                        virial[i][j] += r_cm[i] * factor * dr[j];
                    }
                }
            }

            /* Check for nan in the energy calculation */
            /* Write all the things! */
            if(u != u) {
                fprintf(stderr, "NaN encountered in anchor_potential_bd\n");
                fprintf(stderr, "Step: %d\n", properties->i_current_step);
                fprintf(stderr, "anchor: %d i_bond: %d u: %g\n", i_anchor, i_bond, u);
                fprintf(stderr, "k: %g dr_mag2: %g \n", k, dr_mag2);
                fprintf(stderr, "dr: {%g %g %g}\n", dr[0], dr[1], dr[2]);
                fprintf(stderr, "length: %g pos: {%g %g %g}\n", length[i_bond], pos[0], pos[1], pos[2]);
                fprintf(stderr, "r_bond: {%g %g %g} u_bond: {%g %g %g}\n", 
                        r_bond[i_bond][0], r_bond[i_bond][1], r_bond[i_bond][2],
                        u_bond[i_bond][0], u_bond[i_bond][1], u_bond[i_bond][2]);
                fprintf(stderr, "origin: {%g %g %g}\n", origin[0], origin[1], origin[2]);
                exit(1);
            }
                
            double r_rel[3];
            double tau_i[3];
            for (i = 0; i < n_dim; ++i)
                r_rel[i] = pos[i] - origin[i];

            //Calculates the torque for the rotation of the anchor
            cross_product(r_rel, f_bond[i_bond], tau_i, n_dim);
            for (i = 0; i < 3; ++i)
                tau_local[i_anchor][i] += tau_i[i];

            //Torque on the bond from the anchor
            double r_contact[3] = {-0.5 * u_bond[i_bond][0] * length[i_bond],
                                   -0.5 * u_bond[i_bond][1] * length[i_bond],
                                   -0.5 * u_bond[i_bond][2] * length[i_bond]};
            cross_product(r_contact, f_bond[i_bond], t_bond[i_bond], n_dim);
            
            calc_matrix[i_bond] = 1;
        }
    }

    return u;
}
