/* This routine calculates the interaction energy of two 'square well line' particles,
 defined as a square well potential integrated over two finite line segments.
 This is equivalent to calculating the overlap area of a rectangle and an ellipse.
 See S. Varga and F. J. Vesely, J. Chem. Phys. 131, 194506 (2009).
 
 input: number of spatial dimensions (n_dim)
 number of periodic dimensions (n_periodic)
 square well depth (well_depth)
 square well diameter (well_diameter)
 real position of first line (r_1)
 scaled position of first line (s_1)
 director of first line (u_1)
 length of first line (length_1)
 real position of second line (r_2)
 scaled position of second line (s_2)
 director of second line (u_2)
 length of second line (length_2)
 
 output: potential energy of pair of interacting lines (return value). */

#include "bob.h"

#define SMALL 1.0e-12

double antiparallel_overlap(int n_dim, int n_periodic, double **h,
                                       double well_diameter,
                                       double *r_1, double *s_1, double *u_1, double length_1,
                                       double *r_2, double *s_2, double *u_2, double length_2)
{
    int i, j;
    double half_length_1, half_length_2, well_diameter2, dr2, d, alpha, beta, b, c, lambda_0, mu_0,
        r_min2, a2, a, gamma_0, delta_0, fact1, fact2,
        gamma_A, gamma_B, delta_A, delta_B, delta_C, delta_D,
        gamma_l, gamma_u, delta_l, delta_u, fact3, fact4,
        gamma_1, gamma_2, gamma_3, gamma_4,
        gamma_a, gamma_b, 
        u, z, z2, rho2, lambda_1, lambda_2;
    double ds[3], dr[3], r_min[3], u_2_temp[3];

    /* Compute various constants. */
    half_length_1 = 0.5 * length_1;
    half_length_2 = 0.5 * length_2;
    well_diameter2 = SQR(well_diameter);

    /* Compute pair separation vector. */
    for (i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
        ds[i] = s_2[i] - s_1[i];
        ds[i] -= NINT(ds[i]);
    }
    for (i = 0; i < n_periodic; ++i) {
        dr[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            dr[i] += h[i][j] * ds[j];
    }
    for (i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
        dr[i] = r_2[i] - r_1[i];

    /* Compute squared pair separation. */
    dr2 = 0.0;
    for (i = 0; i < n_dim; ++i)
        dr2 += SQR(dr[i]);

    /* If the pair separation exceeds the spherocylinder length plus the well diameter,
       the spherocylinders don't interact. */
    if (dr2 >= SQR(half_length_1 + half_length_2 + well_diameter))
        return 0.0;

    /* Compute dot product of directors and change sign of dot product and u_2 if dot product is negative. */
    for (i = 0; i < n_dim; ++i) {
        u_2_temp[i] = u_2[i];
    }
    d = 0.0;
    for (i = 0; i < n_dim; ++i) {
        d += u_1[i] * u_2_temp[i];
    }
    /* Return 0 if the rods aren't antiparallel */
    if (d >= 0) {
        return(0.0);
    }
    if (d < 0) {
        d = -d;
        for (i = 0; i < n_dim; ++i)
            u_2_temp[i] = -u_2[i];
    }
    alpha = 1.0 - SQR(d);
    beta = sqrt(alpha);

    if (alpha > SMALL) {        /* General case - lines aren't parallel. */

        /* Find point of closest approach of infinitely long 'carrier' lines. */
        b = c = 0.0;
        for (i = 0; i < n_dim; ++i) {
            b += dr[i] * u_2_temp[i];
            c += dr[i] * u_1[i];
        }
        lambda_0 = (c - b * d) / alpha;
        mu_0 = (-b + c * d) / alpha;

        /* Compute minimum distance between carrier lines. */
        r_min2 = 0.0;
        if (n_dim == 3) {
            for (i = 0; i < n_dim; ++i) {
                r_min[i] = dr[i] - lambda_0 * u_1[i] + mu_0 * u_2_temp[i];
                r_min2 += SQR(r_min[i]);
            }
        }

        /* If the minimum distance exceeds the interaction range, the lines don't interact. */
        if (r_min2 >= well_diameter2)
            return 0.0;

        /* Calculate interaction parameter. */
        a2 = well_diameter2 - r_min2;
        a = sqrt(a2);

        /* Transform to parameters defined with respect to 'intersection' of
           carrier lines (gamma and delta); gamma_0 and delta_0 are the centers
           of the line segments in terms of these new parameters.  In terms of
           gamma and delta, the integration region is a rectangle, and the
           interaction energy is proportional to the overlap area of this
           rectangle and the ellipse defined by gamma^2 + delta^2 - 2 * gamma *
           delta * d = r_ellipse^2. */
        gamma_0 = -lambda_0;
        delta_0 = -mu_0;

        /* Calculate coordinates of extremal points of ellipse. */
        fact1 = a / beta;
        fact2 = fact1 * d;
        gamma_A = -fact1;
        delta_A = -fact2;
        gamma_B = fact1;
        delta_B = fact2;
        delta_C = -fact1;
        delta_D = fact1;

        /* Determine upper and lower limits of bounding rectangle. */
        gamma_l = MAX(gamma_0 - half_length_1, gamma_A);
        gamma_u = MIN(gamma_0 + half_length_1, gamma_B);
        delta_l = MAX(delta_0 - half_length_2, delta_C);
        delta_u = MIN(delta_0 + half_length_2, delta_D);

        /* If the ellipse and rectangle don't intersect, the lines don't interact. */
        if (gamma_l >= gamma_u || delta_l >= delta_u)
            return 0.0;

        /* Calculate intersection points of horizontal sides of rectangle with ellipse. */
        fact3 = sqrt(ABS(a2 - SQR(delta_l) * alpha));
        fact4 = sqrt(ABS(a2 - SQR(delta_u) * alpha));
        gamma_1 = delta_l * d - fact3;
        gamma_2 = delta_l * d + fact3;
        gamma_3 = delta_u * d - fact4;
        gamma_4 = delta_u * d + fact4;

        /* Now determine limits of integration. */
        if (delta_l > delta_A)
            gamma_a = MAX(gamma_1, gamma_l);
        else {
            if (delta_u > delta_A)
                gamma_a = MAX(gamma_A, gamma_l);
            else
                gamma_a = MAX(gamma_3, gamma_l);
        }
        if (delta_u <= delta_B)
            gamma_b = MIN(gamma_4, gamma_u);
        else {
            if (delta_l <= delta_B)
                gamma_b = MIN(gamma_B, gamma_u);
            else
                gamma_b = MIN(gamma_2, gamma_u);
        }

        /* Calculate the length of the overlap area. */
        if(gamma_b > gamma_a)
            u = gamma_b - gamma_a;
        else
            u = 0.0;
    }
    else {                    /* The case of nearly parallel lines needs to be handled separately. */

        /* Resolve pair separation vector into parallel and perpendicular components. */
        z = 0.0;
        for (i = 0; i < n_dim; ++i)
            z += dr[i] * u_1[i];
        z2 = SQR(z);
        rho2 = dr2 - z2;

        /* If the perpendicular distance exceeds the interaction range, the lines don't interact. */
        if (rho2 >= well_diameter2)
            return 0.0;

        /* Calculate interaction parameter. */
        a2 = well_diameter2 - rho2;
        a = sqrt(a2);

        /* Determine limits of integrations. */
        lambda_1 = MIN(half_length_1, MAX(-half_length_1, -half_length_2 + z - a));
        lambda_2 = MIN(half_length_1, MAX(-half_length_1, half_length_2 + z - a));

        /* Calculate potential energy of two interacting square well lines. */
        u = lambda_2 - lambda_1;
    }

    /* Return potential energy of pair of interacting lines. */
    return u;
}

#undef SMALL
