/* This routine regulates the system pressure using the method of Berendsen et al.
   (J. Chem. Phys. 81, 3684 (1984)).

   Input: pointer to parameters structure (parameters)
          pointer to properties structure (properties)

   Output: the cell matrix and site coordinates are scaled on output */

#include "bob.h"

void berendsen_barostat_sphero(system_parameters * parameters, system_properties * properties)
{
    int n_dim, ortho_scaling, n_bonds, i_site, i, j, k, i_bond, j_site,
        box_check_flag;
    double delta, target_press, press_time, **press_tensor, **r_bond, **s_bond, **dr_tot,
        **h, **h_inv, **a, **b, *a_perp, *volume, beta, *length, half_min_lin, max_length;
    static int first_call = 1;
    static double **h_old, **r_old, **mu;

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    delta = parameters->delta;
    target_press = parameters->press;
    press_time = parameters->press_time;
    ortho_scaling = parameters->ortho_scaling;
    press_tensor = properties->thermo.press_tensor;
    r_bond = properties->bonds.r_bond;
    s_bond = properties->bonds.s_bond;
    dr_tot = properties->sites.dr_tot;
    h = properties->unit_cell.h;
    h_inv = properties->unit_cell.h_inv;
    a = properties->unit_cell.a;
    b = properties->unit_cell.b;
    a_perp = properties->unit_cell.a_perp;
    volume = &(properties->unit_cell.volume);
    n_bonds = properties->bonds.n_bonds;
    length = properties->bonds.length;
    box_check_flag = parameters->box_check_flag;

    /* Allocate memory for local arrays the first time the routine is called. */
    if (first_call) {
        h_old = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
        r_old = (double **) allocate_2d_array(n_bonds, n_dim, sizeof(double));
        mu = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
        first_call = 0;
    }

    /* Save current cell matrix and site positions. */
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j)
            h_old[i][j] = h[i][j];
    for (i_bond = 0; i_bond < n_bonds; ++i_bond)
        for (i = 0; i < n_dim; ++i)
            r_old[i_bond][i] = r_bond[i_bond][i];

    /* Compute scaling matrix. */
    beta = 1.0;                 /* Set compressibility to unity. */
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j) {
            if (i == j)
                mu[i][j] =
                    1.0 - (beta * delta / (n_dim * press_time)) * (target_press -
                                                                   press_tensor[i][j]);
            else {
                if (ortho_scaling)
                    mu[i][j] = 0.0;
                else
                    mu[i][j] = (beta * delta / (n_dim * press_time)) * press_tensor[i][j];
            }
        }
    if (ortho_scaling == 2) {
        double mu_avg = 0.0;
        for (i = 0; i < n_dim; ++i)
            mu_avg += mu[i][i];
        mu_avg /= n_dim;
        for (i = 0; i < n_dim; ++i)
            mu[i][i] = mu_avg;
    }

    /* Scale unit cell matrix. */
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j)
            h[i][j] = 0.0;
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j)
            for (k = 0; k < n_dim; ++k)
                h[i][j] += mu[i][k] * h_old[k][j];

    /* Compute inverse cell matrix and related quantities. */
    unit_cell_dimensions(n_dim, h, h_inv, a, b, a_perp, volume);

    /* Compute new site positions and add site displacements to displacement accumulators. */
    if (!ortho_scaling) {
        printf("berendsen barostat only currently implemented for ortho_scaling = 1\n");
        exit(1);
    }
    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {
        i_site = 2 * i_bond;
        j_site = i_site + 1;
        for (i = 0; i < n_dim; ++i) {
            r_bond[i_bond][i] = 0.0;
            //            properties->bonds.u_bond[i_bond][i] = 0.0;
            for (j = 0; j < n_dim; ++j) {
                r_bond[i_bond][i] += h[i][j] * s_bond[i_bond][j];
                //              properties->bonds.u_bond[i_bond][i] += mu[i][j] * properties->bonds.v_bond[i_bond][j] / properties->bonds.length[i_bond];
            }
            dr_tot[i_site][i] += r_bond[i_bond][i] - r_old[i_bond][i];
            dr_tot[j_site][i] += r_bond[i_bond][i] - r_old[i_bond][i];
        }
    }

    /* Calculate half the minimum dimension of the unit cell */
    half_min_lin = 0.5 * a_perp[0];
    for (i = 0; i < n_dim; ++i)
        half_min_lin = MIN(half_min_lin, a_perp[i]);
    
    /* Find length of longest rod in the system */
    max_length = length[0];
    for (i_bond = 0; i_bond < n_bonds; ++i_bond)
        max_length = MAX(max_length, length[i_bond]);
    
    /* If length of rod plus its interaction range is greater than half the box
       dimension, particle can  self interact. Exit program. */
    if (max_length + parameters->r_cutoff >
        half_min_lin && box_check_flag) { 
        fprintf(stderr, "Interaction range greater than half the minimum length of the cell. Exiting.\n");
        exit(1); 
    }
    
    properties->control.neighbor_list_flag = 1;

    return;
}
