/* This routine calculates the bond vector and related quantities for a single bond.
 
 Input: number of spatial dimensions (n_dim)
 number of periodic dimensions (n_periodic)
 unit cell matrix (h)
 array of scaled site positions (s)
 array of real site positions (r)
 bond endpoint site labels (site_1, site_2)
 
 Output: bond vector (v_bond)
 unit bond vector (u_bond)
 pointer to bond length (length)
 pointer to squared bond length (length2) */

#include "bob.h"

void bond_vector(int n_dim, int n_periodic, double **h,
                 double **s, double **r, int site_1, int site_2,
                 double *v_bond, double *u_bond, double *length, double *length2) {
    int i, j;
    double scale;
    double sv_bond[3];

    /* Compute bond vector. */
    for (i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
        sv_bond[i] = s[site_2][i] - s[site_1][i];
        sv_bond[i] -= NINT(sv_bond[i]);
    }
    for (i = 0; i < n_periodic; ++i) {
        v_bond[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            v_bond[i] += h[i][j] * sv_bond[j];
    }
    for (i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
        v_bond[i] = r[site_2][i] - r[site_1][i];

    /* Compute length of bond. */
    *length2 = 0.0;
    for (i = 0; i < n_dim; ++i)
        *length2 += SQR(v_bond[i]);
    *length = sqrt(*length2);

    /* Calculate unit vector along bond. */
    scale = 1.0 / *length;
    for (i = 0; i < n_dim; ++i)
        u_bond[i] = scale * v_bond[i];

    return;
}
