/* This routine calculates the forces, potential energy, and virial for spherocylinders
   interacting via a WCA potential, for any type of boundary condition
   (free, periodic, or mixed) and any number of dimensions, using an all-pairs search.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)


   Output: array of forces (f_bond)
   virial (virial)
   array of torques (t_bond)
   calc_matrix (set to 1 if force != 0)
   potential energy (return value) */


#include "minimum_distance.h"
#include "bob.h"
#include "xlink_management.h"

double brownian_sphero_all_pairs(system_parameters *parameters, system_properties *properties,
                                 double **f_bond, double **virial, double **t_bond, int *calc_matrix) {
    int i, j, i_bond, j_bond, n_bonds, n_dim, n_periodic;
    double **h, **r_bond, **s_bond, **u_bond, *length, sigma2, four_epsilon,
        f_cutoff, r_cutoff, r_cutoff2;
    double rho2, rho6, rho12, factor,
        r_min_mag, r_min_mag2, lambda, mu,
        delta, u, u_shift;
    double r_min[3], dr[3];

    n_dim = parameters->n_dim;
    n_periodic = parameters->n_periodic;
    n_bonds = properties->bonds.n_bonds;
    delta = parameters->delta;
    r_bond = properties->bonds.r_bond;
    s_bond = properties->bonds.s_bond;
    u_bond = properties->bonds.u_bond;
    length = properties->bonds.length;
    h = properties->unit_cell.h;
    sigma2 = 1.0;
    four_epsilon = 4.0;
    r_cutoff = parameters->r_cutoff_lj;
    r_cutoff2 = SQR(r_cutoff);
    rho2 = sigma2 / r_cutoff2;
    rho6 = CUBE(rho2);
    rho12 = SQR(rho6);
    u_shift = 1.0;

    for (i_bond = 0; i_bond < n_bonds; ++i_bond)
        for (i = 0; i < n_dim; ++i)
            f_bond[i_bond][i] = 0.0;
    for (i_bond = 0; i_bond < n_bonds; ++i_bond)
        for (i = 0; i < 3; ++i)
            t_bond[i_bond][i] = 0.0;
    for (i = 0; i < n_dim; ++i)
        for(j = 0; j < n_dim; ++j)
            virial[i][j] = 0.0;
    u = 0.0;

    for(i_bond = 0; i_bond < n_bonds - 1; ++i_bond) {
        for(j_bond = i_bond + 1; j_bond < n_bonds; ++j_bond) {
            min_distance_sphero_dr(n_dim, n_periodic, h,
                                   r_bond[i_bond], s_bond[i_bond], u_bond[i_bond], length[i_bond],
                                   r_bond[j_bond], s_bond[j_bond], u_bond[j_bond], length[j_bond],
                                   dr, r_min, &r_min_mag2, &lambda, &mu);

            if (r_min_mag2 < r_cutoff2) {
                /* Calculate WCA potential and forces */
                rho2 = sigma2 / r_min_mag2;
                rho6 = CUBE(rho2);
                rho12 = SQR(rho6);
                u += four_epsilon * (rho12 - rho6) + u_shift;
                factor = 6.0 * four_epsilon * (2.0 * rho12 - rho6) / r_min_mag2;

                f_cutoff = 0.1 / delta *
                    MIN(properties->bonds.gamma_par[i_bond], properties->bonds.gamma_par[j_bond]);

                /* Truncate force if greater than f_cutoff */
                r_min_mag = sqrt(r_min_mag2);
                if (factor * r_min_mag > f_cutoff)
                    factor = f_cutoff / r_min_mag;

                double f_lj[3] = {0.0, 0.0, 0.0};
                for (i = 0; i < n_dim; ++i)
                    f_lj[i] = factor * r_min[i];

                for (i = 0; i < n_dim; ++i) {
                    f_bond[i_bond][i] -= f_lj[i];
                    f_bond[j_bond][i] += f_lj[i];
                }

                /* Calculate torques */
                double r_contact_i[3] = {0.5 * u_bond[i_bond][0] * lambda,
                                         0.5 * u_bond[i_bond][1] * lambda,
                                         0.5 * u_bond[i_bond][2] * lambda};
                double r_contact_j[3] = {0.5 * u_bond[j_bond][0] * mu,
                                         0.5 * u_bond[j_bond][1] * mu,
                                         0.5 * u_bond[j_bond][2] * mu};
                double tau[3];
                cross_product(r_contact_i, f_lj, tau, n_dim);
                for (i = 0; i < 3; ++i)
                    t_bond[i_bond][i] -= tau[i];
                cross_product(r_contact_j, f_lj, tau, n_dim);
                for (i = 0; i < 3; ++i)
                    t_bond[j_bond][i] += tau[i];

                /* Add contribution to virial. */
                for (i = 0; i < n_dim; ++i)
                    for (j = 0; j < n_dim; ++j)
                        virial[i][j] += dr[i] * f_lj[j];

                calc_matrix[i_bond] = calc_matrix[j_bond] = 1;
            }

        }
    }
    return u;
}

/* This routine calculates the forces, potential energy, and virial for spherocylinders
   interacting via a WCA potential, for any type of boundary condition
   (free, periodic, or mixed) and any number of dimensions, using a neighbor list search.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)


   Output: array of forces (f_bond)
   virial (virial)
   orientational displacement (t_bond)
   calc_matrix (set to 1 if force != 0)
   potential energy (return value) */

double brownian_sphero_neighbor_lists(system_parameters *parameters,
                                      system_properties *properties,
                                      double **f_bond, double **virial,
                                      double **t_bond, int *calc_matrix) {
    int i, j, i_bond, j_bond, n_bonds, n_dim, n_periodic;
    nl_list *neighbs;
    double **h, **r_bond, **s_bond, **u_bond, *length, sigma2, four_epsilon,
        r_cutoff, r_cutoff2;
    double rho2, rho6, rho12, factor, lambda, mu, f_cutoff,
        delta, u, u_shift, r_min_mag;
    double r_min[3];

    n_dim = parameters->n_dim;
    n_periodic = parameters->n_periodic;
    n_bonds = properties->bonds.n_bonds;
    delta = parameters->delta;
    r_bond = properties->bonds.r_bond;
    s_bond = properties->bonds.s_bond;
    u_bond = properties->bonds.u_bond;
    length = properties->bonds.length;
    h = properties->unit_cell.h;
    neighbs = properties->neighbors.neighbs;
    sigma2 = 1.0;
    four_epsilon = 4.0;
    r_cutoff = parameters->r_cutoff_lj;
    r_cutoff2 = SQR(r_cutoff);
    rho2 = sigma2 / r_cutoff2;
    rho6 = CUBE(rho2);
    rho12 = SQR(rho6);
    u_shift = -four_epsilon * (rho12 - rho6);

    /* Check neighbor lists to make sure they're up to date */
    if (properties->control.neighbor_list_flag) {
        check_neighbor_lists_sphero(parameters, properties);
        properties->control.neighbor_list_flag = 0;
    }

    /* Zero everything */
    memset(f_bond[0], 0, n_bonds * n_dim * sizeof(double));
    memset(t_bond[0], 0, n_bonds * 3 * sizeof(double));
    if (properties->control.virial_flag)
        memset(virial[0], 0, n_dim * n_dim * sizeof(double));
    u = 0.0;

    
    properties->crosslinks.Clear();

    for(i_bond = 0; i_bond < n_bonds; ++i_bond) {
        for(nl_list::iterator p = neighbs[i_bond].begin();
            p != neighbs[i_bond].end();
            ++p) {

            j_bond = p->label;
            p->value = 0.0;

            // Calculate minimum distance between spherocylinders. Note this
            // updates dr local to neighbor list entry to remove redundant
            // calculations in crosslink routines.
            
            if (j_bond > i_bond) {
                min_distance_sphero_dr(n_dim, n_periodic, h,
                                       r_bond[i_bond], s_bond[i_bond], u_bond[i_bond], length[i_bond],
                                       r_bond[j_bond], s_bond[j_bond], u_bond[j_bond], length[j_bond],
                                       p->dr, r_min, &p->r_min_mag2, &lambda, &mu);

                if (parameters->nl_twoway_flag) {
                    for(nl_list::iterator p_neighbs = neighbs[j_bond].begin();
                        p_neighbs != neighbs[j_bond].end();
                        ++p_neighbs) {
                        if (p_neighbs->label == i_bond) {
                            for (int i = 0; i < 3; ++i)
                                p_neighbs->dr[i] = -p->dr[i];
                            p_neighbs->r_min_mag2 = p->r_min_mag2;;
                            break;
                        }
                    }
                } 
                
                if (p->r_min_mag2 < r_cutoff2) {
                    /* Calculate WCA potential and forces */
                    rho2 = sigma2 / p->r_min_mag2;
                    rho6 = CUBE(rho2);
                    rho12 = SQR(rho6);

                    u += four_epsilon * (rho12 - rho6) + u_shift;
                    factor = 6.0 * four_epsilon * (2.0 * rho12 - rho6) / p->r_min_mag2;

                    f_cutoff = 0.1 / delta *
                        MIN(properties->bonds.gamma_par[i_bond], properties->bonds.gamma_par[j_bond]);
                    /* Truncate force if greater than f_cutoff */
                    r_min_mag = sqrt(p->r_min_mag2);
                    if (factor * r_min_mag > f_cutoff)
                        factor = f_cutoff / r_min_mag;

                    double f_lj[3] = {0.0, 0.0, 0.0};
                    for (i = 0; i < n_dim; ++i)
                        f_lj[i] = factor * r_min[i];

                    /* Add lj force to accumulator */
                    for (i = 0; i < n_dim; ++i) {
                        f_bond[i_bond][i] -= f_lj[i];
                        f_bond[j_bond][i] += f_lj[i];
                    }

                    /* Calculate torques */
                    double r_contact_i[3] = {0.0, 0.0, 0.0};
                    for (i = 0; i < n_dim; ++i)
                        r_contact_i[i] = u_bond[i_bond][i] * lambda;
                    
                    double r_contact_j[3] = {0.0, 0.0, 0.0};
                    for (i = 0; i < n_dim; ++i)
                        r_contact_j[i] = u_bond[j_bond][i] * mu;

                    double tau[3];
                    cross_product(r_contact_i, f_lj, tau, n_dim);
                    for (i = 0; i < 3; ++i)
                        t_bond[i_bond][i] -= tau[i];
                    cross_product(r_contact_j, f_lj, tau, n_dim);
                    for (i = 0; i < 3; ++i)
                        t_bond[j_bond][i] += tau[i];

                    if (properties->control.virial_flag == 1)
                        for (i = 0; i < n_dim; ++i)
                            for (j = 0; j < n_dim; ++j)
                                virial[i][j] += p->dr[i] * f_lj[j];
                    
                    calc_matrix[i_bond] = calc_matrix[j_bond] = 1;
                  
                }
                if (properties->crosslinks.attachment_model_ == 1) {
                    /* FIXME cutoff wrong for one-stage */
                    properties->crosslinks.
                        PushOneStageNeighb(n_dim, i_bond, *p, u_bond, length);
                }
            }
        }
    }
    
    return u;
}
