/* This routine calculates the forces, potential energy, and virial for spherocylinders
   interacting via a WCA potential, for any type of boundary condition
   (free, periodic, or mixed) and any number of dimensions, using a neighbor list search.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)


   Output: array of forces (f_bond)
   virial (virial)
   torque (t_bond)
   calc_matrix (set to 1 if force != 0)
   potential energy (return value)

*/

#include "minimum_distance.h"
#include "bob.h"
#include "xlink_management.h"

#include <iostream>

double brownian_sphero_neighbor_lists_mp(system_parameters *parameters,
                                         system_properties *properties,
                                         double **f_bond, double **virial,
                                         double **t_bond, int *calc_matrix) {
    const double sigma2 = 1.0;
    const double four_epsilon = 4.0;
    const double r_cutoff = parameters->r_cutoff_lj;
    const double r_cutoff2 = SQR(r_cutoff);
    const double rho2 = sigma2 / r_cutoff2;
    const double rho6 = CUBE(rho2);
    const double rho12 = SQR(rho6);
    const double u_shift = -four_epsilon * (rho12 - rho6);
    const double r_cutoff_crosslink2 = SQR(parameters->r_cutoff);

    double ***f_local = properties->mp_local.f_local;
    double ***t_local = properties->mp_local.t_local;
    int **calc_local = properties->mp_local.calc_local;
    double ***virial_local = properties->mp_local.virial_local;

    /* Check neighbor lists to make sure they're up to date */
    if (properties->control.neighbor_list_flag) {
        check_neighbor_lists_sphero_mp(parameters, properties);
        properties->control.neighbor_list_flag = 0;
    }

    properties->crosslinks.Clear(); // This just clears the NL for one-stage, two-stage is copied every time, so unnecessary
    double u = 0.0;
    #ifdef ENABLE_OPENMP
    #pragma omp parallel
    #endif
    {
        /* Get current thread number to know which accumulator to work on */
        int i_thr = 0;
        #ifdef ENABLE_OPENMP
        i_thr = omp_get_thread_num();
        #endif

        memset(&(f_local[i_thr][0][0]), 0, properties->bonds.n_bonds *
               parameters->n_dim * sizeof(double));
        memset(&(t_local[i_thr][0][0]), 0, properties->bonds.n_bonds * 3 * sizeof(double));
        memset(calc_local[i_thr], 0, properties->bonds.n_bonds * sizeof(int));
        if (properties->control.virial_flag)
            memset(&(virial_local[i_thr][0][0]), 0,
                   parameters->n_dim * parameters->n_dim * sizeof(double));


        #ifdef ENABLE_OPENMP
        #pragma omp for reduction(+:u) schedule(runtime)
        #endif
        for(int i_bond = 0; i_bond < properties->bonds.n_bonds; i_bond++) {
            for (nl_list::iterator p = properties->neighbors.neighbs[i_bond].begin();
                 p < properties->neighbors.neighbs[i_bond].end();
                 ++p) {

                double lambda, mu, r_min[3];
                int j_bond = p->label;
                p->value = 0.0;
                min_distance_sphero_dr(parameters->n_dim, parameters->n_periodic,
                                       properties->unit_cell.h,
                                       properties->bonds.r_bond[i_bond],
                                       properties->bonds.s_bond[i_bond],
                                       properties->bonds.u_bond[i_bond],
                                       properties->bonds.length[i_bond],
                                       properties->bonds.r_bond[j_bond],
                                       properties->bonds.s_bond[j_bond],
                                       properties->bonds.u_bond[j_bond],
                                       properties->bonds.length[j_bond],
                                       p->dr, r_min, &p->r_min_mag2, &lambda, &mu);

                if (j_bond > i_bond && p->r_min_mag2 < r_cutoff2) {

                    /* Calculate WCA potential and forces */
                    double rho2 = sigma2 / p->r_min_mag2;
                    double rho6 = CUBE(rho2);
                    double rho12 = SQR(rho6);

                    u += four_epsilon * (rho12 - rho6) + u_shift;
                    double factor = 6.0 * four_epsilon * (2.0 * rho12 - rho6) / p->r_min_mag2;

                    //std::cout << "Step: " << properties->i_current_step << " ibond: " << i_bond << ", j_bond: " << j_bond << ", rminmag2: " << p->r_min_mag2 << ", cutoff2: " << r_cutoff2 << std::endl;

                    double f_cutoff = 0.1 / parameters->delta *
                        MIN(properties->bonds.gamma_par[i_bond],
                            properties->bonds.gamma_par[j_bond]);
                    //std::cout << "  fcutoff: " << f_cutoff << std::endl;

                    /* Truncate force if greater than f_cutoff */
                    double r_min_mag = sqrt(p->r_min_mag2);
                    if (factor * r_min_mag > f_cutoff) {
                        factor = f_cutoff / r_min_mag;
                        printf(" *** Force exceeded f_cutoff brownian_sphero_neighbor_lists_mp***\n");
                        //exit(1);
                    }
                    double f_lj[3] = {0.0, 0.0, 0.0};
                    for (int i = 0; i < parameters->n_dim; ++i)
                        f_lj[i] = factor * r_min[i];

                    /* Add lj force to accumulator */
                    for (int i = 0; i < parameters->n_dim; ++i) {
                        f_local[i_thr][i_bond][i] -= f_lj[i];
                        f_local[i_thr][j_bond][i] += f_lj[i];
                    }

                    /* Calculate torques */
                    double r_contact_i[3] = {0.0, 0.0, 0.0};
                    for (int i = 0; i < parameters->n_dim; ++i)
                        r_contact_i[i] = properties->bonds.u_bond[i_bond][i] * lambda;

                    double r_contact_j[3] = {0.0, 0.0, 0.0};
                    for (int i = 0; i < parameters->n_dim; ++i)
                        r_contact_j[i] = properties->bonds.u_bond[j_bond][i] * mu;
                    double tau[3];

                    cross_product(r_contact_i, f_lj, tau, parameters->n_dim);
                    for (int i = 0; i < 3; ++i)
                        t_local[i_thr][i_bond][i] -= tau[i];
                    cross_product(r_contact_j, f_lj, tau, parameters->n_dim);
                    for (int i = 0; i < 3; ++i)
                        t_local[i_thr][j_bond][i] += tau[i];

                    if (properties->control.virial_flag == 1)
                        for (int i = 0; i < parameters->n_dim; ++i)
                            for (int j = 0; j < parameters->n_dim; ++j)
                                virial_local[i_thr][i][j] += p->dr[i] * f_lj[j];

                    /* Set inverse drag matrix to be calculated during integration time */
                    calc_local[i_thr][i_bond] = 1;
                    calc_local[i_thr][j_bond] = 1;
                }
                if (properties->crosslinks.attachment_model_ == 1) {
                    properties->crosslinks.
                        PushOneStageNeighb(parameters->n_dim, i_bond, *p,
                                           properties->bonds.u_bond,
                                           properties->bonds.length);
                }
            }
        }
    }

    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));

    // Reduce the forces and torques to the bonds
    #ifdef ENABLE_OPENMP
    #pragma omp parallel for
    #endif
    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        for (int i_thr = 0; i_thr < properties->mp_local.n_threads; ++i_thr) {
            if (calc_local[i_thr][i_bond]) {
                for (int i = 0; i < parameters->n_dim; ++i) {
                    f_bond[i_bond][i] += f_local[i_thr][i_bond][i];
                }
                for (int i = 0; i < 3; ++i) {
                    t_bond[i_bond][i] += t_local[i_thr][i_bond][i];
                }
                calc_matrix[i_bond] = 1;
            }
        }
    }

    if (properties->control.virial_flag) {
        memset(virial[0], 0, parameters->n_dim * parameters->n_dim * sizeof(double));

        for (int i_thr = 0; i_thr < properties->mp_local.n_threads; ++i_thr)
            for (int i = 0; i < parameters->n_dim; ++i)
                for (int j = 0; j < parameters->n_dim; ++j)
                    virial[i][j] += virial_local[i_thr][i][j];
    }

    return u;
}
