#include "bob.h"
#include "chromosome_management.h"
#include "kinetochore.h"
#include "lookup_table_advanced.h"
#include "macros.h"
#include "minimum_distance.h"
#include "probabilities.h"
#include "triangle_mesh.h"

#include <gsl/gsl_integration.h>

#include <iostream>
#include <iomanip>

#define SKIN_ 8.0

ChromosomeManagement::ChromosomeManagement(system_parameters *parameters,
                                           system_properties *properties,
                                           const char* chromosome_file,
                                           const char* chromosome_config) {
    Init(parameters, properties, chromosome_file, chromosome_config);
}

ChromosomeManagement::ChromosomeManagement() {
}


void ChromosomeManagement::Init(system_parameters *parameters,
                                system_properties *properties,
                                const char* chromosome_file,
                                const char* chromosome_config) {
    if (chromosome_file == NULL) {
        std::cerr << "Warning: No chromosome file provided" << std::endl;
        return;
    }
    if (print_info_) {
        std::cout << "reading chromosome structure from " << chromosome_file << std::endl;
    }

    YAML::Node node = YAML::LoadFile(chromosome_file);

    Init(parameters, properties, &node);
}

void ChromosomeManagement::Init(system_parameters *parameters,
                                system_properties *properties,
                                YAML::Node* pnode) {
    parameters_ = parameters;
    properties_ = properties;

    nbonds_ = properties->bonds.n_bonds;
    ndim_   = parameters->n_dim;

    node_ = *pnode;

    ParseParams();

    if (set_biorient_flag_) {
        BiorientForceKinetochores(parameters_,
                                  properties_,
                                  true);
    }
	// XXX below added by ya boi shane: 
	if(align_flag_){
		AttachKinetochoreMicrotubules();
	}

    if (!enabled_) {
        nchromosomes_ = 0;
        nkcs_ = 0;
        return;
    }

    // Calculate the cutoff stuff
    double max_length = 0.0;
    if (parameters->dynamic_instability_flag) {
        max_length = (parameters->n_periodic == 0) ? properties->unit_cell.h[0][0] :
            0.5 * properties->unit_cell.h[0][0];
    }
    CalcCutoff(max_length);

    // Build the integration tables
    BuildTables();

    // Update the neighbor lists for all
    InitNeighborLists();

    Print();
}

void ChromosomeManagement::ParseParams() {

    // Find teh number of chromosomes
    nchromosomes_ = node_["chromosomes"]["chromosome"].size();
    if (node_["chromosomes"]["properties"]["replicate"]) {
        nchromosomes_ = node_["chromosomes"]["properties"]["replicate"].as<int>();
    }
    nkcs_ = 2*nchromosomes_;
    if (print_info_) {
        std::cout << "Number of chromosomes: " << nchromosomes_ << std::endl;
        std::cout << "Number of sister kinetochores: " << nkcs_ << std::endl;
    }

    if (nkcs_ == 0) {
        std::cout << "No kinetochores present, thus no chromosomes.  Returning before checking!\n";
        enabled_ = false;
        return;
    }
    enabled_ = true;

    try {
        if (node_["chromosomes"]["properties"]["filename"]) {
            filename_ = node_["chromosomes"]["properties"]["filename"].as<std::string>();
        }

        if (node_["chromosomes"]["properties"]["write_posit"]) {
            write_posit_ = node_["chromosomes"]["properties"]["write_posit"].as<bool>();
        }

        if (node_["chromosomes"]["properties"]["thermal_diffusion"]) {
            thermal_diffusion_ = node_["chromosomes"]["properties"]["thermal_diffusion"].as<bool>();
        }

        if (node_["chromosomes"]["properties"]["AF_tip_crowd"]) {
            af_tip_crowd_ = node_["chromosomes"]["properties"]["AF_tip_crowd"].as<bool>();
        }

        if (node_["chromosomes"]["properties"]["reconfigure"]) {
            reconfigure_ = node_["chromosomes"]["properties"]["reconfigure"].as<bool>();
        }

        if (node_["chromosomes"]["properties"]["AuroraB"]) {
            aurora_b_effects_ = node_["chromosomes"]["properties"]["AuroraB"].as<int>();
            if (aurora_b_effects_ == 1) {
                aurora_b_factor_ = node_["chromosomes"]["properties"]["AuroraB_factor"].as<double>();
            } else if ((aurora_b_effects_ == 2) || (aurora_b_effects_ == 3) || (aurora_b_effects_ == 4)) {
                aurora_b_force_dependence_ = node_["chromosomes"]["properties"]["AuroraB_force_dependence"].as<double>(); 
                aurora_b_force_plateau_ = node_["chromosomes"]["properties"]["AuroraB_force_plateau"].as<double>();
            }
        }

        if (node_["chromosomes"]["properties"]["use_poisson_process"]) {
            use_poisson_process_ = node_["chromosomes"]["properties"]["use_poisson_process"].as<int>();
        }

        if (node_["chromosomes"]["properties"]["AF_tip_tracking"]) {
            af_tip_tracking_ = node_["chromosomes"]["properties"]["AF_tip_tracking"].as<double>();
        }

        if (node_["chromosomes"]["properties"]["Dam1"]) {
            dam1_effects_ = node_["chromosomes"]["properties"]["Dam1"].as<int>();
            dam1_catastrophe_factor_ = node_["chromosomes"]["properties"]["Dam1_catastrophe_factor"].as<double>();
        }

        if (node_["chromosomes"]["properties"]["use_kc_potential"]) {
            use_kc_potential_ = node_["chromosomes"]["properties"]["use_kc_potential"].as<int>();
        }
        if (node_["chromosomes"]["properties"]["use_soft_chromatin_potential"]) {
            use_soft_chromatin_potential_ = node_["chromosomes"]["properties"]["use_soft_chromatin_potential"].as<int>();
        }

        if (node_["chromosomes"]["properties"]["progressive_locking"]) {
            prog_lock_ = node_["chromosomes"]["properties"]["progressive_locking"].as<int>();
        }

        // Check to see if we are going to proceed into anaphase when bi-orientation occurs
        if (node_["chromosomes"]["properties"]["Anaphase"]) {
            do_anaphase_ = (bool)node_["chromosomes"]["properties"]["Anaphase"].as<int>();
            sac_delay_ = node_["chromosomes"]["properties"]["SAC_delay"].as<int>();
            anaphase_rate_ = node_["chromosomes"]["properties"]["anaphase_rate"].as<double>();
            anaphase_depoly_ = node_["chromosomes"]["properties"]["anaphase_depoly"].as<double>();
        }

        nkctriverts_ = node_["chromosomes"]["properties"]["kc_triangulation_verticies"].as<int>();
        kc_triangulation_type_ = node_["chromosomes"]["properties"]["kc_triangulation_type"].as<std::string>();
        kc_diameter_ = node_["chromosomes"]["properties"]["kc_diameter"].as<double>();
        if (kc_triangulation_type_ == "rectangle") {
            kc_dimensions_[0] = node_["chromosomes"]["properties"]["kc_dimensions"][0].as<double>();
            kc_dimensions_[1] = node_["chromosomes"]["properties"]["kc_dimensions"][1].as<double>();
        } else if (kc_triangulation_type_ == "point") {
            kc_dimensions_[0] = node_["chromosomes"]["properties"]["kc_diameter"].as<double>();
            kc_dimensions_[1] = node_["chromosomes"]["properties"]["kc_diameter"].as<double>();
        }
        if (node_["chromosomes"]["properties"]["translational_drag"]) {
            if (print_info_) {
                std::cout << "NOTE: Enabling direct access to translational and rotational drag for chromosomes!\n";
            }
            gamma_t_ = node_["chromosomes"]["properties"]["translational_drag"].as<double>();
            gamma_r_ = node_["chromosomes"]["properties"]["rotational_drag"].as<double>();
        } else {
            // Stoke's formula with eta = 1/3pi for a sphere to make units nice
            double kc_rad = 0.5 * kc_diameter_;
            gamma_t_ = 2 * kc_rad;
            gamma_r_ = (8./3.) *kc_rad*kc_rad*kc_rad;
        }
        nkmc_ = node_["chromosomes"]["properties"]["nkmc"].as<int>();
        delta_kmc_ = nkmc_ * parameters_->delta;
        kc_spb_max_distance_ = node_["chromosomes"]["properties"]["kc_spb_max_distance"].as<double>();
        kc_spb_min_distance_ = node_["chromosomes"]["properties"]["kc_spb_min_distance"].as<double>();
        chromatin_r0_= node_["chromosomes"]["properties"]["chromatin_rest_length"].as<double>();
        chromatin_k_ = node_["chromosomes"]["properties"]["chromatin_spring"].as<double>();
        chromatid_length_ = node_["chromosomes"]["properties"]["chromatid_length"].as<double>();
        chromatid_diameter_ = node_["chromosomes"]["properties"]["chromatid_diameter"].as<double>();
        chromatid_kc_offset_ = node_["chromosomes"]["properties"]["chromatid_kc_offset"].as<double>();
        if (node_["chromosomes"]["properties"]["common_chromatin_angular_spring"]) {
            chromatin_ktor_=   node_["chromosomes"]["properties"]["common_chromatin_angular_spring"].as<double>();
            chromatid_kvtor_ = node_["chromosomes"]["properties"]["common_chromatin_angular_spring"].as<double>();
        } else  {
            chromatin_ktor_= node_["chromosomes"]["properties"]["chromatin_torsion_spring"].as<double>();
            chromatid_kvtor_ = node_["chromosomes"]["properties"]["chromatid_kv_spring"].as<double>();
        }
        chromatid_mt_repulsion_= node_["chromosomes"]["properties"]["chromatid_mt_repulsion"].as<double>();
        // Set the mt repulsion sigma at 5 sigma of the total, which will be the cutoff
        chromatid_mt_sigma_ = (chromatid_diameter_/2. + 0.5)/5.;
        chromatid_mt_cutoff_ = 5. * chromatid_mt_sigma_;
        chromatid_mt_cutoff2_ = SQR(chromatid_mt_cutoff_);
        chromatid_mt_fc_distance_ = node_["chromosomes"]["properties"]["chromatid_mt_fc_distance"].as<double>();
        chromatid_mt_fc_factor_ = node_["chromosomes"]["properties"]["chromatid_mt_fc_factor"].as<double>();
        //af_location_radius_ = node_["chromosomes"]["properties"]["AF_location_radius"].as<double>();
        af_exclusion_radius_ = node_["chromosomes"]["properties"]["AF_exclusion_radius"].as<double>();
        naf_ = node_["chromosomes"]["properties"]["AF_number_complexes"].as<int>();
        af_k_ = node_["chromosomes"]["properties"]["AF_spring"].as<double>();
        //if (prog_lock_ == 0) {
        //    af_kr_ = node_["chromosomes"]["properties"]["AF_spring_angular"].as<double>();
        //}
        af_r0_ = node_["chromosomes"]["properties"]["AF_rest_length"].as<double>() + 0.5;
        if (node_["chromosomes"]["properties"]["AF_characteristic_length"]) {
            af_xc_assemble_ = node_["chromosomes"]["properties"]["AF_characteristic_length"].as<double>();
            af_xc_disemble_ = node_["chromosomes"]["properties"]["AF_characteristic_length"].as<double>();
            af_xc_side_     = node_["chromosomes"]["properties"]["AF_characteristic_length"].as<double>();
        } else {
            af_xc_assemble_ = node_["chromosomes"]["properties"]["AF_xc_assemble"].as<double>();
            af_xc_disemble_ = node_["chromosomes"]["properties"]["AF_xc_disemble"].as<double>();
            af_xc_side_     = node_["chromosomes"]["properties"]["AF_xc_side"].as<double>();
        }
        af_chic_ = node_["chromosomes"]["properties"]["AF_characteristic_angular_factor"].as<double>();
        if (node_["chromosomes"]["properties"]["AF_potential_type"]) {
            af_potential_type_ = node_["chromosomes"]["properties"]["AF_potential_type"].as<std::string>(); 
        } else {
            af_potential_type_ = "harmonic";
        }
        af_tip_eps_eff_ = node_["chromosomes"]["properties"]["AF_tip_concentration"].as<double>();
        if (node_["chromosomes"]["properties"]["AF_tip_on_rate"]) {
            af_tip_on_rate_assemble_ = node_["chromosomes"]["properties"]["AF_tip_on_rate"].as<double>();
            af_tip_on_rate_disemble_ = node_["chromosomes"]["properties"]["AF_tip_on_rate"].as<double>();
        } else {
            af_tip_on_rate_assemble_ = node_["chromosomes"]["properties"]["AF_tip_on_rate_assemble"].as<double>();
            af_tip_on_rate_disemble_ = node_["chromosomes"]["properties"]["AF_tip_on_rate_disemble"].as<double>();
        }
        af_tip_diffusion_ = node_["chromosomes"]["properties"]["AF_tip_diffusion"].as<double>();
        af_side_eps_eff_ = node_["chromosomes"]["properties"]["AF_side_concentration"].as<double>();
        af_side_on_rate_ = node_["chromosomes"]["properties"]["AF_side_on_rate"].as<double>();
        af_side_diffusion_ = node_["chromosomes"]["properties"]["AF_side_diffusion"].as<double>();
        af_tip_distance_ = node_["chromosomes"]["properties"]["AF_tip_distance"].as<double>();
        //// Tip enhancement
        //af_tip_enhancement_ = node_["chromosomes"]["properties"]["AF_tip_enhancement"].as<double>();
        //af_tip_distance_ = node_["chromosomes"]["properties"]["AF_tip_distance"].as<double>();
        //af_diffusion_tip_ = node_["chromosomes"]["properties"]["AF_tip_diffusion"].as<double>();
        // Stabilization
        // Only check if enabled
        if (node_["chromosomes"]["properties"]["AF_stabilization_length"]) {
            af_l_stabilize_ = node_["chromosomes"]["properties"]["AF_stabilization_length"].as<double>();
            af_f_stabilize_vg_ = node_["chromosomes"]["properties"]["AF_stabilization_force_vg"].as<double>();
            af_f_stabilize_vs_ = node_["chromosomes"]["properties"]["AF_stabilization_force_vs"].as<double>();
            af_f_stabilize_fc_ = node_["chromosomes"]["properties"]["AF_stabilization_force_fc"].as<double>();
            af_f_stabilize_fr_ = node_["chromosomes"]["properties"]["AF_stabilization_force_fr"].as<double>();
            af_mtpoly_vmax_ = node_["chromosomes"]["properties"]["AF_vmax"].as<double>();
            af_mtpoly_dlmax_ = af_mtpoly_vmax_ * parameters_->delta;
        }
        // velocity
        if (node_["chromosomes"]["properties"]["AF_velocity"]) {
            af_velocity_ = node_["chromosomes"]["properties"]["AF_velocity"].as<double>();
            af_stall_force_ = node_["chromosomes"]["properties"]["AF_stall_force"].as<double>();
        }

        Allocate();

        // Check for progressive locking variable
        if (prog_lock_ != 0) {
            for (int i = 0; i < naf_+1; ++i) {
                af_kr_[i] = node_["chromosomes"]["properties"]["AF_kr"][i].as<double>();
            }
        } else {
            af_kr_[0] = node_["chromosomes"]["properties"]["AF_spring_angular"].as<double>();
        }

        if (node_["chromosomes"]["properties"]["color_chromatid"]) {
            color_chromatid_[0] = node_["chromosomes"]["properties"]["color_chromatid"][0].as<float>();
            color_chromatid_[1] = node_["chromosomes"]["properties"]["color_chromatid"][1].as<float>();
            color_chromatid_[2] = node_["chromosomes"]["properties"]["color_chromatid"][2].as<float>();
            color_chromatid_[3] = node_["chromosomes"]["properties"]["color_chromatid"][3].as<float>();
        }

        if (node_["chromosomes"]["properties"]["r_eplicate"]) {
            // Fake the initiailization of the first one over and over
            for (int ic = 0; ic < nchromosomes_; ++ic) {
                color_[ic][0] = node_["chromosomes"]["chromosome"][0]["color"][0].as<float>();
                color_[ic][1] = node_["chromosomes"]["chromosome"][0]["color"][1].as<float>();
                color_[ic][2] = node_["chromosomes"]["chromosome"][0]["color"][2].as<float>();
                color_[ic][3] = node_["chromosomes"]["chromosome"][0]["color"][3].as<float>();
                CreateKinetochores(ic, true);
            }
        } else {
            // initialize each
            for (int ic = 0; ic < nchromosomes_; ++ic) {
                color_[ic][0] = node_["chromosomes"]["chromosome"][ic]["color"][0].as<float>();
                color_[ic][1] = node_["chromosomes"]["chromosome"][ic]["color"][1].as<float>();
                color_[ic][2] = node_["chromosomes"]["chromosome"][ic]["color"][2].as<float>();
                color_[ic][3] = node_["chromosomes"]["chromosome"][ic]["color"][3].as<float>();
                for (int icolor = 0; icolor < 4; ++icolor) {
                    if (node_["chromosomes"]["chromosome"][ic]["color_af"][0]) {
                        color_af_[ic][icolor] = node_["chromosomes"]["chromosome"][ic]["color_af"][icolor].as<float>();
                    } else {
                        color_af_[ic][icolor] = color_[ic][icolor];
                    }
                }
                CreateKinetochores(ic);
            }
        }
    } catch (std::exception &ex) {
        std::cerr << "Exception thrown during parsing of Chromosome YAML file, check: \n";
        std::cerr << "   " << ex.what() << std::endl;
        enabled_ = false;
        return;
    }
}

void ChromosomeManagement::Allocate() {
    // Generate the chromosome vector, then let it initialize itself
    kinetochores_ = new Kinetochore[nkcs_];
    neighbors_ = new nl_list[nkcs_];
    color_ = (float**) allocate_2d_array(nkcs_, 4, sizeof(float));
    color_chromatid_ = (float*) allocate_1d_array(4, sizeof(float));
    color_af_ = (float**) allocate_2d_array(nkcs_, 4, sizeof(float));
    second_sister_ = (bool*) allocate_1d_array(nkcs_, sizeof(bool));
    if (prog_lock_) {
        af_kr_ = (double*) allocate_1d_array(naf_+1, sizeof(double));
    } else {
        af_kr_ = (double*) allocate_1d_array(1, sizeof(double));
    }
    f_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    f_interkc_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    t_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    r_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    u_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    ueff_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    v_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    w_ = (double**) allocate_2d_array(nkcs_, ndim_, sizeof(double));
    vertex_rel_ = (double ***) allocate_3d_array(nkcs_, nkctriverts_, 3, sizeof(double));
    dr_tot_kc_ = (double ***) allocate_3d_array(nkcs_, nkctriverts_, 3, sizeof(double));
    dr_tot_mt_ = (double **) allocate_2d_array(properties_->bonds.n_bonds*2, ndim_, sizeof(double));
    anchors_ = new al_list[nkcs_];
    tris_ = new triangle_mesh[nkcs_];
    fkcmttip_ = (double**)allocate_2d_array(properties_->bonds.n_bonds, ndim_, sizeof(double));
    kcattachmentstatus_ = (int*) allocate_1d_array(nkcs_, sizeof(int));
    chromosome_orientation_status_ = (int*) allocate_1d_array(nchromosomes_, sizeof(int));

    n_exp_ = (double*) allocate_1d_array(nkcs_, sizeof(double));
    n_bound_ = (int*) allocate_1d_array(nkcs_, sizeof(int));
    n_exp_lookup_ = (LookupTableAdvanced*) allocate_1d_array(1, sizeof(LookupTableAdvanced));

    // Any initial conditions
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        n_bound_[ikc] = 0;
    }

    int nthreads = 1;
    #ifdef ENABLE_OPENMP
    nthreads = omp_get_num_threads();
    #endif
    gsl_workspaces_ = new gsl_integration_workspace*[nthreads];

    for (int ithread = 0; ithread < nthreads; ++ithread) {
        gsl_workspaces_[ithread] = gsl_integration_workspace_alloc(100000);
    }
}

// Creates the kinetochores and chromatid variables
void ChromosomeManagement::CreateKinetochores(int ichromo,
                                              bool replicate) {
    int kc0 = 2*ichromo;
    int kc1 = 2*ichromo+1;
    YAML::Node subnode;
    set_biorient_flag_ = false;
    if (replicate) {
        subnode = node_["chromosomes"]["chromosome"][0];
    } else {
        subnode = node_["chromosomes"]["chromosome"][ichromo];
    }

    // Generate the intial position of the kinetochore pair
    std::string insertion_type = subnode["insertion_type"].as<std::string>();
	int n_chromos = (int)node_["chromosomes"]["chromosome"].size();
    double r[3] = {0.0};
    double u[3] = {0.0};
    double v[3] = {0.0};
    double w[3] = {0.0};
    double offset[3] = {0.0};
    double vshiftA = 0.0;
    double vshiftB = 0.0;
    if (insertion_type == "spb") {
        double rbox = 0.5 * properties_->unit_cell.h[0][0];
        // Inner radius to prevent overlap with nucleus
        double rinner = rbox - kc_diameter_;
        double rmag2;

        bool near_spb = false;
        bool inside = false;
        bool overlap = true;
        do {
            inside = false;
            near_spb = false;
            // Generate random position
            rmag2 = 0.0;
            for (int i = 0; i < ndim_; ++i) {
                r[i] = (gsl_rng_uniform(properties_->rng.r) - 0.5) *
                       (2*rinner);
                rmag2 += SQR(r[i]);
            }

            // Check if inside the boundary
            if (rmag2 < SQR(rinner)) {
                inside = true;
            }

            // Check if near spb
            for (int ispb = 0; ispb < properties_->anchors.n_anchors; ++ispb) {
                double dr_spb[3] = {0.0};
                double dr_spb_mag2 = 0.0;
                for (int i = 0; i < ndim_; ++i) {
                    dr_spb[i] = properties_->anchors.r_anchor[ispb][i] - r[i];
                    dr_spb_mag2 += SQR(dr_spb[i]);
                }

                if ((dr_spb_mag2 < SQR(kc_spb_max_distance_)) && (dr_spb_mag2 > SQR(kc_spb_min_distance_))) {
                    near_spb = true;
                }
            }
        } while (!inside || !near_spb);

        // Now generate a random orientation
        generate_random_unit_vector(ndim_, u, properties_->rng.r);

        // Set the variables
        for (int i = 0; i < ndim_; ++i) {
            // Do an offset of each of 0.5 requil
            r_[kc0][i] = r[i] + 0.5 * chromatin_r0_ * u[i];
            r_[kc1][i] = r[i] - 0.5 * chromatin_r0_ * u[i];;
            u_[kc0][i] = u[i];
            u_[kc1][i] = u[i];
        }

    } else if (insertion_type == "spblink") {
        // Place the KC near the SPB AWAY from the other SPB 
        double rbox = 0.5 * properties_->unit_cell.h[0][0];
        // Inner radius to prevent overlap with nucleus
        double rinner = rbox - kc_diameter_;
        double rmag2;

        bool near_spb = false;
        bool inside = false;
        bool overlap = true;

        do {
            inside = false;
            near_spb = false;
            // Generate random position
            rmag2 = 0.0;
            for (int i = 0; i < ndim_; ++i) {
                r[i] = (gsl_rng_uniform(properties_->rng.r) - 0.5) *
                       (2*rinner);
                rmag2 += SQR(r[i]);
            }

            // Check if inside the boundary
            if (rmag2 < SQR(rinner)) {
                inside = true;
            }

            // Check if near spb1, and not near SPB2
            double dr_spb0[3] = {0.0};
            double dr_spb1[3] = {0.0};
            double dr_spb0_mag2 = 0.0;
            double dr_spb1_mag2 = 0.0;
            for (int i = 0; i < ndim_; ++i) {
                dr_spb0[i] = properties_->anchors.r_anchor[0][i] - r[i];
                dr_spb1[i] = properties_->anchors.r_anchor[1][i] - r[i];
            }
            dr_spb0_mag2 = dot_product(ndim_, dr_spb0, dr_spb0);
            dr_spb1_mag2 = dot_product(ndim_, dr_spb1, dr_spb1);

            if ((dr_spb1_mag2 > (dr_spb0_mag2 + 2.5 * properties_->anchors.diameter[0] * properties_->anchors.diameter[0])) && 
                (dr_spb0_mag2 < SQR(kc_spb_max_distance_)) && 
                (dr_spb0_mag2 > SQR(kc_spb_min_distance_))) {
                near_spb = true;
            }
        } while (!inside || !near_spb);

        // Now generate a random orientation
        generate_random_unit_vector(ndim_, u, properties_->rng.r);

        // If SPBlink, point the kinetochore at the damn SPB
        if (insertion_type == "spblink") {
            for (int i = 0; i < 3; ++i) {
                u[i] = properties_->anchors.r_anchor[0][i] - r[i];
            }
            double unormfactor = sqrt(dot_product(3, u, u));
            for (int i = 0; i < 3; ++i) {
                u[i] /= unormfactor;
            }
        }

        // Set the variables
        for (int i = 0; i < ndim_; ++i) {
            // Do an offset of each of 0.5 requil
            r_[kc0][i] = r[i] + 0.5 * chromatin_r0_ * u[i];
            r_[kc1][i] = r[i] - 0.5 * chromatin_r0_ * u[i];;
            u_[kc0][i] = u[i];
            u_[kc1][i] = u[i];
        }
    } else if (insertion_type == "xyz") {
        for (int i = 0; i < ndim_; ++i) {
            r[i] = subnode["x0"][i].as<double>();
            u[i] = subnode["u0"][i].as<double>();
            offset[i] = subnode["offset"][i].as<double>();
        }

        for (int i = 0; i < ndim_; ++i) {
            r_[kc0][i] = r[i];
            r_[kc1][i] = r[i] + offset[i];
            u_[kc0][i] = u[i];
            u_[kc1][i] = u[i];
        }

    } else if (insertion_type == "aligned") {
	
		align_flag_ = true;
        // Place the KC near the SPB AWAY from the other SPB 
		double rbox = 0.5 * properties_->unit_cell.h[0][0];
		// Inner radius to prevent overlap with nucleus
		double rinner = rbox - kc_diameter_;
		double rmag2;

		bool inside = false;
		// Distribute chromosomes uniformly along azimuthal plane 
		double alpha = ichromo * (6.283/n_chromos); 
		double r_0 = rinner/3;
		r[0] = r_0*cos(alpha); 
		r[1] = 0;
		r[2] = r_0*sin(alpha);

		// Check if inside the boundary
		rmag2 = 0.0;
		for(int i(0); i < ndim_; i++){
			rmag2 += SQR(r[i]);
		}
		if (rmag2 < SQR(rinner)) {
			inside = true;
		}
		if(!inside){
			printf("WOAH BUDDY - CANT PUT THEM CHROMOS THERE!\n");
			exit(1);
		}
		// Align kinetochore 
		for (int i = 0; i < 3; ++i) {
			u[i] = properties_->anchors.r_anchor[0][i] - r[i];
		}
		double unormfactor = sqrt(dot_product(3, u, u));
		for (int i = 0; i < 3; ++i) {
			u[i] /= unormfactor;
		}

		// Set the variables
		for (int i = 0; i < ndim_; ++i) {
            // Do an offset of each of 0.5 requil
            r_[kc0][i] = r[i] + 0.5 * chromatin_r0_ * u[i];
            r_[kc1][i] = r[i] - 0.5 * chromatin_r0_ * u[i];;
            u_[kc0][i] = u[i];
            u_[kc1][i] = u[i];
		}
	} else if (insertion_type == "fakeinsert") {
        for (int i = 0; i < ndim_; ++i) {
            r[i] = subnode["x0"][i].as<double>();
            u[i] = subnode["u0"][i].as<double>();
        }
        for (int i = 0; i < ndim_; ++i) {
            // Do an offset of each of 0.5 requil
            r_[kc0][i] = r[i] + 0.5 * chromatin_r0_ * u[i];
            r_[kc1][i] = r[i] - 0.5 * chromatin_r0_ * u[i];;
            u_[kc0][i] = u[i];
            u_[kc1][i] = u[i];
        }
    } else if (insertion_type == "fullspecify") {
        for (int i = 0; i < ndim_; ++i) {
            r_[kc0][i] = subnode["rA"][i].as<double>();
            u_[kc0][i] = subnode["uA"][i].as<double>();
            r_[kc1][i] = subnode["rB"][i].as<double>();
            u_[kc1][i] = subnode["uB"][i].as<double>();
        }
        // Check for vshifts
        if (subnode["vshiftA"]) {
            vshiftA = subnode["vshiftA"].as<double>();
        }
        if (subnode["vshiftB"]) {
            vshiftB = subnode["vshiftB"].as<double>();
        }
    } else if (insertion_type == "random") {
        for (int i = 0; i < ndim_; ++i) {
            r_[kc0][i] = (gsl_rng_uniform(properties_->rng.r) - 0.5) * 4 * chromatin_r0_;
            r_[kc1][i] = (gsl_rng_uniform(properties_->rng.r) - 0.5) * 4 * chromatin_r0_;
        }
        generate_random_unit_vector(ndim_, u_[kc0], properties_->rng.r);
        generate_random_unit_vector(ndim_, u_[kc1], properties_->rng.r);
    } else {
        std::cerr << "No other kinetochore insertion available yet, exiting\n";
        exit(1);
    }

    if (subnode["sites"]) {
        sites_specified_ = true;
    }

    // Common kinetochore initialization
    kinetochores_[kc0].Init(this,
                            kc0,
                            ichromo,
                            &(second_sister_[kc0]),
                            r_[kc0],
                            u_[kc0],
                            ueff_[kc0],
                            v_[kc0],
                            w_[kc0],
                            vshiftA);
    kinetochores_[kc1].Init(this,
                            kc1,
                            ichromo,
                            &(second_sister_[kc1]),
                            r_[kc1],
                            u_[kc1],
                            ueff_[kc1],
                            v_[kc1],
                            w_[kc1],
                            vshiftB);

    if (subnode["special"]) {
        std::string special_instructions = subnode["special"].as<std::string>();

        if (special_instructions == "forcetipattach") {
            // Force kinetochore attachments
            std::cout << "Forcing kinetochore attachments\n";
            std::vector<int> bondlist0 {0, 1, 2};
            kinetochores_[kc0].ForceAttach(parameters_,
                                           properties_,
                                           bondlist0);
            n_bound_[kc0] = naf_;

            std::vector<int> bondlist1 {3, 4, 5};
            kinetochores_[kc1].ForceAttach(parameters_,
                                           properties_,
                                           bondlist1);
            n_bound_[kc1] = naf_;
            PrintFrame();
        } else if (special_instructions == "force_attach_single") {
            std::cout << "Forcing kinetochore single attachment\n";
            std::vector<int> bondlist0 = {0};
            kinetochores_[kc0].ForceAttach(parameters_,
                                           properties_,
                                           bondlist0);
            n_bound_[kc0] = 1;
            PrintFrame();
        } else if (special_instructions == "biorient") {
            set_biorient_flag_ = true;
        }
    }

    if(insertion_type == "spblink"){
        // Link the kinetochores to the correct pole
        LinkSPBKinetochores(parameters_,
                            properties_,
                            kc0,
                            kc1);
    }
}

void ChromosomeManagement::CalcCutoff(double max_length) {
    if (print_info_) {
        std::cout << "   Calculating KC 1->2 cutoff\n";
    }
    for (int ibond = 0; ibond < nbonds_; ++ibond) {
        max_length = MAX(properties_->bonds.length[ibond], max_length);
    }
    const double eps = 1E-3; // Tolerance for number of binding sites between pair
    const double temp = 1.0;
    double eps_eff = std::max(af_tip_eps_eff_, af_side_eps_eff_);
    double xc = MAX(fabs(af_xc_assemble_), fabs(af_xc_disemble_));
    double af_kr = 0.0;
    for (int isite = 0; isite < naf_+1; ++isite) {
        af_kr = MAX(af_kr, af_kr_[isite]);
    }
    double epseffprime = eps_eff * exp(0.5 * af_k_ * xc * xc + 0.5 * af_kr * af_chic_ * af_chic_);
    double rc_0 = sqrt(2.0 / (af_k_) *
                       temp *
                       log(epseffprime * max_length / eps *sqrt(2.0 * temp / af_k_)));
    double angle_shift = 0.0;
    double alpha = af_k_ / 2.;
    double alpha2 = af_kr / 2.;
    const double eps2 = 1e-5;
    double a_cutoff = exp(alpha*xc*xc + alpha2*af_chic_*af_chic_)/sqrt(alpha) * my_erfinv(1 - 4.0*sqrt(alpha/M_PI)*eps2) +
        af_r0_ + xc;
    if (af_kr > 0.0) {
        angle_shift = 1.0;
    }
    rcutoff_ = rc_0 + af_r0_ + xc + angle_shift;
    acutoff_ = a_cutoff + angle_shift;
    std::cout << "      Rmin cutoff (Y): " << rcutoff_ << std::endl;
    std::cout << "      Xlim cutoff (X): " << acutoff_ << std::endl;
    rcutoff2_ = rcutoff_*rcutoff_;
}

void ChromosomeManagement::BuildTables() {
    std::cout << "WARNING: Lookup Tables in Chromosomes are disabled in favor of coarser kMC methods\n";
    return;
    // Only build if r0 isn't 0.0
    //if (af_r0_ != 0.0) {
    //    std::vector<double> x[4];
    //    double bin_size = 0.05;
    //    double alpha = af_k_ / 2.;
    //    double alpha2 = af_kr_ / 2.;
    //    double eps = 1e-5;
    //    double a_cutoff = exp(alpha*af_xc_assemble_*af_xc_assemble_)/sqrt(alpha) * my_erfinv(1 - 4.0*sqrt(alpha/M_PI)*eps) +
    //        af_r0_ + af_xc_assemble_;
    //    double angle_shift = 0.0;
    //    if (af_kr_ > 0.0) {
    //        //angle_shift = 1.0/sqrt(alpha2) * my_erfinv(1 - 4.0*sqrt(alpha2/M_PI)*eps);
    //        angle_shift = 0.0;
    //    }
    //    //std::cout << "A Angle shift: " << angle_shift << std::endl;
    //    a_cutoff = a_cutoff + angle_shift;
    //    double y_cutoff = rcutoff_;

    //    kc_params params;
    //    params.alpha = alpha;
    //    params.alpha2 = alpha2;
    //    params.r0 = af_r0_;
    //    params.xc = af_xc_assemble_;

    //    std::cout << "   KC 1-> a_cutoff: " << a_cutoff << std::endl;
    //    std::cout << "   KC 1-> y_cutoff: " << y_cutoff << std::endl;

    //    // our x values can be less than zero, and the potential can be asymmetric, so call the correct init function
    //    for (double a = -a_cutoff; a <= a_cutoff; a += bin_size) {
    //        x[0].push_back(a);
    //    }

    //    for (double y0 = 0.0; y0 <= y_cutoff; y0 += bin_size) {
    //        x[1].push_back(y0);
    //    }

    //    for (double theta0 = -1.0; theta0 <= 1.0; theta0 += bin_size) {
    //        x[2].push_back(theta0);
    //        x[3].push_back(theta0);
    //    }

    //    n_exp_lookup_->InitAsymmetric(4, x, &prob_1_2_advanced_upper, &prob_1_2_advanced_lower, &params);
    //    //n_exp_lookup_->Init(4, x, &prob_1_2_advanced, &params);
    //}
}

void ChromosomeManagement::Print() {
    if (!print_info_) {
        return;
    }
    std::cout << "Chromosome Common Properties\n";
    std::cout << "   --- kMC Properties ---" << std::endl;
    std::cout << "   n kmc: " << nkmc_ << std::endl;
    std::cout << "   delta kmc: " << delta_kmc_ << std::endl;
    std::cout << "   use kc potential: " << use_kc_potential_ << std::endl;
    std::cout << "   use soft chromatin potential: " << use_soft_chromatin_potential_ << std::endl;
    std::cout << "   --- Kinetochore Properties ---" << std::endl;
    std::cout << "   geometry verticies: " << kc_triangulation_type_ << std::endl;
    if (kc_triangulation_type_ == "rectangle") {
        std::cout << "   kc dimensions: [" << kc_dimensions_[0] << ", " << kc_dimensions_[1] << "]\n";
    } else if (kc_triangulation_type_ == "point") {
        std::cout << "   kc radius: " << kc_dimensions_[0] << "\n";
    }
    std::cout << "   number verticies: " << nkctriverts_ << std::endl;
    std::cout << "   kc diameter: " << kc_diameter_ << std::endl;
    std::cout << "   thermal diffusion: " << thermal_diffusion_ << std::endl;
    std::cout << "   translational drag: " << gamma_t_ << std::endl;
    std::cout << "   rotational drag: " << gamma_r_ << std::endl;
    std::cout << "   --- Chromatin Properties --- " << std::endl;
    std::cout << "   chromatin spring: " << chromatin_k_ << std::endl;
    std::cout << "   chromatin rest length: " << chromatin_r0_ << std::endl;
    std::cout << "   chromatin kc angular spring: " << chromatin_ktor_ << std::endl;
    std::cout << "   --- Chromatid Properties ---" << std::endl;
    std::cout << "   chromatid length: " << chromatid_length_ << std::endl;
    std::cout << "   chromatid diameter: " << chromatid_diameter_ << std::endl;
    std::cout << "   chromatid kc offset: " << chromatid_kc_offset_ << std::endl;
    std::cout << "   chromatid kv spring: " << chromatid_kvtor_ << std::endl;
    std::cout << "   chromatid mt repulsion: " << chromatid_mt_repulsion_ << std::endl;
    std::cout << "   chromatid mt sigma: " << chromatid_mt_sigma_ << std::endl;
    std::cout << "   chromatid mt cutoff (5x sigma): " << chromatid_mt_cutoff_ << std::endl;
    std::cout << "   chromatid mt catastrophe distance: " << chromatid_mt_fc_distance_ << std::endl;
    std::cout << "   chromatid mt catastrophe factor: " << chromatid_mt_fc_factor_ << std::endl;
    std::cout << "   chromatid color = [" << color_chromatid_[0] << ", "
                                          << color_chromatid_[1] << ", "
                                          << color_chromatid_[2] << ", "
                                          << color_chromatid_[3] << "]\n";
    std::cout << "   --- Signaling Proteins ---" << std::endl;
    std::cout << "   Aurora B Effects: " << aurora_b_effects_ << std::endl;
    if (aurora_b_effects_ == 1) {
        std::cout << "   Aurora B Effect is biorientation-dependent, uses misaligned destabilization factor\n";
    } else if (aurora_b_effects_ == 2) {
        std::cout << "   Aurora B Effect is inter-kinetochore force-dependent, off-rate only\n";
    } else if (aurora_b_effects_ == 3) {
        std::cout << "   Aurora B Effect is inter-kinetochore force-dependent, on- and off-rates\n";
    } else if (aurora_b_effects_ == 4) {
        std::cout << "   Aurora B Effect is inter-kinetochore force-dependent, off-rate tip only\n";
    }
    if (aurora_b_effects_ == 1) {
        std::cout << "   Aurora B Factor: " << aurora_b_factor_ << std::endl;
    } else if ((aurora_b_effects_ == 2) || (aurora_b_effects_ == 3) || (aurora_b_effects_ == 4)) {
        std::cout << "   Aurora B Force-dependence: " << aurora_b_force_dependence_ << std::endl;
        std::cout << "   Aurora B Force plateau: " << aurora_b_force_plateau_ << std::endl;
    }
    std::cout << "   Dam1 Effects: " << dam1_effects_ << std::endl;
    std::cout << "   Dam1 Catastrophe: " << dam1_catastrophe_factor_ << std::endl;
    std::cout << "   --- Anaphase Properties ---" << std::endl;
    std::cout << "   Do Anaphase: " << do_anaphase_ << std::endl;
    std::cout << "   SAC delay: " << sac_delay_ << std::endl;
    std::cout << "   Attachment_rate: " << anaphase_rate_ << std::endl;
    std::cout << "   Depoly speed: " << anaphase_depoly_ << std::endl;
    std::cout << "   --- Kinetochore Microtubule Attachment Properties ---" << std::endl;
    std::cout << "   Number Attachment Factors: " << naf_ << std::endl;
    std::cout << "   Use poisson process for detachment: " << use_poisson_process_ << std::endl;
    std::cout << "   AF Potential Type: " << af_potential_type_ << std::endl;
    std::cout << "   AF exclusion radius: " << af_exclusion_radius_ << std::endl;
    std::cout << "   AF spring: " << af_k_ << std::endl;
    if (prog_lock_ == 0) {
        std::cout << "   AF angular spring: " << af_kr_[0] << std::endl;
    } else {
        std::cout << "   AF angular spring: [";
        for (int isite = 0; isite < naf_+1; ++isite) {
            std::cout << af_kr_[isite] << ", ";
        }
        std::cout << "]\n";
    }
    std::cout << "   AF rest length (real): " << af_r0_ - 0.5 << " (simadj: " << af_r0_ << ")\n";
    std::cout << "   AF characteristic length assemble: " << af_xc_assemble_ << std::endl;
    std::cout << "   AF characteristic length disemble: " << af_xc_disemble_ << std::endl;
    std::cout << "   AF characteristic length side: " << af_xc_side_ << std::endl;
    std::cout << "   AF characteristic angular factor: " << af_chic_ << std::endl;
    std::cout << "   AF tip distance: " << af_tip_distance_ << std::endl;
    std::cout << "   AF tip crowding: " << (int)af_tip_crowd_ << std::endl;
    std::cout << "   AF tip tracking: " << af_tip_tracking_ << std::endl;
    std::cout << "   AF tip eps eff: " << af_tip_eps_eff_ << std::endl;
    std::cout << "   AF tip on rate assemble: " << af_tip_on_rate_assemble_ << std::endl;
    std::cout << "   AF tip on rate disemble: " << af_tip_on_rate_disemble_ << std::endl;
    std::cout << "   AF tip diffusion: " << af_tip_diffusion_ << std::endl;
    std::cout << "   AF side eps eff: " << af_side_eps_eff_ << std::endl;
    std::cout << "   AF side on rate: " << af_side_on_rate_ << std::endl;
    std::cout << "   AF side diffusion: " << af_side_diffusion_ << std::endl;
    std::cout << "   AF rcutoff: " << rcutoff_ << std::endl;
    std::cout << "   AF stabilization length: " << af_l_stabilize_ << std::endl;
    std::cout << "   AF stabilization force vg: " << af_f_stabilize_vg_ << std::endl;
    std::cout << "   AF stabilization force vs: " << af_f_stabilize_vs_ << std::endl;
    std::cout << "   AF stabilization force fc: " << af_f_stabilize_fc_ << std::endl;
    std::cout << "   AF stabilization force fr: " << af_f_stabilize_fr_ << std::endl;
    std::cout << "   AF microtubule max poly velocity: " << af_mtpoly_vmax_ << std::endl;
    std::cout << "   AF velocity: " << af_velocity_ << std::endl;
    std::cout << "   AF stall force: " << af_stall_force_ << std::endl;
    for (int ic = 0; ic < nchromosomes_; ++ic) {
        std::cout << "Chromosome " << ic << ":" << std::endl;
        PrintPositions(ic);
    }
}

void ChromosomeManagement::PrintPositions(int ic) {
    // Just print the position information
    std::cout << "   kc0:" << std::endl;
    std::cout << "      x = [" << kinetochores_[2*ic].r_[0] << ", "
                               << kinetochores_[2*ic].r_[1] << ", "
                               << kinetochores_[2*ic].r_[2] << "]\n";
    std::cout << "      u = [" << kinetochores_[2*ic].u_[0] << ", "
                               << kinetochores_[2*ic].u_[1] << ", "
                               << kinetochores_[2*ic].u_[2] << "]\n";
    std::cout << "      ueff = [" << kinetochores_[2*ic].ueff_[0] << ", "
                                  << kinetochores_[2*ic].ueff_[1] << ", "
                                  << kinetochores_[2*ic].ueff_[2] << "]\n";
    std::cout << "      v = [" << kinetochores_[2*ic].v_[0] << ", "
                               << kinetochores_[2*ic].v_[1] << ", "
                               << kinetochores_[2*ic].v_[2] << "]\n";
    std::cout << "      w = [" << kinetochores_[2*ic].w_[0] << ", "
                               << kinetochores_[2*ic].w_[1] << ", "
                               << kinetochores_[2*ic].w_[2] << "]\n";
    double flipsister = kinetochores_[2*ic].second_sister_ ? -1.0 : 1.0;
    std::cout << "      c = [" << kinetochores_[2*ic].r_[0] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic].u_[0] << ", "
                               << kinetochores_[2*ic].r_[1] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic].u_[1] << ", "
                               << kinetochores_[2*ic].r_[2] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic].u_[2] << "]\n";
    for (auto p = anchors_[2*ic].begin();
             p != anchors_[2*ic].end();
             ++p) {
        std::cout << "         af = [" << p->pos[0] << ", "
                                       << p->pos[1] << ", "
                                       << p->pos[2] << "]\n";
    }
    std::cout << "   kc1:" << std::endl;
    std::cout << "      x = [" << kinetochores_[2*ic+1].r_[0] << ", "
                               << kinetochores_[2*ic+1].r_[1] << ", "
                               << kinetochores_[2*ic+1].r_[2] << "]\n";
    std::cout << "      u = [" << kinetochores_[2*ic+1].u_[0] << ", "
                               << kinetochores_[2*ic+1].u_[1] << ", "
                               << kinetochores_[2*ic+1].u_[2] << "]\n";
    std::cout << "      ueff = [" << kinetochores_[2*ic+1].ueff_[0] << ", "
                                  << kinetochores_[2*ic+1].ueff_[1] << ", "
                                  << kinetochores_[2*ic+1].ueff_[2] << "]\n";
    std::cout << "      v = [" << kinetochores_[2*ic+1].v_[0] << ", "
                               << kinetochores_[2*ic+1].v_[1] << ", "
                               << kinetochores_[2*ic+1].v_[2] << "]\n";
    std::cout << "      w = [" << kinetochores_[2*ic+1].w_[0] << ", "
                               << kinetochores_[2*ic+1].w_[1] << ", "
                               << kinetochores_[2*ic+1].w_[2] << "]\n";
    flipsister = kinetochores_[2*ic+1].second_sister_ ? -1.0 : 1.0;
    std::cout << "      c = [" << kinetochores_[2*ic+1].r_[0] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic+1].u_[0] << ", "
                               << kinetochores_[2*ic+1].r_[1] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic+1].u_[1] << ", "
                               << kinetochores_[2*ic+1].r_[2] - flipsister * chromatid_kc_offset_ * kinetochores_[2*ic+1].u_[2] << "]\n";
    for (auto p = anchors_[2*ic+1].begin();
             p != anchors_[2*ic+1].end();
             ++p) {
        std::cout << "         af = [" << p->pos[0] << ", "
                                       << p->pos[1] << ", "
                                       << p->pos[2] << "]\n";
    }
    std::cout << "   color = [" << color_[ic][0] << ", "
                                << color_[ic][1] << ", "
                                << color_[ic][2] << ", "
                                << color_[ic][3] << "]\n";
    std::cout << "   color AF = [" << color_af_[ic][0] << ", "
                                   << color_af_[ic][1] << ", "
                                   << color_af_[ic][2] << ", "
                                   << color_af_[ic][3] << "]\n";
    std::cout << std::endl;
}

// Print out the current frame information
void ChromosomeManagement::PrintFrame() {
    std::cout << "****************\n";
    std::cout << "Chromosomes Print Frame: " << properties_->i_current_step << std::endl;
    double spb_sep[3] = {0.0};
    if (properties_->anchors.n_anchors > 0) {
        for (int i = 0; i < 3; ++i) {
            spb_sep[i] = properties_->anchors.r_anchor[1][i] - properties_->anchors.r_anchor[0][i];
        }
    }
    double rspindle = sqrt(dot_product(3, spb_sep, spb_sep));
    std::cout << "rSpindle: " << rspindle << std::endl;
    for (int ic = 0; ic < nchromosomes_; ++ic) {
        std::cout << "Chromosome[" << ic << "] biorientation: " << chromosome_orientation_status_[ic] << std::endl; 
    }
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);
        kc->PrintFrame();
    }
    std::cout << "SAC(" << do_anaphase_ << "): " << sac_status_ << std::endl;
    std::cout << "  bioriented times: " << sac_bioriented_times_ << std::endl;
}

// Do a consistency check on everybody
void ChromosomeManagement::ConsistencyCheck() {
    // Farm out the work to all the kinetochores
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);
        kc->ConsistencyCheck();
    }
}

void ChromosomeManagement::UpdatePositions() {
    if (parameters_ == nullptr) {
        return;
    }
    // Update the position of these stupid things
    double delta = parameters_->delta;
    double temp = parameters_->temp;

    //// FIXME XXX
    //// Super duper dangerous
    ////
    ////t_[0][1] = 100.0;
    //f_[0][1] += 20.0; // pull the first kc away
    //// FIXME XXX

    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        //std::cout << "Kinetochore [" << ikc << "] update:\n";
        double dr[3] = {0.0};

        // Deterministic forces
        //std::cout << "   force: (" << t_[ikc][0] << ", " << t_[ikc][1] << ", " << t_[ikc][2] << ")\n";
        for (int i = 0; i < ndim_; ++i) {
            dr[i] += f_[ikc][i] * delta / gamma_t_;
        }

        // Random forces from gaussian noise
        if (thermal_diffusion_) {
            for (int i = 0; i < ndim_; ++i) {
                dr[i] += gsl_ran_gaussian_ziggurat(properties_->rng.r, sqrt(2.0 * temp * delta / gamma_t_));
            }
        }
        //std::cout << "   dr: ("
        //          << dr[0] << ", "
        //          << dr[1] << ", "
        //          << dr[2] << ")\n";

        //if (sqrt(dot_product(ndim_, dr, dr)) > 0.1) {
        //    std::cerr << "ERROR step: " << properties_->i_current_step << std::endl;
        //    std::cerr << "ERROR in chromosome motion, dr: " << sqrt(dot_product(ndim_, dr, dr)) << std::endl;
        //    PrintFrame();
        //    exit(1);
        //}

        //// Deterministic torques and random torques combine together
        //double urand[3] = {0.0};

        //// For the random torque, pick a random unit vector, then a torque about that random unit
        //// vector with the proper amount
        //// Can rearrange to get in the same form as a torque, up to a cross product (which is random, so doesn't matter)
        //generate_random_unit_vector(ndim_, urand, properties_->rng.r);
        //double tau_random = gsl_ran_gaussian_ziggurat(properties_->rng.r, sqrt(2.0 * temp * gamma_r_ / delta));
        
        // Two random numbers operate orthogonally, and then we can use a common torque to combine
        // them
        double tau_random[2] = {0.0};
        if (thermal_diffusion_) {
            tau_random[0] = gsl_ran_gaussian_ziggurat(properties_->rng.r, sqrt(2.0 * temp * gamma_r_ / delta));
            tau_random[1] = gsl_ran_gaussian_ziggurat(properties_->rng.r, sqrt(2.0 * temp * gamma_r_ / delta));
        }
        //std::cout << "   trand: [" << tau_random[0] << ", " << tau_random[1] << "]\n";

        // Add this to the main torques
        double tau_full[3] = {0.0};
        for (int i = 0; i < ndim_; ++i) {
            //tau_full[i] = t_[ikc][i] + urand[i] * tau_random;
            tau_full[i] = t_[ikc][i] + tau_random[0] * w_[ikc][i] - tau_random[1] * v_[ikc][i];
        }

        // Deterministic torques for everything now
        // Each of u, v, w rotates according to the torque equation
        // du = 1/gamma * Torque * delta
        double du[3] = {0.0};
        double dv[3] = {0.0};
        double dw[3] = {0.0};
        //std::cout << "   torque: (" << t_[ikc][0] << ", " << t_[ikc][1] << ", " << t_[ikc][2] << ")\n";
        //std::cout << "   tfull : (" << tau_full[0] << ", " << tau_full[1] << ", " << tau_full[2] << ")\n";
        cross_product(tau_full, u_[ikc], du, 3);
        cross_product(tau_full, v_[ikc], dv, 3);
        cross_product(tau_full, w_[ikc], dw, 3);
        // Scale
        for (int i = 0; i < ndim_; ++i) {
            du[i] *= delta / gamma_r_;
            dv[i] *= delta / gamma_r_;
            dw[i] *= delta / gamma_r_;
        }
        //std::cout << "   du: ("
        //          << du[0] << ", "
        //          << du[1] << ", "
        //          << du[2] << ")\n";
        //std::cout << "   dv: ("
        //          << dv[0] << ", "
        //          << dv[1] << ", "
        //          << dv[2] << ")\n";
        //std::cout << "   dw: ("
        //          << dw[0] << ", "
        //          << dw[1] << ", "
        //          << dw[2] << ")\n";
        
        // Update information
        // Update position and orieintation
        for (int i = 0; i < ndim_; ++i) {
            r_[ikc][i] += dr[i];
            u_[ikc][i] += du[i];
            if (second_sister_[ikc]) {
                ueff_[ikc][i] = -u_[ikc][i];
            } else {
                ueff_[ikc][i] = u_[ikc][i];
            }
            v_[ikc][i] += dv[i];
            w_[ikc][i] += dw[i];
        }
        // Normalize
        double normu = sqrt(dot_product(ndim_, u_[ikc], u_[ikc]));
        double normv = sqrt(dot_product(ndim_, v_[ikc], v_[ikc]));
        double normw = sqrt(dot_product(ndim_, w_[ikc], w_[ikc]));
        for (int i = 0; i < ndim_; ++i) {
            u_[ikc][i] /= normu;
            v_[ikc][i] /= normv;
            w_[ikc][i] /= normw;
        }

        // Generate the new anchor positions
        int isite = 0;
        for (al_list::iterator p = anchors_[ikc].begin();
                 p != anchors_[ikc].end();
                 ++p, ++isite) {
            // Generate new lab frame coordinate
            double original_pos[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                original_pos[i] = p->pos[i];
                p->pos[i] = r_[ikc][i] +
                            p->pos_rel[0] * u_[ikc][i] +
                            p->pos_rel[1] * v_[ikc][i] +
                            p->pos_rel[2] * w_[ikc][i];
                kinetochores_[ikc].dr_tot_[isite][i] += p->pos[i] - original_pos[i];
            }
        }

        // We also have to update the mesh locations
        // The vertex number does not change, we just have to update it, then
        // call something that updates the global change of variables
        for (int iv = 0; iv < nkctriverts_; ++iv) {
            //std::cout << "iv[" << iv << "] previous ("
            //    << tris_[ikc].verts[0][iv] << ", "
            //    << tris_[ikc].verts[1][iv] << ", "
            //    << tris_[ikc].verts[2][iv] << ")\n";
            for (int i = 0; i < ndim_; ++i) {
                double dr = tris_[ikc].verts[i][iv]; // set dr first
                tris_[ikc].verts[i][iv] = r_[ikc][i] +
                    vertex_rel_[ikc][iv][0] * u_[ikc][i] +
                    vertex_rel_[ikc][iv][1] * v_[ikc][i] +
                    vertex_rel_[ikc][iv][2] * w_[ikc][i];
                dr_tot_kc_[ikc][iv][i] += tris_[ikc].verts[i][iv] - dr;
            }
            //std::cout << "iv[" << iv << "] next ("
            //    << tris_[ikc].verts[0][iv] << ", "
            //    << tris_[ikc].verts[1][iv] << ", "
            //    << tris_[ikc].verts[2][iv] << ")\n";
        }
        update_polygon(&(tris_[ikc]));
    }

    //Print();
}

void ChromosomeManagement::CheckKCmeshNL() {
    double skin = parameters_->skin;
    double half_skin2 = SQR(0.5 * skin);

    bool nl_flag = false;
    // Loop over all the kinetochore mesh displacements, and if any have
    // violated the thing, update
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        for (int iv = 0; iv < nkctriverts_; ++iv) {
            double dr2 = 0.0;
            for (int i = 0; i < ndim_; ++i) {
                dr2 += SQR(dr_tot_kc_[ikc][iv][i]);
            }
            nl_flag = dr2 > half_skin2;
            if (nl_flag) {
                //std::cout << "KC dr2 = " << dr2 << " (half skin " << half_skin2 << ")\n";
                break;
            }
        }
        if (nl_flag) {
            break;
        }
    }

    //// Also loop over the MT displacements, and if any have violated the skin
    //// update
    //// REMEMBER this is over the sites of the bonds, not the bonds themselves
    //double **dr_tot = properties_->sites.dr_tot;
    //for (int ibond = 0; ibond < 2*nbonds_; ++ibond) {
    //    // Early exit if nl_flag already set
    //    if (nl_flag) {
    //        break;
    //    }
    //    // piggyback off of dr_tot from the sites originally
    //    double dr2 = 0.0;
    //    for (int i = 0; i < ndim_; ++i) {
    //        dr2 += SQR(dr_tot[ibond][i]);
    //    }
    //    nl_flag = dr2 > half_skin2;
    //    if (nl_flag) {
    //        std::cout << "MT dr2 = " << dr2 << " (half skin " << half_skin2 << ")\n";
    //        break;
    //    }
    //}

    // Also have to independently check the MT accumulators, as they may have also moved
    // too far....
    for (int ibond = 0; ibond < 2*nbonds_; ++ibond) {
        // Early exit if nl_flag already set
        if (nl_flag) {
            break;
        }
        // Cannot piggyback off the accumulator, so use our own
        double dr2 = 0.0;
        for (int i = 0; i < ndim_; ++i) {
            dr2 += SQR(dr_tot_mt_[ibond][i]);
        }
        nl_flag = dr2 > half_skin2;
        if (nl_flag) {
            //std::cout << "MT dr2 = " << dr2 << " (half skin " << half_skin2 << ")\n";
            break;
        }
    }

    if (nl_flag) {
        // Purge the neighbor lists and reset them
        for (int ikc = 0; ikc < nkcs_; ++ikc) {
            neighbors_[ikc].clear();

            for (int ibond = 0; ibond < properties_->bonds.n_bonds; ++ibond) {
                // Check the overlap 
                double rmin[3], r_min_mag2, rcontact[3], mu;
                min_distance_sphero_polygon(parameters_->n_dim,
                                            parameters_->n_periodic,
                                            properties_->unit_cell.h,
                                            properties_->bonds.r_bond[ibond],
                                            properties_->bonds.s_bond[ibond],
                                            properties_->bonds.u_bond[ibond],
                                            properties_->bonds.length[ibond],
                                            &(tris_[ikc]),
                                            rmin, &r_min_mag2, rcontact, &mu);
                if (r_min_mag2 < half_skin2) {
                    //std::cout << "KC " << ikc << " adding bond " << ibond << std::endl;
                    nl_entry new_neighbor;
                    new_neighbor.label = ibond;
                    neighbors_[ikc].push_back(new_neighbor);
                }
            }

            // Also reset the dr_tot_kc amount
            for (int iv = 0; iv < nkctriverts_; ++iv) {
                for (int i = 0; i < ndim_; ++i) {
                    dr_tot_kc_[ikc][iv][i] = 0.0;
                }
            }
        } // every KC

        // Reset accumulators for all of the MTs as well
        for (int ibond = 0; ibond < 2*nbonds_; ++ibond) {
            for (int i = 0; i < ndim_; ++i) {
                dr_tot_mt_[ibond][i] = 0.0;
            }
        }
    } // if tripped the update
}

void ChromosomeManagement::StepKMC(system_parameters *parameters,
                                   system_properties *properties) {
    if (!enabled_) {
        return;
    }
    //std::cout << "DEBUG, StepKMC\n";
    PrepKMC(parameters, properties);
    RunKMC(parameters, properties);
}

void ChromosomeManagement::PrepKMC(system_parameters *parameters,
                                   system_properties *properties) {
    Update_1_2_Probability(parameters, properties);
    DetermineAttachmentTypes(parameters, properties);
}

void ChromosomeManagement::Update_1_2_Probability(system_parameters *parameters,
                                                  system_properties *properties) {
    if (properties->i_current_step % nkmc_ == 0) {
        // Zero out everything
        for (int ikc = 0; ikc < nkcs_; ++ikc) {
            n_exp_[ikc] = 0.0;
        }
    }

    // Loop over each kinetochore
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc_iter = &(kinetochores_[ikc]); // just alias it
        kc_iter->Step(ndim_,
                      parameters->n_periodic,
                      properties->bonds.n_bonds,
                      properties->unit_cell.h,
                      properties->bonds.r_bond,
                      properties->bonds.s_bond,
                      properties->bonds.u_bond,
                      properties->bonds.length,
                      parameters->delta,
                      properties->rng.r);
        if (properties->i_current_step % nkmc_ == 0) {
            kc_iter->Update_1_2(ndim_,
                                parameters_->n_periodic,
                                properties->bonds.n_bonds,
                                properties->unit_cell.h,
                                properties->bonds.r_bond,
                                properties->bonds.s_bond,
                                properties->bonds.u_bond,
                                properties->bonds.length);
            n_exp_[ikc] += kc_iter->n_exp_tot_;
        }
    }

    // FIXME XXX Debug on the nexp
    //for (int i = 0; i < nkcs_; ++i) {
    //    std::cout << "KC[" << i << "] nexp: " << n_exp_[i] << std::endl;
    //}
}

// Determine the attachment type for all of the kinetochores/chromosomes
void ChromosomeManagement::DetermineAttachmentTypes(system_parameters *parameters,
                                                    system_properties *properties) {
    int n_chromosomes_bioriented = 0;
    if (properties->i_current_step % nkmc_ == 0) {
        for (int ic = 0; ic < nchromosomes_; ++ic) {
            // Ask the kinetochores to determine their attachment status
            //std::cout << "Step[" << properties->i_current_step << "] chromosomes determining attachment type\n";
            int kc0 = 2*ic;
            int kc1 = 2*ic+1;
            kinetochores_[kc0].DetermineAttachmentStatus(parameters,
                                                         properties);
            kinetochores_[kc1].DetermineAttachmentStatus(parameters,
                                                         properties);
            // Now, figure out what kind of attachment we happen to be
            if (kcattachmentstatus_[kc0] == -1 && kcattachmentstatus_[kc1] == -1) {
                // Unattached means both are -1
                chromosome_orientation_status_[ic] = 0;
            } else if (kcattachmentstatus_[kc0] == 2 || kcattachmentstatus_[kc1] == 2) {
                // Merotelic means either of them is merotelic
                chromosome_orientation_status_[ic] = 2;
            } else if (kcattachmentstatus_[kc0] == kcattachmentstatus_[kc1]) {
                // Syntelic (they are attached to the same pole)
                chromosome_orientation_status_[ic] = 3;
            } else {
                // Remaining options are to be amphitelic (attached to different poles), or monotelic (only one is attached)
                if (kcattachmentstatus_[kc0] == -1 || kcattachmentstatus_[kc1] == -1) {
                    // One of them is unattached, and so this must be a monotelic state
                    chromosome_orientation_status_[ic] = 1;
                } else {
                    // Only remaining option is to be in amphitelic attachment
                    chromosome_orientation_status_[ic] = 4;
                }
            }
            //std::cout << "Attachment Status: " << chromosome_orientation_status_[ic] << std::endl;
            if (chromosome_orientation_status_[ic] == 4) {
                n_chromosomes_bioriented++;
            }
        }

        if (do_anaphase_) {
            //// Check if the SAC should be silenced
            //if (n_chromosomes_bioriented == nchromosomes_) {
            //    sac_bioriented_times_ += nkmc_;
            //} else {
            //    sac_bioriented_times_ = 0; // reset if there is a break
            //}
            if (n_chromosomes_bioriented == nchromosomes_) {
                sac_bioriented_times_ += nkmc_; // Just accumulate the time
            }

            //// Make sure to turn off the Aurora B destabilization after anaphase
            //if (sac_status_ == 0) {
            //    for (int ic = 0; ic < nchromosomes_; ++ic) {
            //        chromosome_orientation_status_[ic] = 4; // Force into having ABK disabled
            //    }
            //}

            if ((sac_bioriented_times_ > sac_delay_) && (sac_status_ == 1)) {
                // Turn off the SAC
                sac_status_ = 0;
                std::cout << "Turning off the SAC, step: " << properties->i_current_step << std::endl;
                std::cout << "  SAC bi-orientation time: " << sac_bioriented_times_ << std::endl;
                std::cout << "  SAC status: " << sac_status_ << std::endl;
                std::cout << "  bi-oriented: " << n_chromosomes_bioriented << std::endl;
            }
        }
    }
}

void ChromosomeManagement::InitNeighborLists() {
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);
        double r_cutoff2 = SQR(rcutoff_ + SKIN_);
        kc->UpdateNeighbors(parameters_->n_dim,
                            parameters_->n_periodic,
                            properties_->bonds.n_bonds,
                            properties_->unit_cell.h,
                            properties_->bonds.r_bond,
                            properties_->bonds.s_bond,
                            properties_->bonds.u_bond,
                            properties_->bonds.length,
                            r_cutoff2);
    }

    // Also, force the neighbor list update for KC mesh interactions
    double skin = parameters_->skin;
    double half_skin2 = SQR(0.5 * skin);
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        neighbors_[ikc].clear();

        for (int ibond = 0; ibond < properties_->bonds.n_bonds; ++ibond) {
            // Check the overlap 
            double rmin[3], r_min_mag2, rcontact[3], mu;
            min_distance_sphero_polygon(parameters_->n_dim,
                                        parameters_->n_periodic,
                                        properties_->unit_cell.h,
                                        properties_->bonds.r_bond[ibond],
                                        properties_->bonds.s_bond[ibond],
                                        properties_->bonds.u_bond[ibond],
                                        properties_->bonds.length[ibond],
                                        &(tris_[ikc]),
                                        rmin, &r_min_mag2, rcontact, &mu);
            if (r_min_mag2 < half_skin2) {
                //std::cout << "KC " << ikc << " adding bond " << ibond << std::endl;
                nl_entry new_neighbor;
                new_neighbor.label = ibond;
                neighbors_[ikc].push_back(new_neighbor);
            }
        }
    }
}

void ChromosomeManagement::RunKMC(system_parameters *parameters,
                                  system_properties *properties) {
    if (properties->i_current_step % nkmc_ == 0) {
        // Randomly decide to do detach->attach or reverse
        int g[2] = {0, 1};
        for (int i = 0; i < 2; ++i) {
            int j = gsl_rng_uniform_int(properties->rng.r, 2);
            int swap = g[i];
            g[i] = g[j];
            g[j] = swap;
        }

        for (int i = 0; i < 2; ++i) {
            switch (g[i]) {
                case 0:
                    KMC_1_2(parameters, properties);
                    break;
                case 1:
                    KMC_2_1(parameters, properties);
                    break;
            }
        }

        // Check if the kinetochores need to rebuild their neighbor list
        for (int ikc = 0; ikc < nkcs_; ++ikc) {
            Kinetochore *kc = (&kinetochores_[ikc]); 
            if (kc->trip_update_on_removal_) {
                kc->trip_update_on_removal_ = false;
                double r_cutoff2 = SQR(rcutoff_ + SKIN_);
                kc->UpdateNeighbors(parameters->n_dim,
                                    parameters->n_periodic,
                                    properties->bonds.n_bonds,
                                    properties->unit_cell.h,
                                    properties->bonds.r_bond,
                                    properties->bonds.s_bond,
                                    properties->bonds.u_bond,
                                    properties->bonds.length,
                                    r_cutoff2); 
            }
        }

        // DEBUG
        //for (int ikc = 0; ikc < nkcs_; ++ikc) {
        //    std::cout << "KC[" << ikc << "], nbound: " << n_bound_[ikc] << std::endl;
        //}
    }
}

void ChromosomeManagement::KMC_1_2(system_parameters *parameters,
                                   system_properties *properties) {
    //std::cout << "DEBUG, KMC_1_2\n";
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        // How many attach?
        int Ntot = gsl_ran_poisson(properties->rng.r,
                                   n_exp_[ikc] * delta_kmc_);
        // Now do the attachment
        for (int itrial = 0; itrial < Ntot; ++itrial) {
            //if (properties->i_current_step >= 7577191) {
            //    std::cout << "(" << properties->i_current_step << ")";
            //    std::cout << "KC[" << ikc << "] attaching trial: " << itrial << std::endl;
            //    std::cout << "  nbound before new attach: " << n_bound_[ikc] << std::endl;
            //}
            Kinetochore *kc = &(kinetochores_)[ikc];
            bool successful_insert = kc->Insert_1_2(parameters,
                                                    properties);
            if (successful_insert) {
                n_bound_[ikc]++;
                //PrintFrame();
            }
        }
    }
}

void ChromosomeManagement::KMC_2_1(system_parameters *parameters,
                                   system_properties *properties) {
    //std::cout << "DEBUG, KMC_2_1\n";
    if (af_xc_assemble_ == 0.0 && af_r0_ == 0.0) {
        KMC_2_1_FIndep(parameters, properties);
    } else {
        KMC_2_1_FDep(parameters, properties);
    }
}

void ChromosomeManagement::KMC_2_1_FIndep(system_parameters *parameters,
                                          system_properties *properties) {
    // Force independent, just detach via probability distribution
    gsl_rng *r = properties->rng.r;

    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        std::cout << "CM Shouldn't be here...\n";
        exit(1);
        unsigned int noff = gsl_ran_binomial(r, af_tip_on_rate_assemble_ * delta_kmc_, n_bound_[ikc]);

        for (unsigned int ioff = 0; ioff < noff; ++ioff) {
            //std::cout << "KC[" << ikc << "] detaching trial: "<< ioff << std::endl;
            Kinetochore *kc = &(kinetochores_[ikc]);
            bool did_remove = kc->Remove_2_1(parameters,
                                             properties);

            if (!did_remove) {
                std::cout << "Something funny happened during KC detachment\n";
                exit(1);
            }
        }

        // Assuming everythign went well, and we removed, rebulid the neighbor list
        // if need be
        if (noff > 0) {
            Kinetochore *kc = &(kinetochores_[ikc]);
            kc->trip_update_on_removal_ = true;
            //double r_cutoff2 = SQR(rcutoff_ + SKIN_);
            //kc->UpdateNeighbors(parameters->n_dim,
            //                    parameters->n_periodic,
            //                    properties->bonds.n_bonds,
            //                    properties->unit_cell.h,
            //                    properties->bonds.r_bond,
            //                    properties->bonds.s_bond,
            //                    properties->bonds.u_bond,
            //                    properties->bonds.length,
            //                    r_cutoff2);
        }
    }
}

void ChromosomeManagement::KMC_2_1_FDep(system_parameters *parameters,
                                        system_properties *properties) {
    // Force dependent unbinding
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = (&kinetochores_[ikc]); 
        int noff = kc->Remove_2_1_Fdep(parameters,
                                       properties);

        // Check for crowding effects
        if (af_tip_crowd_) {
            for (int ikc2 = 0; ikc2 < nkcs_; ++ikc2) {
                Kinetochore *kc2 = &(kinetochores_[ikc2]);
                
                for (int isite1 = 0; isite1 < naf_; ++isite1) {
                    for (int isite2 = 0; isite2 < naf_; ++isite2) {
                        if (isite1 == isite2 && ikc == ikc2) {
                            continue; // Don't worry about ourselves
                        }
                        // If they aren't both attached, continue
                        if (kc->attach_[isite1] == -1 && kc2->attach_[isite2] == -1) {
                            continue;
                        }

                        int ibond1 = kc->attach_[isite1];
                        int ibond2 = kc2->attach_[isite2];

                        // Check if on the same bond...
                        if (ibond1 != ibond2) {
                            continue;
                        }

                        double dstab1 = properties->bonds.length[ibond1] - kc->cross_pos_[isite1];
                        double dstab2 = properties->bonds.length[ibond2] - kc2->cross_pos_[isite2];

                        // Only check unbinding of myself (isite1 and ikc/kc)
                        if (dstab2 < af_tip_distance_ && dstab1 < af_tip_distance_) {
                            if (dstab1 > dstab2) {
                                // Unbind me
                                //std::cout << "KC[" << ikc << "]{" << isite1 << "} MT[" << ibond1 << "] position: " << kc->cross_pos_[isite1]
                                //    << ", crowding against KC[" << ikc2 << "]{" << isite2 << "} position: " << kc2->cross_pos_[isite2] << std::endl;
                                n_bound_[ikc]--;
                                kc->attach_[isite1] = -1;
                                kc->cross_pos_[isite1] = 0.0;
                                noff++;
                            }
                        }
                    }
                }
            }
        }

        if (noff > 0) {
            //std::cout << "KC[" << ikc << "] detached trials: " << noff << std::endl;
            kc->trip_update_on_removal_ = true;
            //PrintFrame();
            //double r_cutoff2 = SQR(rcutoff_ + SKIN_);
            //kc->UpdateNeighbors(parameters->n_dim,
            //                    parameters->n_periodic,
            //                    properties->bonds.n_bonds,
            //                    properties->unit_cell.h,
            //                    properties->bonds.r_bond,
            //                    properties->bonds.s_bond,
            //                    properties->bonds.u_bond,
            //                    properties->bonds.length,
            //                    r_cutoff2); 
        }
    }
}

void ChromosomeManagement::WriteState(system_parameters *paramters,
                                      system_properties *properties,
                                      const char* dumpfile) {
    if (!write_posit_) {
        return;
    }

    // Write out everything about the chromosomes
    if (outfile_ == nullptr) {
        outfile_ = gfopen(filename_.c_str(), "w");
    }

    if (dumpfile != nullptr) {
        outfile_ = gfopen(dumpfile, "w");
    }

    // The fundamental unit is really the kinetochore, not the chromosome
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);
        //std::cout << "kc[" << ikc << "]\n";
        // Positions and orientations
        float pos[3] = {0.0};
        std::copy(r_[ikc], r_[ikc]+3, pos);
        //std::cout << "  r: (" << pos[0] << ", " << pos[1] << ", " << pos[2] << ")\n";
        fwrite(pos, sizeof(float), ndim_, outfile_);
        std::copy(u_[ikc], u_[ikc]+3, pos);
        //std::cout << "  u: (" << pos[0] << ", " << pos[1] << ", " << pos[2] << ")\n";
        fwrite(pos, sizeof(float), ndim_, outfile_);
        std::copy(v_[ikc], v_[ikc]+3, pos);
        //std::cout << "  v: (" << pos[0] << ", " << pos[1] << ", " << pos[2] << ")\n";
        fwrite(pos, sizeof(float), ndim_, outfile_);
        std::copy(w_[ikc], w_[ikc]+3, pos);
        //std::cout << "  w: (" << pos[0] << ", " << pos[1] << ", " << pos[2] << ")\n";
        fwrite(pos, sizeof(float), ndim_, outfile_);

        // Number bound
        //std::cout << "  nbound: " << n_bound_[ikc] << std::endl;
        fwrite(&(n_bound_[ikc]), sizeof(int), 1, outfile_);

        // Site information
        for (int isite = 0; isite < naf_; ++isite) {
            std::copy(anchors_[ikc][isite].pos, anchors_[ikc][isite].pos+3, pos);
            //std::cout << "  af: r: (" << pos[0] << ", " << pos[1] << ", " << pos[2] << ")\n";
            fwrite(pos, sizeof(float), ndim_, outfile_);
            int attachid = kc->attach_[isite];
            //std::cout << "  af: attach: " << attachid << std::endl;
            fwrite(&attachid, sizeof(int), 1, outfile_);
            float crosspos = kc->cross_pos_[isite];
            //std::cout << "  af: crosspos: " << crosspos << std::endl;
            fwrite(&crosspos, sizeof(float), 1, outfile_);
        }
    }

    fflush(outfile_);
}

std::vector<std::vector<double>> ChromosomeManagement::GenerateAFForces() {
    std::vector<std::vector<double>> forces;
    for (int ipole = 0; ipole < properties_->anchors.n_anchors; ++ipole) {
        std::vector<double> fakeforces(3, 0.0);
        forces.push_back(fakeforces);
    }

    int nmts_per_spb = nbonds_ / properties_->anchors.n_anchors;
    // Loop over KCs
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);
        // Loop over sites
        for (int isite = 0; isite < naf_; ++isite) {
            // Get the bond information
            int ibond = kc->attach_[isite];

            if (ibond == -1) {
                continue;
            }

            // Get the pole information from the bond
            int ipole = ibond / nmts_per_spb;

            // Load the force with the -= to mimic what is in af_mt_harmonic_potential
            for (int i = 0; i < ndim_; ++i) {
                forces[ipole][i] -= kc->flink_[isite][i];
            }
        }
    }

    return forces;
}

std::vector<std::vector<double>> ChromosomeManagement::GenerateInterKCForces() {
    std::vector<std::vector<double>> forces;
    for (int ic = 0; ic < nchromosomes_; ++ic) {
        std::vector<double> fakeforces(3, 0.0);
        forces.push_back(fakeforces);
    }

    // Loop over the chromosomes, and add the forces
    for (int ic = 0; ic < nchromosomes_; ++ic) {
        for (int i = 0; i < ndim_; ++i) {
            forces[ic][i] += f_interkc_[2*ic][i];
        }
    }

    return forces;
}

void ChromosomeManagement::ReadState(system_parameters *parameters,
                                     system_properties *properties) {
    if (!write_posit_) {
        return;
    }
    if (infile_ == nullptr) {
        infile_ = gfopen(filename_.c_str(), "r+");
    }
    // Read a single state from the file
    for (int ikc = 0; ikc < nkcs_; ++ikc) {
        Kinetochore *kc = &(kinetochores_[ikc]);

        float tmp = 0.0;
        for (int i = 0; i < ndim_; ++i) {
            fread(&tmp, sizeof(float), 1, infile_);
            r_[ikc][i] = tmp;
        }

        for (int i = 0; i < ndim_; ++i) {
            fread(&tmp, sizeof(float), 1, infile_);
            u_[ikc][i] = tmp;
        }

        for (int i = 0; i < ndim_; ++i) {
            fread(&tmp, sizeof(float), 1, infile_);
            v_[ikc][i] = tmp;
        }

        for (int i = 0; i < ndim_; ++i) {
            fread(&tmp, sizeof(float), 1, infile_);
            w_[ikc][i] = tmp;
        }
            
        int nbound;
        fread(&nbound, sizeof(int), 1, infile_);
        n_bound_[ikc] = nbound;

        for (int isite = 0; isite < naf_; ++isite) {
            for (int i = 0; i < ndim_; ++i) {
                fread(&tmp, sizeof(float), 1, infile_);
                anchors_[ikc][isite].pos[i] = tmp;
            }

            int attach;
            fread(&attach, sizeof(int), 1, infile_);
            kc->attach_[isite] = attach;
            fread(&tmp, sizeof(float), 1, infile_);
            kc->cross_pos_[isite] = tmp;

            if (attach != -1) {
                // Update r_cross_
                kc->UpdateSiteVector(ndim_,
                                     parameters->n_periodic,
                                     properties->unit_cell.h,
                                     isite,
                                     properties->bonds.r_bond[attach],
                                     properties->bonds.s_bond[attach],
                                     properties->bonds.u_bond[attach],
                                     properties->bonds.length[attach]);
                //n_bound_[ikc]++; // No need, already know how many are bound per KC
            }
        }
        for (int iv = 0; iv < nkctriverts_; ++iv) {
            //std::cout << "iv[" << iv << "] previous ("
            //    << tris_[ikc].verts[0][iv] << ", "
            //    << tris_[ikc].verts[1][iv] << ", "
            //    << tris_[ikc].verts[2][iv] << ")\n";
            for (int i = 0; i < ndim_; ++i) {
                tris_[ikc].verts[i][iv] = r_[ikc][i] +
                    vertex_rel_[ikc][iv][0] * u_[ikc][i] +
                    vertex_rel_[ikc][iv][1] * v_[ikc][i] +
                    vertex_rel_[ikc][iv][2] * w_[ikc][i];
            }
            //std::cout << "iv[" << iv << "] next ("
            //    << tris_[ikc].verts[0][iv] << ", "
            //    << tris_[ikc].verts[1][iv] << ", "
            //    << tris_[ikc].verts[2][iv] << ")\n";
        }
        update_polygon(&(tris_[ikc]));
    }
}

void ChromosomeManagement::CloseAll() {
    if (outfile_ != nullptr) {
        fflush(outfile_);
        fclose(outfile_);
        outfile_ = nullptr;
    }

    if (infile_ != nullptr) {
        fclose(infile_);
        infile_ = nullptr;
    }
}

void ChromosomeManagement::Restart(system_parameters *parameters,
                                   system_properties *properties,
                                   const std::string& restart_name) {
    // Close out everyting
    if (!reconfigure_) {
        CloseAll();
        std::string oldname = filename_;
        filename_ = restart_name;
        filename_.append(".posit");

        // Reload the data
        ReadState(parameters, properties);

        // Reset to original stuff
        CloseAll();
        filename_ = oldname;
    } else {
        Reconfigure(parameters, properties);
    }

    // Reset the chromosome neighbor lists
    InitNeighborLists();
    DetermineAttachmentTypes(parameters, properties);
    PrintFrame();
}

void ChromosomeManagement::Restart(system_parameters *parameters,
                                   system_properties *properties,
                                   int nframes) {
    // Close out everything
    CloseAll();

    // Load as many states as we need...
    for (int iframe = 0; iframe < nframes; ++iframe) {
        ReadState(parameters, properties);
    }

    // This SHOULD reset the file pointers to the correct places, if we opened the file apprpriately
}

// XXX BASIS HERE
void ChromosomeManagement::Reconfigure(system_parameters *parameters,
                                       system_properties *properties) {
    std::string config_type = node_["chromosomes"]["properties"]["reconfigure_type"].as<std::string>();
    std::cout << "RECONFIGURING Chromosomes to type " << config_type << std::endl;

    if (config_type == "biorientation") {
        // Biorient the kineotochores in the spindle
        BiorientForceKinetochores(parameters, properties, false);
    } else if (config_type == "biorientation_attach") {
        // Biorient and attach the kinetochores in the spindle 
        BiorientForceKinetochores(parameters, properties, true);
    } else {
        std::cerr << "Chromosome reconfiguration type: " << config_type << " not implemented, exiting!\n";
        exit(11);
    }

    Print();
    PrintFrame();
}

void ChromosomeManagement::BiorientForceKinetochores(system_parameters *parameters,
                                                     system_properties *properties,
                                                     bool force_attach) {
    std::cout << "Forcing Biorientation of kinetochore\n";
    // Figure out where the middle of the spindle is from the spbs, along with it's orientation
    double r_spindle[3] = {0.0};
    double u_spindle[3] = {0.0};
    double central_displacement = node_["chromosomes"]["properties"]["reconfigure_displacement"].as<double>();

    for (int i = 0; i < ndim_; ++i) {
        r_spindle[i] = (properties->anchors.r_anchor[1][i] + properties->anchors.r_anchor[0][i]) / 2.0;
        u_spindle[i] = properties->anchors.r_anchor[1][i] - properties->anchors.r_anchor[0][i];
    }
    double norm_factor = sqrt(dot_product(ndim_, u_spindle, u_spindle));
    for (int i = 0; i < ndim_; ++i) {
        u_spindle[i] = u_spindle[i] / norm_factor;
    }

    std::cout << "Spindle r(" << r_spindle[0] << ", "
                              << r_spindle[1] << ", "
                              << r_spindle[1] << ")\n";
    std::cout << "        u(" << u_spindle[0] << ", "
                              << u_spindle[1] << ", "
                              << u_spindle[1] << ")\n";

    // Loop over and set each chromosomes
    for (int ichromo = 0; ichromo < nchromosomes_; ++ichromo) {
        int ikc0 = 2*ichromo;
        int ikc1 = 2*ichromo + 1;

        anchors_[ikc0].clear();
        deallocate_triangle_mesh(&(tris_[ikc0]));
        anchors_[ikc1].clear();
        deallocate_triangle_mesh(&(tris_[ikc1]));

        // Displace the chromosome some distance away from the central axis of the spindle
        double tmp[3] = {0.0};
        generate_random_unit_vector(ndim_, tmp, properties_->rng.r);
        double rdisp[3] = {0.0};
        cross_product(u_spindle, tmp, rdisp, ndim_);
        double rkcmid[3] = {0.0};
        norm_factor = sqrt(dot_product(ndim_, rdisp, rdisp));
        for (int i = 0; i < ndim_; ++i) {
            rdisp[i] = rdisp[i] / norm_factor;
            rkcmid[i] = r_spindle[i] + central_displacement * rdisp[i];
        }
        std::cout << "Chromo[" << ichromo << "] r(" << rkcmid[0] << ", "
                                                  << rkcmid[1] << ", "
                                                  << rkcmid[1] << ")\n";
        for (int i = 0; i < ndim_; ++i) {
            r_[ikc0][i] = rkcmid[i] + 0.5 * chromatin_r0_ * u_spindle[i];
            r_[ikc1][i] = rkcmid[i] - 0.5 * chromatin_r0_ * u_spindle[i];
            u_[ikc0][i] = u_spindle[i];
            u_[ikc1][i] = u_spindle[i];
        }

        // Now, reset the valid information for the kinetochores
        kinetochores_[ikc0].ReInit();
        kinetochores_[ikc1].ReInit();
    }

    // Check to see if we are forcing the attachments
    if (force_attach) {
        std::set<int> boundset;
        // Loop over all the AFs and find the nearest microtubule tip from the correct pole
        int nafpole = naf_ * nkcs_ / 2;

        // Pole 1 (kinetochore 0)
        for (int ichromo = 0; ichromo < nchromosomes_; ++ichromo) {
            int ikc0 = 2*ichromo;
            int ikc1 = 2*ichromo + 1;
            std::vector<int> bondlist0;
            std::vector<int> bondlist1;

            for (int isite = 0; isite < naf_; ++isite) {
                double mindist2 = 1e10;
                int minlabel = -1;
                // Compute the minimum distance to microtubules
                for (int ibond = properties->bonds.n_bonds/2; ibond < properties->bonds.n_bonds; ++ibond) {
                    double bondtip[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        bondtip[i] = properties->bonds.r_bond[ibond][i] + 0.5 * properties->bonds.length[ibond] * properties->bonds.u_bond[ibond][i];
                    }
                    double dr[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        dr[i] = bondtip[i] - anchors_[ikc0][isite].pos[i];
                    }
                    double rminmag2tip = dot_product(ndim_, dr, dr);
                    //// Do it by the closest microtubule
                    //double dr[3];
                    //double s[3];
                    //double rminmag2;
                    //double mu;
                    //min_distance_sphere_sphero(parameters->n_dim,
                    //                           parameters->n_periodic,
                    //                           properties->unit_cell.h,
                    //                           anchors_[ikc0][isite].pos,
                    //                           s,
                    //                           properties->bonds.r_bond[ibond],
                    //                           properties->bonds.s_bond[ibond],
                    //                           properties->bonds.u_bond[ibond],
                    //                           properties->bonds.length[ibond],
                    //                           dr,
                    //                           &rminmag2,
                    //                           &mu);
                    //// Do it by the smallest length one
                    double rminmag2 = properties->bonds.length[ibond];
                    double objective = 1.0 * rminmag2 + 0.0 * rminmag2tip;

                    // If minimum, grab label
                    if ((objective < mindist2) && (boundset.find(ibond) == boundset.end())) {
                        mindist2 = rminmag2;
                        minlabel = ibond;
                    }
                } // Loop over bonds

                // Now, attach to the minimum of the bond list
                bondlist0.push_back(minlabel);
                boundset.insert(minlabel);
            } // End of the sets of this kinetochores (0)
            for (int isite = 0; isite < naf_; ++isite) {
                double mindist2 = 1e10;
                int minlabel = -1;
                // Compute the minimum distance to microtubules
                for (int ibond = 0; ibond < properties->bonds.n_bonds/2; ++ibond) {
                    double bondtip[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        bondtip[i] = properties->bonds.r_bond[ibond][i] + 0.5 * properties->bonds.length[ibond] * properties->bonds.u_bond[ibond][i];
                    }
                    double dr[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        dr[i] = bondtip[i] - anchors_[ikc0][isite].pos[i];
                    }
                    double rminmag2tip = dot_product(ndim_, dr, dr);
                    //// Do it by the closest microtubule
                    //double dr[3];
                    //double s[3];
                    //double rminmag2;
                    //double mu;
                    //min_distance_sphere_sphero(parameters->n_dim,
                    //                           parameters->n_periodic,
                    //                           properties->unit_cell.h,
                    //                           anchors_[ikc0][isite].pos,
                    //                           s,
                    //                           properties->bonds.r_bond[ibond],
                    //                           properties->bonds.s_bond[ibond],
                    //                           properties->bonds.u_bond[ibond],
                    //                           properties->bonds.length[ibond],
                    //                           dr,
                    //                           &rminmag2,
                    //                           &mu);
                    //// Do it by the smallest length one
                    double rminmag2 = properties->bonds.length[ibond];
                    double objective = 1.0 * rminmag2 + 0.0 * rminmag2tip;

                    // If minimum, grab label
                    if ((objective < mindist2) && (boundset.find(ibond) == boundset.end())) {
                        mindist2 = rminmag2;
                        minlabel = ibond;
                    }
                } // Loop over bonds

                // Now, attach to the minimum of the bond list
                bondlist1.push_back(minlabel);
                boundset.insert(minlabel);
            } // End of the sets of this kinetochores (1)

            // Force the attachment of the first kinetochore
            kinetochores_[ikc0].ForceAttach(parameters,
                                            properties,
                                            bondlist0);
            kinetochores_[ikc1].ForceAttach(parameters,
                                            properties,
                                            bondlist1);

        }
    }
}


void ChromosomeManagement::LinkSPBKinetochores(system_parameters *parameters,
                                               system_properties *properties,
                                               int ikc0,
                                               int ikc1) {
    // Link a single binding site on either kinetochore to the nearest MT tip present in the
    // entire set of microtubules (choose SPB0, the other was the one that was inserted)
    double mindist2 = 1e10;
    int minlabel = -1;
    int whichkc = -1;
    int whichsite = -1;

    std::set<int> boundset;
    // Loop over all KCs in system and build list of those already attached to MTs
    for (int ikc = 0; ikc < ikc0; ++ikc) {
        for (int isite = 0; isite < naf_; ++isite) {
            if (kinetochores_[ikc].attach_[isite] != -1) {
                boundset.insert(kinetochores_[ikc].attach_[isite]);
            }
        }
    }

    // Loop over sites
    for (int isite = 0; isite < naf_; ++isite) {
        for (int ibond = 0; ibond < properties->bonds.n_bonds/2; ++ibond) {
            if (boundset.find(ibond) != boundset.end()) {
                continue;
            }
            double bondtip[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                bondtip[i] = properties->bonds.r_bond[ibond][i] + 0.5 * properties->bonds.length[ibond] * properties->bonds.u_bond[ibond][i];
            }
            double dr0[3] = {0.0};
            double dr1[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                dr0[i] = bondtip[i] - anchors_[ikc0][isite].pos[i];
                dr1[i] = bondtip[i] - anchors_[ikc1][isite].pos[i];
            }
            double rminmag1tip = dot_product(ndim_, dr0, dr0);
            double rminmag2tip = dot_product(ndim_, dr1, dr1);

            if (rminmag1tip < mindist2) {
                mindist2 = rminmag1tip;
                minlabel = ibond;
                whichkc = ikc0;
                whichsite = isite;
            }
            if (rminmag2tip < mindist2) {
                mindist2 = rminmag2tip;
                minlabel = ibond;
                whichkc = ikc1;
                whichsite = isite;
            }
        }
    }

    // Ise the minlabel and site to force attachment
    kinetochores_[whichkc].ForceAttachSingle(parameters,
                                       properties,
                                       minlabel,
                                       whichsite);
    n_bound_[whichkc] += 1;
}

void ChromosomeManagement::LinkOppositeKinetochoreSites(system_parameters *parameters,
                                                system_properties *properties,
                                                int ikc0,
                                                int ikc1, 
												int i_site) {
	int n_kcs_loc = 2;
	int n_kcs_tot = 2*nchromosomes_;
	int minlabel[n_kcs_loc];
	double mindistsq[n_kcs_loc];
	for(int i_kc(0); i_kc < n_kcs_loc; i_kc++){
		minlabel[i_kc] = -1;
		mindistsq[i_kc] = 1e10;
	}
	std::set<int> boundset;
	// Loop over all KCs in system and build list of those already attached to MTs
	for(int ikc = 0; ikc < n_kcs_tot; ++ikc) {
		for(int isite = 0; isite < naf_; ++isite) {
			if(kinetochores_[ikc].attach_[isite] != -1) {
				boundset.insert(kinetochores_[ikc].attach_[isite]);
				//printf("kc0: %i | %i is bound to %i\n", ikc0, kinetochores_[ikc].attach_[isite], ikc);
			}
		}
	}
	for(int i_bond = 0; i_bond < properties->bonds.n_bonds/2; ++i_bond){
		if(boundset.find(i_bond) != boundset.end()){
			continue;
		}
		double bondtip[3] = {0.0};
		for (int i = 0; i < ndim_; ++i) {
			bondtip[i] = properties->bonds.r_bond[i_bond][i] + 0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
		}
		double dr0[3] = {0.0};
		for (int i = 0; i < ndim_; ++i) {
			dr0[i] = bondtip[i] - anchors_[ikc0][i_site].pos[i];
		}
		double rminmag1tip = dot_product(ndim_, dr0, dr0);
		if (rminmag1tip < mindistsq[0]) {
			mindistsq[0] = rminmag1tip;
			minlabel[0] = i_bond;
		}
	}
	for (int ibond = properties->bonds.n_bonds/2; ibond < properties->bonds.n_bonds; ++ibond) {
		if (boundset.find(ibond) != boundset.end()) {
			continue;
		}
		//printf("ibond is %i\n", ibond);
		double bondtip[3] = {0.0};
		for (int i = 0; i < ndim_; ++i) {
			bondtip[i] = properties->bonds.r_bond[ibond][i] + 0.5 * properties->bonds.length[ibond] * properties->bonds.u_bond[ibond][i];
		}
		double dr1[3] = {0.0};
		for (int i = 0; i < ndim_; ++i) {
			dr1[i] = bondtip[i] - anchors_[ikc1][i_site].pos[i];
		}
		double rminmag2tip = dot_product(ndim_, dr1, dr1);
		if (rminmag2tip < mindistsq[1]) {
			mindistsq[1] = rminmag2tip;
			minlabel[1] = ibond;
		}
	}
	kinetochores_[ikc0].ForceAttachSingle(parameters,
			properties,
			minlabel[0],
			i_site);
	n_bound_[ikc0] += 1;
	kinetochores_[ikc1].ForceAttachSingle(parameters,
			properties,
			minlabel[1],
			i_site);
	n_bound_[ikc1] += 1;
}

void ChromosomeManagement::AttachKinetochoreMicrotubules(){

	for(int i_site(0); i_site < naf_; i_site++){
		for(int i_chromo(0); i_chromo < nchromosomes_; i_chromo++){
			int ikc0 = 2*i_chromo;
			int ikc1 = 2*i_chromo + 1;
			LinkOppositeKinetochoreSites(parameters_, properties_, 
					ikc0, ikc1, i_site);
		}
	}
}
