#ifndef _CHROMOSOME_MANAGEMENT_H_
#define _CHROMOSOME_MANAGEMENT_H_

#include <vector>
#include <yaml-cpp/yaml.h>

class Kinetochore;

#ifndef _PARAMETERS_H
typedef struct system_parameters system_parameters;
#endif
#ifndef _SYSTEM_PROPERTIES_H
typedef struct system_properties system_properties;
#endif
#ifndef _TRIANGLE_MESH_H_
typedef struct _triangle_mesh triangle_mesh;
#endif

#include "anchor_list.h"
#include "helpers.h"
#include "neighbors.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_integration.h>

#ifndef _LOOKUP_TABLE_ADVANCED_H_
class LookupTableAdvanced;
#endif

class ChromosomeManagement {
  private:

    std::string filename_ = "chromosomes.posit";
    FILE *outfile_ = nullptr;
    FILE *infile_ = nullptr;

  public:

	bool align_flag_ = false;
    bool sites_specified_ = false;
    bool write_posit_ = false;
    bool thermal_diffusion_ = true;
    bool enabled_ = false;
    bool print_info_ = true;
    bool af_tip_crowd_ = false; // If AFs experience crowding near the tip of the microtubule
    bool reconfigure_ = false; // Reconfigure the simulation to use/not use chromosomes
    bool set_biorient_flag_ = false; // force beginning biorientation or not
    bool do_anaphase_ = false; // do anaphase separation or not
    int use_poisson_process_ = 0; // use a poisson process for unbinding calculation of AFs
    int use_kc_potential_ = 1; // Turn on the hard interaction between MTs and KCs
    int use_soft_chromatin_potential_ = 1; // Turn on the soft chromatin/kc interactions

    int ndim_ = 3;
    int nbonds_ = 0;
    int nchromosomes_ = 0;
    int nkcs_ = 0;
    int naf_ = 0;
    int nkctriverts_ = 6; // kc triangulation type, number of verticies
    int nkmc_ = 1; // number of timesteps to wait before running kmc routine
    int aurora_b_effects_ = 0; // If there are Aurora B stabilization effects (0 for no effect, 1 for on, 2 for decaying force off rate, 3 for decaying force on and off rate, 4 as in number 2, but only for tip-associated attachments)
    int dam1_effects_ = 0; // Dam1 effects on kMTs (promoting destabilization of kMTs)
    int prog_lock_ = 0; // Progressive locking of the attachment factors down (0 for no effect, 1 for on)
    int sac_delay_ = 0; // number of timesteps to delay deactivating the SAC
    int sac_status_ = 1; // SAC status, 1 is SAC active, 0 is SAC silenced
    int sac_bioriented_times_ = 0; // amount of timesteps (contiguous) that have been bioriented

    double delta_kmc_ = 0.0; // delta timestep for the kMC based on nkmc
    double kc_diameter_ = 0.0;
    double kc_dimensions_[2] = {0.0}; // dimensions for rectangular kinetochore
    double gamma_t_ = 0.0;
    double gamma_r_ = 0.0;
    double kc_spb_max_distance_ = 0.0;
    double kc_spb_min_distance_ = 0.0;
    double chromatin_r0_ = 0.0;
    double chromatin_k_ = 0.0;
    double chromatin_ktor_ = 0.0;
    double chromatid_length_ = 0.0;
    double chromatid_diameter_ = 0.0;
    double chromatid_kc_offset_ = 0.0;
    double chromatid_kvtor_ = 0.0;
    double chromatid_mt_repulsion_ = 0.0;
    double chromatid_mt_sigma_ = 0.0;
    double chromatid_mt_cutoff_ = 0.0;
    double chromatid_mt_cutoff2_ = 0.0;
    double chromatid_mt_fc_distance_ = 0.0;
    double chromatid_mt_fc_factor_ = 0.0;
    double aurora_b_factor_ = 1.0; ///< Aurora B destabilization factor
    double aurora_b_force_dependence_ = 0.0; ///< Aurora B force-dependent factor
    double aurora_b_force_plateau_ = 0.0; ///< Aurora B force-dependent factor plateau distance
    double dam1_catastrophe_factor_ = 1.0;
    double af_exclusion_radius_ = 1.0;
    double af_k_ = 0.0;
    //double af_kr_ = 0.0;
    double af_r0_ = 0.0;
    double af_xc_assemble_ = 0.0; //tip is implied here
    double af_xc_disemble_ = 0.0;
    double af_xc_side_ = 0.0;
    double af_chic_ = 0.0;
    double af_tip_eps_eff_ = 0.0;
    double af_tip_on_rate_assemble_ = 0.0;
    double af_tip_on_rate_disemble_ = 0.0;
    double af_tip_diffusion_ = 0.0;
    double af_side_eps_eff_ = 0.0;
    double af_side_on_rate_ = 0.0;
    double af_side_diffusion_ = 0.0;
    double af_tip_distance_ = 0.0;
    double af_l_stabilize_ = 0.0;
    double af_f_stabilize_vg_ = 0.0;
    double af_f_stabilize_vs_ = 0.0;
    double af_f_stabilize_fc_ = 0.0;
    double af_f_stabilize_fr_ = 0.0;
    double af_mtpoly_vmax_ = 0.0;
    double af_mtpoly_dlmax_ = 0.0;
    double af_velocity_ = 0.0;
    double af_stall_force_ = 0.0;
    double af_tip_tracking_ = 0.0;

    double anaphase_rate_ = 1.0;
    double anaphase_depoly_ = 1.0;

    double rcutoff_ = 0.0;
    double rcutoff2_ = 0.0;
    double acutoff_ = 0.0; // x cutoff along mt

    std::string kc_triangulation_type_;
    std::string af_potential_type_;

    YAML::Node node_;

    bool *second_sister_; // are we the second kinetochore sister
    int *n_bound_; // total number bound
    double *n_exp_; // total number expected
    int *kcattachmentstatus_; // kinetochore attachment status
                              // -1 is unattached
                              // 0 or 1 is the attached pole
                              // 2 is attached to both poles (merotelic)
    int *chromosome_orientation_status_; // Chromosome orientation type
                                       // 0 is unattached
                                       // 1 is Monotelic
                                       // 2 is Merotelic
                                       // 3 is Syntelic
                                       // 4 is Amphitelic

    float **color_;
    float *color_chromatid_;
    float **color_af_;
    double *af_kr_; // Progressive locking of angular spring
    double **fkcmttip_;
    double **f_; // force on each kinetochore
    double **f_interkc_; // Interkinetochore potential on each kinetochore
    double **t_; // torque on each kinetochore
    double **r_; // location of kinetochore
    double **u_; // orientation 1 of each kinetochore
    double **ueff_; // effective orientation of kinetochore
    double **v_; // orientation 2 of each kinetochore
    double **w_; // orientation 3 of each kinetochore
    double ***dr_tot_kc_; // dr total for each kinetochore, each vetex position
    double **dr_tot_mt_; // dr total for each microtubule site position
    double ***vertex_rel_; // relative vertex position for each kinetochore
    
    Kinetochore *kinetochores_;
    al_list *anchors_;
    nl_list *neighbors_; // neighbors to the kinetochore meshes
    LookupTableAdvanced *n_exp_lookup_;
    triangle_mesh *tris_; // triangles meshes for all kinetochores

    system_parameters *parameters_ = nullptr;
    system_properties *properties_ = nullptr;

    gsl_integration_workspace **gsl_workspaces_;
  
  private:

    void CalcCutoff(double max_length);
    void BuildTables();
    void InitNeighborLists();

  public:

    ChromosomeManagement();
    ChromosomeManagement(system_parameters *parameters,
                         system_properties *properties,
                         const char* chromosome_file,
                         const char* chromosome_config);
    void Init(system_parameters *parameters,
              system_properties *properties,
              const char* chromosome_file,
              const char* chromosome_config);
    void Init(system_parameters *parameters,
              system_properties *properties,
              YAML::Node *pnode);

    void Allocate();
    void ParseParams();
    void CreateKinetochores(int ichromo,
                            bool replicate = false);
    void Print();
    void PrintPositions(int ic);
    void PrintFrame();
    void UpdatePositions();
    void CheckKCmeshNL();
    void ChangeFilename(const std::string &filename) {
        filename_ = filename;
    }

    // KMC functions
    void PrepKMC(system_parameters *parameters,
                 system_properties *properties);
    void StepKMC(system_parameters *parameters,
                 system_properties *properties);
    void RunKMC(system_parameters *parameters,
                system_properties *properties);
    void Update_1_2_Probability(system_parameters *parameters,
                                system_properties *properties);
    void DetermineAttachmentTypes(system_parameters *parameters,
                                  system_properties *properties);
    void KMC_1_2(system_parameters *parameters,
                 system_properties *properties);
    void KMC_2_1(system_parameters *parameters,
                 system_properties *properties);
    void KMC_2_1_FIndep(system_parameters *parameters,
                        system_properties *properties);
    void KMC_2_1_FDep(system_parameters *parameters,
                      system_properties *properties);
    void WriteState(system_parameters *parameters,
                    system_properties *properties,
                    const char* dumpfile = nullptr);
    void ReadState(system_parameters *parameters,
                   system_properties *properties);
    void Restart(system_parameters *parameters,
                 system_properties *properties,
                 const std::string& restart_name);
    void Reconfigure(system_parameters *parameters,
                     system_properties *properties);
    void Restart(system_parameters *parameters,
                 system_properties *properties,
                 int nframes);
    void BiorientForceKinetochores(system_parameters *parameters,
                                   system_properties *properties,
                                   bool force_attach);
    void LinkSPBKinetochores(system_parameters *parameters,
                             system_properties *properties,
                             int kc0,
                             int kc1);
	void LinkOppositeKinetochoreSites(system_parameters *parameters, 
							  		  system_properties *properties, 
							  		  int kc0, 
							  		  int kc1, 
									  int i_site);
	void AttachKinetochoreMicrotubules();
    void CloseAll();
    void ConsistencyCheck();

    // Generate the AF force information or the inter-kc potential version
    std::vector<std::vector<double>> GenerateAFForces();
    std::vector<std::vector<double>> GenerateInterKCForces();
};

#endif
