#include <iostream>
#include <string>
#include <yaml-cpp/yaml.h>
#include "bob.h"

void configurator_diffusion_analysis_free(system_parameters *parameters, system_properties *properties, char *config_file);
void configurator_xlink_analysis(system_parameters *parameters, system_properties *properties, char *config_file);


void configurator_generic(system_parameters *parameters, system_properties *properties, char *config_file) {
    // Print message to standard output.
    std::cout << "Configurator starting using file " << config_file << "\n";

    //read in info from yaml file
    YAML::Node node = YAML::LoadFile(config_file);

    std::string config_type = "diffusion_analysis_free";
    if (node["config_type"]) {
        config_type = node["config_type"].as<std::string>();
    }

    if (config_type == "diffusion_analysis_free") {
        configurator_diffusion_analysis_free(parameters, properties, config_file);
    } else if (config_type == "xlink_analysis") {
        configurator_xlink_analysis(parameters, properties, config_file);
    } else {
        std::cout << "Not yet bae!\n";
        exit(1);
    }
}

void configurator_diffusion_analysis_free(system_parameters *parameters, system_properties *properties, char *config_file) {
    //read in info from yaml file
    YAML::Node node = YAML::LoadFile(config_file);

    double sys_diameter = node["system_diameter"].as<double>();
    double sys_radius = sys_diameter/2.0;
    double sphere_diameter = node["sphere_diameter"].as<double>();

    int n_kinetochores = 0;
    int n_mts = 0;
    int n_spbs = 0;

    if (node["kc"]) {
        n_kinetochores = node["kc"]["number"].as<int>(); 
    }
    if (node["mt"]) {
        n_mts = node["mt"]["number"].as<int>();
    }
    if (node["spb"]) {
        n_mts = node["spb"]["number"].as<int>();
    }

    int n_dim = parameters->n_dim = node["n_dim"].as<int>();
    parameters->sphere_diameter = sphere_diameter;
    parameters->n_spheres = n_kinetochores;
    parameters->n_anchors = properties->anchors.n_anchors = n_spbs;
    parameters->n_spheros = properties->bonds.n_bonds = n_mts;

    std::cout << "\n";
    std::cout << "Basic Parameters:\n";
    std::cout << "   n_dim = " << n_dim << "\n";
    std::cout << "   sys_diameter = " << sys_diameter << "\n";
    std::cout << "   kc_diameter = " << sphere_diameter << "\n";
    std::cout << "   n_kinetochores = " << n_kinetochores << "\n";
    std::cout << "   n_mts = " << n_mts << "\n";
    std::cout << "   n_spbs = " << n_spbs << "\n\n";

    // Allocate memory for structures
    int n_sites = properties->sites.n_sites = n_kinetochores + 2*n_mts;
    properties->unit_cell.h = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
    for (int i = 0; i < n_dim; ++i) {
        properties->unit_cell.h[i][i] = sys_diameter;
    }

    // Allocate the unit cell structure
    init_unit_cell_structure(parameters, properties);

    // Allocate site structure memory
    properties->sites.v = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
    properties->sites.r = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
    init_site_structure_sphero(parameters, properties);

    // Allocate bonds structure
    init_bond_structure_sphero(parameters, properties);

    // Allocate anchor objects
    init_anchor_structure(parameters, properties);

    // Insert microtubules (rods)
    for (int imt = 0; imt < n_mts; ++imt) {
        std::string insertion_type = "zero";
        if (node["mt"]["insertion_type"]) {
            insertion_type = node["mt"]["insertion_type"].as<std::string>();
        }
        double rlength = node["mt"]["length"].as<double>();

        if (insertion_type == "zero") {
            for (int i = 0; i < n_dim; ++i) {
                properties->bonds.r_bond[imt][i] = 0.0;
                properties->bonds.u_bond[imt][i] = 0.0;
            }
            properties->bonds.u_bond[imt][0] = 1.0;
            properties->bonds.length[imt] = rlength;
        } else {
            std::cout << "nonzero mts not allowed yet\n";
            exit(1);
        }
        std::cout << "New microtubule " << imt << ":\n"
                  << "   location = {"
                  << properties->bonds.r_bond[imt][0] << " "
                  << properties->bonds.r_bond[imt][1] << " "
                  << properties->bonds.r_bond[imt][2] << "}\n"
                  << "   orientation = {"
                  << properties->bonds.u_bond[imt][0] << " "
                  << properties->bonds.u_bond[imt][1] << " "
                  << properties->bonds.u_bond[imt][2] << "}\n"
                  << "   length = " << properties->bonds.length[imt] 
                  << "\n\n";

    }

    // Insert kinetochores (spheres)
    int i_site = n_mts*2;
    for (int ikc = 0; ikc < n_kinetochores; ++ikc) {
        std::string insertion_type = "random";
        if (node["kc"]["insertion_type"]) {
            insertion_type = node["kc"]["insertion_type"].as<std::string>();
        }

        if (insertion_type == "random") {
            // Just insert the kc inside the box
            for (int i = 0; i < n_dim; ++i) {
                properties->sites.r[i_site][i] = 2.0 * sys_radius *
                    gsl_rng_uniform_pos(properties->rng.r);
            }
        } else if (insertion_type == "zero") {
            for (int i = 0; i < n_dim; ++i) {
                properties->sites.r[i_site][i] = 0.0;
            }
        } else {
            std::cout << "Nope, not yet!\n";
            exit(1);
        }
        std::cout << "New kinetochore " << ikc << ":\n"
                  << "   location = {"
                  << properties->sites.r[i_site][0] << " "
                  << properties->sites.r[i_site][1] << " "
                  << properties->sites.r[i_site][2] << "}\n\n";

        i_site++;
    }
}

void configurator_xlink_analysis(system_parameters *parameters, system_properties *properties, char *config_file) {
    //read in info from yaml file
    YAML::Node node = YAML::LoadFile(config_file);

    double box[3] = {0.0};

    box[0] = node["box"][0].as<double>();
    box[1] = node["box"][1].as<double>();
    box[2] = node["box"][2].as<double>();

    double sphere_diameter = node["sphere_diameter"].as<double>();

    int n_dim = parameters->n_dim = node["n_dim"].as<int>();
    int n_mts = (int)node["mt"]["rod"].size();
    int n_kinetochores = 0;
    int n_spbs = 0;
    parameters->sphere_diameter = sphere_diameter;
    parameters->n_spheres = n_kinetochores;
    parameters->n_anchors = properties->anchors.n_anchors = n_spbs;
    parameters->n_spheros = properties->bonds.n_bonds = n_mts;

    int n_sites = properties->sites.n_sites = 2*n_mts;

    std::cout << "\n";
    std::cout << "Basic Parameters:\n";
    std::cout << "   n_dim = " << n_dim << "\n";
    std::cout << "   box = (" << box[0] << ", " << box[1] << ", " << box[2] << ")\n";
    std::cout << "   n_spheros = " << n_mts << "\n";

    // Allocate memory for structures
    properties->unit_cell.h = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
    for (int i = 0; i < n_dim; ++i) {
        properties->unit_cell.h[i][i] = box[i];
    }

    init_unit_cell_structure(parameters, properties);

    // Allocate site structure memory
    properties->sites.v = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
    properties->sites.r = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
    init_site_structure_sphero(parameters, properties);

    // Allocate bonds structure
    init_bond_structure_sphero(parameters, properties);

    // Allocate anchor objects
    init_anchor_structure(parameters, properties);

    // Insert microtubules (rods)
    for (int imt = 0; imt < n_mts; ++imt) {
        std::string insertion_type = "xyz";
        if (node["mt"]["insertion_type"]) {
            insertion_type = node["mt"]["insertion_type"].as<std::string>();
        }

        if (insertion_type == "xyz") {
            for (int i = 0; i < n_dim; ++i) {
                properties->bonds.r_bond[imt][i] = node["mt"]["rod"][imt]["x"][i].as<double>();
                properties->bonds.u_bond[imt][i] = node["mt"]["rod"][imt]["u"][i].as<double>();
            }
            properties->bonds.length[imt] = node["mt"]["rod"][imt]["l"].as<double>();
        } else {
            std::cout << "nonxyz mts not allowed yet\n";
            exit(1);
        }
        std::cout << "New microtubule " << imt << ":\n"
                  << "   location = {"
                  << properties->bonds.r_bond[imt][0] << " "
                  << properties->bonds.r_bond[imt][1] << " "
                  << properties->bonds.r_bond[imt][2] << "}\n"
                  << "   orientation = {"
                  << properties->bonds.u_bond[imt][0] << " "
                  << properties->bonds.u_bond[imt][1] << " "
                  << properties->bonds.u_bond[imt][2] << "}\n"
                  << "   length = " << properties->bonds.length[imt] 
                  << "\n\n";

    }
}
