#include <iostream>
#include <string>
#include <yaml-cpp/yaml.h>
#include "bob.h"

int sphere_overlap(int n_dim, double sphere_diameter, double *r1, double *r2) {

    double dist_mag2 = 0.0;

    for (int i = 0; i < n_dim; i++) {
        dist_mag2 += SQR(r1[i]-r2[i]);
    }

    return (dist_mag2 < SQR(sphere_diameter));
}

int spb_overlap(int n_dim, double *r_spb1, double diameter1,
            double *r_spb2, double diameter2,
            double radius) {

    double alpha1, alpha2, alpha, r1, r2;
    r1 = 0.5*diameter1;
    r2 = 0.5*diameter2;

    alpha1 = asin(r1/radius);
    alpha2 = asin(r2/radius);
    alpha = acos( dot_product(n_dim, r_spb1, r_spb2)/SQR(radius) );

    return  (alpha <= alpha1+alpha2);
}

void insert_random_kc(int n_dim, int i_site, double kc_diameter, system_properties *properties) {

    int overlap;
    do {
        double r_mag2;
        double radius = 0.5*properties->unit_cell.h[0][0] - 0.5 * kc_diameter;
        do {
            r_mag2 = 0.0;
            for (int i = 0; i < n_dim; i++) {
                properties->sites.r[i_site][i] = 2.0 * radius * (gsl_rng_uniform_pos(properties->rng.r) - 0.5);
                r_mag2 += SQR(properties->sites.r[i_site][i]);
            }
        } while (r_mag2 >= SQR(radius));

        double diameter_12 = 0.5 * (1.0 + kc_diameter);
        overlap = 0;
        for (int i_bond = 0; i_bond < properties->bonds.n_bonds; i_bond++) {
            overlap += sphere_sphero_overlap(n_dim, 0,
                                             properties->unit_cell.h, 0.0, diameter_12,
                                             properties->sites.r[i_site],
                                             properties->sites.s[i_site],
                                             properties->bonds.r_bond[i_bond],
                                             properties->bonds.s_bond[i_bond],
                                             properties->bonds.u_bond[i_bond],
                                             properties->bonds.length[i_bond]);
        }

        for (int j_site = 2*properties->bonds.n_bonds; j_site < i_site; j_site++) {
            overlap += sphere_overlap(n_dim, kc_diameter,
                                      properties->sites.r[i_site],
                                      properties->sites.r[j_site]);
        }
    } while (overlap != 0);
}

void insert_random_mt(int n_dim, int i_bond, int i_spb, system_properties *properties) {
    double diameter = properties->unit_cell.h[0][0];
    double radius = 0.5*diameter;
    double spb_dia = properties->anchors.attach_diameter[i_spb];

    double u_spb[3];
    for (int i = 0; i < n_dim; i++) {
        u_spb[i] = properties->anchors.r_anchor[i_spb][i]/radius;
    }

    int overlap;
    do {
        /* generate random anchor position within spb */
        double alpha;
        double alpha_max = asin(spb_dia/diameter);
        double pos_vec[3];
        do {
            generate_random_unit_vector(n_dim, pos_vec, properties->rng.r);
            alpha = acos(dot_product(n_dim, u_spb, pos_vec));
        } while (alpha > alpha_max);

        /* Insert mt with random orientation */
        double r_mag2;
        do {
            r_mag2 = 0.0;
            generate_random_unit_vector(n_dim, properties->bonds.u_bond[i_bond], properties->rng.r);
            for (int i = 0; i < n_dim; i++) {
                properties->bonds.v_bond[i_bond][i] =
                    properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

                r_mag2 += SQR(radius * pos_vec[i] + properties->bonds.v_bond[i_bond][i] +
                              properties->bonds.u_bond[i_bond][i] * properties->anchors.r0[i_spb]);
            }
        } while (r_mag2 >= SQR(radius - 0.5) || dot_product(n_dim, u_spb, properties->bonds.u_bond[i_bond]) > 0);

        /* update bond position*/
        for (int i = 0; i < n_dim; i++) {
            properties->bonds.r_bond[i_bond][i] = radius * pos_vec[i] +
                0.5 * properties->bonds.v_bond[i_bond][i] +
                properties->bonds.u_bond[i_bond][i] * properties->anchors.r0[i_spb];
        }

        overlap = 0;
        for (int j_bond = 0; j_bond < i_bond-1; j_bond++) {
            overlap += sphero_overlap(n_dim, 0, properties->unit_cell.h, 0.0, 1.0,
                                      properties->bonds.r_bond[i_bond],
                                      properties->bonds.s_bond[i_bond],
                                      properties->bonds.u_bond[i_bond],
                                      properties->bonds.length[i_bond],
                                      properties->bonds.r_bond[j_bond],
                                      properties->bonds.s_bond[j_bond],
                                      properties->bonds.u_bond[j_bond],
                                      properties->bonds.length[j_bond]);
        }
    } while (overlap != 0);
}

void update_spb_ref_vectors(int i_spb, double sys_diameter,
                            anchor_properties *anchors) {
    double *r_anchor = anchors->r_anchor[i_spb];
    double *u_anchor = anchors->u_anchor[i_spb];
    double *v_anchor = anchors->v_anchor[i_spb];
    double *w_anchor = anchors->w_anchor[i_spb];

    for (int i = 0; i < 3; ++i)
        u_anchor[i] = -2.0 * r_anchor[i] / sys_diameter;

    double vect1[3] = { 1.0, 0, 0.0 };
    double vect2[3] = { 0.0, 1.0, 0.0 };
    if (1.0 - ABS(u_anchor[0]) > 1e-2)
        cross_product(u_anchor, vect1, v_anchor, 3);
    else
        cross_product(u_anchor, vect2, v_anchor, 3);

    double norm_factor = sqrt(1.0/dot_product(3, v_anchor, v_anchor));
    for (int i = 0; i < 3; ++i)
        v_anchor[i] *= norm_factor;
    
    cross_product(u_anchor, v_anchor, w_anchor, 3);
}

void update_spb_drag_constants(int i_spb, system_parameters* parameters, anchor_properties *anchors) {
    /* FIXME */
    // double eps = anchors->diameter[i_spb] / 4.0;
    // double ln_2e = log(2/eps);
    // double scale = 10.0; //fudge factor
    // anchors->gamma_tra[i_spb] = scale * 16 / 3.0 /
    //     (ln_2e - 0.5772 + 4.0 / eps - 0.5 * SQR(eps) * ln_2e);
    double spb_diffusion_coefficient = parameters->spb_diffusion;

    anchors->gamma_tra[i_spb] = 1.0 / spb_diffusion_coefficient;
    anchors->gamma_rot[i_spb] = anchors->gamma_tra[i_spb]*SQR(0.5 * anchors->attach_diameter[i_spb]);
}

void insert_spb_random(int n_dim, int i_spb, double sys_radius, double spb_diameter,
                       system_properties *properties) {
    if( i_spb == 0 ) {
        properties->anchors.r_anchor[i_spb][0] = 0.0;
        properties->anchors.r_anchor[i_spb][1] = 0.0;
        properties->anchors.r_anchor[i_spb][2] = sys_radius;
    }
    else {
        /* Initialize overlap variable, if 0 no overlap */
        int overlap;
        do {
            double spb_rand[3];
            generate_random_unit_vector(n_dim, spb_rand, properties->rng.r);

            for (int i = 0; i < n_dim; i++) {
                properties->anchors.r_anchor[i_spb][i] = sys_radius*spb_rand[i];
            }

            /* ensure no overlap */
            overlap = 0;
            for (int i = 0; i < i_spb; i++) {
                //need to include spb_overlap function
                overlap += spb_overlap(n_dim,
                                       properties->anchors.r_anchor[i],
                                       properties->anchors.diameter[i],
                                       properties->anchors.r_anchor[i_spb],
                                       spb_diameter,
                                       sys_radius);
            }
        } while( overlap != 0);
    }
}

void insert_spb_rtp(double sys_radius, double spb_theta, double spb_phi, double *r_anchor) {
    r_anchor[0] = sys_radius * sin(spb_theta) * cos(spb_phi);
    r_anchor[1] = sys_radius * sin(spb_theta) * sin(spb_phi);
    r_anchor[2] = sys_radius * cos(spb_theta);
}

void print_anchor_properties(int i_spb, anchor_properties *anchors) {
    //output parameters to screen
    // First print the generic anchor properties, then the specifics
    if (anchors->interaction_type_int == 0) {
        printf("SPB interaction is WCA\n");
    } else {
        printf("SPB interaction is LJ12-6\n");
        printf("  strength: %g\n", anchors->spb_interaction_strength);
        printf("  range: %g\n", anchors->spb_interaction_range);
    }
    // Specific anchor properties
    printf ("New anchor %d:\n", i_spb);
    printf ("   location = {%g %g %g}\n",
            anchors->r_anchor[i_spb][0],
            anchors->r_anchor[i_spb][1],
            anchors->r_anchor[i_spb][2]);
    printf ("   reference_u = {%g %g %g}\n",
            anchors->u_anchor[i_spb][0],
            anchors->u_anchor[i_spb][1],
            anchors->u_anchor[i_spb][2]);
    printf ("   reference_v = {%g %g %g}\n",
            anchors->v_anchor[i_spb][0],
            anchors->v_anchor[i_spb][1],
            anchors->v_anchor[i_spb][2]);
    printf ("   reference_w = {%g %g %g}\n",
            anchors->w_anchor[i_spb][0],
            anchors->w_anchor[i_spb][1],
            anchors->w_anchor[i_spb][2]);

    printf ("   diameter = %g\n", anchors->diameter[i_spb]);
    printf ("   attach_diameter = %g\n", anchors->attach_diameter[i_spb]);
    printf ("   k = %g\n", anchors->k[i_spb]);
    printf ("   kr = %g\n", anchors->kr[i_spb]);
    printf ("   r0 = %g\n", anchors->r0[i_spb]);
    printf ("   translational_gamma = %g\n", anchors->gamma_tra[i_spb]);
    printf ("   rotational_gamma = %g\n\n", anchors->gamma_rot[i_spb]);
}

void print_mt(int i_mt, int i_spb, double *r_bond, double length, al_entry *new_anchor) {
    printf(  "   New bond %d:\n", i_mt);
    printf(  "      location = {%g %g %g}\n",
             r_bond[0],
             r_bond[1],
             r_bond[2]);
    printf(  "      binding site = {%g %g %g}\n",
             new_anchor->pos[0],
             new_anchor->pos[1],
             new_anchor->pos[2]);
    printf(  "      length %g\n", length);
    printf(  "      parent anchor = %d\n\n", i_spb);
}



void insert_mt_rtp(int n_dim, int i_bond, double mt_theta, double mt_phi, double sys_radius,
                   double *r_bond, double *v_bond, double *u_bond, double length) {
    double u_mag = sqrt(dot_product(n_dim, u_bond, u_bond));
    for (int i = 0; i < n_dim; i++) {
        u_bond[i] /= u_mag;
    }

    /* set r_bond */
    r_bond[0] = sys_radius * sin(mt_theta) * cos(mt_phi) + 0.5 * length * u_bond[0];
    r_bond[1] = sys_radius * sin(mt_theta) * sin(mt_phi) + 0.5 * length * u_bond[1];
    r_bond[2] = sys_radius * cos(mt_theta) + 0.5 * length * u_bond[2];

    /* set v_bond */
    for (int i = 0; i < n_dim; ++i) {
        v_bond[i] = length * u_bond[i];
    }
}

void init_mt_anchor(system_parameters *parameters, system_properties *properties, int n_dim, 
					int i_bond, int i_spb, int i_mt, int sys_diameter, double lensradius, 
					double anchorpos[3], bool XYZ_insert){

	// XXX: does this typecasting matter?
	int sys_radius = sys_diameter / 2.0;
	/* Add entry to attach MT to SPB via "anchor" list */
	al_entry new_anchor;
	new_anchor.label = i_bond;
	for (int i = 0; i < n_dim; ++i) {
		new_anchor.pos[i] = properties->bonds.r_bond[i_bond][i] -
			properties->bonds.u_bond[i_bond][i] * (0.5 * properties->bonds.length[i_bond] + properties->anchors.r0[i_spb]);
	}
	if (XYZ_insert) {
		// Carry along anchor information too
		for (int i = 0; i < n_dim; ++i) {
			new_anchor.pos[i] = anchorpos[i];
		}
	}
	double r_rel[3];
	for (int i = 0; i < n_dim; ++i)
		r_rel[i] = new_anchor.pos[i] - properties->anchors.r_anchor[i_spb][i];
	double u_proj = dot_product(n_dim, r_rel, properties->anchors.u_anchor[i_spb]);
	double v_proj = dot_product(n_dim, r_rel, properties->anchors.v_anchor[i_spb]);
	double w_proj = dot_product(n_dim, r_rel, properties->anchors.w_anchor[i_spb]);
	for (int i = 0; i < 3; ++i) {
		new_anchor.pos_rel[i] =
			properties->anchors.u_anchor[i_spb][i] * u_proj +
			properties->anchors.v_anchor[i_spb][i] * v_proj +
			properties->anchors.w_anchor[i_spb][i] * w_proj;
	}

	// relative orientation of attachment anchor information
	if (parameters->spb_angular_spring_flag == 0 || parameters->spb_angular_spring_flag == 2) {
		// The case where we either don't have an orientation and don't care, or are random based on the MT
		// insertion orientation
		double u_rel[3];
		for (int i = 0; i < n_dim; ++i) {
			new_anchor.u[i] = properties->bonds.u_bond[i_bond][i];
			u_rel[i] = new_anchor.u[i];
		}
		double u_proj_rel = dot_product(n_dim, u_rel, properties->anchors.u_anchor[i_spb]);
		double v_proj_rel = dot_product(n_dim, u_rel, properties->anchors.v_anchor[i_spb]);
		double w_proj_rel = dot_product(n_dim, u_rel, properties->anchors.w_anchor[i_spb]);
		new_anchor.u_rel[0] = u_proj_rel;
		new_anchor.u_rel[1] = v_proj_rel;
		new_anchor.u_rel[2] = w_proj_rel;
	} else if (parameters->spb_angular_spring_flag == 1) {
		// inward pointing normal for spring!
		for (int i = 0; i < 3; ++i) {
			new_anchor.u[i] = -2.0 * new_anchor.pos[i] / sys_diameter;
		}
		double unorm = dot_product(n_dim, new_anchor.u, new_anchor.u);

		double u_rel[3];
		for (int i = 0; i < 3; ++i) {
			new_anchor.u[i] = new_anchor.u[i] / sqrt(unorm);
			u_rel[i] = new_anchor.u[i];
		}
		double u_proj_rel = dot_product(n_dim, u_rel, properties->anchors.u_anchor[i_spb]);
		double v_proj_rel = dot_product(n_dim, u_rel, properties->anchors.v_anchor[i_spb]);
		double w_proj_rel = dot_product(n_dim, u_rel, properties->anchors.w_anchor[i_spb]);
		new_anchor.u_rel[0] = u_proj_rel;
		new_anchor.u_rel[1] = v_proj_rel;
		new_anchor.u_rel[2] = w_proj_rel;
	} else if (parameters->spb_angular_spring_flag == 3) {
		// Solve for the two distances we need to get the lens placed correctly
		double rd = 0.5 * properties->anchors.diameter[i_spb];
		double x1 = sqrt(sys_radius*sys_radius - rd*rd);
		double x2 = sqrt(lensradius*lensradius - rd*rd);

		std::cout << "x1: " << x1 << ", x2: " << x2 << std::endl;
		double xshift[3] = {0.0};
		for (int i = 0; i < 3; ++i) {
			xshift[i] = properties->anchors.r_anchor[i_spb][i] * (x1 + x2) / sys_radius;
			std::cout << "xshift[" << i << "] = " << xshift[i] << std::endl;
		}
		double rprime[3] = {0.0};
		for (int i = 0; i < 3; ++i) {
			rprime[i] = new_anchor.pos[i] - xshift[i];
			std::cout << "rprime[" << i << "] = " << rprime[i] << std::endl;
		}
		double rprimenorm = sqrt(dot_product(n_dim, rprime, rprime));
		for (int i = 0; i < 3; ++i) {
			rprime[i] = rprime[i] / rprimenorm;
			std::cout << "rprime[" << i << "] = " << rprime[i] << std::endl;
			new_anchor.u[i] = rprime[i];
		}
		double u_rel[3];
		for (int i = 0; i < 3; ++i) {
			u_rel[i] = new_anchor.u[i];
		}
		double u_proj_rel = dot_product(n_dim, u_rel, properties->anchors.u_anchor[i_spb]);
		double v_proj_rel = dot_product(n_dim, u_rel, properties->anchors.v_anchor[i_spb]);
		double w_proj_rel = dot_product(n_dim, u_rel, properties->anchors.w_anchor[i_spb]);
		new_anchor.u_rel[0] = u_proj_rel;
		new_anchor.u_rel[1] = v_proj_rel;
		new_anchor.u_rel[2] = w_proj_rel;
	}
	std::cout << "Anchor u:    [" << new_anchor.u[0] << ", " << new_anchor.u[1] << ", " << new_anchor.u[2] << "]\n";
	std::cout << "Anchor urel: [" << new_anchor.u_rel[0] << ", " << new_anchor.u_rel[1] << ", " << new_anchor.u_rel[2] << "]\n";
	properties->anchors.anchor_list[i_spb].push_back(new_anchor);
	/* Tell us what just happened! */
	print_mt(i_mt, i_spb, properties->bonds.r_bond[i_bond],
			properties->bonds.length[i_bond],
			&new_anchor);
}

void configurator_spb_dynamics(system_parameters *parameters, system_properties *properties, char *config_file) {
    /* Print message to standard output. */
    std::cout << "Configurator starting using file " << config_file << "\n";

    //read in info from yaml file
    YAML::Node node = YAML::LoadFile(config_file);

    if (node["vesicles"]){
        properties->vesicles.enabled_= true;
        properties->vesicles.Init(parameters, properties, &node);
        configurator_vesicles(parameters, properties, node);
        return;
    }

    /* Setup default values */
    double capture_length = 0.0;
    double capture_diameter = 0.0;
    double lateral_capture = 1;

	// XXX: does this typecasting matter?
    int sys_diameter = node["system_diameter"].as<double>();
    int sys_radius = sys_diameter / 2.0;
    double sphere_diameter = node["sphere_diameter"].as<double>();

    int n_spbs = (int)node["spb"].size();
    int n_kinetochores = (int)node["kc"].size();

    int n_mts(0);
    for (int i_spb(0); i_spb < n_spbs; i_spb++){
		for(int i_mt(0); i_mt < node["spb"][i_spb]["mt"].size(); i_mt++){
			std::string insert_type = node["spb"][i_spb]["mt"][i_mt]["insertion_type"].as<std::string>();
			if(insert_type.substr(0,6) == "Bundle"){
				n_mts += node["spb"][i_spb]["mt"][i_mt]["count"].as<int>();
			}
			else{
			   	n_mts += 1;
			}
		}
	}
    n_mts += node["mt"].size();

    if (capture_diameter == 0.0) {
        capture_diameter = sphere_diameter;
    }
    int n_dim = parameters->n_dim = node["n_dim"].as<int>();
    parameters->sphere_diameter = sphere_diameter;
    parameters->capture_diameter = capture_diameter;
    parameters->capture_length = capture_length;
    parameters->lateral_capture = lateral_capture;
    parameters->n_spheres = n_kinetochores;
    parameters->n_anchors = properties->anchors.n_anchors = n_spbs;
    parameters->n_spheros = properties->bonds.n_bonds = n_mts;
    if (parameters->n_periodic == -1)
        parameters->n_periodic = 0;

    std::cout << "\n";
    std::cout << "Basic Parameters:\n";
    std::cout << "   n_dim = " << n_dim << "\n";
    std::cout << "   sys_diameter = " << sys_diameter << "\n";
    std::cout << "   kc_diameter = " << sphere_diameter << "\n";
    std::cout << "   capture_diameter = " << capture_diameter << "\n";
    std::cout << "   capture_length = " << capture_length << "\n";
    std::cout << "   lateral_capture = " << lateral_capture << "\n";
    std::cout << "   n_kinetochores = " << n_kinetochores << "\n";
    std::cout << "   n_mts = " << n_mts << "\n";
    std::cout << "   n_spbs = " << n_spbs << "\n\n";

    //Make sure tubes extruded from NE are not too big.
    //tube diameters cannot be more than 2/3 the size of the membrane.
    if (sqrt(parameters->ne_ratio/2.0) > sys_diameter/3 && parameters->wall_potential_flag == 5 ){
        std::cout << "  ne_ratio is too large to be physical.\n"
                  << "  Reducing size so extruded tubes diameters are 2/3 diameter of NE.\n\n";
        parameters->ne_ratio = SQR(sys_diameter/3)*2;
    }

    //Make sure tubes extruded from NE are not too small. 
    //tube diameters cannot be less than diameter of MT.
    else if (parameters->wall_potential_flag == 5 && sqrt(parameters->ne_ratio/2.0) < .5 ){
        std::cout << "  ne_ratio is too large to be physical.\n"
                  << "  Reducing size so extruded tubes diameters are 2/3 diameter of NE.\n\n";
        parameters->ne_ratio = SQR(.5)*2;
    }

    { /* Allocate memory for structures */
        int n_sites = properties->sites.n_sites = n_kinetochores + 2*n_mts;

        properties->unit_cell.h = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
        for (int i = 0; i < n_dim; ++i) {
            properties->unit_cell.h[i][i] = sys_diameter;
        }

        /* Allocate unit cell structure */
        init_unit_cell_structure(parameters, properties);

        /* Allocate memory for arrays in site structure. */
        properties->sites.v = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
        properties->sites.r = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
        init_site_structure_sphero(parameters, properties);

        /* Allocate bonds structure */
        init_bond_structure_sphero(parameters, properties);

        /* Allocate anchor objects */
        init_anchor_structure(parameters, properties);
    } /* End allocation block */

    /* Loop over SPBs and initialize their values */
    int i_bond = 0;
    for (int i_spb = 0; i_spb < n_spbs; i_spb++) {
        properties->anchors.diameter[i_spb] = node["spb"][i_spb]["properties"]["diameter"].as<double>();

        // Check for common properties first
        // r0
        if (node["common_r0"]) {
            properties->anchors.r0[i_spb] = node["common_r0"].as<double>();
        } else {
            properties->anchors.r0[i_spb] = node["spb"][i_spb]["properties"]["r0"].as<double>();
        }

        //k 
        if (node["common_spring"]) {
            properties->anchors.k[i_spb] = node["common_spring"].as<double>();
        } else {
            properties->anchors.k[i_spb] = node["spb"][i_spb]["properties"]["spring"].as<double>();
        }

        // kr
        if (node["common_kr"]) {
            properties->anchors.kr[i_spb] = node["common_kr"].as<double>();
        } else {
            if (node["spb"][i_spb]["properties"]["kr"]) {
                properties->anchors.kr[i_spb] = node["spb"][i_spb]["properties"]["kr"].as<double>();
            } else {
                properties->anchors.kr[i_spb] = 0.0;
            }
        }
        // Also grab the type of angular spring
        parameters->spb_angular_spring_flag = 0;
        if (node["angular_spring_type"]) {
            std::string angular_spring_type = node["angular_spring_type"].as<std::string>();
            if (angular_spring_type == "none") {
                parameters->spb_angular_spring_flag = 0;
            } else if (angular_spring_type == "inward") {
                parameters->spb_angular_spring_flag = 1;
                std::cout << "NOTE: Angular SPB-MT spring inward pointing!\n";
            } else if (angular_spring_type == "original") {
                parameters->spb_angular_spring_flag = 2;
                std::cout << "NOTE: Angular SPB-MT spring original orientation pointing!\n";
            } else if (angular_spring_type == "lens") {
                parameters->spb_angular_spring_flag = 3;
                std::cout << "NOTE: Angular SPB-MT spring lens pointing!\n";
            }
        }

        // Check to see what kind of interaction we are giong to be using for SPBs, 0 = WCA, 1 = LJ126
        if (node["spb_interaction_potential"]) {
            std::string spb_interaction_potential = node["spb_interaction_potential"].as<std::string>();
            if (spb_interaction_potential == "WCA") {
                properties->anchors.interaction_type_int = 0;
            } else if (spb_interaction_potential == "LJ126") {
                properties->anchors.interaction_type_int = 1;
                properties->anchors.spb_interaction_strength = node["spb_interaction_strength"].as<double>();
                properties->anchors.spb_interaction_range = node["spb_interaction_range"].as<double>();
            } else {
                std::cerr << "ERROR: SPB interaction type " << spb_interaction_potential << " not supported, exiting\n";
                exit(1);
            }
        } else {
            // Default is WCA potential
            properties->anchors.interaction_type_int = 0;
        }

        properties->anchors.attach_diameter[i_spb] = node["spb"][i_spb]["properties"]["attach_diameter"].as<double>();

        std::string insert_type;// = node["insertion_type"].as<std::string>();
        if(node["spb"][i_spb]["properties"]["insertion_type"]) {
            insert_type = node["spb"][i_spb]["properties"]["insertion_type"].as<std::string>();
        } else {
            insert_type = "RTP";
        }

        // Get the anchor color from file, or set to default
        float default_spb_color[4] = {1, 0.64453125, 0.0, 1.0};
        if (node["spb"][i_spb]["properties"]["color"]) {
            for (int icolor = 0; icolor < 4; ++icolor) {    
                default_spb_color[icolor] = node["spb"][i_spb]["properties"]["color"][icolor].as<float>();
            }
        }
        for (int icolor = 0; icolor < 4; ++icolor) {
            properties->anchors.color_[i_spb][icolor] = default_spb_color[icolor];
        }

        if ( insert_type.compare("RTP") == 0 ){
            double spb_theta = node["spb"][i_spb]["properties"]["theta"].as<double>();
            double spb_phi  = node["spb"][i_spb]["properties"]["phi"].as<double>();
            insert_spb_rtp(sys_radius, spb_theta, spb_phi, properties->anchors.r_anchor[i_spb]);
        }
        else if ( insert_type.compare("Random") == 0 ) {
            /* Special case for the first spb */
            insert_spb_random(n_dim, i_spb, sys_radius,
                              properties->anchors.diameter[i_spb], properties);
        }

        /* Rotate through the y axis to get a ref vector */
        update_spb_ref_vectors(i_spb, sys_diameter, &properties->anchors);

        /* Generate the drag coefficient values */
        update_spb_drag_constants(i_spb, parameters, &properties->anchors);
        /* FIXME */

        /* Print anchor properties */
        print_anchor_properties(i_spb, &properties->anchors);

        /* Read in and initialize "microtubules" */
        for(unsigned int i_mt = 0; i_mt < node["spb"][i_spb]["mt"].size(); i_mt++) {
            
            std::string insert_type;
            if(node["spb"][i_spb]["mt"][i_mt]["insertion_type"]) {
                insert_type = node["spb"][i_spb]["mt"][i_mt]["insertion_type"].as<std::string>();
            } else {
                insert_type = "RTP";
            }

			/* Initialize properties for mts */
            double anchorpos[3] = {0.0};
			double lensradius(0);
			if(parameters->spb_angular_spring_flag == 3){ 
				lensradius = node["lens_radius"].as<double>();
			}
            if(insert_type == "RTP") { /* R-theta-phi insertion */
    			properties->bonds.length[i_bond] = node["spb"][i_spb]["mt"][i_mt]["length"].as<double>();
                double mt_theta = node["spb"][i_spb]["mt"][i_mt]["theta"].as<double>();
                double mt_phi = node["spb"][i_spb]["mt"][i_mt]["phi"].as<double>();

                for (int i = 0; i < n_dim; i++) {
                    properties->bonds.u_bond[i_bond][i] =
                        node["spb"][i_spb]["mt"][i_mt]["orient"][i].as<double>();
                }

                insert_mt_rtp(n_dim, i_bond, mt_theta, mt_phi, sys_radius,
                              properties->bonds.r_bond[i_bond],
                              properties->bonds.v_bond[i_bond],
                              properties->bonds.u_bond[i_bond],
                              properties->bonds.length[i_bond]);
				init_mt_anchor(parameters, properties, n_dim, i_bond,
						i_spb, i_mt, sys_diameter, lensradius, 
						anchorpos, false);
				i_bond++;
            } /* End of R-theta-phi insertion */
			else if(insert_type == "Bundle_RTP"){	/* Insert MTs in a circular bundle */
				double n_mts_bundled = node["spb"][i_spb]["mt"][i_mt]["count"].as<double>();
				double bundle_center_theta = node["spb"][i_spb]["mt"][i_mt]["center_theta"].as<double>();
				double bundle_center_phi = node["spb"][i_spb]["mt"][i_mt]["center_phi"].as<double>();
				double r_at_spb = node["spb"][i_spb]["mt"][i_mt]["radius_at_spb"].as<double>();
				double cone_angle = node["spb"][i_spb]["mt"][i_mt]["cone_angle"].as<double>();
				double alpha = 6.283 / n_mts_bundled; 
				for(int i_mt_bundle(0); i_mt_bundle < n_mts_bundled; i_mt_bundle++){
    				properties->bonds.length[i_bond] = node["spb"][i_spb]["mt"][i_mt]["length"].as<double>();
					double dx_theta = r_at_spb * cos((double)i_spb*alpha/n_spbs + alpha*i_mt_bundle);
					double dx_phi = r_at_spb * sin((double)i_spb*alpha/n_spbs + alpha*i_mt_bundle);
					double mt_theta = bundle_center_theta + dx_theta;
					double mt_phi = bundle_center_phi - dx_phi;
					double main_dir(0);
					int side_dir_index[2];
					int i_scratch(0);
					for(int i_dim = 0; i_dim < n_dim; i_dim++){
						double orientation = node["spb"][i_spb]["mt"][i_mt]["orient"][i_dim].as<double>();
						if(abs(orientation) == 1){
							main_dir = orientation; 
							properties->bonds.u_bond[i_bond][i_dim] = orientation;
						}
						else side_dir_index[i_scratch++] = i_dim; 
					}
					double side_orientation[2];
					side_orientation[0] = cone_angle * cos((double)i_spb*alpha/n_spbs + alpha*i_mt_bundle);
					side_orientation[1] = -1 * main_dir * cone_angle * sin((double)i_spb*alpha/n_spbs + alpha*i_mt_bundle);
					for(int i_dim(0); i_dim < n_dim-1; i_dim++){
						int recorded_index = side_dir_index[i_dim];
						properties->bonds.u_bond[i_bond][recorded_index] = side_orientation[i_dim];

					}
					insert_mt_rtp(n_dim, i_bond, mt_theta, mt_phi, sys_radius,
							properties->bonds.r_bond[i_bond],
							properties->bonds.v_bond[i_bond],
							properties->bonds.u_bond[i_bond],
							properties->bonds.length[i_bond]);
					init_mt_anchor(parameters, properties, n_dim, i_bond,
							i_spb, i_mt, sys_diameter, lensradius, 
							anchorpos, false);
					i_bond++;
				}

			} /* End of Bundle-RTP insertion */ 
            else if(insert_type == "Random") { /* Randomly insert MT on SPB */
    			properties->bonds.length[i_bond] = node["spb"][i_spb]["mt"][i_mt]["length"].as<double>();
				insert_random_mt(n_dim, i_bond, i_spb, properties);
				init_mt_anchor(parameters, properties, n_dim, i_bond,
						i_spb, i_mt, sys_diameter, lensradius, 
						anchorpos, false);
				i_bond++;
            } /* End of Random MT insertion */
            else if(insert_type == "XYZ") {
    			properties->bonds.length[i_bond] = node["spb"][i_spb]["mt"][i_mt]["length"].as<double>();
                for (int i = 0; i < n_dim; ++i) {
                    properties->bonds.r_bond[i_bond][i] = 
                        node["spb"][i_spb]["mt"][i_mt]["r"][i].as<double>();
                    properties->bonds.u_bond[i_bond][i] = 
                        node["spb"][i_spb]["mt"][i_mt]["u"][i].as<double>();
                    anchorpos[i] =
                        node["spb"][i_spb]["mt"][i_mt]["anchorpos"][i].as<double>();
                }
                double umag = dot_product(n_dim, properties->bonds.u_bond[i_bond], properties->bonds.u_bond[i_bond]);
                for (int i = 0; i < n_dim; ++i) {
                    properties->bonds.u_bond[i_bond][i] /= umag;
                }
                for (int i = 0; i < n_dim; ++i) {
                    properties->bonds.v_bond[i_bond][i] = properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
                }
				init_mt_anchor(parameters, properties, n_dim, i_bond,
						i_spb, i_mt, sys_diameter, lensradius, 
						anchorpos, true);
				i_bond++;
            } /* End of XYZ insertion */
        }
    } /* End of SPB insertion loop */

    /* Read in and initialize free "microtubules" */
    for(unsigned int i_mt = 0; i_mt < node["mt"].size(); i_mt++){
        /* Initialize properties for mts */

       for (int i = 0; i < n_dim; i++) {
            properties->bonds.r_bond[i_bond][i] =
                node["mt"][i_mt]["position"][i].as<double>();
        }
        for (int i = 0; i < n_dim; i++) {
            properties->bonds.u_bond[i_bond][i] =
                node["mt"][i_mt]["orient"][i].as<double>();
        }
        properties->bonds.length[i_bond] = node["mt"][i_mt]["length"].as<double>();

        printf(  "New free bond %d:\n", i_mt);
        printf(  "   location = {%g %g %g}\n",
                 properties->bonds.r_bond[i_bond][0],
                 properties->bonds.r_bond[i_bond][1],
                 properties->bonds.r_bond[i_bond][2]);
        printf(  "   orientation = {%g %g %g}\n",
                 properties->bonds.u_bond[i_bond][0],
                 properties->bonds.u_bond[i_bond][1],
                 properties->bonds.u_bond[i_bond][2]);
        printf(  "   length = %g\n\n",
                 properties->bonds.length[i_bond]);
        
        i_bond++;
    }    

    /* Insert all of our kinetochores */
    int i_site = n_mts*2;
    for( int i_kc = 0; i_kc < n_kinetochores; i_kc++){
        std::string insert_type;// = node["insertion_type"].as<std::string>();
        if(node["kc"][i_kc]["properties"]["insertion_type"]) {
            insert_type = node["kc"][i_kc]["properties"]["insertion_type"].as<std::string>();
        } else {
            insert_type = "RTP";
        }

        
        if (insert_type == "RTP") {
            for (int i = 0; i < n_dim; ++i) {
                properties->sites.r[i_site][i] = node["kc"][i_kc]["position"][i].as<double>();
            }
        }

        else if (insert_type == "Random") {
            insert_random_kc(n_dim, i_site, sphere_diameter, properties);
        }
        std::cout << "New kinetochore " << i_kc << ":\n"
                  << "   location = {"
                  << properties->sites.r[i_site][0] << " "
                  << properties->sites.r[i_site][1] << " "
                  << properties->sites.r[i_site][2] << "}\n\n";

        i_site++;
    } /* Done with KC insertion.  */
}

void configurator_vesicles( system_parameters *parameters, system_properties *properties, 
                            YAML::Node &node ) {
    double capture_length = 0.0;
    double capture_diameter = 0.0;
    double lateral_capture = 1;

    int sys_diameter = node["system_diameter"].as<double>();
    //int sys_radius = sys_diameter / 2.0;
    double sphere_diameter = node["sphere_diameter"].as<double>();

    int n_spbs = 0;
    int n_kinetochores = 0;

    int n_mts = 0;
    if (node["mt"]) n_mts += node["mt"].size();

    int n_vesicles = (int)node["vesicles"]["vesicle"].size();


    if (capture_diameter == 0.0) {
        capture_diameter = sphere_diameter;
    }

    int n_dim = parameters->n_dim = node["n_dim"].as<int>();
    parameters->sphere_diameter = sphere_diameter;
    parameters->capture_diameter = capture_diameter;
    parameters->capture_length = capture_length;
    parameters->lateral_capture = lateral_capture;
    parameters->n_spheres = n_kinetochores;
    parameters->n_anchors = properties->anchors.n_anchors = n_spbs;
    parameters->n_spheros = properties->bonds.n_bonds = n_mts;
    if (parameters->n_periodic == -1) parameters->n_periodic = 0;

    // Other parameters to set for later
    parameters->dynamic_instability_flag = 0;
    parameters->mt_diffusion_flag = 0;

    std::cout << "\n";
    std::cout << "Basic Parameters:\n";
    std::cout << "   n_dim = " << n_dim << "\n";
    std::cout << "   sys_diameter = " << sys_diameter << "\n";
    std::cout << "   n_mts = " << n_mts << "\n";
    std::cout << "   n_vesicles = " << n_vesicles << "\n";

    //Make sure tubes extruded from NE are not too big.
    //tube diameters cannot be more than 2/3 the size of the membrane.
    if (sqrt(parameters->ne_ratio/2.0) > sys_diameter/3 && parameters->wall_potential_flag == 5 ){
        std::cout << "  ne_ratio is too large to be physical.\n"
                  << "  Reducing size so extruded tubes diameters are 2/3 diameter of NE.\n\n";
        parameters->ne_ratio = SQR(sys_diameter/3)*2;
    }

    //Make sure tubes extruded from NE are not too small. 
    //tube diameters cannot be less than diameter of MT.
    else if (parameters->wall_potential_flag == 5 && sqrt(parameters->ne_ratio/2.0) < .5 ){
        std::cout << "  ne_ratio is too large to be physical.\n"
                  << "  Reducing size so extruded tubes diameters are 2/3 diameter of NE.\n\n";
        parameters->ne_ratio = SQR(.5)*2;
    }

    { /* Allocate memory for structures */
        int n_sites = properties->sites.n_sites = n_kinetochores + 2*n_mts;

        properties->unit_cell.h = (double **) allocate_2d_array(n_dim, n_dim, sizeof(double));
        for (int i = 0; i < n_dim; ++i) {
            properties->unit_cell.h[i][i] = sys_diameter;
        }

        /* Allocate unit cell structure */
        init_unit_cell_structure(parameters, properties);

        /* Allocate memory for arrays in site structure. */
        properties->sites.v = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
        properties->sites.r = (double **) allocate_2d_array(n_sites, n_dim, sizeof(double));
        init_site_structure_sphero(parameters, properties);

        /* Allocate bonds structure */
        init_bond_structure_sphero(parameters, properties);

        /* Allocate anchor objects */
        init_anchor_structure(parameters, properties);
    } /* End allocation block */

    /* Read in and initialize free "microtubules" */
    for(unsigned int i_mt = 0; i_mt < node["mt"].size(); i_mt++){
        /* Initialize properties for mts */

       for (int i = 0; i < n_dim; i++) {
            properties->bonds.r_bond[i_mt][i] =
                node["mt"][i_mt]["position"][i].as<double>();
        }
        for (int i = 0; i < n_dim; i++) {
            properties->bonds.u_bond[i_mt][i] =
                node["mt"][i_mt]["orient"][i].as<double>();
        }
        properties->bonds.length[i_mt] = node["mt"][i_mt]["length"].as<double>();

        printf(  "New free bond %d:\n", i_mt);
        printf(  "   location = {%g %g %g}\n",
                 properties->bonds.r_bond[i_mt][0],
                 properties->bonds.r_bond[i_mt][1],
                 properties->bonds.r_bond[i_mt][2]);
        printf(  "   orientation = {%g %g %g}\n",
                 properties->bonds.u_bond[i_mt][0],
                 properties->bonds.u_bond[i_mt][1],
                 properties->bonds.u_bond[i_mt][2]);
        printf(  "   length = %g\n\n",
                 properties->bonds.length[i_mt]);
    }    
    return;
}



