/* Utility program to generate appropriate header file and source code for declaring
   and parsing input parameters for molecular simulation library.

   Command-line input: name of parameter configuration file (config_file)

   Output: header file containing parameter structure declaration (parameters.h)
           body of parameter parsing routine (parse_parameters_body.cpp) */

#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <string.h>

#define F_MAX 2048
#define __LINE_MAX 2048

/* Provide function prototypes. */
void create_header_file(char *config_file);
void create_source_file(char *config_file);

int main(int argc, char *argv[])
{
   char config_file[F_MAX];

   /* Get parameters from command-line input. */
   if (argc != 2) {
      fprintf(stderr, "Usage: %s config_file\n", argv[0]);
      exit(1);
   }
   strcpy(config_file, argv[1]);

   /* Create header file. */
   create_header_file(config_file);

   /* Create source file. */
   create_source_file(config_file);

   exit(0);
}

/* Read parameter configuration file and create header file containing parameters structure. */
void create_header_file(char *config_file)
{
   FILE *f_config, *f_header;
   char line[__LINE_MAX], line_tmp[__LINE_MAX], *param_name, *param_type, *comment;
   int n_lines, n_params;

   /* Open header file and write preamble to it. */
   f_header = fopen("parameters.h", "w");
   fputs("#ifndef _PARAMETERS_H\n", f_header);
   fputs("#define _PARAMETERS_H\n", f_header);
   fputs("/* The <system_parameters> structure contains thermodynamic, control, and\n", f_header);
   fputs("   configuration parameters for the simulation. */\n\n", f_header);
   fputs("typedef struct {\n\n", f_header);

   /* Read through parameter configuration file and add entries to header file. */
   f_config = fopen(config_file, "r");
   n_lines = 0;
   n_params = 0;
   while (fgets(line, __LINE_MAX, f_config) != NULL) {

      /* Increment line counter. */
      ++n_lines;

      /* Copy line into scratch space. */
      strcpy(line_tmp, line);

      /* Get parameter name, parameter type, and comments. Fields are assumed to be white space
         (space and tab) delimited. */
      param_name = strtok(line_tmp, " \t");
      param_type = strtok(NULL, " \t");
      comment = strtok(NULL, "\t\n");

      /* Append entry to header file. */
      if (strcmp(param_type, "int") == 0)
          fprintf(f_header, "   %s %s;\t\t/* %s */\n\n", "int", param_name, comment);
      else if (strcmp(param_type, "long") == 0)
          fprintf(f_header, "   %s %s;\t\t/* %s */\n\n", "long", param_name, comment);
      else if (strcmp(param_type, "float") == 0)
          fprintf(f_header, "   %s %s;\t\t/* %s */\n\n", "float", param_name, comment);
      else if (strcmp(param_type, "double") == 0)
          fprintf(f_header, "   %s %s;\t\t/* %s */\n\n", "double", param_name, comment);
      else if (strcmp(param_type, "string") == 0)
          fprintf(f_header, "   %s *%s;\t\t/* %s */\n\n", "char", param_name, comment);
      else if (strcmp(param_type, "double_list") == 0)
          fprintf(f_header, "   %s %s;\t\t/* %s */\n\n", "double_list", param_name, comment);
      else {
         fprintf(stderr, "error parsing parameter configuration file on line %d:\n", n_lines);
         fprintf(stderr, "%s", line);
         exit(1);
      }

      /* Increment parameter counter. */
      ++n_params;
   }

   /* Close parameter configuration file. */
   fclose(f_config);

   /* Write tail to header file and close it. */
   fputs("} system_parameters;\n", f_header);
   fputs("#endif\n", f_header);

   fclose(f_header);

   return;
}

/* Read parameter configuration file and create source code for parsing input parameter file. */
void create_source_file(char *config_file)
{
   FILE *f_config, *f_source;
   char line[__LINE_MAX], line_tmp[__LINE_MAX], *param_name, *param_type;
   int n_lines, n_params;

   /* Open source file and write preamble to it. */
   f_source = fopen("parse_parameters_body.h", "w");

   /* Read through parameter configuration file and add entries to source file. */
   f_config = fopen(config_file, "r");
   n_lines = 0;
   n_params = 0;
   while (fgets(line, __LINE_MAX, f_config) != NULL) {

      /* Increment line counter. */
      ++n_lines;

      /* Copy line into scratch space. */
      strcpy(line_tmp, line);

      /* Get parameter name, parameter type, and comments. Fields are assumed to be white space
         (space and tab) delimited. */
      param_name = strtok(line_tmp, " \t");
      param_type = strtok(NULL, " \t");

      /* Append entry to header file. */
      fprintf(f_source, "if (strcmp(param_name, \"%s\") == 0) {\n", param_name);
      if (strcmp(param_type, "int") == 0) {
         fprintf(f_source, "   parameters -> %s = atoi(param_value);\n", param_name);
         fprintf(f_source, "   fprintf(stdout, \"   %s = %%d\\n\", parameters -> %s);\n", param_name, param_name);
      }
      else if (strcmp(param_type, "long") == 0) {
         fprintf(f_source, "   parameters -> %s = atol(param_value);\n", param_name);
         fprintf(f_source, "   fprintf(stdout, \"   %s = %%ld\\n\", parameters -> %s);\n", param_name, param_name);
      }
      else if (strcmp(param_type, "float") == 0 || strcmp(param_type, "double") == 0) {
         fprintf(f_source, "   parameters -> %s = atof(param_value);\n", param_name);
         fprintf(f_source, "   fprintf(stdout, \"   %s = %%g\\n\", parameters -> %s);\n", param_name, param_name);
      }
      else if (strcmp(param_type, "string") == 0) {
         fprintf(f_source, "   parameters -> %s = (char *) gmalloc((strlen(param_value) + 1) * sizeof(char));\n", param_name);
         fprintf(f_source, "   strcpy(parameters -> %s, param_value);\n", param_name);
         fprintf(f_source, "   fprintf(stdout, \"   %s = %%s\\n\", parameters -> %s);\n", param_name, param_name);
      }
      else if (strcmp(param_type, "double_list") == 0 || strcmp(param_type, "double_list") == 0) {
          fprintf(f_source, "   parameters -> %s = parse_double_list(param_value);\n", param_name);
          fprintf(f_source, "   fprintf(stdout, \"   %s = \");\n", param_name);
          fprintf(f_source, "   print_double_list(parameters -> %s);\n", param_name);
      }
      else {
         fprintf(stderr, "error parsing parameter configuration file on line %d:\n", n_lines);
         fprintf(stderr, "%s", line);
         exit(1);
      }
      fputs("}\n", f_source);

      /* Increment parameter counter. */
      ++n_params;
   }

   /* Close parameter configuration file. */
   fclose(f_config);

   /* Close source file. */
   fclose(f_source);

   return;
}

#undef F_MAX
#undef __LINE_MAX
