/* This routine calculates the forces, potential energy, and virial for sites
   interacting via a lj plus generalized ramp potential, for any type of boundary condition
   (free, periodic, or mixed) and any number of dimensions, using an all-pairs search.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   potential/force component index (i_comp)

   Output: array of forces (f)
   virial (virial)
   potential energy (return value) */

#include "bob.h"
#include "xlink_entry.h"

double crosslink_interaction_bd(system_parameters *parameters,
                                system_properties *properties, double **f,
                                double **virial, double **t_bond, int *calc_matrix) {
    /* Set up shortcuts to data structures. */
    xlink_list **stage_2_xlinks = properties->crosslinks.stage_2_xlinks_;
    int n_dim = parameters->n_dim;
    int n_bonds = properties->bonds.n_bonds;
    
    /* Zero generalized ramp forces, potential energy and virial. */
    memset(f[0], 0, n_bonds * n_dim * sizeof(double));
    memset(t_bond[0], 0, n_bonds * 3 * sizeof(double));
    if (properties->control.virial_flag)
        memset(virial[0], 0, n_dim * n_dim * sizeof(double));
    double u = 0.0;
    
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        for (int i_type = 0; i_type < properties->crosslinks.n_types_; ++i_type) {
            for (xlink_list::iterator xlink = stage_2_xlinks[i_type][i_bond].begin();
                 xlink < stage_2_xlinks[i_type][i_bond].end();
                 xlink++) {
                if(xlink->IsActive()) {
                    double f_link[3] = {0.0, 0.0, 0.0};
                    u += xlink->CalcForce(parameters->n_dim,
                                          parameters->n_periodic,
                                          properties->unit_cell.h,
                                          properties->bonds.r_bond,
                                          properties->bonds.s_bond,
                                          properties->bonds.u_bond,
                                          properties->bonds.length,
                                          f_link);
                    
                    int bond_1 = xlink->head_parent_[0];
                    int bond_2 = xlink->head_parent_[1];
                    for (int i = 0; i < n_dim; ++i)
                        f[bond_1][i] += f_link[i];
                    for (int i = 0; i < n_dim; ++i) 
                        f[bond_2][i] -= f_link[i];

                    /* Calculate torques */
                    double lambda = xlink->cross_position_[0] - 0.5 * properties->bonds.length[bond_1];
                    double mu     = xlink->cross_position_[1] - 0.5 * properties->bonds.length[bond_2];
                    double r_contact_i[3];
                    double r_contact_j[3];
                    if (n_dim == 2) {
                        r_contact_i[0] = properties->bonds.u_bond[bond_1][0] * lambda;
                        r_contact_i[1] = properties->bonds.u_bond[bond_1][1] * lambda;
                        r_contact_i[2] = 0.0;
                        r_contact_j[0] = properties->bonds.u_bond[bond_2][0] * mu;
                        r_contact_j[1] = properties->bonds.u_bond[bond_2][1] * mu;
                        r_contact_j[2] = 0.0;
                    }
                    else if (n_dim == 3) {
                        r_contact_i[0] = properties->bonds.u_bond[bond_1][0] * lambda;
                        r_contact_i[1] = properties->bonds.u_bond[bond_1][1] * lambda;
                        r_contact_i[2] = properties->bonds.u_bond[bond_1][2] * lambda;
                        r_contact_j[0] = properties->bonds.u_bond[bond_2][0] * mu;
                        r_contact_j[1] = properties->bonds.u_bond[bond_2][1] * mu;
                        r_contact_j[2] = properties->bonds.u_bond[bond_2][2] * mu;
                    }
                    
                    double tau[3];
                    cross_product(r_contact_i, f_link, tau, n_dim);
                    for (int i = 0; i < 3; ++i)
                        t_bond[bond_1][i] += tau[i];
                    cross_product(r_contact_j, f_link, tau, n_dim);
                    for (int i = 0; i < 3; ++i)
                        t_bond[bond_2][i] -= tau[i];

                    calc_matrix[bond_1] = calc_matrix[bond_2] = 1;
                    
                    /* Add contribution to virial. */
                    if (properties->control.virial_flag == 1)
                        for (int i = 0; i < n_dim; ++i)
                            for (int j = 0; j < n_dim; ++j)
                                virial[i][j] -= xlink->dr_[i] * f_link[j];

                    /* Walk xlink along spherocylinders */
                    xlink->Step(parameters->n_dim,
                                parameters->delta,
                                properties->bonds.u_bond,
                                properties->bonds.length,
                                f_link, properties->rng.r);
                }
            }
        }
    }
    
    return u;
}
