/* Brownian dynamics program to simulate mixtures of soft spherocylinders
   and explicit crosslinkers.

   Command-line input: name of parameter file (param_file)

   Output: various thermodynamic quantities are written to standard output */

#include "bob.h"
#include <sys/time.h>
#include "diffusion_properties.h"
#ifndef NOGRAPH
#include "graphics.h"
#endif

#include "parse_flags.h"

#include <iostream>

/* Return 1 if the difference is negative, otherwise 0.  */
int timeval_subtract(struct timeval *result, struct timeval *t2, struct timeval *t1)
{
    long int diff = (t2->tv_usec + 1000000 * t2->tv_sec) - (t1->tv_usec + 1000000 * t1->tv_sec);
    result->tv_sec = diff / 1000000;
    result->tv_usec = diff % 1000000;

    return (diff<0);
}

void timeval_print(struct timeval *tv)
{
    char buffer[30];
    time_t curtime;

    printf("%ld.%06ld", (long int) tv->tv_sec, (long int) tv->tv_usec);
    curtime = tv->tv_sec;
    strftime(buffer, 30, "%m-%d-%Y  %T", localtime(&curtime));
    printf(" = %s.%06ld\n", buffer, (long int) tv->tv_usec);
}

int main(int argc, char *argv[])
{
    system_parameters parameters;
    system_properties properties;
    system_potential potential;
    FILE *f_posit, *f_thermo, *f_opt;
    char default_file[F_MAX], param_file[F_MAX], opt_file[F_MAX], *color_file;
    int n_dim, n_posit, n_thermo, n_graph, graph_flag, n_rgb,
        barostat_option, opt_flag, n_opt, write_crosslink_flag,
        n_steps, i_step, n_types, i_type;
    int i_final_frame;
    double delta, **h, **r, *skin, skin_old, md_efficiency,
        opt_time, opt_time_old, elapsed_time;

    // Use the new run_options structure to do everything
    run_options run_opts = parse_opts(argc, argv, 4);

    strcpy(default_file, run_opts.default_file.c_str());
    strcpy(param_file, run_opts.equil_file.c_str());
    strcpy(opt_file, run_opts.opt_file.c_str());

    ///* Get name of parameter file from command-line input. */
    //if (argc != 4) {
    //    fprintf(stderr, "Usage: %s default_file param_file opt_file\n", argv[0]);
    //    exit(1);
    //}
    //strcpy(default_file, argv[1]);
    //strcpy(param_file, argv[2]);
    //strcpy(opt_file, argv[3]);

    /* Read in default parameters. */
    parse_parameters(default_file, &parameters);

    /* Read in run-specific input parameters. */
    parse_parameters(param_file, &parameters);

    /* Read in optimized parameters. */
    parse_parameters(opt_file, &parameters);

    parameters.xlink_test_flag = run_opts.xlink_test;

    /* Initialize variables and allocate memory. */
    init_crosslink_sphero_bd_mp(&parameters, &properties, &potential);

    /* Set up shortcuts to data structures. */
    n_dim = parameters.n_dim;
    n_steps = parameters.n_steps;
    barostat_option = parameters.barostat_option;
    n_posit = parameters.n_posit;
    n_thermo = parameters.n_thermo;
    n_graph = parameters.n_graph;
    graph_flag = parameters.graph_flag;
    color_file = parameters.color_file;
    opt_flag = parameters.opt_flag;
    n_opt = parameters.n_opt;
    delta = parameters.delta;
    h = properties.unit_cell.h;
    r = properties.sites.r;
    n_types = properties.crosslinks.n_types_;

    skin = &(parameters.skin);

    int crosslink_flag = parameters.crosslink_flag;
    write_crosslink_flag = parameters.write_crosslink_flag;

    // Restarting the crosslinks requires that we check the bond and site vectors
    if (run_opts.restart) {
        // Check on the posit file first, this is a read that is different than the 
        // read_config because the velocities aren't read
        std::cout << "RESTART Enabled, starting from previously defined files!\n";
        std::string checkpointfile = run_opts.posit_file_names[0];
        std::cout << "   POSITIONS: " << checkpointfile << std::endl;
        f_posit = gfopen(checkpointfile.c_str(), "rb+");
        properties.read_header_func(&parameters, &properties, f_posit);

        // Do some arithmetic to figure out how many frames there are, and which one we
        // are going to load, loop until we hit the required frame number
        int iframe = 0;
        for (iframe = 0; iframe < run_opts.startframe+1; ++iframe) {
            std::cout << "Reading posit frame: " << iframe << std::endl;
            read_positions(parameters.n_dim,
                           properties.sites.n_sites,
                           properties.unit_cell.h,
                           properties.sites.r,
                           f_posit);
        }
        i_final_frame = iframe;

        // Update the bond site positions
        update_bond_vectors(n_dim,
                            parameters.n_periodic,
                            properties.bonds.n_bonds,
                            h,
                            properties.sites.s,
                            properties.sites.r,
                            properties.bonds.bond_site_1,
                            properties.bonds.bond_site_2,
                            properties.bonds.v_bond,
                            properties.bonds.u_bond,
                            properties.bonds.length,
                            properties.bonds.length2);
        update_bond_positions(n_dim,
                              parameters.n_periodic,
                              properties.bonds.n_bonds,
                              h,
                              properties.unit_cell.h_inv,
                              properties.sites.r,
                              properties.bonds.bond_site_1,
                              properties.bonds.v_bond,
                              properties.bonds.r_bond,
                              properties.bonds.s_bond);
        update_bond_site_positions_mp(n_dim, parameters.n_periodic,
                                      properties.bonds.n_bonds,
                                      properties.sites.n_sites, h,
                                      properties.unit_cell.h_inv,
                                      properties.bonds.bond_site_1,
                                      properties.bonds.bond_site_2,
                                      properties.bonds.r_bond, 
                                      properties.bonds.u_bond,
                                      properties.bonds.
                                      length, r, properties.sites.s);
        // Crosslinks, defer to the crosslink file
        std::string xlink_restart = run_opts.posit_file_names[1];
        std::cout << "   XLINKS: " << xlink_restart << ".posit.stage().type()\n";
        properties.crosslinks.Restart(&parameters, &properties, iframe);
        f_thermo = gfopen("sphero.thermo", "w");
        //print_simulation_step(&parameters, &properties);
    } else {
        // Open the normal output files and run stuff as per the usual
        // Write the initial config
        write_config(&parameters, &properties, "sphero.initial_config");

        // Open the output files for writing
        f_posit = gfopen("sphero.posit", "w");
        f_thermo = gfopen("sphero.thermo", "w");

        // Write header to trajectory file
        properties.write_header_func(&parameters, &properties, f_posit);
    }

#ifndef NOGRAPH
    /* Activate graphics if graph_flag == 1. */
    Graphics graphics;
    if (graph_flag == 1) {
        graphics.Init(&parameters, parameters.n_dim,
                      properties.unit_cell.h, 0);
        graphics.ResizeWindow(800,800);
        graphics.SetBoundaryType("cube");

        if (!crosslink_flag){
            graphics.DrawLoop(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length );
        }

        else {
            graphics.DrawLoop(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_2_xlinks_);
        }
        //printf(" -- Total number of active crosslinks "
    }
#endif
    
    /* Initialize optimization parameters. */
    opt_time = 0.0;
    if (opt_flag) {
        opt_time = cpu();
        skin_old = 0.0;
        md_efficiency = 0.0;
    }

    DiffusionProperties diffusion(parameters.n_dim,
                                  properties.bonds.n_bonds,
                                  1,
                                  properties.bonds.r_bond,
                                  properties.bonds.u_bond);
    FILE *f_thermo_ext = gfopen("sphero.thermo_ext", "w");

        fprintf(f_thermo_ext, "%s ", "ndir_val");
        for (int i = 0; i < n_dim; ++i) 
            fprintf(f_thermo_ext, "%s%d ", "ndir_comp_", i);
        fprintf(f_thermo_ext, "\n");
        fflush(f_thermo_ext);


    evaluate_forces_bd(&parameters, &properties, &potential);

    struct timeval tvBegin, tvEnd, tvDiff;

    // begin
    gettimeofday(&tvBegin, NULL);

    /* Loop over integration timesteps. */
    for (i_step = 1; i_step <= n_steps; ++i_step) {
        if (i_step % n_thermo == 0)
            properties.control.virial_flag = 1;
        else
            properties.control.virial_flag = 0;

        properties.i_current_step = i_step;

        /* Insert and remove crosslinks */
        if (crosslink_flag)
            properties.crosslinks.StepKMC(&parameters, &properties);

        /* Optimize neighbor list skin every n_opt steps. */
        if (opt_flag && i_step % n_opt == 0) {
            opt_time_old = opt_time;
            opt_time = cpu();
            elapsed_time = opt_time - opt_time_old;
            fprintf(stdout, "optimizing neighbor list skin, timesteps %d - %d:\n\n",
                    i_step - n_opt + 1, i_step);
            optimize_skin(n_opt, delta, skin, &skin_old, elapsed_time, &md_efficiency);

            update_neighbor_lists_sphero_mp(&parameters, &properties);

            fflush(stdout);
        }

        /* Carry out brownian dynamics step step. */
        if (run_opts.xlink_test)
            crosslink_interaction_bd_mp(&parameters, &properties, 
                                        properties.bonds.f_bond, 
                                        properties.thermo.virial, 
                                        properties.bonds.t_bond, 
                                        potential.calc_matrix);
        else
            position_step_bd_mp(&parameters, &properties, &potential);

        /* Apply berendsen barostat to equilibrate pressure */
        if (barostat_option)
            berendsen_barostat_sphero(&parameters, &properties);

        properties.time = i_step * delta;

        #ifndef NOGRAPH
        /* Display configuration. */
        if (graph_flag && i_step % n_graph == 0) {
            if (graphics.color_switch_ == 2)
                local_order_cl(&parameters, &properties);
        }
        #endif
        if (i_step % n_thermo == 0) {
            /* Calculate stress and pressure tensors */
            thermodynamics_sphero(&parameters, &properties);

            diffusion.Update_OP_Tensors(properties.bonds.u_bond);
            eigensystem director = diffusion.GetCurrentDirector();
            write_thermo_sphero(&parameters, &properties, &potential, f_thermo);
            fprintf(f_thermo_ext, "%g ", director.value);
            for (int i = 0; i < n_dim; ++i) 
                fprintf(f_thermo_ext, "%g ", director.vector[i]);
            fprintf(f_thermo_ext, "\n");
            fflush(f_thermo_ext);
        }
        if (i_step % n_posit == 0) {
            update_bond_site_positions_mp(n_dim, parameters.n_periodic, properties.bonds.n_bonds,
                                          properties.sites.n_sites, h, properties.unit_cell.h_inv,
                                          properties.bonds.bond_site_1, properties.bonds.bond_site_2,
                                          properties.bonds.r_bond, properties.bonds.u_bond, properties.bonds.
                                          length, r, properties.sites.s);
            write_positions(n_dim, properties.sites.n_sites, h, r, f_posit);
            i_final_frame++;

            if (crosslink_flag) {
                if (write_crosslink_flag){
                    properties.crosslinks.WriteState(&parameters, &properties);
                }
            }

            //print_simulation_step(&parameters, &properties);
        }
        #ifndef NOGRAPH
        /* Display configuration. */
        if (graph_flag && i_step % n_graph == 0) {
            if (graphics.color_switch_ == 2)
                local_order_cl(&parameters, &properties);

            graphics.Draw(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_2_xlinks_);

            if (parameters.grab_flag == 1)
                grabber(graphics.width_fb_, graphics.height_fb_, parameters.grab_file, (int) i_step/n_graph);
        }
        #endif
    }

    //end
    gettimeofday(&tvEnd, NULL);

    // diff
    timeval_subtract(&tvDiff, &tvEnd, &tvBegin);
    printf("%ld.%06ld\n", (long int) tvDiff.tv_sec, (long int) tvDiff.tv_usec);

    std::cout << "Closing files and writing final configuration!\n";
    std::cout << "Final frame number (including the restart): " << i_final_frame << std::endl;

    /* Close output files. */
    fclose(f_posit);
    fclose(f_thermo);

    if (crosslink_flag) {
        properties.crosslinks.CloseAll();
    }

    /* Write final configuration to output file. */
    update_bond_site_positions_mp(n_dim, parameters.n_periodic, properties.bonds.n_bonds,
                                  properties.sites.n_sites, h, properties.unit_cell.h_inv,
                                  properties.bonds.bond_site_1, properties.bonds.bond_site_2,
                                  properties.bonds.r_bond, properties.bonds.u_bond, 
                                  properties.bonds.length, r, properties.sites.s);
    write_config(&parameters, &properties, "sphero.final_config");

    if (crosslink_flag && write_crosslink_flag) {
        // Write out xlink informaiton to new filenames
        properties.crosslinks.xlink_filename_ = "crosslinks.final";
        properties.crosslinks.WriteState(&parameters, &properties);
    }

    /* Write optimized parameters to output file. */
    f_opt = gfopen("sphero_bd.opt", "w");
    fprintf(f_opt, "skin = %g\n", *skin);
    fclose(f_opt);
    
    remove("sphero.checkpoint");

    /* Normal termination. */
    std::cout << "Successful run!\n";
    exit(0);
}
