#ifndef _DATA_COLLECTION_H
#define _DATA_COLLECTION_H

#include "bob.h"
#include <fstream>
#include <iostream>
#include <yaml-cpp/yaml.h>
#include <vector>
#include <map>
#include "xlink_entry.h"

//object data structures
struct xlink_struct{/*{{{*/
    bool is_used=0;/*{{{*/
    int bin_num,
        stage_1_xlinks,    //number of stage 1 xlinks 
        stage_2_xlinks,    //number of stage 2 xlinks 
        xlink_mt;          //index of bond that we are concerned with (TODO could make this an array)
    std::vector<int> xlink_dist; //Distribution of stage 2 slinks on mt (in 100 nm bins)

    inline void Init(system_parameters *parameters, system_properties *properties){
        if (parameters->dynamic_instability_flag == 1)
            bin_num = int(parameters->max_length/4);
        else
            bin_num = int(properties->bonds.length[xlink_mt]/4);
        Clear();
    };

    inline void Clear(){
        if (is_used){
            xlink_dist.clear();
            xlink_dist.resize(bin_num, 0);
            stage_1_xlinks = 0;
            stage_2_xlinks = 0;
        }
    };
};/*}}}*/

struct bead_struct{
    bool is_used=0;/*{{{*/
    int *n_traps = NULL;
    int n_dim;
    double extension;      //Seperation of SPBs

    std::vector< std::vector<double> > f_trap;      //Trap force arrays         
    std::vector< std::vector<double> > r_bead;      //Trap position arrays        

    inline void Clear(){
        if (is_used){
            f_trap.clear();
            r_bead.clear();
            f_trap.resize(*n_traps, std::vector<double>(n_dim,0));
            r_bead.resize(*n_traps, std::vector<double>(n_dim,0));
            extension = 0;
        }
    };

    inline void Init(int n_dim_, int* n_traps_){
        n_dim=n_dim_; n_traps=n_traps_; 
        Clear();
    };
};/*}}}*/

struct mt_struct{
    bool is_used=0;/*{{{*/
    int n_dim;
    int orientation;

    inline void Clear(){
        if (is_used){
            orientation = 0;
        }
    };

    inline void Init(int n_dim_){ n_dim=n_dim_; Clear();};
};/*}}}*/

struct spb_struct{
    bool is_used=0;/*{{{*/
    int n_dim;
    double separation;

    inline void Clear(){
        if (is_used){
            separation = 0;
        }
    };

    inline void Init(int n_dim_){ n_dim = n_dim_; Clear();};
};/*}}}*/

struct wall_struct{
    bool is_used=0;/*{{{*/
    int n_dim;
    int mt_index;   //Index of MT of interest
    double dr;      //distance the tip is from the wall
    double vel;     //magnitude of velocity with which the tip is traveling
    double f_par_total;
    double f_par;   //magnitude of the force exerted on the tip
    double t0;      //Time of previous
    double t;      //Current time
    double *pos0;   //Previous position of tip
    double *pos;    //Current position

    void Clear(){if (is_used){
        dr = vel = f_par = f_par_total = 0;
        mt_index = 0;
        for (int i=0; i<n_dim; i++) pos0[i] = pos[i];
        t0 = t;
        std::fill(pos, pos+n_dim, 0);}
    };

    void Init(int n_dim_){
        n_dim = n_dim_;
        pos0 = new double[n_dim_];
        pos = new double[n_dim_];
        t0 = t = 0;
        std::fill(pos, pos+n_dim, 0);
        Clear();
    };

    ~wall_struct(){
        delete[] pos0;
        delete[] pos;
    };
};/*}}}*/
/*}}}*/

class DataCollection
{
    private:
        int n_dim_,
            n_steps_,
            avg_flag_,
            avg_steps_,
            threshold_;

        xlink_struct xl_data_;
        bead_struct bead_data_;
        mt_struct mt_data_;
        spb_struct spb_data_;
        wall_struct w_data_;

        double overlap_;
        double extension_;
        double orientation_;        //Cos(theta) of the first two mts

        YAML::Node node_;
        system_properties *properties_;
        system_parameters *parameters_;

        std::map<std::string, std::ofstream> outfiles_;
        std::map<std::string, std::vector<std::string> > data_types_;

        void ClearData(); //check
        void InitData();  //check

        /*Creates Maps and Headers*/
        void CreateOutFileMap(); //check
        std::string GetOpticalTrapHeader(YAML::const_iterator *ot_node); //check
        std::string GetCrosslinkHeader(YAML::const_iterator *ot_node); //check
        std::string GetWallTestHeader(YAML::const_iterator *ot_node); //check
        std::string GetSPBHeader(YAML::const_iterator *ot_node);

        /*Output functions*/
        void WriteOpticalTrapOutputs(); //check
        void WriteCrosslinkOutputs(); //check
        void WriteWallOutputs(); //check
        
        /*Data Collection and calculations*/
        void AddOpticalTrapData(); //check
        void AddCrosslinkData();  //check
        void AddWallData(); //check
        //Optical trap calculations
        void CalcExtension();       //Get the distance between the first 2 beads. If there is only one bead this get the distance between the bead and trap
        void CalcOrientation();     //Get the dot product between the first 2 mts
        void CalcOverlap();         //Get the projection of first MT over the second
        void CalcTrapForce();       //Get the force the trap is excerting check
        void CalcBeadPosition();    //Get the position of the bead relative to the lab frame
        //Crosslink calculations
        void CalcXlinkDistribution(); //check
        void CalcStage1Xlink();  //check
        void CalcStage2Xlink();  //check

        //Wall Calculations
        void CalcTipPos(); //check
        void CalcProtrusion(); //check
        void CalcVelocity();  //check
        void CalcForce();

    public:
        DataCollection();
        DataCollection( system_parameters *parameters, system_properties *properties ); //check
        int GetAvgSteps();
        void WriteOutputs(); //check
        void CalculateValues(); //check
        int GetThreshold(); //check
        bool WallIsUsed(); //check
        void GetWallForce(double f_par, int index); //check
        void Close();
};









#endif
