#include "bob.h"
#include <gsl/gsl_math.h>
#include <gsl/gsl_eigen.h>
#include <gsl/gsl_blas.h>
#include "diffusion_properties.h"

DiffusionProperties::DiffusionProperties(int n_dim, int n_objs,
                                         int n_blocks_max, double **r, double **u) {
    for (int i = 0; i < 9; ++i)
        op_tensor_[i] = 0.0;

    n_objs_ = n_objs;
    n_points_ = n_blocks_max;
    n_dim_ = n_dim;

    // n_bins_t_ = 100; // Number of time domain bins for displacement distribution
    // n_bins_r_ = 100; // Number of spatial domain bins for displacement distribution
    // r_max_ = 100; // Maximum distance to track for displacement distribution

    // ddf_x_.resize(n_points_ * n_bins_t * n_bins_r);
    // std::fill(ddf_x_.begin(), ddf_x_.end(), 0);

    // ddf_y_.resize(n_points_ * n_bins_t * n_bins_r);
    // std::fill(ddf_y_.begin(), ddf_y_.end(), 0);
    

    n_measurements_ = (int*) allocate_1d_array(n_blocks_max, sizeof(int));
    r_mol_avg2_ = (double*) allocate_1d_array(n_blocks_max, sizeof(double));
    r_mol_par_avg2_ = (double*) allocate_1d_array(n_blocks_max, sizeof(double));
    r_mol_perp_avg2_ = (double*) allocate_1d_array(n_blocks_max, sizeof(double));
    ui_dot_u0_ = (double*) allocate_1d_array(n_blocks_max, sizeof(double));
    theta2_ = (double*) allocate_1d_array(n_blocks_max, sizeof(double));

    u_0_ = (double**) allocate_2d_array(n_objs, n_dim, sizeof(double));
    u_old_ = (double**) allocate_2d_array(n_objs, n_dim, sizeof(double));
    dr_ = (double**) allocate_2d_array(n_objs, n_dim, sizeof(double));
    r_0_ = (double**) allocate_2d_array(n_objs, n_dim, sizeof(double));
    theta_ = (double*) allocate_1d_array(n_objs, sizeof(double));

    eval_ = gsl_vector_alloc (n_dim_);
    evec_ = gsl_matrix_alloc (n_dim_, n_dim_);
    w_ = gsl_eigen_symmv_alloc (n_dim_);

    this->Update(r, u, 1);
}

void DiffusionProperties::Calc_OP_Tensor(double **u_obj) {

    for (int i = 0; i < SQR(n_dim_); ++i)
        op_tensor_curr_[i] = 0.0;

    /* input order parameter tensor, data is linear to appease GSL */
    for (int i_obj = 0; i_obj < n_objs_; ++i_obj) {
        for (int i = 0; i < n_dim_; ++i)
            for (int j = 0; j < n_dim_; ++j) {
                op_tensor_curr_[i * n_dim_ + j] += u_obj[i_obj][i] * u_obj[i_obj][j];
            }
    }

    double factor;
    if (n_dim_ == 2)
        factor = 2.0 / n_objs_;
    else
        factor = 1.5 / n_objs_; 
    for (int i = 0; i < SQR(n_dim_); ++i) {
        op_tensor_curr_[i] = factor * op_tensor_curr_[i];
    }
    
    for (int i = 0; i < n_dim_; ++i)
        op_tensor_curr_[i * n_dim_ + i] = op_tensor_curr_[i * n_dim_ + i] - 0.5 - 0.5 * (n_dim_ == 2);
}

void DiffusionProperties::Update_OP_Tensors(double **u_obj) {
    this->Calc_OP_Tensor(u_obj);

    for (int i = 0; i < SQR(n_dim_); ++i) 
        op_tensor_[i] += op_tensor_curr_[i];

    director_ = CalcDirectorOrderParameter(op_tensor_curr_);
    curr_point_++;
}

eigensystem DiffusionProperties::GetAverageDirector() {
    double op_tensor[9];
    for (int i = 0; i < SQR(n_dim_); ++i)
        op_tensor[i] = op_tensor_[i] / curr_point_;

    return CalcDirectorOrderParameter(op_tensor);
}

eigensystem DiffusionProperties::GetCurrentDirector() {
    return CalcDirectorOrderParameter(op_tensor_curr_);
}

void DiffusionProperties::FinalizeDirector() {
    director_ = this->GetAverageDirector();
}

eigensystem DiffusionProperties::CalcDirectorOrderParameter(double *op_tensor) {
    if (n_dim_ < 1) {
        fprintf(stderr, "Dimension for calc_director_order_parameter must be > 0\n");
        exit(1);
    }
    
    /* Setup GSL workspaces to diagonalize order parameter tensor */
    gsl_matrix_view m = gsl_matrix_view_array(op_tensor, n_dim_, n_dim_);
    
    gsl_eigen_symmv(&m.matrix, eval_, evec_, w_);
    gsl_eigen_symmv_sort(eval_, evec_, GSL_EIGEN_SORT_VAL_DESC);
    
    /* Copy result into eigensystem structure */
    eigensystem director;
    for (int i = 0; i < n_dim_; ++i)
        director.vector[i] = evec_->data[i * n_dim_];
    director.value = gsl_vector_get(eval_, 0);

    /* Reorient director such that it lies in the upper hemisphere */
    if (director.vector[n_dim_ - 1] < 0) {
        for (int i = 0; i < n_dim_; ++i)
            director.vector[i] *= -1.0;
    }
    
    return director;
}

void DiffusionProperties::Update(double **r_obj,
                                 double **u_obj,
                                 int init_flag // 1 if at new time origin, 0 otherwise
                                 ) {
    if (init_flag) {
        for (int i_obj = 0; i_obj < n_objs_; ++i_obj) {
            theta_[i_obj] = 0.0;

            for (int i = 0; i < n_dim_; ++i) {
                u_old_[i_obj][i] = u_0_[i_obj][i] = u_obj[i_obj][i];
                r_0_[i_obj][i] = r_obj[i_obj][i];
                dr_[i_obj][i] = 0.0;
            }

        }
        curr_point_ = 0;
        
        return;
    }

    double ui_dot_u0 = 0.0;
    double theta2 = 0.0;
    for(int i_obj = 0; i_obj < n_objs_; ++i_obj) {
        for(int i = 0; i < n_dim_; ++i)
            dr_[i_obj][i] = r_obj[i_obj][i] - r_0_[i_obj][i];

        if (n_dim_ == 2) {
            double costheta = dot_product(n_dim_, u_obj[i_obj], u_old_[i_obj]); 
            if (costheta > 1.0)
                costheta = 1.0;
            double d_theta = acos(costheta);
            
            if (u_obj[i_obj][0]*u_old_[i_obj][1] - u_obj[i_obj][1] * u_old_[i_obj][0] < 0)
                d_theta = -d_theta;
            
            theta_[i_obj] += d_theta;
        }
        else if (n_dim_ == 3) {
            double norm[3];
            cross_product(u_0_[i_obj], u_old_[i_obj], norm, 3);
            double u_planar[3];
            double factor = dot_product(3, u_obj[i_obj], norm);
            double u_planar_norm_factor = 0.0;
            for (int i = 0; i < 3; ++i) {
                u_planar[i] = u_obj[i_obj][i] - factor * norm[i];
                u_planar_norm_factor += SQR(u_planar[i]);
            }
            u_planar_norm_factor = 1.0 / sqrt(u_planar_norm_factor);
            for (int i = 0; i < 3; ++i)
                u_planar[i] *= u_planar_norm_factor;
            double rot_norm[3];
            cross_product(u_old_[i_obj], u_planar, rot_norm, 3);
            double costheta = dot_product(3, u_planar, u_old_[i_obj]);

            if (costheta > 1.0) costheta = 1.0;
            else if (costheta < -1.0) costheta = -1.0;
            
            double d_theta = acos(dot_product(3, u_planar, u_old_[i_obj]));
            if (dot_product(3, rot_norm, norm) < 0)
                d_theta = - d_theta;

            theta_[i_obj] += d_theta;
        }

        theta2 += SQR(theta_[i_obj]);

        for (int i = 0; i < n_dim_; ++i)
            u_old_[i_obj][i] = u_obj[i_obj][i];

        ui_dot_u0 += dot_product(n_dim_, u_obj[i_obj], u_0_[i_obj]);
    }
    theta2 /= n_objs_;
    ui_dot_u0 /= n_objs_;
    
    double r_mol_avg2 = 0.0;
    double r_mol_par_avg2 = 0.0;
    double r_mol_perp_avg2 = 0.0;
    for (int i_obj = 0; i_obj < n_objs_; ++i_obj) {
        for (int i = 0; i < n_dim_; ++i) {
            r_mol_avg2 += SQR(dr_[i_obj][i]);
        }
        r_mol_par_avg2 += SQR(dot_product(n_dim_, dr_[i_obj], director_.vector));
    }
    r_mol_avg2 /= n_objs_;
    r_mol_par_avg2 /= n_objs_;
    r_mol_perp_avg2 = r_mol_avg2 - r_mol_par_avg2;

    r_mol_avg2_[curr_point_] += r_mol_avg2;
    r_mol_par_avg2_[curr_point_] += r_mol_par_avg2;
    r_mol_perp_avg2_[curr_point_] += r_mol_perp_avg2;
    ui_dot_u0_[curr_point_] += ui_dot_u0;
    theta2_[curr_point_] += theta2;
    n_measurements_[curr_point_] += 1;

    curr_point_ += 1;
}


void DiffusionProperties::OutputMSD(double delta_config, int n_configs,
                                    const char* filename) {
    FILE *f_diffusion = gfopen(filename, "w");
    fprintf(f_diffusion, "time r_mol_avg2 r_mol_avg2_err r_mol_par_avg2 " \
            "r_mol_par_avg2_err r_mol_perp_avg2 r_mol_perp_avg2_err theta2 theta2_err costheta\n");
    for (int i_config = 0; i_config < n_points_ - 1; ++i_config) {
        int n_meas = n_measurements_[i_config];
        double r_mol_avg2 = r_mol_avg2_[i_config] /= n_meas;
        double r_mol_par_avg2 = r_mol_par_avg2_[i_config] /= n_meas;
        double r_mol_perp_avg2 = r_mol_perp_avg2_[i_config] /= n_meas;
        double ui_dot_u0 = ui_dot_u0_[i_config] /= n_meas;
        double theta2 = theta2_[i_config] /= n_meas;

        int n = (i_config+1);
        int n_0 = n_meas;
        int n_0_p = n_configs - 1 - n;

        double err_factor = (double) n * ( 4 * n_0_p - n ) / n_objs_;
        err_factor = sqrt(err_factor) / n_0;
        
        fprintf(f_diffusion, "%g %g %g %g %g %g %g %g %g %g\n",
                (i_config + 1) * delta_config,
                r_mol_avg2, r_mol_avg2 /  n_dim_ * err_factor,
                r_mol_par_avg2, r_mol_par_avg2 * err_factor,
                r_mol_perp_avg2, r_mol_perp_avg2 /  (n_dim_ - 1) * err_factor,
                theta2, theta2 / (n_dim_ - 1) * err_factor,
                ui_dot_u0);
    }
    fclose(f_diffusion);
}
