#include "bob.h"
#include "kinetochore.h"
#include "xlink_entry.h"
#include "data_collection.h"
#include <iostream>
#include <iomanip>

//Function used to find index of the wall potential in the system_potential structure
//Update after adding any new all potentials



int find_wall_potential_index(system_parameters *parameters, system_potential *potential)
{
    int wall_comp = -1;
    for (int i_comp = 0; i_comp<potential->n_comp; i_comp++) {
    if (potential->pot_func[i_comp] == wca_sphero_wall_potential_bd ||
         potential->pot_func[i_comp] == soft_sphero_wall_potential_bd ||
         potential->pot_func[i_comp] == linear_sphero_wall_potential_bd ||
         potential->pot_func[i_comp] == quadratic_sphero_wall_potential_bd ||
         potential->pot_func[i_comp] == NM_sphero_wall_potential_bd ) {
        wall_comp = i_comp;

        //fprintf( stdout, "Wall comp address: %p\n", potential->pot_func[wall_comp]);
        //fprintf( stdout, "wca address: %p\n", &wca_sphero_wall_potential_bd);
        //fprintf( stdout, "soft address: %p\n", &soft_sphero_wall_potential_bd);
        //fprintf( stdout, "linear address: %p\n", &linear_sphero_wall_potential_bd);
        //fprintf( stdout, "quad address: %p\n", &quadratic_sphero_wall_potential_bd);
        //fprintf( stdout, "NM address: %p\n", &NM_sphero_wall_potential_bd);
        break;
    }
}

    if (wall_comp == -1) {
        fprintf(stderr, "Could not find specified potential function in system_potential");
        exit(1);
    }
    else
        return wall_comp;

}

// Do the same sort of thing to find the potential for the MT/Chromatid interaction
int find_kc_mt_potential_index(system_parameters *parameters, system_potential *potential) {
    int kc_mt_comp = -1;
    for (int icomp = 0; icomp < potential->n_comp; ++icomp) {
        if (potential->pot_func[icomp] == chromosome_mt_soft_gaussian_potential_allpairs) {
            kc_mt_comp = icomp;
            break;
        }
    }

    if (kc_mt_comp == -1) {
        std::cerr << "Could not find specified potential for KC MT Soft Gaussian\n";
        exit(1);
    }

    return kc_mt_comp;
}

//Returns the speed of polymerization of MTs. 
//See Doorn, G. S., Tănase, C., Mulder, B. M., & Dogterom, M. (2000). 
//On the stall force for growing microtubules. 
//European Biophysics Journal, 29(1), 2-6. doi:10.1007/s002490050245
//f = force exerted on the tip of the MT parallel to the direction the MT
double calc_polymerization_speed(double v_poly, double f_s, double f_par){
//double calc_polymerization_speed(system_parameters *parameters, double f){
    //static double v_0 = 0; //The growth velocity of a single protofilament
    //static double K = 0; //Critical concentration for MT polymerization
                         //Also the Dissociation constant of GTP tubulin
    double v = 0;
    double N = 13; //Number of protofiliments of MT
    double delta = .32; //length of filament subunits
    double sigma = delta / N;
    //double v_poly = parameters->v_poly;
    //double f_s = parameters->f_mt_stall;
    double K = exp(-1.0*sigma*f_s); 

    //K = exp(-1.0*sigma*f_s);
    //TODO make this a parameters value that is calculated
    double factor = N*(1-pow(K,N));
    for (int n=1; n< N; n++)
        factor += (1-K)*double(n)*(1-pow(K,n));
    double v_0 = v_poly/factor;

    v = N*(exp(-1.0*delta*f_par) - pow(K,N));
    for (int n=1; n<N; n++)
        v += (1-K)*double(n)*(exp(-1.0*sigma*n*f_par) - pow(K,n));

    v *= v_0;    
    return v;
}

void update_polymerization_states(system_parameters *parameters,
                                  system_properties *properties,
                                  system_potential *potential)
{
    //std::cout << "DI::update_polymerization_states\n";
    //int i_bond, n_bonds, wall_comp, kc_mt_comp;
    //double delta, p_factor, p_norm, roll;
    //double p_stg0, p_stp0, p_gts0, p_gtp0, p_ptg0, p_pts0;
    //double p_stg, p_stp, p_gts, p_gtp, p_ptg, p_pts;
    //double **u_bond, *bond_force, f_par; 
    //poly_state_enum *poly_state;
    //int *stab_state;
    //gsl_rng *r_rng;

    int n_bonds = properties->bonds.n_bonds;
    double** u_bond = properties->bonds.u_bond;
    gsl_rng* r_rng = properties->rng.r;
    poly_state_enum* poly_state = properties->bonds.poly_state;
    int* stab_state = properties->bonds.stab_state;
    double delta = parameters->delta;
    double p_stg0 = parameters->f_shrink_to_grow * delta;
    double p_stp0 = parameters->f_shrink_to_pause * delta;
    double p_gts0 = parameters->f_grow_to_shrink * delta;
    double p_gtp0 = parameters->f_grow_to_pause * delta;
    double p_ptg0 = parameters->f_pause_to_grow * delta;
    double p_pts0 = parameters->f_pause_to_shrink * delta;
    //double f_par = 0.0;
    int wall_comp = -1;
    
    if (parameters->force_induced_catastrophe_flag == 1 
             && parameters->wall_potential_flag != 0) {
        wall_comp = find_wall_potential_index(parameters, potential);
    }

    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        double roll = gsl_rng_uniform_pos(r_rng);
        double p_factor_r = 1.0;
        double p_factor_c = 1.0;

        bool force_depoly_kc = false;

        // Adjust DI rates if MT is in stable state
        if (stab_state[i_bond] != -1) {
            if (stab_state[i_bond] >= 0) {
                // Crosslink stabilization
                p_factor_r = properties->crosslinks.f_stabilize_fr_[stab_state[i_bond]];
                p_factor_c = properties->crosslinks.f_stabilize_fc_[stab_state[i_bond]];
            } else {
                // Chromosome stabilization
                //std::cout << "Chromosome stabilization!\n";
                //std::cout << "stab state: " << stab_state[i_bond] << std::endl;
                // Decode the stab state
                int rawn = -stab_state[i_bond];
                rawn = rawn / 100; // shift off the chromosome flag
                int ikc = rawn % 10000;
                rawn = rawn / 10000;
                int isite = rawn;
                //std::cout << "Bond[" << i_bond << "] decoded kc[" << ikc << "], isite[" << isite << "]\n";
                // Get the force information
                Kinetochore *kc = &(properties->chromosomes.kinetochores_[ikc]);
                double *kcforce_vec = kc->flink_[isite];
                // Only stabilize if the force is in the same direction as the microtubule direction
                double u_dot_kcf = dot_product(parameters->n_dim, kcforce_vec, u_bond[i_bond]);
                if (u_dot_kcf >= 0.0) {
                    //std::cout << "  KC force not in same direciton as microtubule!\n";
                } else {
                    double kcforce = sqrt(dot_product(parameters->n_dim, kcforce_vec, kcforce_vec));
                    //std::cout << "  KC Force: " << kcforce << std::endl;
                    //std::cout << "  stab fr: " << exp(kcforce / properties->chromosomes.af_f_stabilize_fr_) << std::endl;
                    //std::cout << "  stab fc: " << exp(kcforce / properties->chromosomes.af_f_stabilize_fc_) << std::endl;

                    // Adjust the stabilizations
                    p_factor_r = exp(kcforce / properties->chromosomes.af_f_stabilize_fr_);
                    p_factor_c = exp(kcforce / properties->chromosomes.af_f_stabilize_fc_);

                }

                // If there is Dam1 effects, multiply p_factor_c by the additional factor
                if (properties->chromosomes.dam1_effects_ == 1) {
                    p_factor_c = p_factor_c * properties->chromosomes.dam1_catastrophe_factor_;
                }

                // If we are doing anaphase, and the SAC has been silenced, then force depoly of this
                // MT if KC is tip attached
                if (properties->chromosomes.sac_status_ == 0) {
                    force_depoly_kc = true;
                }
            }
        } 

        double p_stg = p_stg0 * p_factor_r;
        double p_stp = p_stp0 * p_factor_r;
        double p_gts = p_gts0 * p_factor_c;
        double p_gtp = p_gtp0 * p_factor_c;
        double p_ptg = p_ptg0 * p_factor_r;
        double p_pts = p_pts0 * p_factor_c;
        
        // Adjust Catastrophe rates based on wall-mt potential
        if (wall_comp != -1){
            double *bond_force = properties->bonds.fbondtip_ne[i_bond];

            double f_par = -1.0*dot_product(parameters->n_dim, bond_force, u_bond[i_bond]);

            if (f_par > 0.0) {
                double p_factor = exp(0.0828*f_par); // Where does 0.0828 come from?
                p_gts = (p_gts+p_gtp)*p_factor;
                p_pts = p_pts*p_factor;
            }
        }
        // Also adjust based on any tip force from the kinetochores/chromatids
        if (properties->chromosomes.nkcs_ > 0 && parameters->force_induced_catastrophe_flag == 1) {
            double *tip_force = properties->chromosomes.fkcmttip_[i_bond];

            double f_par_kc = -1.0*dot_product(parameters->n_dim, tip_force, u_bond[i_bond]);

            if (f_par_kc > 0.0) {
                double p_factor_kc = exp(properties->chromosomes.chromatid_mt_fc_factor_*f_par_kc);
                p_gts = (p_gts+p_gtp)*p_factor_kc;
                p_pts = p_pts*p_factor_kc;

                //std::cout << "  Tip Force: ("
                //    << tip_force[0] << ", "
                //    << tip_force[1] << ", "
                //    << tip_force[2] << ")\n";
                //std::cout << std::setprecision(16);
                //std::cout << "  p_factor_kc: " << p_factor_kc << ", p_gts: " << p_gts << std::endl;

                //std::cout << "Step[" << properties->i_current_step << "]\n";
                //std::cout << "  bond[" << i_bond << "]\n";
                //std::cout << std::setprecision(16);
                //std::cout << "  p_factor_kc: " << p_factor_kc << std::endl;
                //exit(1);
            }
        }

        // MT Shrinking
        if (poly_state[i_bond] == SHRINK) {
            double p_norm = p_stg + p_stp;
            if (p_norm > 1.0) {
                if (roll < p_stg/p_norm) {
                    poly_state[i_bond] = GROW;
                }
                else {
                    poly_state[i_bond] = PAUSE;
                }
            }
            else {
                if (roll < p_stg) {
                    poly_state[i_bond] = GROW;
                }
                else if (roll >= p_stg && roll < (p_stg + p_stp)) {
                    poly_state[i_bond] = PAUSE;
                }
            }
        }
        
        // MT Growing
        else if (poly_state[i_bond] == GROW) {
            double p_norm = p_gts + p_gtp;
            if (p_norm > 1.0) {
                if (roll < p_gts/p_norm) {
                    poly_state[i_bond] = SHRINK;
                }
                else {
                    poly_state[i_bond] = PAUSE;
                }
            }
            else {
                if (roll < p_gts) {
                    poly_state[i_bond] = SHRINK;
                }
                if (roll >= p_gts && roll < (p_gts + p_gtp)) {
                    poly_state[i_bond] = PAUSE;
                }
            }
        }
        
        // MT Paused
        else if (poly_state[i_bond] == PAUSE) {
            double p_norm = p_ptg + p_pts;
            if (p_norm > 1.0) {
                if (roll < p_ptg/p_norm) {
                    poly_state[i_bond] = GROW;
                }
                else {
                    poly_state[i_bond] = SHRINK;
                }
            }
            else {
                if (roll < p_ptg) {
                    poly_state[i_bond] = GROW;
                }
                if (roll >= p_ptg && roll < (p_ptg + p_pts)) {
                    poly_state[i_bond] = SHRINK;
                }
            }
        }

        // Final check, if the SAC has been silenced, and we are doing anaphase
        // then force_depoly should trip and we can force the shrink state
        if (force_depoly_kc) {
            poly_state[i_bond] = SHRINK;
            //std::cout << "bond[" << i_bond << "] forcing depoly " << force_depoly_kc << " poly state[" << poly_state[i_bond] << "]\n";
        }
    }
}

void grow_bonds(system_parameters * parameters, system_properties * properties, system_potential *potential)
{
    int *site_2 = properties->bonds.bond_site_2;
    double** r = properties->sites.r;
    double** r_bond = properties->bonds.r_bond;
    double** u_bond = properties->bonds.u_bond;
    double** v_bond = properties->bonds.v_bond;
    double* length = properties->bonds.length;
    double* length2 = properties->bonds.length2;
    double delta = parameters->delta;
    int n_dim = parameters->n_dim;
    int n_bonds = properties->bonds.n_bonds;
    double** dr_tot = properties->sites.dr_tot;
    poly_state_enum* poly_state = properties->bonds.poly_state;
    int* stab_state = properties->bonds.stab_state;
    

    /* Loop over bonds. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        double delta_L_poly = parameters->v_poly * delta;
        double delta_L_depoly = -parameters->v_depoly * delta;
        double delta_L = 0;
        /* Get labels of sites in bond. */
        int j_site = site_2[i_bond];

        if (poly_state[i_bond] == GROW){
            if (parameters->wall_potential_flag != 0  &&
                    parameters->force_poly_flag == 1) {
                int wall_comp = find_wall_potential_index(parameters, potential);
                double* bond_force = properties->bonds.fbondtip_ne[i_bond];
                double f_par = -1.0*dot_product(n_dim, bond_force, u_bond[i_bond]);
                if (f_par > 0.0 ) {
                    delta_L_poly = delta * calc_polymerization_speed(parameters->v_poly,
                                                                     parameters->f_mt_stall, f_par);
                }
                delta_L = delta_L_poly;
            }
            else{
                delta_L = delta_L_poly;
            }
        }
        else if (poly_state[i_bond] == SHRINK)
            delta_L = delta_L_depoly;
        else if (poly_state[i_bond] == PAUSE)
            delta_L = 0.0;
        else
            delta_L = 0.0;

        if (stab_state[i_bond] != -1 ) {
            if (stab_state[i_bond] >= 0) {
                // Crosslink stabilization
                if (poly_state[i_bond] == GROW && delta_L_poly > 0) //Second condition added for force dependent polymerization reasons
                    delta_L = delta_L * properties->crosslinks.f_stabilize_vg_[stab_state[i_bond]];
                else if (poly_state[i_bond] == SHRINK)
                    delta_L = delta_L * properties->crosslinks.f_stabilize_vs_[stab_state[i_bond]];
            } else {
                // Chromosome stabilization
                //std::cout << "grow shrink stab state: " << stab_state[i_bond] << std::endl;
                // Decode the stab state
                int rawn = -stab_state[i_bond];
                rawn = rawn / 100; // shift off the chromosome flag
                int ikc = rawn % 10000;
                rawn = rawn / 10000;
                int isite = rawn;
                //std::cout << "Bond[" << i_bond << "] decoded kc[" << ikc << "], isite[" << isite << "]\n";
                // Get the force information
                Kinetochore *kc = &(properties->chromosomes.kinetochores_[ikc]);
                double *kcforce_vec = kc->flink_[isite];
                // Only stabilize if force is in the same direction as the microtubule direction
                double u_dot_kcf = dot_product(parameters->n_dim, kcforce_vec, u_bond[i_bond]);
                if (u_dot_kcf >= 0.0) {
                    //std::cout << "  KC force not in same direciton as microtubule!\n";
                } else {
                    double kcforce = sqrt(dot_product(parameters->n_dim, kcforce_vec, kcforce_vec));
                    double vgstab = 1.0;
                    double vsstab = 1.0;
                    if (properties->chromosomes.af_f_stabilize_vg_ != 0.0) {
                        vgstab = exp(kcforce / properties->chromosomes.af_f_stabilize_vg_);
                    }
                    if (properties->chromosomes.af_f_stabilize_vs_ != 0.0) {
                        vsstab = exp(kcforce / properties->chromosomes.af_f_stabilize_vs_);
                    }
                    if (poly_state[i_bond] == GROW && delta_L_poly > 0) {//Second condition added for force dependent polymerization reasons
                        delta_L = delta_L * vgstab;
                        //if (i_bond == 4) {
                        //    std::cout << "DI bond[" << i_bond << "] start delta_l " << delta_L << std::endl;
                        //}
                        delta_L = MIN(properties->chromosomes.af_mtpoly_dlmax_, delta_L);
                        //if (i_bond == 4) {
                        //    std::cout << "DI bond[" << i_bond << "] end delta_l " << delta_L << std::endl;
                        //}
                        //std::cout << "poly delta l cutoff: " << delta_L << std::endl;
                    }
                    else if (poly_state[i_bond] == SHRINK) {
                        delta_L = delta_L * vsstab;
                        //std::cout << "depoly delta l: " << delta_L << std::endl;

                        // If the depoly has been set and the SAC has been silenced, override thie behavior
                        if (properties->chromosomes.sac_status_ == 0) {
                            //std::cout << "Anaphase begun, bond " << i_bond << " delta L original: " << delta_L << std::endl;
                            delta_L = -properties->chromosomes.anaphase_depoly_ * delta * vsstab;
                            //std::cout << "delta_L new: " << delta_L << std::endl;
                        }
                        delta_L = -MIN(properties->chromosomes.af_mtpoly_dlmax_, fabs(delta_L));
                        //std::cout << "depoly delta l cutoff: " << delta_L << std::endl;
                    }
                }
            }
        }

        //Prevent mt from shrinking too much
        if ( (length[i_bond] + delta_L) < parameters->min_length ){
            delta_L = parameters->min_length - length[i_bond] ;
            length[i_bond] = parameters->min_length;
        }
        else
            length[i_bond] += delta_L;
        
        length2[i_bond] = SQR(length[i_bond]);

        /* grow rod if poly flag, shrink if not */
        for (int i = 0; i < n_dim; ++i) {
            double factor = delta_L * u_bond[i_bond][i];
            r[j_site][i] += factor;
            v_bond[i_bond][i] += factor;
            r_bond[i_bond][i] += 0.5 * factor;
            dr_tot[j_site][i] += factor;
        }
        //if (properties->i_current_step % parameters->n_graph == 0)
            //std::cout<< "(grow) bond "<<i_bond<<" length: " << length[i_bond] << std::endl;
    }
}

void stabilize_bonds(system_parameters * parameters, system_properties * properties)
{
    int *stab_state = properties->bonds.stab_state;

    int n_bonds = properties->bonds.n_bonds;
    double *length = properties->bonds.length;

    //-1 means that no stabilization is taking place
    std::fill(stab_state, stab_state + n_bonds, -1);

    xlink_list **stage_1_xlinks = properties->crosslinks.stage_1_xlinks_;
    xlink_list **stage_2_xlinks = properties->crosslinks.stage_2_xlinks_;
    double *f_stabilize = properties->crosslinks.f_stabilize_vg_;

    for (int i_type = 0; i_type < properties->crosslinks.n_types_; ++i_type) {
        double l_stabilize = properties->crosslinks.l_stabilize_[i_type];

        if (properties->crosslinks.l_stabilize_[i_type] > 0.0) {
            
            /* Loop over bonds. */
            for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
                for(xlink_list::iterator xlink = stage_2_xlinks[i_type][i_bond].begin();
                    xlink < stage_2_xlinks[i_type][i_bond].end();
                    xlink++) {
                    if(xlink->IsActive()) {
                        int bond_1 = xlink->head_parent_[0];
                        int bond_2 = xlink->head_parent_[1];
                        double d1_stabilize = length[bond_1] - xlink->cross_position_[0];
                        double d2_stabilize = length[bond_2] - xlink->cross_position_[1];

                        if (d1_stabilize < l_stabilize) {
                            //If not stabilized, stabilize
                            if (stab_state[bond_1] == -1) {
                                stab_state[bond_1] = i_type;
                            }
                            else {
                                //Choose the larger stabilizing factor
                                stab_state[bond_1] =
                                    (f_stabilize[stab_state[bond_1]] >= f_stabilize[i_type]) ?
                                        (i_type) : stab_state[bond_1];
                            }
                        }

                        if (d2_stabilize < l_stabilize){
                            //If not stabilized, stabilize
                            if (stab_state[bond_2] == -1) {
                                stab_state[bond_2] = i_type;
                            }
                            else {
                                //Choose the larger stabilizing factor
                                stab_state[bond_2] =
                                    (f_stabilize[stab_state[bond_2]] >= f_stabilize[i_type]) ?
                                        (i_type) : stab_state[bond_2];
                            }
                        }
                    }
                }

                //Not physical but possibly necessary for current wild type possibly
                if (parameters->xlink_stage1_stabilization_flag) {
                    if (properties->crosslinks.attachment_model_ == 2 && stab_state[i_bond] == -1) {
                        for(xlink_list::iterator xlink = stage_1_xlinks[i_type][i_bond].begin();
                            xlink < stage_1_xlinks[i_type][i_bond].end(); xlink++) {
                            if(xlink->IsActive()) {
                                double d_stabilize = length[i_bond] - xlink->cross_position_[0];
                                if (d_stabilize < l_stabilize) {
                                    if (stab_state[i_bond] == -1) {
                                        stab_state[i_bond] = i_type;
                                    }
                                    else {
                                        // We have one close enough to the tip to stabilize
                                        stab_state[i_bond] =
                                            (f_stabilize[stab_state[i_bond]] <= f_stabilize[i_type]) ?
                                            (i_type) : stab_state[i_bond];
                                    }
                                }
                            }
                        }
                    }
                }
            } // loop over bonds
        }
    } // loop over crosslinks

    // Check for chromosome behavior
    // FIXME inefficient at the moment, can just get bond from the kinetochore
    if (properties->chromosomes.enabled_) {
        // Chromosomes are turned on, they will always take precedence of stabilization
        double l_stab_af = properties->chromosomes.af_l_stabilize_;
        for (int ibond = 0; ibond < n_bonds; ++ibond) {
            // Loop over kinetochores
            for (int ikc = 0; ikc < properties->chromosomes.nkcs_; ++ikc) {
                // Loop over binding sites
                Kinetochore *kc_iter = &(properties->chromosomes.kinetochores_[ikc]);
                for (int isite = 0; isite < properties->chromosomes.naf_; ++isite) {
                    if (kc_iter->attach_[isite] != ibond) {
                        continue;
                    }

                    //std::cout << "Step: " << properties->i_current_step << ", KC[" << ikc << "] site[" << isite << "] attached bond[" << ibond << "]\n";

                    double dstab = length[ibond] - kc_iter->cross_pos_[isite];
                    if (dstab < l_stab_af) {
                        // No matter what, override the stabiliztion
                        // Encode the chromosome ikc and isite as binary information
                        //std::cout << "...and stabilized!\n";
                        int rawn = 10;
                        // Add the ikc information
                        rawn += ikc * 100;
                        // Add the isite information
                        rawn += isite * 1e6;
                        stab_state[ibond] = -rawn;
                        //std::cout << "  Encoded as " << -rawn << std::endl;
                    }
                }
            }
        }
    }
}

int dynamic_instability(system_parameters *parameters,
                        system_properties *properties,
                        system_potential *potential)
{
    //int n_bonds, i_bond;
    //double *length, min_length, max_length;

    /* Set up shortcuts to data structures. */
    double min_length = parameters->min_length;
    double max_length = parameters->max_length;
    double* length = properties->bonds.length;
    int n_bonds = properties->bonds.n_bonds;

    /* Let crosslinks stabilize filament growth */
    stabilize_bonds(parameters, properties);

    /* Update polymerization states */
    update_polymerization_states(parameters, properties, potential);

    /* Loop over bonds. */
    /* check for bonds that exceed max and min length requirements and deal with accordingly */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        /* If the bonds are outside the min/max, set them appropriately */
        if (length[i_bond] <= min_length) {
            properties->bonds.poly_state[i_bond] = GROW;
        }
        else if (length[i_bond] > max_length) {
            properties->bonds.poly_state[i_bond] = PAUSE;
        }

        /* If any bond has length < 0, flag and exit program */
        if(length[i_bond] <= 0.0) {
            fprintf(stderr, "dynamic_instability length less than zero\nbond: %d length: %g\n", i_bond, length[i_bond]);
            fprintf(stderr, "Step: %d\n", properties->i_current_step);
            exit(1);
        }
    }

    /* grow and shrink existing bonds */
    grow_bonds(parameters, properties, potential);

    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        double p = (length[i_bond] + 1.0);
        double inv_l = 1.0 / p;
        double inv_l2 = SQR(inv_l);
        double log_l = log(p);
        double gamma_0 = p / 3;
        
        properties->bonds.gamma_perp[i_bond] =
            4.0 * gamma_0 / (log_l + 0.839 + 0.185 * inv_l + 0.233 * inv_l2);
        properties->bonds.gamma_par[i_bond] =
            2.0 * gamma_0 / (log_l - 0.207 + 0.98 * inv_l - 0.133 * inv_l2);
        properties->bonds.gamma_rot[i_bond] =
            SQR(p) * gamma_0 / 3.0 / (log_l - 0.662 + 0.917 * inv_l - 0.050 * inv_l2);
        
        properties->bonds.diff_perp[i_bond] =
            parameters->temp / properties->bonds.gamma_perp[i_bond];
        properties->bonds.diff_par[i_bond] =
            parameters->temp / properties->bonds.gamma_par[i_bond];
        properties->bonds.diff_rot[i_bond] =
            parameters->temp / properties->bonds.gamma_rot[i_bond];

    }

    return 0;
}
