/* This routine evaluates the total force on each site, the potential 
   energy, and the virial by summing contributions from individual terms in the
   potential energy expression. The only difference between this routine and
   the pure evaluate forces routine is that it allows for the implementation of 
   torques in a BD sense.

   Input: pointer to parameters structure (parameters)
          pointer to properties structure (properties)
          pointer to potential energy structure (potential)
          pointer to 

   Output: the force array and the system potential energy and virial
           are modified on output */

#include "bob.h"
#include <iostream>

void evaluate_forces_bd(system_parameters *parameters,
                        system_properties *properties, 
                        system_potential *potential) {
    
    int n_sites, n_bonds, n_dim, n_comp, *calc_matrix;
    int n_kcs;
    double u, **f, **f_bond, **virial, **t_bond, *u_comp, ***f_comp, ***virial_comp, ***t_comp;
    double **f_kc, **t_kc, **ftip_kc;
    double (**pot_func) (system_parameters *parameters,
                         system_properties *properties,
                         double **,
                         double **,
                         double **,
                         int *);

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_bonds = properties->bonds.n_bonds;
    n_sites = properties->sites.n_sites;
    f = properties->sites.f;
    f_bond = properties->bonds.f_bond;
    virial = properties->thermo.virial;
    n_comp = potential->n_comp;
    pot_func = potential->pot_func;
    f_comp = potential->f_comp;
    u_comp = potential->u_comp;
    virial_comp = potential->virial_comp;
    calc_matrix = potential->calc_matrix;
    t_bond = properties->bonds.t_bond;
    t_comp = potential->t_comp;
    n_kcs = properties->chromosomes.nkcs_;
    f_kc = properties->chromosomes.f_;
    t_kc = properties->chromosomes.t_;
    ftip_kc = properties->chromosomes.fkcmttip_;

    /* Zero total potential energy, forces, integrated torques, matrix inversion flag and virial. */
    u = 0.0;
    for (int i_site = n_bonds * 2; i_site < n_sites; ++i_site) {
        for (int i = 0; i < n_dim; ++i)
            f[i_site][i] = 0.0;
    }
    for (int i = 0; i < n_dim; ++i)
        for (int j = 0; j < n_dim; ++j)
            virial[i][j] = 0.0;

    memset(f_bond[0], 0, n_bonds*n_dim*sizeof(double));
    memset(t_bond[0], 0, n_bonds*3*sizeof(double));
    memset(calc_matrix, 0, n_bonds * sizeof(int));

    // Make sure that we don't have extra forces/torques on the kcs
    if (n_kcs > 0) {
        memset(f_kc[0], 0, n_kcs*n_dim*sizeof(double));
        memset(t_kc[0], 0, n_kcs*3*sizeof(double));
        memset(ftip_kc[0], 0, n_bonds * n_dim * sizeof(double));
    }

    /* Loop over contributions to potential energy/force/virial. */
    for (int i_comp = 0; i_comp < n_comp; ++i_comp) {
        /* Evaluate potential energy/force/virial contribution. */
        u_comp[i_comp] = pot_func[i_comp] (parameters, properties,
                                           f_comp[i_comp], 
                                           virial_comp[i_comp],
                                           t_comp[i_comp],
                                           calc_matrix);

        /* Add potential energy, force, and virial contributions to accumulators. */
        u += u_comp[i_comp];

        //// FIXME XXX
        //// Check each potential for the pathological forces
        //for (int ibond = 0; ibond < n_bonds; ++ibond) {
        //    if (dot_product(n_dim, f_comp[i_comp][ibond], f_comp[i_comp][ibond]) > 1.0e9) {
        //        std::cerr << "ERROR: fcomp going cra cray\n";
        //        std::cout << "  icomp: " << i_comp << ", bond: " << ibond << std::endl;
        //        if (pot_func[i_comp] == af_mt_harmonic_potential) {
        //            std::cout << "  AF MT harmonic potential\n";
        //        }
        //        if (pot_func[i_comp] == kinetochore_mt_potential_allpairs) {
        //            std::cout << "  Kinetochore MT tip exclusion potential\n";
        //        }
        //        if (pot_func[i_comp] == wca_sphero_wall_potential_bd) {
        //            std::cout << "  MT wall WCA potential\n";
        //        }
        //        std::cout << "  fcomp: (" << f_comp[i_comp][ibond][0] << ", " << f_comp[i_comp][ibond][1] << ", " << f_comp[i_comp][ibond][2] << ")\n";
        //        std::cout << "  length: " << properties->bonds.length[ibond] << std::endl;
        //    }
        //}
        //// FIXME XXX

        // Multithread the summation from the different potentials
        #ifdef ENABLE_OPENMP
        #pragma omp parallel for schedule(runtime)
        #endif
        for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
            for (int i = 0; i < n_dim; ++i) {
                f_bond[i_bond][i] += f_comp[i_comp][i_bond][i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += t_comp[i_comp][i_bond][i];
            }
        }

        /* FIXME: hack */
        for (int i_site = n_bonds*2; i_site < n_sites; ++i_site) {
            int i_obj = i_site - n_bonds;
            for (int i = 0; i < n_dim; ++i)
                f[i_site][i] += f_comp[i_comp][i_obj][i];
        }
        if (properties->control.virial_flag)
            for (int i = 0; i < n_dim; ++i)
                for (int j = 0; j < n_dim; ++j)
                    virial[i][j] += virial_comp[i_comp][i][j];

        /* Check for nan in the force generation */
        if(u_comp[i_comp] != u_comp[i_comp]) {
            fprintf(stderr,"NaN encountered in evaluate_forces_bd\n");
            fprintf(stderr,"Step: %d\n", properties->i_current_step);
            fprintf(stderr,"u: %g potential #: %d\n", u_comp[i_comp], i_comp);
            exit(1);
        }
    }
    /* Save potential energy */
    properties->thermo.u = u;

    return;
}
